/*

 gui-textwidget.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "../ui-common/txt.h"

static gchar *gui_window_line2text(LINE_REC *line)
{
    GString *str;
    gint color;
    gchar *ret, *ptr;

    g_return_val_if_fail(line != NULL, NULL);

    str = g_string_new(NULL);

    color = 0;
    for (ptr = line->text; ; ptr++)
    {
	if (*ptr != 0)
	{
	    g_string_append_c(str, *ptr);
	    continue;
	}

	ptr++;
	if ((*ptr & 0x80) == 0)
	{
	    /* set color */
	    color = *ptr;
	    g_string_sprintfa(str, "\003%c%c", (color & 0x07)+1, ((color & 0xf0) >> 4)+1);
	    if (color & 0x08) g_string_sprintfa(str, "\002");
	}
	else switch ((guchar) *ptr)
	{
	    case LINE_CMD_EOL:
		ret = str->str;
		g_string_free(str, FALSE);
		return ret;
	    case LINE_CMD_CONTINUE:
		ptr = *(gchar **) (ptr+1)-1;
		break;
	    case LINE_CMD_UNDERLINE:
		g_string_append_c(str, 31);
		break;
	    case LINE_CMD_COLOR8:
		g_string_sprintfa(str, "\003%c%c", 9, ((color & 0xf0) >> 4)+1);
		color &= 0xfff0;
		color |= 8|ATTR_COLOR8;
		break;
	    case LINE_CMD_BEEP:
		break;
	    case LINE_CMD_INDENT:
		break;
	}
    }

    return NULL;
}

static gboolean cmd_last(gchar *data)
{
    GSList *list, *line;
    GList *startline;
    gchar *params, *str, *args, *text;
    gchar **tmp, **arglist;
    gboolean only_new;
    struct tm *tm;
    gint level;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST | PARAM_FLAG_OPTARGS, &args, &text);
    if (*text == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    /* level can be specified in arguments.. */
    level = 0; only_new = FALSE;
    arglist = g_strsplit(args, " ", -1);
    for (tmp = arglist; *tmp != NULL; tmp++)
    {
	if (g_strcasecmp(*tmp, "-new") == 0)
	    only_new = TRUE;
	else
	    level |= level2bits(tmp[1]);
    }
    if (level == 0) level = MSGLEVEL_ALL;
    g_strfreev(arglist);

    startline = NULL;
    if (only_new)
    {
	GList *tmp;

	for (tmp = WINDOW_GUI(CHANNEL_PARENT(cur_channel))->lines; tmp != NULL; tmp = tmp->next)
	{
	    LINE_REC *rec = tmp->data;

	    if (rec->level & MSGLEVEL_LASTLOG)
		startline = tmp;
	}
    }
    list = gui_window_find_text(CHANNEL_PARENT(cur_channel), text, startline);

    printformat(NULL, NULL, MSGLEVEL_LASTLOG, IRCTXT_LASTLOG_START);
    for (line = list; line != NULL; line = line->next)
    {
        LINE_REC *rec = line->data;

        if ((rec->level & level) != 0 && (rec->level & MSGLEVEL_LASTLOG) == 0)
        {
	    text = gui_window_line2text(rec);
	    if (setup_get_bool("toggle_show_timestamps"))
		printtext(NULL, NULL, MSGLEVEL_LASTLOG, text);
            else
            {
                tm = localtime(&rec->time);

                str = g_strdup_printf("[%02d:%02d] %s", tm->tm_hour, tm->tm_min, text);
                printtext(NULL, NULL, MSGLEVEL_LASTLOG, str);
                g_free(str);
	    }
	    g_free(text);
        }
    }

    printformat(NULL, NULL, MSGLEVEL_LASTLOG, IRCTXT_LASTLOG_END);

    g_slist_free(list);
    g_free(params);
    return TRUE;
}

static gboolean cmd_scrollback(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    command_runsub("scrollback", data, server, channel);
    return TRUE;
}

static gboolean cmd_scrollback_clear(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    gui_window_clear(CHANNEL_PARENT(channel));
    return TRUE;
}

static void scrollback_goto_pos(WINDOW_REC *window, GList *pos)
{
    GUI_WINDOW_REC *gui;

    g_return_if_fail(window != NULL);
    g_return_if_fail(pos != NULL);

    gui = WINDOW_GUI(window);

    if (g_list_find(gui->bottom_startline, pos->data) == NULL)
    {
	gui->startline = pos;
	gui->subline = 0;
	gui->bottom = FALSE;
    }
    else
    {
	/* reached the last line */
	if (gui->bottom) return;

	gui->bottom = TRUE;
	gui->startline = gui->bottom_startline;
	gui->subline = gui->bottom_subline;
	gui->ypos = last_text_line-first_text_line-1;
    }

    if (is_window_visible(window))
	gui_window_redraw(window);
    signal_emit("gui page scrolled", 1, window);
}

static gboolean cmd_scrollback_goto(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    WINDOW_REC *window;
    GList *pos;
    gchar *params, *arg1, *arg2;
    gint lines;

    window = CHANNEL_PARENT(channel);

    params = cmd_get_params(data, 2, &arg1, &arg2);
    if (*arg2 == '\0' && (*arg1 == '-' || *arg1 == '+'))
    {
	/* go forward/backward n lines */
	if (sscanf(arg1 + (*arg1 == '-' ? 0 : 1), "%d", &lines) == 1)
	    gui_window_scroll(window, lines);
    }
    else if (*arg2 == '\0' && strchr(arg1, ':') == NULL && strchr(arg1, '.') == NULL &&
	     sscanf(arg1, "%d", &lines) == 1)
    {
        /* go to n'th line. */
	pos = g_list_nth(WINDOW_GUI(window)->lines, lines);
	if (pos != NULL)
            scrollback_goto_pos(window, pos);
    }
    else
    {
	struct tm tm;
	time_t stamp;
	gint day, month;

	/* [dd.mm | -<days ago>] hh:mi[:ss] */
	stamp = time(NULL);
	if (*arg1 == '-')
	{
	    /* -<days ago> */
	    if (sscanf(arg1+1, "%d", &day) == 1)
		stamp -= day*3600*24;
	    memcpy(&tm, localtime(&stamp), sizeof(struct tm));
	}
	else if (*arg2 != '\0')
	{
	    /* dd.mm */
	    if (sscanf(arg1, "%d.%d", &day, &month) == 2)
	    {
		month--;
		memcpy(&tm, localtime(&stamp), sizeof(struct tm));

		if (tm.tm_mon < month)
		    tm.tm_year--;
		tm.tm_mon = month;
		tm.tm_mday = day;
		stamp = mktime(&tm);
	    }
	}
	else
	{
            /* move time argument to arg2 */
	    arg2 = arg1;
	}

	/* hh:mi[:ss] */
	memcpy(&tm, localtime(&stamp), sizeof(struct tm));
        tm.tm_sec = 0;
	sscanf(arg2, "%d:%d:%d", &tm.tm_hour, &tm.tm_min, &tm.tm_sec);
	stamp = mktime(&tm);

	/* find the first line after timestamp */
	for (pos = WINDOW_GUI(window)->lines; pos != NULL; pos = pos->next)
	{
	    LINE_REC *rec = pos->data;

	    if (rec->time >= stamp)
	    {
		scrollback_goto_pos(window, pos);
		break;
	    }
	}
    }
    g_free(params);
    return TRUE;
}

static gboolean cmd_scrollback_home(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    GUI_WINDOW_REC *gui;
    WINDOW_REC *window;

    window = CHANNEL_PARENT(channel);
    gui = WINDOW_GUI(window);

    if (gui->bottom_startline == gui->startline)
	return TRUE;

    gui->bottom = FALSE;
    gui->startline = gui->lines;
    gui->subline = 0;

    if (is_window_visible(window))
	gui_window_redraw(window);
    signal_emit("gui page scrolled", 1, window);
    return TRUE;
}

static gboolean cmd_scrollback_end(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    GUI_WINDOW_REC *gui;
    WINDOW_REC *window;

    window = CHANNEL_PARENT(channel);
    gui = WINDOW_GUI(window);

    gui->bottom = TRUE;
    gui->startline = gui->bottom_startline;
    gui->subline = gui->bottom_subline;
    gui->ypos = last_text_line-first_text_line-1;

    if (is_window_visible(window))
	gui_window_redraw(window);
    signal_emit("gui page scrolled", 1, window);
    return TRUE;
}

void gui_textwidget_init(void)
{
    command_bind("last", NULL, (SIGNAL_FUNC) cmd_last);
    command_bind("scrollback", NULL, (SIGNAL_FUNC) cmd_scrollback);
    command_bind("scrollback clear", NULL, (SIGNAL_FUNC) cmd_scrollback_clear);
    command_bind("scrollback goto", NULL, (SIGNAL_FUNC) cmd_scrollback_goto);
    command_bind("scrollback home", NULL, (SIGNAL_FUNC) cmd_scrollback_home);
    command_bind("scrollback end", NULL, (SIGNAL_FUNC) cmd_scrollback_end);
}

void gui_textwidget_deinit(void)
{
    command_unbind("last", (SIGNAL_FUNC) cmd_last);
    command_unbind("scrollback", (SIGNAL_FUNC) cmd_scrollback);
    command_unbind("scrollback clear", (SIGNAL_FUNC) cmd_scrollback_clear);
    command_unbind("scrollback goto", (SIGNAL_FUNC) cmd_scrollback_goto);
    command_unbind("scrollback home", (SIGNAL_FUNC) cmd_scrollback_home);
    command_unbind("scrollback end", (SIGNAL_FUNC) cmd_scrollback_end);
}
