/*
 memdebug.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>

#define ENABLE_BUFFER_CHECKS
#define BUFFER_CHECK_SIZE 5
#define MIN_BUFFER_CHECK_SIZE 2

typedef struct
{
    gpointer p;
    gint size;
    gchar *file;
    gint line;
    gchar *comment;
}
MEM_REC;

static GHashTable *data = NULL, *preallocs = NULL;
static gchar *comment = "";

static void add_flow_checks(guchar *p, gulong size)
{
#ifdef ENABLE_BUFFER_CHECKS
    gint n;

    for (n = 0; n < BUFFER_CHECK_SIZE; n++)
        p[n] = n ^ 0x7f;
    for (n = 0; n < BUFFER_CHECK_SIZE; n++)
        p[size-BUFFER_CHECK_SIZE+n] = n ^ 0x7f;
#endif
}

void ig_memcheck_rec(gpointer key, MEM_REC *rec)
{
    guchar *p;
    gint n;

    if (rec->size != INT_MIN)
    {
	p = rec->p;

	for (n = 0; n < MIN_BUFFER_CHECK_SIZE; n++)
	    if (p[n] != (n ^ 0x7f))
		g_error("buffer underflow, file %s line %d!\n", rec->file, rec->line);

	for (n = 0; n < MIN_BUFFER_CHECK_SIZE; n++)
	    if (p[rec->size-BUFFER_CHECK_SIZE+n] != (n ^ 0x7f))
		g_error("buffer overflow, file %s line %d!\n", rec->file, rec->line);
    }
}

static void mem_check(void)
{
#ifdef ENABLE_BUFFER_CHECKS
    g_hash_table_foreach(data, (GHFunc) ig_memcheck_rec, NULL);
#endif
}

static void data_add(gpointer p, gint size, gchar *file, gint line)
{
    MEM_REC *rec;

    if (size <= 0 && size != INT_MIN)
	g_error("size = %d, file %s line %d", size, file, line);
    
    if (data == NULL)
    {
        data = g_hash_table_new((GHashFunc) g_direct_hash, (GCompareFunc) g_direct_equal);
        preallocs = g_hash_table_new((GHashFunc) g_direct_hash, (GCompareFunc) g_direct_equal);
    }

    if (g_hash_table_lookup(data, p) != NULL)
        g_error("data_add() already malloc()'ed %p (in %s:%d)", p, file, line);

    rec = g_new(MEM_REC, 1);
    g_hash_table_insert(data, p, rec);

    rec->p = p;
    rec->size = size;
    rec->file = file;
    rec->line = line;
    rec->comment = g_strdup(comment);

    if (size == INT_MIN)
	g_hash_table_insert(preallocs, p-BUFFER_CHECK_SIZE, p);
    else
	add_flow_checks(p, size);
    mem_check();
}

static gpointer data_remove(gpointer p, gchar *file, gint line)
{
    MEM_REC *rec;

    mem_check();

    if (g_hash_table_lookup(preallocs, p) != NULL)
    {
	g_hash_table_remove(preallocs, p);
	p += BUFFER_CHECK_SIZE;
    }

    rec = g_hash_table_lookup(data, p);
    if (rec == NULL)
    {
	g_warning("data_remove() data %p not found (in %s:%d)", p, file, line);
	return p+BUFFER_CHECK_SIZE;
    }

    g_hash_table_remove(data, p);
    g_free(rec->comment);
    g_free(rec);

    return p;
}

gpointer ig_malloc(gint size, gchar *file, gint line)
{
#if 1
    gpointer p;

    size += BUFFER_CHECK_SIZE*2;
    p = g_malloc(size);
    data_add(p, size, file, line);
    return p+BUFFER_CHECK_SIZE;
#else
    return g_malloc(size);
#endif
}

gpointer ig_malloc0(gint size, gchar *file, gint line)
{
#if 1
    gpointer p;

    size += BUFFER_CHECK_SIZE*2;
    p = g_malloc0(size);
    data_add(p, size, file, line);
    return p+BUFFER_CHECK_SIZE;
#else
    return g_malloc0(size);
#endif
}

gpointer ig_realloc(gpointer mem, gulong size, gchar *file, gint line)
{
#if 1
    gpointer p;

    size += BUFFER_CHECK_SIZE*2;
    mem -= BUFFER_CHECK_SIZE;
    data_remove(mem, file, line);
    p = g_realloc(mem, size);
    data_add(p, size, file, line);
    return p+BUFFER_CHECK_SIZE;
#else
    return g_realloc(mem, size);
#endif
}

gchar *ig_strdup(const char *str, gchar *file, gint line)
{
    gpointer p;

    p = ig_malloc(strlen(str)+1, file, line);
    strcpy(p, str);

    return p;
}

gchar *ig_strconcat(const char *str, ...)
{
  guint	  l;
  va_list args;
  gchar	  *s;
  gchar	  *concat;

  g_return_val_if_fail (str != NULL, NULL);

  l = 1 + strlen (str);
  va_start (args, str);
  s = va_arg (args, gchar*);
  while (s)
    {
      l += strlen (s);
      s = va_arg (args, gchar*);
    }
  va_end (args);

  concat = ig_malloc(l, "ig_strconcat", 0);
  concat[0] = 0;

  strcat (concat, str);
  va_start (args, str);
  s = va_arg (args, gchar*);
  while (s)
    {
      strcat (concat, s);
      s = va_arg (args, gchar*);
    }
  va_end (args);

  return concat;
}

gchar *ig_strdup_printf(const gchar *format, ...)
{
    gchar *buffer, *p;
    va_list args;

    va_start (args, format);
    buffer = g_strdup_vprintf (format, args);
    va_end (args);

    p = ig_malloc(strlen(buffer)+1, "ig_strdup_printf", 0);
    strcpy(p, buffer);
    return p;
}

gchar *ig_strdup_vprintf(const gchar *format, va_list args)
{
    gchar *buffer, *p;

    buffer = g_strdup_vprintf (format, args);

    p = ig_malloc(strlen(buffer)+1, "ig_strdup_vprintf", 0);
    strcpy(p, buffer);
    return p;
}

void ig_free(gpointer p)
{
#if 1
    p -= BUFFER_CHECK_SIZE;
    p = data_remove(p, "??", 0);
    if (p != NULL)
#endif
	g_free(p);
}

GString *ig_string_new(gchar *str)
{
    GString *ret;

    ret = g_string_new(str);
    data_add(ret, INT_MIN, "g_string_new", 0);
    return ret;
}

void ig_string_free(GString *str, gboolean freeit)
{
    data_remove(str, "g_string_free", 0);
    if (!freeit)
	data_add(str->str, INT_MIN, "ig_string_free (free = FALSE)", 0);
    g_string_free(str, freeit);
}

void ig_profile_line(gpointer key, MEM_REC *rec)
{
    printf("%p in %s:%d (%s)\n", rec->p, rec->file, rec->line, rec->comment);
}

void ig_mem_profile(void)
{
    g_hash_table_foreach(data, (GHFunc) ig_profile_line, NULL);
    g_hash_table_destroy(data);
    g_hash_table_destroy(preallocs);
}

void ig_set_data(gchar *data)
{
    comment = data;
}
