/*
 rawlog.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static gint rawlog_lines;

/* NOTE! str must be dynamically allocated and must not be freed after! */
static void rawlog_add(SERVER_REC *server, gchar *str)
{
    if (server->rawlog_lines < rawlog_lines || rawlog_lines <= 2)
	server->rawlog_lines++;
    else
    {
	g_free(server->rawlog->data);
	server->rawlog = g_list_remove(server->rawlog, server->rawlog->data);
    }

    server->rawlog = g_list_append(server->rawlog, str);
    signal_emit("rawlog", 2, server, str);
}

void rawlog_input(SERVER_REC *server, gchar *str)
{
    g_return_if_fail(server != NULL);
    g_return_if_fail(str != NULL);

    rawlog_add(server, g_strdup_printf(">> %s", str));
}

void rawlog_output(SERVER_REC *server, gchar *str)
{
    g_return_if_fail(server != NULL);
    g_return_if_fail(str != NULL);

    rawlog_add(server, g_strdup_printf("<< %s", str));
}

void rawlog_redirect(SERVER_REC *server, gchar *str)
{
    g_return_if_fail(server != NULL);
    g_return_if_fail(str != NULL);

    rawlog_add(server, g_strdup_printf("--> %s", str));
}

static gboolean sig_disconnected(SERVER_REC *server)
{
    g_return_val_if_fail(server != NULL, FALSE);

    g_list_foreach(server->rawlog, (GFunc) g_free, NULL);
    g_list_free(server->rawlog);

    return TRUE;
}

static gboolean cmd_rawlog(gchar *data, SERVER_REC *server)
{
    GList *tmp;
    gchar *path;
    gint f;

    g_return_val_if_fail(data != NULL, FALSE);
    if (server == NULL) cmd_return_error(CMDERR_NOT_CONNECTED);

    path = convert_home(data);
    f = open(path, O_WRONLY | O_APPEND | O_CREAT, LOG_FILE_CREATE_MODE);
    if (f == -1)
    {
	g_free(path);
        return TRUE;
    }
    g_free(path);

    for (tmp = server->rawlog; tmp != NULL; tmp = tmp->next)
    {
	write(f, tmp->data, strlen((gchar *) tmp->data));
	write(f, "\n", 1);
    }
    close(f);

    return TRUE;
}

static gboolean sig_setup(void)
{
    rawlog_lines = setup_get_int("rawlog_lines");
    return TRUE;
}

void rawlog_init(void)
{
    rawlog_lines = 100;

    signal_add("server disconnected", (SIGNAL_FUNC) sig_disconnected);
    signal_add("startup settings read", (SIGNAL_FUNC) sig_setup);
    signal_add("setup changed", (SIGNAL_FUNC) sig_setup);
    command_bind("rawlog", NULL, (SIGNAL_FUNC) cmd_rawlog);
}

void rawlog_deinit(void)
{
    signal_remove("server disconnected", (SIGNAL_FUNC) sig_disconnected);
    signal_remove("startup settings read", (SIGNAL_FUNC) sig_setup);
    signal_remove("setup changed", (SIGNAL_FUNC) sig_setup);
    command_unbind("rawlog", (SIGNAL_FUNC) cmd_rawlog);
}
