/*
 plugins-setup.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

GList *availplugins;

PLUGIN_SETUP_REC *plugin_setup_add(gchar *path, gchar *name, gchar *desc)
{
    proplist_t pprop, iprop;
    PLUGIN_SETUP_REC *rec;

    g_return_val_if_fail(path != NULL, NULL);
    g_return_val_if_fail(name != NULL, NULL);
    g_return_val_if_fail(desc != NULL, NULL);

    rec = g_new0(PLUGIN_SETUP_REC, 1);
    availplugins = g_list_append(availplugins, rec);
    rec->path = g_strdup(path);
    rec->name = g_strdup(name);
    rec->description = g_strdup(desc);

    /* get plugin settings from config file */
    pprop = config_get_prop(cprop, "plugins");
    iprop = config_get_prop(pprop, name);

    rec->autorun = config_get_bool(iprop, "autorun", FALSE);
    rec->args = config_get_str(iprop, "args", NULL);
    if (rec->args != NULL) rec->args = g_strdup(rec->args);

    return rec;
}

void plugin_setup_remove(PLUGIN_SETUP_REC *rec)
{
    g_return_if_fail(rec != NULL);

    g_free(rec->path);
    g_free(rec->name);
    g_free(rec->description);
    if (rec->args != NULL) g_free(rec->args);
    g_free(rec);

    availplugins = g_list_remove(availplugins, rec);
}

PLUGIN_SETUP_REC *plugin_setup_find(gchar *name)
{
    GList *tmp;

    for (tmp = g_list_first(availplugins); tmp != NULL; tmp = tmp->next)
    {
	PLUGIN_SETUP_REC *rec = tmp->data;

	if (strcmp(rec->name, name) == 0)
	    return rec;
    }

    return NULL;
}

static void scan_plugins(gchar *directory)
{
    PLUGIN_SETUP_REC *rec;
    DIR *dir;
    struct dirent *e;

    g_return_if_fail(directory != NULL);

    dir = opendir(directory);
    if (dir == NULL) return;

    while ((e = readdir(dir)) != NULL)
    {
        if (strncmp(e->d_name, "lib", 3) == 0 &&
            strcmp(e->d_name+strlen(e->d_name)-3, ".so") == 0)
        {
            /* libplugin.so -> plugin */
            gchar *path, *name, *desc;

            name = g_malloc(strlen(e->d_name)-5);
            name[strlen(e->d_name)-6] = '\0';
            memcpy(name, e->d_name+3, strlen(e->d_name)-6);

	    rec = plugin_setup_find(name);
	    if (rec == NULL)
	    {
		desc = plugin_get_description(name);
		if (desc != NULL)
		{
		    path = g_strconcat(directory, "/", e->d_name, NULL);
		    plugin_setup_add(path, name, desc);
		    g_free(desc);
		    g_free(path);
		}
	    }
            g_free(name);
        }
    }

    closedir(dir);
}

static void plugins_setup_autorun(void)
{
    GList *tmp;

    for (tmp = g_list_first(availplugins); tmp != NULL; tmp = tmp->next)
    {
        PLUGIN_SETUP_REC *rec = tmp->data;

        if (rec->autorun)
            plugin_load(rec->path, rec->args);
    }
}

void plugins_setup_scandirs(void)
{
    gchar *plugindir;

    /* first remove all the existing plugins.. */
    while (availplugins != NULL)
        plugin_setup_remove(availplugins->data);

    /* scan for all available plugins, first from home dir */
    plugindir = g_strconcat(g_get_home_dir(), "/.irssi/plugins", NULL);
    scan_plugins(plugindir);
    g_free(plugindir);

    /* then the global dir, ignore plugins that are already
       found from home dir */
    scan_plugins(PLUGINSDIR);
}

static gboolean sig_init_finished(void)
{
    plugins_setup_scandirs();
    plugins_setup_autorun();
    return TRUE;
}

void plugins_setup_init(void)
{
    availplugins = NULL;

    signal_add("irssi init finished", (SIGNAL_FUNC) sig_init_finished);
}

void plugins_setup_deinit(void)
{
    while (availplugins != NULL)
        plugin_setup_remove(availplugins->data);

    signal_remove("irssi init finished", (SIGNAL_FUNC) sig_init_finished);

}
