/*
 setup-servers.c : Irssi settings

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

GList *setupservers; /* list of irc servers */

/* server */
static gint server_reconnect_time; /* reconnect to server no sooner than n seconds */
static gchar *source_host; /* Our own IP to use */
static gboolean toggle_skip_motd;

gboolean source_host_ok; /* Use source_host_ip .. */
IPADDR source_host_ip; /* Resolved address */

static gchar *default_nick, *alternate_nick, *user_name, *real_name;

SETUP_REC server_options[] =
{
    { SETUP_TYPE_SPIN, &server_reconnect_time, "server_reconnect_time", GINT_TO_POINTER(300) },
    { SETUP_TYPE_ENTRY, &source_host, "source_host", "" },
    { SETUP_TYPE_TOGGLEBUTTON, &toggle_skip_motd, "toggle_skip_motd", GINT_TO_POINTER(FALSE) },

    { SETUP_TYPE_ENTRY, &default_nick, "default_nick", NULL },
    { SETUP_TYPE_ENTRY, &alternate_nick, "alternate_nick", NULL },
    { SETUP_TYPE_ENTRY, &user_name, "user_name", NULL },
    { SETUP_TYPE_ENTRY, &real_name, "real_name", NULL },

    { 0, NULL, NULL }
};

static SETUP_SERVER_REC *setupserver_add(GList **servers, proplist_t prop)
{
    SETUP_SERVER_REC *rec;
    gchar *ircnet, *server, *password;
    gint port, cmd_queue_speed;
    gboolean autoconnect;

    g_return_val_if_fail(servers != NULL, NULL);
    g_return_val_if_fail(prop != NULL, NULL);

    ircnet = config_get_str(prop, "ircnet", NULL);
    server = config_get_str(prop, "server", NULL);
    if (ircnet == NULL || server == NULL) return NULL;

    port = config_get_int(prop, "port", 6667);
    password = config_get_str(prop, "password", "");
    autoconnect = config_get_bool(prop, "autoconnect", FALSE);
    cmd_queue_speed = config_get_int(prop, "cmd_queue_speed", 0);

    if (server_setup_find(server, port) != NULL)
    {
	/* already exists - don't let it get there twice or
	   server reconnects will screw up! */
        return NULL;
    }

    rec = g_new0(SETUP_SERVER_REC, 1);
    rec->ircnet = g_strdup(ircnet);
    rec->server = g_strdup(server);
    rec->password = g_strdup(password);
    rec->port = port;
    rec->autoconnect = autoconnect;
    rec->cmd_queue_speed = cmd_queue_speed;

    *servers = g_list_append(*servers, rec);
    return rec;
}

void init_userinfo(void)
{
    /* check if nick/username/realname wasn't read from setup.. */
    if (real_name == NULL || *real_name == '\0')
    {
	if (real_name != NULL) g_free(real_name);
	real_name = g_getenv("IRCNAME");
	real_name = g_strdup(real_name != NULL ? real_name : g_get_real_name());
    }

    /* username */
    if (user_name == NULL || *user_name == '\0')
    {
	if (user_name != NULL) g_free(user_name);
	user_name = g_getenv("IRCUSER");
	user_name = g_strdup(user_name != NULL ? user_name : g_get_user_name());
    }

    /* nick */
    if (default_nick == NULL || *default_nick == '\0')
    {
	if (default_nick != NULL) g_free(default_nick);
	default_nick = g_getenv("IRCNICK");
	default_nick = g_strdup(default_nick != NULL ? default_nick : user_name);
    }

    /* alternate nick */
    if (alternate_nick == NULL || *alternate_nick == '\0')
    {
	if (alternate_nick != NULL) g_free(alternate_nick);
	if (strlen(default_nick) < 9)
	    alternate_nick = g_strconcat(default_nick, "_", NULL);
	else
	{
	    alternate_nick = g_strdup(default_nick);
            alternate_nick[strlen(alternate_nick)-1] = '_';
	}
    }
}

void setup_servers_init(void)
{
    proplist_t sprop, pvalue;
    gint num, max;

    source_host_ok = FALSE;

    /* Read servers */
    sprop = config_get_prop(cprop, "setupservers");
    max = sprop == NULL ? 0 : PLGetNumberOfElements(sprop);
    for (num = 0; num < max; num++)
    {
	pvalue = PLGetArrayElement(sprop, num);
        setupserver_add(&setupservers, pvalue);
    }
}

void setup_servers_deinit(void)
{
    while (setupservers != NULL)
    {
        SETUP_SERVER_REC *rec = setupservers->data;

        setupservers = g_list_remove(setupservers, rec);

        g_free(rec->ircnet);
        g_free(rec->server);
        g_free(rec->password);
        g_free(rec);
    }
}
