/*
 ui-channels.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

static void channel_creation_finish(WINDOW_REC *window, CHANNEL_REC *channel, gboolean automatic)
{
    if (CHANNEL_PARENT(channel) == NULL)
    {
	/* Add the channel to channels list in window */
	channel->ui_data = window;
	window->channels = g_list_append(window->channels, channel);
    }

    signal_emit("gui channel init", 1, channel);

    if (g_list_length(window->channels) == 1)
    {
        /* first channel in this window */
        window->active = channel;
        signal_emit("channel focused", 1, channel);
    }

    if (window->active == channel)
	signal_emit("channel server changed", 1, channel);

    if (!automatic)
    {
	signal_emit("gui window goto", 1,
		    GINT_TO_POINTER(g_list_index(windows, window)));
    }
}

static gboolean signal_channel_created(CHANNEL_REC *channel, gpointer automatic)
{
    WINDOW_REC *window, *window2;
    CHANNEL_REC *destroy;
    gchar *str;
    GList *tmp;

    g_return_val_if_fail(channel != NULL, FALSE);

    /* first check if there's any windows without channels, this happens when
       last channel is closed in window but window isn't destroyed and empty
       channel is just being created .. (when kicked from channel)

       .. also check if there's any empty windows that we could use.. */
    window = window2 = NULL;
    destroy = NULL;

    str = channel->server == NULL ? NULL :
	g_strdup_printf("%s %s", channel->server->tag, channel->name);

    for (tmp = windows; tmp != NULL; tmp = tmp->next)
    {
        WINDOW_REC *rec = tmp->data;

        if (rec->channels == NULL)
        {
            /* found, use this window.. */
            window = rec;
            break;
	}

	if (rec->waiting_channels != NULL && str != NULL)
	{
            /* right channel/server tag combination in some waiting list? */
	    GSList *tmp;

	    for (tmp = rec->waiting_channels; tmp != NULL; tmp = tmp->next)
	    {
		if (g_strcasecmp(tmp->data, str) == 0)
		{
		    g_free(tmp->data);
		    rec->waiting_channels = g_slist_remove(rec->waiting_channels, tmp->data);
		    window = rec;

		    /* if there exists empty channel in this window, destroy it. */
		    destroy = window->channels->data;
		    if (destroy->type != CHANNEL_TYPE_EMPTY)
			destroy = NULL;
		    break;
		}
	    }

	    if (window != NULL)
		break;
	}

        if (rec->channels->next == NULL)
	{
	    /* empty window found, we could use it.. */
            CHANNEL_REC *chan = rec->channels->data;

	    if (window2 == NULL && chan->type == CHANNEL_TYPE_EMPTY && chan->level == 0)
		window2 = rec;
        }
    }
    if (str != NULL) g_free(str);

    if (window == NULL)
    {
	if (window2 != NULL && channel->type != CHANNEL_TYPE_EMPTY)
	{
	    /* use the empty window */
	    window = window2;
	    destroy = window->channels->data;
	}
	else if (cur_channel != NULL && channel->type != CHANNEL_TYPE_EMPTY &&
		 cur_channel->type == CHANNEL_TYPE_EMPTY && cur_channel->level == 0)
	{
	    /* window is empty, use it. */
	    window = CHANNEL_PARENT(cur_channel);
	    destroy = cur_channel;
	}
	else
	{
	    /* create new window to use */
	    window = ui_window_create(channel, GPOINTER_TO_INT(automatic));
	}
    }

    channel_creation_finish(window, channel, GPOINTER_TO_INT(automatic));

    if (destroy != NULL)
    {
        /* destroy the empty channel */
        channel_destroy(destroy);
    }
    return TRUE;
}

static gboolean signal_channel_created_curwin(CHANNEL_REC *channel)
{
    g_return_val_if_fail(channel != NULL, FALSE);

    channel_creation_finish(CHANNEL_PARENT(cur_channel), channel, FALSE);
    signal_emit("gui channel open", 1, channel);
    return FALSE;
}

static gboolean signal_channel_destroyed(CHANNEL_REC *channel)
{
    WINDOW_REC *window;

    g_return_val_if_fail(channel != NULL, FALSE);

    window = CHANNEL_PARENT(channel);
    window->channels = g_list_remove(window->channels, channel);

    if (window->channels != NULL)
    {
	signal_emit("channel child destroyed", 1, channel);
        if (window->active == channel)
        {
            window->active = window->channels->data;
            signal_emit("channel focused", 1, window->active);
        }
    }
    else
    {
        /* no channels left in window .. what do I do? */
        if (!CHANNEL_PARENT(channel)->destroying)
        {
            /* create empty channel into window */
	    CHANNEL_REC *new_channel;

            new_channel = channel_create(channel->server, "(empty)", CHANNEL_TYPE_EMPTY, TRUE);
            new_channel->new_data = channel->new_data;
        }
        else
        {
            /* window is useless, destroy it. */
            ui_window_destroy(window);
        }
    }

    channel->ui_data = NULL;
    if (cur_channel == channel)
    {
	/* Still using this channel?! Change to anything else but this! */
	GList *pos;

	pos = g_list_find(windows, window);
	if (pos != NULL && pos->prev != NULL)
	{
	    /* switch to previous window */
            cur_channel = ((WINDOW_REC *) pos->prev->data)->active;
	}
	else
	{
	    cur_channel = channels == NULL ? NULL : channels->data;
	}
    }

    return TRUE;
}

static gboolean signal_channel_focused(CHANNEL_REC *channel)
{
    if (g_list_length(CHANNEL_PARENT(channel)->channels) > 1)
    {
	printformat(channel->server, channel->name, MSGLEVEL_CLIENTNOTICE,
		    IRCTXT_TALKING_IN, channel->name);
    }
    return TRUE;
}

static gboolean sig_server_looking(SERVER_REC *server)
{
    GList *tmp;

    g_return_val_if_fail(server != NULL, FALSE);

    /* try to keep some server assigned to channels.. */
    for (tmp = channels; tmp != NULL; tmp = tmp->next)
    {
        CHANNEL_REC *rec = tmp->data;

	if (rec->server == NULL && rec->type != CHANNEL_TYPE_SETUP)
	{
	    rec->server = server;
	    signal_emit("channel server changed", 1, rec);
	}
    }

    return TRUE;
}

static gboolean sig_disconnected(SERVER_REC *server)
{
    CHANNEL_REC *channel;
    WINDOW_REC *window;
    GList *tmp;

    g_return_val_if_fail(server != NULL, FALSE);

    for (tmp = channels; tmp != NULL; tmp = tmp->next)
    {
        channel = tmp->data;

        if (channel->server == server && channel->type == CHANNEL_TYPE_CHANNEL)
	{
	    window = CHANNEL_PARENT(channel);

	    window->waiting_channels =
		g_slist_append(window->waiting_channels, g_strdup_printf("%s %s", server->tag, channel->name));

        }
    }
    return TRUE;
}

static gboolean cmd_wjoin(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    signal_add("channel created", (SIGNAL_FUNC) signal_channel_created_curwin);
    signal_emit("command join", 3, data, server, channel);
    signal_remove("channel created", (SIGNAL_FUNC) signal_channel_created_curwin);
    return TRUE;
}

static gboolean cmd_wquery(gchar *data, SERVER_REC *server, CHANNEL_REC *channel)
{
    signal_add("channel created", (SIGNAL_FUNC) signal_channel_created_curwin);
    signal_emit("command query", 3, data, server, channel);
    signal_remove("channel created", (SIGNAL_FUNC) signal_channel_created_curwin);
    return TRUE;
}

void ui_channels_init(void)
{
    signal_add("channel created", (SIGNAL_FUNC) signal_channel_created);
    signal_add("channel destroyed", (SIGNAL_FUNC) signal_channel_destroyed);
    signal_add("channel focused", (SIGNAL_FUNC) signal_channel_focused);
    signal_add("server looking", (SIGNAL_FUNC) sig_server_looking);
    signal_add_last("server disconnected", (SIGNAL_FUNC) sig_disconnected);
    command_bind("wjoin", NULL, (SIGNAL_FUNC) cmd_wjoin);
    command_bind("wquery", NULL, (SIGNAL_FUNC) cmd_wquery);
}

void ui_channels_deinit(void)
{
    signal_remove("channel created", (SIGNAL_FUNC) signal_channel_created);
    signal_remove("channel destroyed", (SIGNAL_FUNC) signal_channel_destroyed);
    signal_remove("channel focused", (SIGNAL_FUNC) signal_channel_focused);
    signal_remove("server looking", (SIGNAL_FUNC) sig_server_looking);
    signal_remove("server disconnected", (SIGNAL_FUNC) sig_disconnected);
    command_unbind("wjoin", (SIGNAL_FUNC) cmd_wjoin);
    command_unbind("wquery", (SIGNAL_FUNC) cmd_wquery);
}
