/*
 ui-common.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "../nls.h"

#include <sys/signal.h>

static gchar *autocon_server;
static gchar *autocon_password;
static gint autocon_port;
static gboolean no_autoconnect;
static gchar *cmdline_nick;

void ui_common_init(void)
{
    static struct poptOption options[] =
    {
        { "connect", 'c', POPT_ARG_STRING, &autocon_server, 0, N_("Automatically connect to server/ircnet"), N_("SERVER") },
        { "password", 'w', POPT_ARG_STRING, &autocon_password, 0, N_("Autoconnect password"), N_("SERVER") },
        { "port", 'p', POPT_ARG_INT, &autocon_port, 0, N_("Autoconnect port"), N_("PORT") },
        { "noconnect", '!', POPT_ARG_NONE, &no_autoconnect, 0, N_("Disable autoconnecting"), NULL },
        { "nick", 'n', POPT_ARG_STRING, &cmdline_nick, 0, N_("Specify nick to use"), NULL },
        { NULL, '\0', 0, NULL }
    };

    autocon_server = NULL;
    autocon_password = NULL;
    autocon_port = 6667;
    no_autoconnect = FALSE;
    cmdline_nick = NULL;
    args_register(options);

    ui_autorun_init();
    ui_channels_init();
    ui_commands_init();
    ui_completion_init();
    ui_ctcp_init();
    ui_dcc_init();
    ui_events_init();
    ui_events_numeric_init();
    ui_hilight_init();
    ui_hilight_text_init();
    ui_history_init();
    ui_keyboard_init();
    ui_printtext_init();
    ui_printtext_plugin_init();
    ui_server_init();
    ui_settings_init();
    ui_themes_init();
    ui_translation_init();
    ui_windows_init();
}

void ui_common_deinit(void)
{
    ui_autorun_deinit();
    ui_channels_deinit();
    ui_commands_deinit();
    ui_completion_deinit();
    ui_ctcp_deinit();
    ui_dcc_deinit();
    ui_events_deinit();
    ui_events_numeric_deinit();
    ui_keyboard_deinit();
    ui_hilight_deinit();
    ui_hilight_text_deinit();
    ui_history_deinit();
    ui_printtext_deinit();
    ui_printtext_plugin_deinit();
    ui_server_deinit();
    ui_settings_deinit();
    ui_themes_deinit();
    ui_translation_deinit();
    ui_windows_deinit();
}

void ui_common_finish_init(void)
{
    CHANNEL_REC *channel;
    GList *tmp, *ircnets;
    gchar *str, **default_nick;

    signal(SIGPIPE, SIG_IGN);

    if (cmdline_nick != NULL)
    {
	/* override nick found from setup */
	default_nick = setup_get_ptr("default_nick");
	if (*default_nick != NULL) g_free(*default_nick);
	*default_nick = g_strdup(cmdline_nick);
    }

    if (setup_get_bool("toggle_use_status_window"))
    {
        channel = channel_create(NULL, "(status)", CHANNEL_TYPE_EMPTY, TRUE);
        channel->level = MSGLEVEL_ALL ^ (setup_get_bool("toggle_use_msgs_window") ? (MSGLEVEL_MSGS|MSGLEVEL_ACTIONS) : 0);
    }

    if (setup_get_bool("toggle_use_msgs_window"))
    {
        channel = channel_create(NULL, "(msgs)", CHANNEL_TYPE_EMPTY, TRUE);
        channel->level = MSGLEVEL_MSGS|MSGLEVEL_ACTIONS;
    }

    if (channels == NULL)
    {
        /* we have to have at least one channel in window.. */
        channel = channel_create(NULL, "(empty)", CHANNEL_TYPE_EMPTY, TRUE);
    }

    if (autocon_server != NULL)
    {
	/* connect to specified server */
	str = g_strdup_printf(autocon_password == NULL ? "%s %d" : "%s %d %s",
			      autocon_server, autocon_port, autocon_password);
	signal_emit("command connect", 1, str);
	g_free(str);
        return;
    }

    if (no_autoconnect)
    {
	/* don't autoconnect */
	return;
    }

    /* connect to autoconnect servers */
    ircnets = NULL;
    for (tmp = g_list_first(setupservers); tmp != NULL; tmp = tmp->next)
    {
        SETUP_SERVER_REC *rec = tmp->data;

        if (rec->autoconnect && (*rec->ircnet == '\0' || glist_find_icase_string(ircnets, rec->ircnet) == NULL))
        {
            if (*rec->ircnet != '\0')
                ircnets = g_list_append(ircnets, rec->ircnet);

            str = g_strdup_printf("%s %d", rec->server, rec->port);
            signal_emit("command connect", 1, str);
            g_free(str);
        }
    }

    g_list_free(ircnets);
}
