/*
 ui-settings.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"
#include "../settings/settings.h"

static gboolean cmd_set(gchar *data)
{
    SETUP_REC *options;
    gchar *ptr, *val, *origdata;
    gboolean found, printed;
    gint n;

    origdata = g_strdup(data);
    for (data = ptr = data; ptr != NULL; data = ptr)
    {
        val = strchr(data, '=');
        if (val != NULL)
        {
            /* all the rest is value.. */
            *val++ = '\0';

            /* skip heading space */
            ptr = strchr(data, ' ');
            if (ptr != NULL) *ptr = '\0';

            /* skip trailing space */
            while (isspace((gint) *val)) val++;

            ptr = NULL;
        }
        else
        {
            /* just browsing, check if there's anything else */
            ptr = strchr(data, ' ');
            if (ptr != NULL) *ptr++ = '\0';
        }

	found = FALSE;
	for (n = 0; n < OPTION_CATEGORIES; n++)
	{
	    options = option_categories[n].rec;

	    printed = FALSE;
	    for (; options->tag != NULL; options++)
	    {
		if (*data == '\0' || ((val != NULL && g_strcasecmp(options->tag, data) == 0) ||
				      (val == NULL && stristr(options->tag, data) != NULL)))
		{
		    if (val != NULL)
			setup_set_data(options, option_categories[n].name, val);

		    if (!printed)
		    {
			printtext(NULL, NULL, MSGLEVEL_CLIENTCRAP, "%_[ %s ]", option_categories[n].name);
                        printed = TRUE;
			found = TRUE;
		    }
		    printtext(NULL, NULL, MSGLEVEL_CLIENTCRAP, "%s = %s", options->tag, setup_tag_data(options));
		}
	    }
	}

        if (!found)
            printtext(NULL, NULL, MSGLEVEL_CLIENTERROR, "Unknown setting %s", data);
    }
    if (!readonly) PLSave(cprop, TRUE);
    signal_emit("setup changed", 0);

    g_free(origdata);
    return TRUE;
}

static proplist_t config_array_add(proplist_t prop, gchar *hkey, gchar *hvalue, gchar *key, gchar *value)
{
    proplist_t dict;

    dict = PLMakeDictionaryFromEntries(NULL, NULL);
    prop = PLAppendArrayElement(prop, dict);

    dict = config_set_str(dict, hkey, key);
    dict = config_set_str(dict, hvalue, value);
    return prop;
}

static gboolean cmd_alias(gchar *data)
{
    proplist_t aprop;
    LIST_REC *rec;
    gchar *params, *alias, *value;
    gint num;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &alias, &value);
    if (*alias == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);
    if (strchr(alias, '=') != NULL || *value == '=')
    {
        printtext(NULL, NULL, MSGLEVEL_CLIENTERROR, "Use space instead of '=' character when defining alias!");
        g_free(params);
        return TRUE;
    }

    aprop = config_list_section(&cprop, "aliases");
    num = config_list_find(aprop, "alias", alias);
    if (num != -1) PLRemoveArrayElement(aprop, num);

    if (*value != '\0')
    {
        rec = list_find(aliases, alias);
        if (rec != NULL) list_remove(&aliases, rec);
	list_add(&aliases, alias, value);
	aprop = config_array_add(aprop, "alias", "command", alias, value);
    }
    else
    {
        rec = list_find(aliases, alias);
	if (rec != NULL) list_remove(&aliases, rec);
    }
    if (!readonly) PLSave(cprop, TRUE);

    g_free(params);
    return TRUE;
}

static gboolean cmd_unalias(gchar *data)
{
    gchar *params, *alias;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 1, &alias);
    cmd_alias(alias);
    g_free(params);

    return TRUE;
}

static gboolean cmd_ignore(gchar *data)
{
    proplist_t iprop;
    LIST_REC *rec;
    gchar *params, *mask, *type, *str;
    gint num;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &mask, &type);
    if (*mask == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    /* find from ignores list */
    rec = list_find(ignores, mask);

    if (*type == '\0')
    {
	/* display ignore */
	if (rec == NULL)
	    printformat(NULL, NULL, MSGLEVEL_CLIENTERROR, IRCTXT_IGNORE_NOT_FOUND, mask);
	else
	    printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_IGNORED, rec->value, mask);
        g_free(params);
	return TRUE;
    }

    if (rec != NULL)
    {
	/* add to list of previous ignores */
	type = g_strdup_printf("%s %s", rec->value, type);
    }

    /* check that it's ok. */
    str = bits2level(level2bits(type));
    if (rec != NULL) g_free(type);
    type = str;

    if (*type == '\0')
    {
	/* invalid level, abort.. */
	g_free(type);

	type = bits2level(MSGLEVEL_ALL+MSGLEVEL_NO_ACT); /* no_act just so that bits2level() doesn't return simply "ALL" */
        str = g_strconcat("ALL ", type, NULL); g_free(type);
	printformat(NULL, NULL, MSGLEVEL_CLIENTERROR, IRCTXT_LEVELS_ERROR, str);
	g_free(str);

        g_free(params);
	return TRUE;
    }

    /* add to ignores list */
    if (rec == NULL)
	list_add(&ignores, mask, type);
    else
	list_set_value(rec, type);

    /* save to config file */
    iprop = config_list_section(&cprop, "ignores");
    num = config_list_find(iprop, "mask", mask);
    if (num != -1) PLRemoveArrayElement(iprop, num);

    iprop = config_array_add(iprop, "mask", "level", mask, type);
    if (!readonly) PLSave(cprop, TRUE);

    /* ..and display in screen */
    printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_IGNORED, type, mask);

    g_free(type);
    g_free(params);
    return TRUE;
}

static gboolean cmd_unignore(gchar *data, SERVER_REC *server)
{
    proplist_t iprop;
    LIST_REC *rec;
    gchar *params, *mask, *type;
    gboolean found;
    gint num;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &mask, &type);
    if (*mask == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);
    if (*type == '\0') type = "ALL";

    found = server == NULL ? FALSE : autoignore_remove(server, mask, type);

    rec = list_find(ignores, mask);
    if (rec == NULL && !found)
    {
        printformat(NULL, NULL, MSGLEVEL_CLIENTERROR, IRCTXT_IGNORE_NOT_FOUND, mask);
        g_free(params);
        return TRUE;
    }

    printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_UNIGNORED, type, mask);
    if (rec != NULL)
    {
        guint level, newlevel;

        level = level2bits(type);
        if (level == 0) level = MSGLEVEL_ALL;

	iprop = config_list_section(&cprop, "ignores");
	num = config_list_find(iprop, "mask", mask);
	if (num != -1) PLRemoveArrayElement(iprop, num);

        newlevel = level2bits(rec->value) & (~level);
        if (newlevel == 0)
        {
            /* unignored everything */
            list_remove(&ignores, rec);
        }
        else
        {
            /* unignore just some ignore levels.. */
            type = bits2level(newlevel);
	    iprop = config_array_add(iprop, "mask", "level", mask, type);
            list_set_value(rec, type);
            g_free(type);
        }
	if (!readonly) PLSave(cprop, TRUE);
    }

    g_free(params);
    return TRUE;
}


/* add the nick of a hostmask to list if it isn't there already */
static GList* mask_add_nickonce(GList *list, gchar *mask)
{
    gchar *s;
    int len;
    g_return_val_if_fail(mask!=NULL, NULL);

    len = strlen(mask);
    if ((s=strchr(mask, '!')) != NULL) len = s-mask;
    s = g_malloc(len+1);
    g_memmove(s, mask, len);
    s[len] = '\0';
    if (g_list_find_custom(list, s, (GCompareFunc) g_strcasecmp)==NULL)
	return g_list_append(list, s);
    g_free(s);
    return list;
}

/* search for online people, print them and update offline list */
static void print_notify_onserver(SERVER_REC *server, GList *notify,
				GList **offline, int textid, gchar *desc)
{
    GList *online;
    GList *nicks;

    g_return_if_fail( server!=NULL );
    g_return_if_fail( offline!=NULL );
    g_return_if_fail( desc!=NULL );

    online = NULL;

    for (nicks = g_list_first(notify); nicks != NULL; nicks = nicks->next)
    {
	gchar *nick = nicks->data;
	if (!notifylist_ison_server(server, nick)) continue;
	/* this nick is on the net */
	online = g_list_append (online, nick);
	*offline = g_list_remove(*offline, nick);
    }
    if (online!=NULL) {
	gchar *s = list2str(online, ", ");
	printformat(NULL, NULL, MSGLEVEL_CLIENTNOTICE, textid, desc, s);
	g_free(s);
    }

    g_list_free(online);
}

/* show the notify list, displaying who is on which net */
static gboolean cmd_notify_show()
{

    GList *tmp;
    GList *notify;
    GList *offline;
    SERVER_REC *server;
    
    /* build a list containing only the nicks */
    notify = NULL;
    for (tmp = notifies; tmp != NULL; tmp = tmp->next)
    {
	LIST_REC *rec = tmp->data;
	notify = mask_add_nickonce (notify, rec->key);
    }
    offline = g_list_copy(notify);

    for (tmp = ircnets; tmp != NULL; tmp = tmp->next)
    {
	/* look which nicks from notifylist are currently on this net */
	server = server_find_ircnet(tmp->data);
	if (server==NULL) continue;
	print_notify_onserver(server, notify, &offline, 
	    IRCTXT_NOTIFY_ONLINE, tmp->data);
    }
    /* now look on servers without a specified ircnet */
    for (tmp = servers; tmp != NULL; tmp = tmp->next)
    {
	server = tmp->data;
	if (server->ircnet != NULL) continue;
	print_notify_onserver(server, notify, &offline, 
	    IRCTXT_NOTIFY_ONLINE, server->tag);
    }
    /* now print offline people */
    if (offline!=NULL) {
	gchar *s = list2str(offline, ", ");
	printformat(NULL,NULL, MSGLEVEL_CLIENTNOTICE, IRCTXT_NOTIFY_OFFLINE, s);
	g_free(s);
    }
    g_list_free(offline);

    g_list_foreach(notify, (GFunc) g_free, NULL);
    g_list_free(notify);

    return TRUE;
}

static gboolean cmd_notify(gchar *data)
{
    proplist_t nprop;
    LIST_REC *rec;
    gchar *params, *mask, *ircnets;
    gint num;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 2 | PARAM_FLAG_GETREST, &mask, &ircnets);
    if (*mask == '\0')
    {
	g_free(params);
	return cmd_notify_show();
    }

    rec = list_find(notifies, mask);
    if (rec != NULL) list_remove(&notifies, rec);
    list_add(&notifies, mask, ircnets);

    nprop = config_list_section(&cprop, "notify");
    num = config_list_find(nprop, "mask", mask);
    if (num != -1) PLRemoveArrayElement(nprop, num);

    nprop = config_array_add(nprop, "mask", "ircnets", mask, ircnets);
    if (!readonly) PLSave(cprop, TRUE);

    g_free(params);
    return TRUE;
}

static gboolean cmd_unnotify(gchar *data)
{
    proplist_t nprop;
    LIST_REC *rec;
    gchar *params, *mask;
    gint num;

    g_return_val_if_fail(data != NULL, FALSE);

    params = cmd_get_params(data, 1, &mask);
    if (*mask == '\0') cmd_param_error(CMDERR_NOT_ENOUGH_PARAMS);

    rec = list_find(notifies, mask);
    if (rec != NULL) list_remove(&notifies, rec);

    nprop = config_list_section(&cprop, "notify");
    num = config_list_find(nprop, "mask", mask);
    if (num != -1) PLRemoveArrayElement(nprop, num);
    if (!readonly) PLSave(cprop, TRUE);

    g_free(params);
    return TRUE;
}

void ui_settings_init(void)
{
    command_bind("set", NULL, (SIGNAL_FUNC) cmd_set);
    command_bind("alias", NULL, (SIGNAL_FUNC) cmd_alias);
    command_bind("unalias", NULL, (SIGNAL_FUNC) cmd_unalias);
    command_bind("ignore", NULL, (SIGNAL_FUNC) cmd_ignore);
    command_bind("unignore", NULL, (SIGNAL_FUNC) cmd_unignore);
    command_bind("notify", NULL, (SIGNAL_FUNC) cmd_notify);
    command_bind("unnotify", NULL, (SIGNAL_FUNC) cmd_unnotify);
}

void ui_settings_deinit(void)
{
    command_unbind("set", (SIGNAL_FUNC) cmd_set);
    command_unbind("alias", (SIGNAL_FUNC) cmd_alias);
    command_unbind("unalias", (SIGNAL_FUNC) cmd_unalias);
    command_unbind("ignore", (SIGNAL_FUNC) cmd_ignore);
    command_unbind("unignore", (SIGNAL_FUNC) cmd_unignore);
    command_unbind("notify", (SIGNAL_FUNC) cmd_notify);
    command_unbind("unnotify", (SIGNAL_FUNC) cmd_unnotify);
}


/* vim: set shiftwidth=4 sts=4: */
