/*
 ui-themes.c : irssi

    Copyright (C) 1999 Timo Sirainen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "irssi.h"

GList *themes;
THEME_REC *current_theme;

THEME_REC *theme_create(gchar *path, gchar *name)
{
    THEME_REC *rec;

    g_return_val_if_fail(path != NULL, NULL);
    g_return_val_if_fail(name != NULL, NULL);

    rec = g_new0(THEME_REC, 1);
    rec->path = g_strdup(path);
    rec->name = g_strdup(name);
    rec->format = g_new0(gchar*, TEXTS);
    rec->plugins = g_hash_table_new((GHashFunc) g_istr_hash, (GCompareFunc) g_istr_equal);
    signal_emit("theme created", 1, rec);

    return rec;
}

static void theme_destroy_hash(gchar *key, PLUGIN_THEME_REC *rec)
{
    gint n, max;

    max = g_list_length(rec->formatlist);
    for (n = 0; n < max; n++)
	if (rec->format[n] != NULL)
	    g_free(rec->format[n]);
    g_free(rec->format);

    g_free(rec->formatlist->data);
    g_list_free(rec->formatlist);

    g_free(rec->plugin);
    g_free(rec);
}

void theme_destroy(THEME_REC *rec)
{
    gint n;

    signal_emit("theme destroyed", 1, rec);
    g_hash_table_foreach(rec->plugins, (GHFunc) theme_destroy_hash, NULL);
    g_hash_table_destroy(rec->plugins);

    for (n = 0; n < TEXTS; n++)
	if (rec->format[n] != NULL)
	    g_free(rec->format[n]);
    g_free(rec->format);

    if (rec->bg_pixmap != NULL) g_free(rec->bg_pixmap);
    if (rec->font != NULL) g_free(rec->font);
    g_free(rec->path);
    g_free(rec->name);
    g_free(rec);
}

void ui_themes_init(void)
{
    gchar *str;

    /* first there's default theme.. */
    str = g_strdup_printf("%s/.irssi/default.theme", g_get_home_dir());
    current_theme = theme_create(str, "default");
    current_theme->default_color = 15;
    themes = g_list_append(NULL, current_theme);
    g_free(str);
}

void ui_themes_deinit(void)
{
    /* free memory used by themes */
    g_list_foreach(themes, (GFunc) theme_destroy, NULL);
    g_list_free(themes);
    themes = NULL;
}
