#include <jmp-config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <jmp.h>
#include <cls.h>
#include <method.h>
#include <jmpthread.h>

/** Thread comparators */

/** Compare two jmpthreads by there name. */
int jmpthread_compr_name (const void* v1, const void* v2) {
    jmpthread **t1 = (jmpthread**) v1;
    jmpthread **t2 = (jmpthread**) v2;
    return strcmp (jmpthread_get_thread_name(*t1), jmpthread_get_thread_name(*t2));
}

int jmpthread_compr_name_r (const void* v1, const void* v2) {
    return jmpthread_compr_name (v2, v1);
}

/** compare threads based on thread group */
int jmpthread_compr_group (const void* v1, const void* v2) {
    jmpthread** t1 = (jmpthread**)v1;
    jmpthread** t2 = (jmpthread**)v2;
    return (strcmp (jmpthread_get_group_name (*t1), jmpthread_get_group_name (*t2)));
}

int jmpthread_compr_group_r (const void* v1, const void* v2) {
    return jmpthread_compr_group (v2, v1);
}

/** compare threads based on thread parent */
int jmpthread_compr_parent (const void* v1, const void* v2) {
    jmpthread** t1 = (jmpthread**)v1;
    jmpthread** t2 = (jmpthread**)v2;
    return (strcmp (jmpthread_get_parent_name (*t1), jmpthread_get_parent_name (*t2)));
}

int jmpthread_compr_parent_r (const void* v1, const void* v2) {
    return jmpthread_compr_parent (v2, v1);
}

/** Compare threads based on contenation time for the threads. */
int jmpthread_compr_contenation (const void* v1, const void* v2) {
    jlong l;
    jmpthread** t1 = (jmpthread**)v1;
    jmpthread** t2 = (jmpthread**)v2;
    l = (*t1)->timerstack->contendtime - (*t2)->timerstack->contendtime;
    if (l < 0)
	return 1;
    else if (l > 0)
	return -1;
    return 0;
}

int jmpthread_compr_contenation_r (const void* v1, const void* v2) {
    return jmpthread_compr_contenation (v2, v1);
}

/** Compare threads based on consumed cpu time. */
int jmpthread_compr_state (const void* v1, const void* v2) {
    jmpthread** t1 = (jmpthread**)v1;
    jmpthread** t2 = (jmpthread**)v2;
    return (*t1)->state - (*t2)->state;
}

int jmpthread_compr_state_r (const void* v1, const void* v2) {
    return jmpthread_compr_state (v2, v1);
}

/** Compare threads based on consumed cpu time. */
int jmpthread_compr_cputime (const void* v1, const void* v2) {
    jlong l;
    jmpthread** t1 = (jmpthread**)v1;
    jmpthread** t2 = (jmpthread**)v2;
    l = (*t1)->timerstack->cpu_time - (*t2)->timerstack->cpu_time;
    if (l < 0)
	return 1;
    else if (l > 0)
	return -1;
    return 0;
}

int jmpthread_compr_cputime_r (const void* v1, const void* v2) {
    return jmpthread_compr_cputime (v2, v1);
}


/** class comparators... */

/** compare classes based on class name */
int cls_compr_name (const void* v1, const void* v2) {
    cls** c1 = (cls**)v1;
    cls** c2 = (cls**)v2;
    return (strcmp (cls_get_name (*c1), cls_get_name (*c2)));
}

int cls_compr_name_r (const void* v1, const void* v2) {
    return cls_compr_name (v2, v1);
}

/** compare classes based on instance count */
int cls_compr_instance (const void* v1, const void* v2) {
    cls** c1 = (cls**)v1;
    cls** c2 = (cls**)v2;
    return (cls_get_instances (*c2) - cls_get_instances (*c1));
}

int cls_compr_instance_r (const void* v1, const void* v2) {
    return cls_compr_instance (v2, v1);
}

/** compare classes based on maximum instance count */
int cls_compr_max_instance (const void* v1, const void* v2) {
    cls** c1 = (cls**)v1;
    cls** c2 = (cls**)v2;
    return (cls_get_max_instances (*c2) - cls_get_max_instances (*c1));
}

int cls_compr_max_instance_r (const void* v1, const void* v2) {
    return cls_compr_max_instance (v2, v1);
}

/** compare classes based on current size in bytes */
int cls_compr_size (const void* v1, const void* v2) {
    cls** c1 = (cls**)v1;
    cls** c2 = (cls**)v2;
    return (cls_get_size (*c2) - cls_get_size (*c1));
}

int cls_compr_size_r (const void* v1, const void* v2) {
    return cls_compr_size (v2, v1);
}

/** compare classes based on number of garbage collected instances */
int cls_compr_instance_gc (const void* v1, const void* v2) {
    cls** c1 = (cls**)v1;
    cls** c2 = (cls**)v2;
    return (cls_get_total_gc (*c2) - cls_get_total_gc (*c1));
}

int cls_compr_instance_gc_r (const void* v1, const void* v2) {
    return cls_compr_instance_gc (v2, v1);
}

/** compare classes based on number of garbage collected instances */
int cls_compr_tenure (const void* v1, const void* v2) {
    cls** c1 = (cls**)v1;
    cls** c2 = (cls**)v2;
    return (cls_get_tenure (*c1) - cls_get_tenure (*c2));
}

int cls_compr_tenure_r (const void* v1, const void* v2) {
    return cls_compr_tenure (v2, v1);
}


/** Method comparators. */

/** Compare methods based on signature. */
int method_compr_signature (const void* v1, const void* v2) {
    method** m1 = (method**)v1;
    method** m2 = (method**)v2;
    return (strcmp (method_get_method_signature (*m1), method_get_method_signature (*m2)));
}

int method_compr_signature_r (const void* v1, const void* v2) {
    return method_compr_signature (v2, v1);
}

/** Compare methods based on method name (and signature if needed) */
int method_compr_name (const void* v1, const void* v2) {
    method** m1 = (method**)v1;
    method** m2 = (method**)v2;
    int c = 0; 
    c = (strcmp (method_get_method_name (*m1), method_get_method_name (*m2)));
    if (c) 
	return c;
    return method_compr_signature (v1, v2);
}

int method_compr_name_r (const void* v1, const void* v2) {
    return method_compr_name (v2, v1);
}

/** Compare method based on class name, method name and signature (in that order) */
int method_compr_class (const void* v1, const void* v2) {
    cls* c1;
    cls* c2;
    int c;
    method** m1 = (method**)v1;
    method** m2 = (method**)v2;
    c1 = method_get_owner (*m1);
    c2 = method_get_owner (*m2);
    c = (strcmp (cls_get_class_name (c1), cls_get_class_name (c2)));    
    if (c)
	return c;
    return method_compr_name (v1, v2);
}

int method_compr_class_r (const void* v1, const void* v2) {
    return method_compr_class (v2, v1);
}

/** Compare methods based on used time in the method. */
int method_compr_time (const void* v1, const void* v2) {
    jlong l;
    method** m1 = (method**)v1;
    method** m2 = (method**)v2;
    l = (*m1)->time_used.tv - (*m2)->time_used.tv;
    /* l is a long long, dont want it to be auto casted to an int, 
     * so check and return by ourselfs..
     */
    if (l < 0)
	return 1;
    else if (l > 0)
	return -1;
    return 0;
}

int method_compr_time_r (const void* v1, const void* v2) {
    return method_compr_time (v2, v1);
}

/** Compare methods based on number of calls. */
int method_compr_calls (const void* v1, const void* v2) {
    method** m1 = (method**)v1;
    method** m2 = (method**)v2;
    return method_get_calls (*m2) - method_get_calls (*m1);
}

int method_compr_calls_r (const void* v1, const void* v2) {
    return method_compr_calls (v2, v1);
}

/** Compare methods based on time used in method called from this method. */
int method_compr_hold_time (const void* v1, const void* v2) {
    jlong l;
    method** m1 = (method**)v1;
    method** m2 = (method**)v2;
    
    l = (*m1)->time_used.tv_hold - (*m2)->time_used.tv_hold;
    if (l < 0) 
	return 1;
    else if (l > 0)
	return -1;
    return 0;
}

int method_compr_hold_time_r (const void* v1, const void* v2) {
    return method_compr_hold_time (v2, v1);
}

/** Compare methods based on time used in this method and method called from this method. */
int method_compr_total_time (const void* v1, const void* v2) {
    jlong diff, tot1, tot2;
    method** m1 = (method**)v1;
    method** m2 = (method**)v2;

    tot1 = (*m1)->time_used.tv + (*m1)->time_used.tv_hold;
    tot2 = (*m2)->time_used.tv + (*m2)->time_used.tv_hold;
    
    if ((diff = (tot1 - tot2)) < 0) 
	return 1;
    else if (diff > 0) 
	return -1;
    return 0;
}

int method_compr_total_time_r (const void* v1, const void* v2) {
    return method_compr_total_time (v2, v1);
}

/** Compare methods based on allocated objects */
int method_compr_objects (const void* v1, const void* v2) {
    method** m1 = (method**) v1;
    method** m2 = (method**) v2;
    return (*m2)->allocated_objects - (*m1)->allocated_objects;
}

int method_compr_objects_r (const void* v1, const void* v2) {
    return method_compr_objects (v2, v1);
}

/** Compare methods based on allocated objects per call */
int method_compr_objpercall (const void* v1, const void* v2) {
    int ao1, ao2, mc1, mc2;
    method** m1 = (method**) v1;
    method** m2 = (method**) v2;
    ao2 = (*m2)->allocated_objects;
    ao1 = (*m1)->allocated_objects;
    mc2 = method_get_calls (*m2);
    mc1 = method_get_calls (*m1);

    if(mc2>0)
	    ao2 = ao2/mc2;
    if(mc1>0)
	    ao1 = ao1/mc1;
    
    return ao2 - ao1;
}

int method_compr_objpercall_r (const void* v1, const void* v2) {
    return method_compr_objpercall (v2, v1);
}

/** Compare methods based on total time per call. */
int method_compr_total_time_per_call (const void* v1, const void* v2) {
    jlong diff, tot1, tot2;
    int mc1, mc2;
    method** m1 = (method**)v1;
    method** m2 = (method**)v2;

    tot1 = (*m1)->time_used.tv + (*m1)->time_used.tv_hold;
    tot2 = (*m2)->time_used.tv + (*m2)->time_used.tv_hold;

    mc2 = method_get_calls (*m2);
    mc1 = method_get_calls (*m1);
    
    if(mc2>0)
	    tot2 = tot2/mc2;
    if(mc1>0)
	    tot1 = tot1/mc1;
    
    if ((diff = (tot1 - tot2)) < 0) 
	return 1;
    else if (diff > 0) 
	return -1;
    return 0;
}

int method_compr_total_time_per_call_r (const void* v1, const void* v2) {
    return method_compr_total_time_per_call (v2, v1);
}

/** Compare methods based on allocated memory */
int method_compr_bytes (const void* v1, const void* v2) {
    jlong diff;
    method** m1 = (method**) v1;
    method** m2 = (method**) v2;
    
    diff = (*m2)->allocated_memory - (*m1)->allocated_memory;

    if (diff<0) return -1;
    else if (diff==0) return 0;
    else return 1;
}

int method_compr_bytes_r (const void* v1, const void* v2) {
    return method_compr_bytes (v2, v1);
}

/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
