/*
 *  hash.h
 *
 *  Hash table handling..
 */

#ifndef HASH_H__
#define HASH_H__

#include <sys/types.h>
#include <monitor_type.h>

typedef struct hashtab   hashtab;
typedef size_t (*jmphash_f) (void *,
			     size_t);
typedef int    (*jmphash_cmp_f) (void *,
				 void *);
typedef void   (*jmphash_iter_f) (void *);
typedef void   (*jmphash_iter_fa) (void *, void *);

/*
 * jmphash_new
 * returns a new empty hash table.
 * h is the has function,
 * cmp is the compare function.
 */
hashtab *
jmphash_new (
    size_t        size,
    jmphash_f     h,
    jmphash_cmp_f cmp,
    char*         name);

/*
 * jmphash_free
 * clear the memory used by the hash table.
 */
void
jmphash_free (
    hashtab   *htab);


/*
 * jmphash_clear
 * clears all entries in the hash table.
 */
void 
jmphash_clear (
    hashtab   *htab);

/*
 * jmphash_insert
 * inserts the element el into the hash table.
 * return 0 on success, 1 on failure.
 */
int
jmphash_insert (
    void      *el,
    hashtab   *htab);

/*
 * jmphash_search
 * if any element in the hash table is equal to el it is returned.
 */
void *
jmphash_search (
    void      *el,
    hashtab   *htab);


/*
 * jmphash_remove
 * remove the element equal to el from the hash table.
 * return the element removed (or NULL)
 */
void *
jmphash_remove (
    void      *el,
    hashtab   *htab);

/*
 * jmphash_cardinal
 *  return the number of elements in the hash table.
 */
size_t
jmphash_cardinal (
    hashtab   *htab);


/*
 * jmphash_for_each
 *  call proc on every element in the hash table.
 */
void
jmphash_for_each (
    jmphash_iter_f    proc,
    hashtab       *htab);


/*
 * jmphash_for_each
 *  call proc on every element in the hash table.
 *  Extra data argument may be send...
 */
void
jmphash_for_each_with_arg (
    jmphash_iter_fa   proc,
    hashtab       *htab,
    void          *data);

/** Lock a hash table to avoid multi thread access problems */
void jmphash_lock (
    hashtab       *htab, int which);

/** Unlock a hash table */
void jmphash_unlock (
    hashtab       *htab, int which);

/** Lock a hash table to avoid multi thread access problems (maybe nested) */
void jmphash_lock_nested (
    hashtab       *htab, int which);

/** Unlock a hash table (maybe nested) */
void jmphash_unlock_nested (
    hashtab       *htab, int which);

monitor_t jmphash_get_monitor (
    hashtab       *htab);

#endif /* HASH_H__ */

/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
