#ifndef K3DSDK_BITMAP_H
#define K3DSDK_BITMAP_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include "color.h"
#include "result.h"

#include <algorithm>
#include <cmath>
#include <functional>
#include <iostream>
#include <string>

namespace k3d
{

/// Defines data measured in pixels
typedef unsigned long pixel_size_t;

/////////////////////////////////////////////////////////////////////////////
// basic_bitmap

/// Encapsulates a bitmap image
template<typename PixelType>
class basic_bitmap
{
public:
	typedef PixelType pixel_type;
	typedef PixelType* iterator;
	typedef const PixelType* const_iterator;
	typedef basic_bitmap<pixel_type> this_type;

	/// Creates an empty bitmap
	basic_bitmap() :
		m_width(0),
		m_height(0),
		m_data(0)
	{
	}

	/// Creates a new bitmap with given width and height in pixels
	basic_bitmap(const pixel_size_t Width, const pixel_size_t Height) :
		m_width(Width),
		m_height(Height),
		m_data(static_cast<pixel_type*>(std::malloc(m_width * m_height * sizeof(pixel_type))))
	{
		// Sanity checks ...
		assert_warning(m_width);
		assert_warning(m_height);
		assert_warning(m_data);
	}

	/// Creates a new bitmap, copying "old fashioned" C-style data
	basic_bitmap(void* Data, const pixel_size_t Width, const pixel_size_t Height) :
		m_width(Width),
		m_height(Height),
		m_data(static_cast<pixel_type*>(std::malloc(m_width * m_height * sizeof(pixel_type))))
	{
		// Sanity checks ...
		assert_warning(m_width);
		assert_warning(m_height);
		assert_warning(m_data);
		assert_warning(Data);

		memcpy(m_data, Data, m_width * m_height * sizeof(pixel_type));
	}

	/// Copy constructor for bitmaps of similar type
	basic_bitmap(const this_type& RHS) :
		m_width(RHS.m_width),
		m_height(RHS.m_height),
		m_data(static_cast<pixel_type*>(std::malloc(m_width * m_height * sizeof(pixel_type))))
	{
		memcpy(m_data, RHS.m_data, m_width * m_height * sizeof(pixel_type));
	}
	/// Copy constructor for bitmaps of dissimilar type
	template<typename ForeignType>
	basic_bitmap(const basic_bitmap<ForeignType>& RHS) :
		m_width(RHS.width()),
		m_height(RHS.height()),
		m_data(static_cast<pixel_type*>(std::malloc(m_width * m_height * sizeof(pixel_type))))
	{
		std::copy(RHS.data(), RHS.data() + m_width * m_height, m_data);
	}

	/// Destructor
	virtual ~basic_bitmap()
	{
		clear();
	}

	/// Returns the bitmap data width in pixels
	pixel_size_t width() const
	{
		return m_width;
	}

	/// Returns the bitmap data height in pixels
	pixel_size_t height() const
	{
		return m_height;
	}

	/// Returns the raw bitmap data array
	const pixel_type* const data() const
	{
		return m_data;
	}

	/// Returns the raw bitmap data array
	pixel_type* const data()
	{
		return m_data;
	}

	void clear()
	{
		if(m_data)
			std::free(m_data);

		m_width = 0;
		m_height = 0;
		m_data = 0;
	}

	void reset(const pixel_size_t Width, const pixel_size_t Height)
	{
		// Sanity checks ...
		assert_warning(Width);
		assert_warning(Height);

		pixel_type* const data = static_cast<pixel_type*>(std::malloc(Width * Height * sizeof(pixel_type)));
		assert_warning(data);

		clear();

		m_width = Width;
		m_height = Height;
		m_data = data;
	}

	iterator begin()
	{
		return m_data;
	}

	const_iterator begin() const
	{
		return m_data;
	}

	iterator end()
	{
		return m_data + (m_width * m_height);
	}

	const_iterator end() const
	{
		return m_data + (m_width * m_height);
	}

private:
	/// Stores the bitmap width in pixels
	pixel_size_t m_width;
	/// Stores the bitmap height in pixels
	pixel_size_t m_height;
	/// Stores the bitmap data as a 1D array of pixels
	PixelType* m_data;
};

/// We define the standard pixel to be floating-point RGBA data
typedef basic_rgba<half> pixel;
/// We define the standard bitmap to be floating-point RGBA data
typedef basic_bitmap<pixel> bitmap;

/// Copies the source bitmap to the destination, scaling it to the destination dimensions
template<typename source_t, typename destination_t>
void scale_bitmap(const source_t& Source, destination_t& Destination)
{
	// Sanity checks ...
	if(!Source.width() || !Source.height())
		return;

	const double source_width = Source.width() - 1;
	const double source_height = Source.height() - 1;
	const double destination_width = Destination.width() - 1;
	const double destination_height = Destination.height() - 1;

	typename destination_t::iterator destination = Destination.begin();

	// For each row in the destination ...
	for(pixel_size_t y = 0; y < Destination.height(); ++y)
		{
			// Find the closest row in the source ...
			const pixel_size_t j = static_cast<pixel_size_t>((y / destination_height) * source_height);

			// For each pixel in the destination row ...
			for(pixel_size_t x = 0; x < Destination.width(); ++x)
				{
					// Find the closest pixel in the source row ...
					const pixel_size_t i = static_cast<pixel_size_t>((x / destination_width) * source_width);

					// Copy the source pixel to the destination pixel ...
					*destination = *(Source.begin() + (j * Source.width()) + i);
					++destination;
				}
		}
}

/// Fills a bitmap with a checkerboard pattern with given check dimensions and colors
template<typename bitmap_t, typename even_color_t, typename odd_color_t>
void checkerboard_fill(bitmap_t& Bitmap, const pixel_size_t CheckWidth, const pixel_size_t CheckHeight, const even_color_t EvenColor, const odd_color_t OddColor)
{
	typename bitmap_t::iterator target = Bitmap.begin();

	for(pixel_size_t y = 0; y < Bitmap.height(); ++y)
		{
			const bool even_row = (y / CheckHeight) % 2;

			for(pixel_size_t x = 0; x < Bitmap.width(); ++x)
				{
					const bool even_column = (x / CheckWidth) % 2;

					if((even_row + even_column) % 2)
						*target = EvenColor;
					else
						*target = OddColor;

					++target;
				}
		}
}

} // namespace k3d

#endif // !K3DSDK_BITMAP_H


