#ifndef K3DSDK_COMMAND_NODE_H
#define K3DSDK_COMMAND_NODE_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares commandnode, which provides a default implementation of icommand_node
		\author Tim Shead (tshead@k-3d.com)
*/

#include "icommand_node.h"

namespace k3d
{

// Forward declarations
class idocument;
class iobject;

/////////////////////////////////////////////////////////////////////////////
// record_command

/// Records a command for use in a macro
void record_command(icommand_node& CommandNode, const icommand_node::command_t::type_t Type, const std::string& Command, const std::string& Arguments = "");

/////////////////////////////////////////////////////////////////////////////
// command_node_path

/// Returns the location of a command node within the command tree as a filesystem-like absolute path
const std::string command_node_path(icommand_node& Node);

/////////////////////////////////////////////////////////////////////////////
// get_command_node

/// Returns a command node based on its absolute path (could return NULL)
icommand_node* get_command_node(const std::string& Path);
/// Returns a command node relative to its parent (could return NULL)
icommand_node* get_command_node(icommand_node& Parent, const std::string& Child);

/////////////////////////////////////////////////////////////////////////////
// get_document

/// Returns the document (if any) that owns a command-node (could return NULL)
idocument* get_document(icommand_node& Node);

/////////////////////////////////////////////////////////////////////////////
// get_object

/// Returns the object (if any) that owns a command-node (could return NULL)
iobject* get_object(icommand_node& Node);

/////////////////////////////////////////////////////////////////////////////
// get_parent

/// Returns the parent command-node that owns the given command-node (note: the root node returns a reference to itself - don't use this function for walking the hierarchy!)
icommand_node& get_parent(icommand_node& Node);

/////////////////////////////////////////////////////////////////////////////
// is_descendant

/// Returns true iff Descendant is rooted at (or *is*) Parent
bool is_descendant(icommand_node& Parent, icommand_node& Descendant);

/////////////////////////////////////////////////////////////////////////////
// command_node

/// Provides a default implementation of icommand_node
class command_node :
	public icommand_node
{
public:
	command_node(const std::string& Name);
	virtual ~command_node();

	const std::string command_node_name();
	const commands_t commands();
	bool execute_command(const std::string& Command, const std::string& Arguments);

protected:
	/// Sets the name of the command-node
	void set_command_node_name(const std::string& Name);
	/// Registers a command
	void register_command(const std::string& Name, const std::string& Description, const command_t::type_t Type, const std::string& Command);
	
private:
	/// Stores the name of this command-node
	std::string m_name;
	/// Stores commands exposed by this node
	commands_t m_commands;
};

} // namespace k3d

#endif // K3DSDK_COMMAND_NODE_H


