#ifndef K3DSDK_IOPTIONS_H
#define K3DSDK_IOPTIONS_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares k3d::ioptions, an abstract interface for global user options
		\author Tim Shead (tshead@k-3d.com)
*/

#include "iunknown.h"
#include "vectors.h"

#include <boost/filesystem/path.hpp>

#include <list>
#include <string>

namespace k3d
{

/// Abstract interface for global user options
class ioptions :
	public virtual iunknown
{
public:
	/// Used to store/retrieve data about a render engine
	class render_engine
	{
	public:
		render_engine(const std::string& Type, const std::string& Engine, const std::string& Name, const std::string& Render, const std::string& ShaderBinary, const std::string& CompileShader) :
			type(Type),
			engine(Engine),
			name(Name),
			render(Render),
			shader_binary(ShaderBinary),
			compile_shader(CompileShader)
		{
		}

		const std::string type;
		const std::string engine;
		const std::string name;
		const std::string render;
		const std::string shader_binary;
		const std::string compile_shader;
	};
	/// A collection of render engines
	typedef std::list<render_engine> render_engines_t;

	/// Sets default options so the user always has everything they need
	virtual void set_defaults() = 0;

	/// Returns the user's preferred language
	virtual const std::string preferred_language() = 0;
	/// Sets the user's preferred language
	virtual void set_preferred_language(const std::string& Language) = 0;
	/// Returns true iff a Blackbox Recorder should automatically be opened at startup
	virtual const bool blackbox_recorder_at_startup() = 0;
	/// Sets whether a Blackbox Recorder should automatically be opened at setup
	virtual void set_blackbox_recorder_at_startup(const bool State) = 0;
	/// Returns whether the Tutorials menu should be displayed at startup
	virtual const bool tutorial_menu_at_startup() = 0;
	/// Sets whether the Tutorials menu should be displayed at startup
	virtual void set_tutorial_menu_at_startup(const bool State) = 0;
	/// Returns the tutorial speed
	virtual const double tutorial_speed() = 0;
	/// Sets the tutorial speed
	virtual void set_tutorial_speed(const double Speed) = 0;
	/// Returns whether a window's size & position should be restored when opened
	virtual const bool restore_window_geometry() = 0;
	/// Sets whether a window's size & position should be restored when opened
	virtual void set_restore_window_geometry(const bool State) = 0;
	/// Returns the user's preferred HTML viewer command line
	virtual const std::string html_viewer() = 0;
	/// Sets the user's preferred HTML viewer command line
	virtual void set_html_viewer(const std::string& CommandLine) = 0;
	/// Returns the user's preferred bitmap viewer command line
	virtual const std::string bitmap_viewer() = 0;
	/// Sets the user's preferred bitmap viewer command line
	virtual void set_bitmap_viewer(const std::string& CommandLine) = 0;
	/// Returns the user's preferred shading language pre-processor command line
	virtual const std::string sl_preprocessor() = 0;
	/// Sets the user's preferred shading language pre-processor command line
	virtual void set_sl_preprocessor(const std::string& CommandLine) = 0;
	/// Returns the user's preferred default render engine
	virtual void default_render_engine(std::string& Type, std::string& Engine, std::string& Name) = 0;
	/// Sets the user's preferred default render engine
	virtual void set_default_render_engine(const std::string& Type, const std::string& Engine) = 0;
	/// Returns the path to the render farm's shared filesystem
	virtual const boost::filesystem::path render_farm_path() = 0;
	/// Sets the path to the render farm's shared filesystem
	virtual void set_render_farm_path(const boost::filesystem::path& Path) = 0;
	/// Returns the user's collection of configured render engines
	virtual const render_engines_t render_engines() = 0;
	/// Returns the most-recently-used path of a given type (e.g. textures path, documents path, scripts path, etc.)
	virtual const boost::filesystem::path most_recent_path(const std::string& Type) = 0;
	/// Sets the most-recently-used path of a given type (e.g. textures path, documents path, scripts path, etc.)
	virtual void set_most_recent_path(const std::string& Type, const boost::filesystem::path& Path) = 0;
	virtual const vector4 window_geometry(const std::string& WindowName) = 0;
	virtual void set_window_geometry(const std::string& WindowName, const vector4& Geometry) = 0;
	
	/// Commits any changes to user options to persistent storage
	virtual void commit_options() = 0;

protected:
	ioptions() {}
	ioptions(const ioptions& RHS) {}
	ioptions& operator = (const ioptions& RHS) { return *this; }
	virtual ~ioptions() {}
};

} // namespace k3d

#endif // !K3DSDK_IOPTIONS_H


