#ifndef K3D_IPERSISTENT_H
#define K3D_IPERSISTENT_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares ipersistent, an abstract interface implemented by all objects that can be serialized to/from a K-3D document
		\author Tim Shead (tshead@k-3d.com)
*/

#include "iunknown.h"

/**

The methods in ipersistent will be called in a specific order at various stages of an object's lifetime:

When the object is saved as part of a document:

1. save()

When the object is reloaded from a document:

1. load()
2. load_complete() (After load() has been called for all other objects)

*/

// Forward declarations
namespace sdpxml { class Document; }
namespace sdpxml { class Element; }

namespace k3d
{

// Forward declarations
class idependencies;

/// Abstract interface implemented by objects that can be serialized to/from a K-3D document
class ipersistent :
	public virtual k3d::iunknown
{
public:
	/// Called once during document loading
	virtual void load(sdpxml::Document& Document, sdpxml::Element& Element) = 0;
	/// Called once after Load has been called for all persistent objects
	virtual void load_complete() = 0;
	/// Called once during document save
	virtual void save(sdpxml::Element& Element, k3d::idependencies& Dependencies) = 0;

protected:
	ipersistent() {}
	ipersistent(const ipersistent&) {}
	ipersistent& operator=(const ipersistent&) { return *this; }
	virtual ~ipersistent() {}
};

} // namespace k3d

#endif // K3D_IPERSISTENT_H

