#ifndef K3DUI_TEXT_EDITORS_H
#define K3DUI_TEXT_EDITORS_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares the k3dTextEditor, k3dShaderEditor, and k3dScriptEditor classes, which provide extensible user interfaces for working with text files
		\author Tim Shead (tshead@k-3d.com)
*/

#include "k3ddialog.h"

/////////////////////////////////////////////////////////////////////////////
// k3dTextEditor

/// Provides a simple, standardized dialog for editing text
class k3dTextEditor :
	public k3dDialog
{
public:
	typedef k3dDialog base;

	k3dTextEditor(k3d::iunknown* ParentCommandNode, const std::string& CommandNodeName);
	virtual ~k3dTextEditor();

protected:
	/// Call this in derived classes to load a GTKML template and hook-up all events
	virtual bool LoadGTKMLTemplate(const std::string& Template);
	/// Overrides sdpGtkObjectContainer to handle custom events
	void OnEvent(sdpGtkEvent* Event);

	/// Returns the current filepath (could be empty)
	const boost::filesystem::path filepath();
	/// Returns the current editor contents (could be empty string, of course)
	const std::string get_text();
	/// Replaces the current editor contents
	void set_text(const std::string& Text);
	/// Returns the GtkText control for the editor
	sdpGtkText text_control();
	/// Updates the window titlebar
	void update_titlebar();

	// Template design pattern methods that can be overridden by derivatives

	/// Returns the document title
	virtual const std::string get_title();
	/// Returns the text to display in the window titlebar
	virtual const std::string get_titlebar_text();
	/// Returns the type of path that this editor reads / writes
	virtual const std::string get_path_category();

	/// Called whenever a new document is created
	virtual void on_create_new_document();

	/// Called to see if it's OK to overwrite the current editor contents
	bool safe_to_overwrite();

private:
	/// Loads the editor contents from a file
	void load(const boost::filesystem::path& Filepath);
	/// Loads the editor contents from a stream
	void load(std::istream& Stream);
	/// Saves the editor contents to a file
	void save(const boost::filesystem::path& Filepath);
	/// Saves the editor contents to a stream
	void save(std::ostream& Stream);
	/// Called by the WM "close" button
	void OnDelete(sdpGtkEvent* Event);
	/// Called to begin a new script
	void on_file_new();
	/// Called to load a script
	void on_file_open();
	/// Called to save the script
	void on_file_save();
	/// Called to save the script under a new filename
	void on_file_save_as();
	/// Called to revert to a previously-saved version of the file
	void on_file_revert();
	/// Called to close the editor window
	void on_file_close();
	/// Called when the text in the editor window changes
	void on_changed();

	bool file_save();
	bool file_save_as();

	/// The current editor file
	boost::filesystem::path m_file;
	/// Contains "true" if there are unsaved changes to the text
	bool m_changed;
};

/////////////////////////////////////////////////////////////////////////////
// k3dScriptEditor

/// Implements the application user interface
class k3dScriptEditor :
	public k3dTextEditor
{
public:
	typedef k3dTextEditor base;

	k3dScriptEditor(k3d::iunknown* ParentCommandNode, const std::string& CommandNodeName);

protected:
	// Template design pattern methods
	const std::string get_title();
	const std::string get_titlebar_text();
	const std::string get_path_category();

	void start_running();
	void stop_running();

	/// Overrides sdpGtkObjectContainer to handle custom events
	virtual void OnEvent(sdpGtkEvent* Event);

private:
	/// Called to play the current script
	virtual void on_play();

	/// Set to "true" while a script is executing
	bool m_running;
};

/////////////////////////////////////////////////////////////////////////////
// k3dShaderEditor

/// Implements the application user interface
class k3dShaderEditor :
	public k3dTextEditor
{
public:
	typedef k3dTextEditor base;

	k3dShaderEditor(k3d::iunknown* ParentCommandNode, const std::string& CommandNodeName);

protected:
	// Template design pattern methods
	const std::string get_title();
	const std::string get_path_category();

	/// Overrides sdpGtkObjectContainer to handle custom events
	virtual void OnEvent(sdpGtkEvent* Event);
	/// Called to compile the current script
	virtual void on_compile();
};

#endif // K3DUI_TEXT_EDITORS_H

