// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the Axes K-3D object, which provides a configurable set of axes to help in visualizing the 3D workspace
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/bounded.h>
#include <k3dsdk/classes.h>
#include <k3dsdk/iapplication.h>
#include <k3dsdk/transformable.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/module.h>
#include <k3dsdk/transform.h>
#include <k3dsdk/vectors.h>
#include <k3dsdk/viewport.h>

#include <sdpgl/sdpgl.h>

namespace
{

const double XAxisColor[4] = {1.0, 0.0, 0.0, 1.0};
const double YAxisColor[4] = {1.0, 1.0, 0.0, 1.0};
const double ZAxisColor[4] = {0.0, 1.0, 0.0, 1.0};
const double GridColor[4] = {0.4, 0.4, 0.4, 1.0};

/////////////////////////////////////////////////////////////////////////////
// axes_implementation

class axes_implementation :
	public k3d::bounded<k3d::viewport::drawable<k3d::transformable<k3d::persistent<k3d::object> > > >
{
	typedef k3d::bounded<k3d::viewport::drawable<k3d::transformable<k3d::persistent<k3d::object> > > > base;

public:
	axes_implementation(k3d::idocument& Document) :
		base(Document),
		m_Axes(k3d::init_name("axes") + k3d::init_description("Display XYZ axes [boolean]") + k3d::init_value(true) + k3d::init_document(Document)),
		m_XYPlane(k3d::init_name("xyplane") + k3d::init_description("Display XY plane as a grid [boolean]") + k3d::init_value(false) + k3d::init_document(Document)),
		m_YZPlane(k3d::init_name("yzplane") + k3d::init_description("Display YZ plane as a grid [boolean]") + k3d::init_value(false) + k3d::init_document(Document)),
		m_XZPlane(k3d::init_name("xzplane") + k3d::init_description("Display XZ plane as a grid [boolean]") + k3d::init_value(true) + k3d::init_document(Document)),
		m_GridSize(k3d::init_name("gridsize") + k3d::init_description("The size of each grid square [number]") + k3d::init_value(2.0) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance))),
		m_GridCount(k3d::init_name("gridcount") + k3d::init_description("Number of squares along each grid [positive integer]") + k3d::init_value(5) + k3d::init_document(Document) + k3d::init_precision(0) + k3d::init_step_increment(1.0) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_constraint(k3d::constraint::minimum(1UL)))
	{
		enable_serialization(k3d::persistence::proxy(m_Axes));
		enable_serialization(k3d::persistence::proxy(m_XYPlane));
		enable_serialization(k3d::persistence::proxy(m_YZPlane));
		enable_serialization(k3d::persistence::proxy(m_XZPlane));
		enable_serialization(k3d::persistence::proxy(m_GridSize));
		enable_serialization(k3d::persistence::proxy(m_GridCount));

		register_property(m_Axes);
		register_property(m_XYPlane);
		register_property(m_YZPlane);
		register_property(m_XZPlane);
		register_property(m_GridSize);
		register_property(m_GridCount);
		
		m_Axes.changed_signal().connect(SigC::slot(*this, &axes_implementation::async_redraw_all));
		m_XYPlane.changed_signal().connect(SigC::slot(*this, &axes_implementation::async_redraw_all));
		m_YZPlane.changed_signal().connect(SigC::slot(*this, &axes_implementation::async_redraw_all));
		m_XZPlane.changed_signal().connect(SigC::slot(*this, &axes_implementation::async_redraw_all));
		m_GridSize.changed_signal().connect(SigC::slot(*this, &axes_implementation::async_redraw_all));
		m_GridCount.changed_signal().connect(SigC::slot(*this, &axes_implementation::async_redraw_all));
		
		m_position.changed_signal().connect(SigC::slot(*this, &axes_implementation::async_redraw_all));
		m_orientation.changed_signal().connect(SigC::slot(*this, &axes_implementation::async_redraw_all));
		m_scale.changed_signal().connect(SigC::slot(*this, &axes_implementation::async_redraw_all));
	}

	const k3d::bounding_box extents()
	{
		const double size = m_GridSize.property_value() * m_GridCount.property_value();
		return k3d::bounding_box(size, -size, size, -size, size, -size);
	}

	void on_viewport_draw(const k3d::viewport::render_state& State)
	{
		sdpgl::store_attributes attributes();

		glListBase(State.gl_ascii_font_list_base);
		
		glDisable(GL_LIGHTING);
		glDisable(GL_TEXTURE_1D);
		glDisable(GL_TEXTURE_2D);
		glDisable(GL_BLEND);

		glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
		glDisable(GL_CULL_FACE);

		glLineWidth(1.0f);
		glDisable(GL_LINE_STIPPLE);

		const long grid_count = m_GridCount.property_value();
		const double grid_size = m_GridSize.property_value();
		const double size = grid_count * grid_size;

		// Draw axes and labels
		if(m_Axes.property_value())
			{
				// Draw X axis
				glColor3dv(XAxisColor);
				glBegin(GL_LINE_LOOP);
				glVertex3d(-size, 0.0, 0.0);
				glVertex3d(size, 0.0, 0.0);
				glEnd();

				// Draw Y axis
				glColor3dv(YAxisColor);
				glBegin(GL_LINE_LOOP);
				glVertex3d(0.0, -size, 0.0);
				glVertex3d(0.0, size, 0.0);
				glEnd();

				// Draw Z axis
				glColor3dv(ZAxisColor);
				glBegin(GL_LINE_LOOP);
				glVertex3d(0.0, 0.0, -size);
				glVertex3d(0.0, 0.0, size);
				glEnd();

				// Draw axis labels ...
				glColor3dv(GridColor);

				double labelposition = size * 1.1;

				glRasterPos3d(labelposition, 0, 0);
				glCallLists(2, GL_UNSIGNED_BYTE, (GLubyte*)"+X");
				glRasterPos3d(0, labelposition, 0);
				glCallLists(2, GL_UNSIGNED_BYTE, (GLubyte*)"+Y");
				glRasterPos3d(0, 0, labelposition);
				glCallLists(2, GL_UNSIGNED_BYTE, (GLubyte*)"+Z");

				glRasterPos3d(-labelposition, 0, 0);
				glCallLists(2, GL_UNSIGNED_BYTE, (GLubyte*)"-X");
				glRasterPos3d(0, -labelposition, 0);
				glCallLists(2, GL_UNSIGNED_BYTE, (GLubyte*)"-Y");
				glRasterPos3d(0, 0, -labelposition);
				glCallLists(2, GL_UNSIGNED_BYTE, (GLubyte*)"-Z");
			}

		// Setup grid color
		glColor3dv(GridColor);

		// Draw XY plane
		if(m_XYPlane.property_value())
			{
				glBegin(GL_LINES);
				for(long i = -grid_count; i <= grid_count; ++i)
					{
						glVertex3d(i * grid_size, -size, 0.0);
						glVertex3d(i * grid_size, size, 0.0);
						glVertex3d(-size, i * grid_size, 0.0);
						glVertex3d(size, i * grid_size, 0.0);
					}
				glEnd();
			}

		// Draw YZ plane
		if(m_YZPlane.value())
			{
				glBegin(GL_LINES);
				for(long i = -grid_count; i <= grid_count; ++i)
					{
						glVertex3d(0.0, i * grid_size, -size);
						glVertex3d(0.0, i * grid_size, size);
						glVertex3d(0.0, -size, i * grid_size);
						glVertex3d(0.0, size, i * grid_size);
					}
				glEnd();
			}

		// Draw XZ plane
		if(m_XZPlane.value())
			{
				glBegin(GL_LINES);
				for(long i = -grid_count; i <= grid_count; ++i)
					{
						glVertex3d(i * grid_size, 0.0, -size);
						glVertex3d(i * grid_size, 0.0, size);
						glVertex3d(-size, 0.0, i * grid_size);
						glVertex3d(size, 0.0, i * grid_size);
					}
				glEnd();
			}
	}

	void on_viewport_select(const k3d::viewport::render_state& State)
	{
	}

	void snap(const k3d::vector3& InputCoordinates, k3d::vector3& SnapCoordinates, std::string& SnapDescription)
	{
		// Convert coordinates to our frame ...
		k3d::vector3 input_coordinates = k3d::world_to_object_matrix(*this) * InputCoordinates;

		// Snap coordinates ...
		input_coordinates[0] = k3d::round(input_coordinates[0] / m_GridSize.value()) * m_GridSize.value();
		input_coordinates[1] = k3d::round(input_coordinates[1] / m_GridSize.value()) * m_GridSize.value();
		input_coordinates[2] = k3d::round(input_coordinates[2] / m_GridSize.value()) * m_GridSize.value();

		// Convert coordinates back to world frame ...
		SnapCoordinates = k3d::object_to_world_matrix(*this) * input_coordinates;
		SnapDescription = "Axes Grid";
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<axes_implementation>,
			k3d::interface_list<k3d::itransform_source,
			k3d::interface_list<k3d::itransform_sink > > >factory(
			k3d::classes::Axes(),
			"Axes",
			"Configurable set of axes to help in visualizing the 3D workspace",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_Axes;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_XYPlane;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_YZPlane;
	k3d_data_property(bool, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_XZPlane;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_GridSize;
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_GridCount;
};

} // namespace

namespace libk3dcore
{

/////////////////////////////////////////////////////////////////////////////
// axes_factory

k3d::iplugin_factory& axes_factory()
{
	return axes_implementation::get_factory();
}

} // namespace libk3dcore




