// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <k3dsdk/basic_math.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/utility.h>

#include <boost/multi_array.hpp>
#include <iomanip>
#include <iterator>

namespace
{

/// Returns every point within the polyhedron that that is an "interior" point, along with the list of edges that adjoin it
typedef std::map<k3d::point*, k3d::polyhedron::edges_t > point_edges_map_t;
point_edges_map_t interior_points(k3d::polyhedron& Polyhedron)
{
	point_edges_map_t results;
	
	for(k3d::polyhedron::edges_t::iterator edge = Polyhedron.edges.begin(); edge != Polyhedron.edges.end(); ++edge)
		{
			if(0 == (*edge)->vertex)
				continue;
				
			if(results.count((*edge)->vertex))
				continue;
				
			k3d::polyhedron::edges_t edges;
			for(k3d::split_edge* edge2 = *edge; edge2 && edge2->companion; edge2 = edge2->companion->face_clockwise)
				{
					edges.push_back(edge2);
					
					if(edge2->companion->face_clockwise == *edge)
						{
							results[(*edge)->vertex] = edges;
							break;
						}
				}
		}
	
	return results;
}

/// Returns true iff the given object is selected
struct is_selected
{
	template<typename T>
	bool operator()(T Iterator)
	{
		return Iterator.first->selected;
	}
};

} // namespace

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// bevel_points_implementation

class bevel_points_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	bevel_points_implementation(k3d::idocument& Document) :
		base(Document),
		m_distance(k3d::init_name("distance") + k3d::init_description("Distance [number]") + k3d::init_value(0.1) + k3d::init_document(Document) + k3d::init_precision(2) + k3d::init_step_increment(0.01) + k3d::init_units(typeid(k3d::measurement::distance)))
	{
		enable_serialization(k3d::persistence::proxy(m_distance));
		register_property(m_distance);
		
		m_input_mesh.changed_signal().connect(SigC::slot(*this, &bevel_points_implementation::on_reset_geometry));
		m_distance.changed_signal().connect(SigC::slot(*this, &bevel_points_implementation::on_reset_geometry));
		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &bevel_points_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// Get the input geometry ...
		k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Create output geometry ...
		k3d::mesh* const output = new k3d::mesh();
		k3d::deep_copy(*input, *output);

		update_geometry(*output);

		return output;
	}

	void update_geometry(k3d::mesh& Mesh)
	{
		const double distance = m_distance.property_value();

		// For each polyhedron ...
		for(k3d::mesh::polyhedra_t::iterator polyhedron = Mesh.polyhedra.begin(); polyhedron != Mesh.polyhedra.end(); ++polyhedron)
			{
				// Get the set of all bevel-able points ...
				point_edges_map_t points = ::interior_points(**polyhedron);
				
				// Get the set of all selected bevel-able points ...
				point_edges_map_t selected_points;
				k3d::copy_if(points.begin(), points.end(), std::inserter(selected_points, selected_points.end()), ::is_selected());
				if(selected_points.empty())
					selected_points = points;

				// For each point ...
				for(point_edges_map_t::iterator selected_point = selected_points.begin(); selected_point != selected_points.end(); ++selected_point)
					{
						k3d::point* const point = selected_point->first;
						k3d::polyhedron::edges_t& edges = selected_point->second;
						const unsigned int edge_count = edges.size();

						assert_warning(edge_count);

						k3d::mesh::points_t new_points;
						k3d::polyhedron::edges_t new_bevel_edges;
						k3d::polyhedron::edges_t new_face_edges;
						
						// Average adjacent edges to get a vector that will be perpendicular to the bevel ...
						k3d::vector3 bevel_direction;
						for(unsigned int i = 0; i != edge_count; ++i)
							bevel_direction += k3d::normalize(edges[i]->companion->vertex->position - point->position);

						// If this fails, it means that the adjacent faces are coplanar, so bevelling isn't going to work anyway ...
						if(0 == bevel_direction.Length2())
							continue;
						bevel_direction = k3d::normalize(bevel_direction);

						// Create a new point for each adjacent edge ...
						for(unsigned int i = 0; i != edge_count; ++i)
							{
								const k3d::vector3 edge_vector = edges[i]->companion->vertex->position - point->position;
								return_if_fail(0 != edge_vector.Length2());
									
								const k3d::vector3 edge_direction = k3d::normalize(edge_vector);
								const double costheta = bevel_direction * edge_direction;
								return_if_fail(0 != costheta);

								new_points.push_back(new k3d::point(point->position + (distance * (edge_direction / std::abs(costheta)))));
							}

						// Create two new edges for each point ...
						for(unsigned int i = 0; i != edge_count; ++i)
							{
								new_bevel_edges.push_back(new k3d::split_edge(new_points[i], 0, 0));
								new_face_edges.push_back(new k3d::split_edge(new_points[(edge_count - i) % edge_count], 0, 0));
							}

						// Hook-up the new topology ...
						for(unsigned int i = 0; i != edge_count; ++i)
							{
								edges[i]->vertex = new_points[i];
								
								new_bevel_edges[i]->face_clockwise = edges[i]->companion->face_clockwise;
								edges[i]->companion->face_clockwise = new_bevel_edges[i];

								new_face_edges[i]->face_clockwise = new_face_edges[(i + 1) % edge_count];
								k3d::join_edges(*new_face_edges[i], *new_bevel_edges[(edge_count-1) - i]);
							}
							
						// Get rid of the old point ...
						Mesh.points.erase(std::remove(Mesh.points.begin(), Mesh.points.end(), point), Mesh.points.end());
						delete point;
					
						// Create the new face ...
						k3d::face* const new_face = new k3d::face(new_face_edges.front());

						Mesh.points.insert(Mesh.points.end(), new_points.begin(), new_points.end());
						(*polyhedron)->edges.insert((*polyhedron)->edges.end(), new_bevel_edges.begin(), new_bevel_edges.end());
						(*polyhedron)->edges.insert((*polyhedron)->edges.end(), new_face_edges.begin(), new_face_edges.end());
						(*polyhedron)->faces.push_back(new_face);
					}
						
				assert_warning(is_valid(**polyhedron));
				assert_warning(is_solid(**polyhedron));
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bevel_points_implementation>,
				k3d::interface_list<k3d::imesh_source,
				k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x22212ffb, 0xc9124b38, 0x9e756603, 0x33fae278),
				"BevelPoints",
				"Bevels a surface at each selected point",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_distance;
};

/////////////////////////////////////////////////////////////////////////////
// bevel_points_factory

k3d::iplugin_factory& bevel_points_factory()
{
	return bevel_points_implementation::get_factory();
}

} // namespace libk3dmesh

