// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/color.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace k3d
{

inline bool operator < (const k3d::color& a, const k3d::color& b)
{
	if(a.red < b.red)
		return true;
	if(a.red > b.red)
		return false;
	if(a.green < b.green)
		return true;
	if(a.green > b.green)
		return false;
	if(a.blue < b.blue)
		return true;

	return false;
}

} // namespace k3d

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// filter_faces_by_color_implementation

class filter_faces_by_color_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	filter_faces_by_color_implementation(k3d::idocument& Document) :
		base(Document),
		m_color_number(k3d::init_name("color_number") + k3d::init_description("Color number [number]") + k3d::init_value(1) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_color_number));

		register_property(m_color_number);

		m_input_mesh.changed_signal().connect(SigC::slot(*this, &filter_faces_by_color_implementation::on_reset_geometry));

		m_color_number.changed_signal().connect(SigC::slot(*this, &filter_faces_by_color_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &filter_faces_by_color_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.property_value();
		return_val_if_fail(input, 0);

		// Otherwise, we convert input mesh polyhedra to blobbies ...
		k3d::mesh* const output = new k3d::mesh();
		update_geometry(*input, *output);

		return output;
	}

	void update_geometry(const k3d::mesh& Input, k3d::mesh& Output)
	{
		// Find Nth color
		const unsigned long color_number = m_color_number.property_value();
		unsigned long current_color = 0;
		std::map<k3d::color, unsigned long> color_map;
		bool found = false;
		k3d::color selected_color;

		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = Input.polyhedra.begin(); polyhedron != Input.polyhedra.end(); ++polyhedron)
			{
				for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
					{
						k3d::parameters_t::const_iterator color = (*face)->uniform_data.find("Cs");
						if(color != (*face)->uniform_data.end())
							{
								k3d::color face_color = boost::any_cast<k3d::color>(color->second);
								if(color_map.find(face_color) != color_map.end())
									// Color was already indexed
									continue;
								else
									{
										// Save color
										color_map[face_color] = current_color++;
										if(current_color == color_number)
											{
												found = true;
												selected_color = face_color;
												break;
											}
									}
							}
					}

				if(found)
					break;
			}

		// If specified color wasn't found, return
		if(!found)
			return;

		// Copy faces of the same color
		k3d::polyhedron* output_polyhedron = new k3d::polyhedron();
		return_if_fail(output_polyhedron);
		Output.polyhedra.push_back(output_polyhedron);

		typedef std::map<k3d::point*, k3d::point*> point_map_t;
		point_map_t point_map;
		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = Input.polyhedra.begin(); polyhedron != Input.polyhedra.end(); ++polyhedron)
			{
				for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
					{
						k3d::parameters_t::const_iterator color = (*face)->uniform_data.find("Cs");
						if(color != (*face)->uniform_data.end())
							{
								k3d::color face_color = boost::any_cast<k3d::color>(color->second);
								if(face_color != selected_color)
									continue;

								// Copy face ...
								k3d::split_edge* first_edge = (*face)->first_edge;
								k3d::split_edge* current_edge = first_edge;

								k3d::polyhedron::edges_t edges;
								do
									{
										k3d::point* point = current_edge->vertex;
										k3d::point* new_point;
										point_map_t::const_iterator point_equivalence = point_map.find(point);
										if(point_equivalence != point_map.end())
											{
												new_point = point_equivalence->second;
											}
										else
											{
												new_point = new k3d::point(point->position);
												Output.points.push_back(new_point);
												return_if_fail(new_point);
											}

										edges.push_back(new k3d::split_edge(new_point));
										current_edge = current_edge->face_clockwise;
									}
								while(current_edge != first_edge);

								k3d::loop_edges(edges.begin(), edges.end());
								output_polyhedron->edges.insert(output_polyhedron->edges.end(), edges.begin(), edges.end());

								k3d::face* const face = new k3d::face(*edges.begin());
								return_if_fail(face);
								face->uniform_data["Cs"] = selected_color;

								output_polyhedron->faces.push_back(face);
							}
					}
			}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<filter_faces_by_color_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink> > > factory(
				k3d::uuid(0x1efa74fc, 0x6e5d4ab8, 0x9110d288, 0xdbb975ed),
				"FilterFacesByColor",
				"Filter faces by color",
				"Objects",
				k3d::iplugin_factory::EXPERIMENTAL);

		return factory;
	}

private:
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_color_number;
};

/////////////////////////////////////////////////////////////////////////////
// filter_faces_by_color_factory

k3d::iplugin_factory& filter_faces_by_color_factory()
{
	return filter_faces_by_color_implementation::get_factory();
}

} // namespace libk3dmesh


