// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_filter.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/selection.h>
#include <k3dsdk/transform.h>

#include <iterator>

namespace libk3dmesh
{

/////////////////////////////////////////////////////////////////////////////
// points_to_blobby_implementation

class points_to_blobby_implementation :
	public k3d::mesh_filter<k3d::persistent<k3d::object> >
{
	typedef k3d::mesh_filter<k3d::persistent<k3d::object> > base;

public:
	points_to_blobby_implementation(k3d::idocument& Document) :
		base(Document),
		m_radius(k3d::init_name("radius") + k3d::init_description("Radius [number]") + k3d::init_document(Document) + k3d::init_value(1.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)))
	{
		enable_serialization(k3d::persistence::proxy(m_radius));

		register_property(m_radius);
		
		m_input_mesh.changed_signal().connect(SigC::slot(*this, &points_to_blobby_implementation::on_reset_geometry));
		m_radius.changed_signal().connect(SigC::slot(*this, &points_to_blobby_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &points_to_blobby_implementation::on_create_geometry));
	}
	
	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}
	
	k3d::mesh* on_create_geometry()
	{
		// If we don't have any input mesh, we're done ...
		const k3d::mesh* const input = m_input_mesh.property_value();
		if(!input)
			return 0;

		// Otherwise, we convert input mesh polyhedra to blobbies ...
		k3d::mesh* const output = new k3d::mesh();
		update_geometry(*input, *output);

		return output;
	}

	void update_geometry(const k3d::mesh& Input, k3d::mesh& Output)
	{
		const double radius = m_radius.property_value();
		const k3d::matrix4 matrix = k3d::scaling3D(k3d::vector3(radius, radius, radius));
	
		for(k3d::mesh::points_t::const_iterator point = Input.points.begin(); point != Input.points.end(); ++point)
			Output.points.push_back(new k3d::point(**point));

		k3d::blobby::add* const new_add = new k3d::blobby::add();
		for(k3d::mesh::points_t::iterator point = Output.points.begin(); point != Output.points.end(); ++point)
			new_add->add_operand(new k3d::blobby::ellipsoid(*point, matrix));

		Output.blobbies.push_back(new k3d::blobby(new_add));
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<points_to_blobby_implementation>,
			k3d::interface_list<k3d::imesh_source,
			k3d::interface_list<k3d::imesh_sink > > > factory(
				k3d::uuid(0x9d5d69d9, 0xfe994aa0, 0x9b7dee22, 0x1823bd2c),
				"PointsToBlobby",
				"Converts input points to a parametric surface",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_radius;
};

/////////////////////////////////////////////////////////////////////////////
// points_to_blobby_factory

k3d::iplugin_factory& points_to_blobby_factory()
{
	return points_to_blobby_implementation::get_factory();
}

} // namespace libk3dmesh

