// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/axis.h>
#include <k3dsdk/bitmap_filter.h>
#include <k3dsdk/file_filter.h>
#include <k3dsdk/ibitmap_write_format.h>
#include <k3dsdk/irenderman.h>
#include <k3dsdk/irender_farm.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>

#include <iterator>

namespace libk3drenderman
{

/////////////////////////////////////////////////////////////////////////////
// lat_long_environment_map_implementation

class lat_long_environment_map_implementation :
	public k3d::bitmap_filter<k3d::persistent<k3d::object> >,
	public k3d::ri::itexture
{
	typedef k3d::bitmap_filter<k3d::persistent<k3d::object> > base;

public:
	lat_long_environment_map_implementation(k3d::idocument& Document) :
		base(Document),
		m_filter(k3d::init_name("filter") + k3d::init_description("Filter [string]") + k3d::init_value(k3d::ri::RI_GAUSSIAN()) + k3d::init_document(document()) + k3d::init_values(filter_values())),
		m_swidth(k3d::init_name("swidth") + k3d::init_description("Filter S Width [positive real]") + k3d::init_value(2.0) + k3d::init_document(document()) + k3d::init_constraint(k3d::constraint::minimum(std::numeric_limits<double>::epsilon())) + k3d::init_precision(2) + k3d::init_step_increment(1.0) + k3d::init_units(typeid(k3d::measurement::scalar))),
		m_twidth(k3d::init_name("twidth") + k3d::init_description("Filter T Width [positive real]") + k3d::init_value(2.0) + k3d::init_document(document()) + k3d::init_constraint(k3d::constraint::minimum(std::numeric_limits<double>::epsilon())) + k3d::init_precision(2) + k3d::init_step_increment(1.0) + k3d::init_units(typeid(k3d::measurement::scalar)))
	{
		enable_serialization(k3d::persistence::proxy(m_filter));
		enable_serialization(k3d::persistence::proxy(m_swidth));
		enable_serialization(k3d::persistence::proxy(m_twidth));
		
		register_property(m_filter);
		register_property(m_swidth);
		register_property(m_twidth);
	
		m_input.changed_signal().connect(SigC::slot(*this, &lat_long_environment_map_implementation::on_reset_bitmap));
		m_output.need_data_signal().connect(SigC::slot(*this, &lat_long_environment_map_implementation::on_create_bitmap));
	}
	
	void setup_renderman_texture(k3d::irender_frame& Frame, k3d::ri::irender_engine& Engine)
	{
		m_ri_image_path = boost::filesystem::path();
		m_ri_texture_path = boost::filesystem::path();

		// If we don't have an input bitmap, we're done ...
		const k3d::bitmap* const input = m_input.property_value();
		if(!input)
			return;

		m_ri_image_path = Frame.add_input_file("texture");
		return_if_fail(!m_ri_image_path.empty());
		
		m_ri_texture_path = Frame.add_input_file("texture");
		return_if_fail(!m_ri_texture_path.empty());

/*
		// If "render from source" is enabled, just copy the source file to the frame directory ...
		if(m_render_from_source.property_value())
			{
				// Copy the file ...
#ifndef K3D_PLATFORM_WIN32_NATIVE
				const std::string copycommand = "cp " + m_image_path.value().native_file_string() + " " + m_ri_image_path.native_file_string();
#else // !K3D_PLATFORM_WIN32_NATIVE
				const std::string copycommand = "copy " + m_image_path.value().native_file_string() + " " + m_ri_image_path.native_file_string();
#endif // K3D_PLATFORM_WIN32_NATIVE
				system(copycommand.c_str());

			}
		// Otherwise, save the in-memory image data as a TIFF file ...
		else
*/
			{
				k3d::ibitmap_write_format* const filter = k3d::file_filter<k3d::ibitmap_write_format>("TIFFWriter");
				return_if_fail(filter);
				return_if_fail(filter->write_file(m_ri_image_path, *input));
			}
			
		Engine.RiMakeLatLongEnvironmentV(m_ri_image_path.native_file_string(), m_ri_texture_path.native_file_string(), m_filter.property_value(), m_swidth.property_value(), m_twidth.property_value());
	}

	const boost::filesystem::path renderman_texture_path(const k3d::ri::render_state& State)
	{
		return m_ri_texture_path;
	}

	void on_reset_bitmap()
	{
		m_output.reset();
	}
	
	k3d::bitmap* on_create_bitmap()
	{
		// If we don't have any input bitmap, we're done ...
		const k3d::bitmap* const input = m_input.property_value();
		if(!input)
			return 0;

		// Otherwise, we make a copy of the input bitmap and modify the copy ...
		k3d::bitmap* const output = new k3d::bitmap(*input);

		return output;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<lat_long_environment_map_implementation>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink,
			k3d::interface_list<k3d::ri::itexture> > > > factory(
				k3d::uuid(0x04354dba, 0x91a7411c, 0x88b73725, 0x5808a415),
				"RenderManLatLongEnvironmentMap",
				"Converts a bitmap into a RenderMan LatLong Environment Map",
				"Objects",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_list_property(std::string, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_filter;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_swidth;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_twidth;

	const k3d::ilist_property<std::string>::values_t& filter_values()
	{
		static k3d::ilist_property<std::string>::values_t values;
		if(values.empty())
			{
				values.push_back(k3d::ri::RI_GAUSSIAN());
				values.push_back(k3d::ri::RI_BOX());
				values.push_back(k3d::ri::RI_TRIANGLE());
				values.push_back(k3d::ri::RI_CATMULL_ROM());
				values.push_back(k3d::ri::RI_SINC());
			}
		return values;
	}

	/// Stores the absolute path to the saved TIFF version of this lat_long_environment_map (for use during RenderMan rendering)
	boost::filesystem::path m_ri_image_path;
	/// Stores the absolute path to the version of this lat_long_environment_map converted by the renderer (via RiMakeLatLongEnvironment())
	boost::filesystem::path m_ri_texture_path;
};

/////////////////////////////////////////////////////////////////////////////
// lat_long_environment_map_factory

k3d::iplugin_factory& lat_long_environment_map_factory()
{
	return lat_long_environment_map_implementation::get_factory();
}

} // namespace libk3drenderman

