// SDPGL
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include "platform.h"

#ifdef SDPGL_WIN32
 #include <windows.h>
#else // SDPGL_WIN32
 #include <GL/glx.h>
#endif // !SDPGL_WIN32

#include <GL/gl.h>
#include <GL/glu.h>

#include <iostream>
#include <sstream>
#include <stack>

namespace
{

/// Stores a pointer to the desired output stream (defaults to cerr)
static std::ostream* g_output_stream = &std::cerr;

/// Maintains the current tracing status (on/off) on a stack
static std::stack<bool> g_output_stack;

/// Performs the actual work of sending text to the trace output
void raw_trace(const std::string Text)
{
	// See if tracing has been explicitly disabled ...
	if(g_output_stack.size())
		if(!g_output_stack.top())
			return;

	// Otherwise, trace the output ...
	(*g_output_stream) << Text;
	g_output_stream->flush();
}

void begin_function_trace(const std::string Function, const std::string Arguments)
{
	raw_trace(Function + "(" + Arguments);
}

void result_trace(const std::string Text)
{
	raw_trace("=" + Text);
}

void end_function_trace()
{
	raw_trace(");\n");
}

template<typename type>
std::string string_cast(const type& RHS)
{
	std::stringstream buffer;
	buffer << RHS;
	return buffer.str();
}

/////////////////////////////////////////////////////////////////////////////
// Argument formatting ...

std::string sdpGLvoid(const GLvoid* Arg)
{
	return string_cast(Arg);
}

std::string sdpGLsizei(const GLsizei Arg)
{
	return string_cast(Arg);
}

std::string sdpGLint(const GLint Arg)
{
	return string_cast(Arg);
}

std::string sdpGLuint(const GLuint Arg)
{
	return string_cast(Arg);
}

std::string sdpGLbyte(const GLbyte Arg)
{
	return string_cast(Arg);
}

std::string sdpGLubyte(const GLubyte Arg)
{
	return string_cast(Arg);
}

std::string sdpGLshort(const GLshort Arg)
{
	return string_cast(Arg);
}

std::string sdpGLushort(const GLushort Arg)
{
	return string_cast(Arg);
}

std::string sdpGLdouble(const GLdouble Arg)
{
	return string_cast(Arg);
}

std::string sdpGLfloat(const GLfloat Arg)
{
	return string_cast(Arg);
}

std::string sdpGLclampf(const GLclampf Arg)
{
	return string_cast(Arg);
}

std::string sdpGLclampd(const GLclampd Arg)
{
	return string_cast(Arg);
}

std::string sdpGLbitfield(const GLbitfield Arg)
{
	return string_cast(Arg);
}

std::string sdpGLboolean(const GLboolean Arg)
{
	return std::string(Arg ? "GL_TRUE" : "GL_FALSE");
}

std::string sdpGLMatrixd(const GLdouble* Arg)
{
	std::string result;
	result += "[" + sdpGLdouble(Arg[0]) + "," + sdpGLdouble(Arg[1]) + "," + sdpGLdouble(Arg[2]) + "," + sdpGLdouble(Arg[3]) + "]";
	result += "[" + sdpGLdouble(Arg[4]) + "," + sdpGLdouble(Arg[5]) + "," + sdpGLdouble(Arg[6]) + "," + sdpGLdouble(Arg[7]) + "]";
	result += "[" + sdpGLdouble(Arg[8]) + "," + sdpGLdouble(Arg[9]) + "," + sdpGLdouble(Arg[10]) + "," + sdpGLdouble(Arg[11]) + "]";
	result += "[" + sdpGLdouble(Arg[12]) + "," + sdpGLdouble(Arg[13]) + "," + sdpGLdouble(Arg[14]) + "," + sdpGLdouble(Arg[15]) + "]";

	return result;
}

std::string sdpGLenum(const GLenum Arg)
{
	switch(Arg)
		{
			case 0x0000:
				return "GL_POINTS";
			case 0x0001:
				return "GL_LINES";
			case 0x0002:
				return "GL_LINE_LOOP";
			case 0x0003:
				return "GL_LINE_STRIP";
			case 0x0004:
				return "GL_TRIANGLES";
			case 0x0005:
				return "GL_TRIANGLE_STRIP";
			case 0x0006:
				return "GL_TRIANGLE_FAN";
			case 0x0007:
				return "GL_QUADS";
			case 0x0008:
				return "GL_QUAD_STRIP";
			case 0x0009:
				return "GL_POLYGON";
			case 0x0100:
				return "GL_ACCUM";
			case 0x0101:
				return "GL_LOAD";
			case 0x0102:
				return "GL_RETURN";
			case 0x0103:
				return "GL_MULT";
			case 0x0104:
				return "GL_ADD";
			case 0x0200:
				return "GL_NEVER";
			case 0x0201:
				return "GL_LESS";
			case 0x0202:
				return "GL_EQUAL";
			case 0x0203:
				return "GL_LEQUAL";
			case 0x0204:
				return "GL_GREATER";
			case 0x0205:
				return "GL_NOTEQUAL";
			case 0x0206:
				return "GL_GEQUAL";
			case 0x0207:
				return "GL_ALWAYS";
			case 0x0300:
				return "GL_SRC_COLOR";
			case 0x0301:
				return "GL_ONE_MINUS_SRC_COLOR";
			case 0x0302:
				return "GL_SRC_ALPHA";
			case 0x0303:
				return "GL_ONE_MINUS_SRC_ALPHA";
			case 0x0304:
				return "GL_DST_ALPHA";
			case 0x0305:
				return "GL_ONE_MINUS_DST_ALPHA";
			case 0x0306:
				return "GL_DST_COLOR";
			case 0x0307:
				return "GL_ONE_MINUS_DST_COLOR";
			case 0x0308:
				return "GL_SRC_ALPHA_SATURATE";
			case 0x0400:
				return "GL_FRONT_LEFT";
			case 0x0401:
				return "GL_FRONT_RIGHT";
			case 0x0402:
				return "GL_BACK_LEFT";
			case 0x0403:
				return "GL_BACK_RIGHT";
			case 0x0404:
				return "GL_FRONT";
			case 0x0405:
				return "GL_BACK";
			case 0x0406:
				return "GL_LEFT";
			case 0x0407:
				return "GL_RIGHT";
			case 0x0408:
				return "GL_FRONT_AND_BACK";
			case 0x0409:
				return "GL_AUX0";
			case 0x040A:
				return "GL_AUX1";
			case 0x040B:
				return "GL_AUX2";
			case 0x040C:
				return "GL_AUX3";
			case 0x0500:
				return "GL_INVALID_ENUM";
			case 0x0501:
				return "GL_INVALID_VALUE";
			case 0x0502:
				return "GL_INVALID_OPERATION";
			case 0x0503:
				return "GL_STACK_OVERFLOW";
			case 0x0504:
				return "GL_STACK_UNDERFLOW";
			case 0x0505:
				return "GL_OUT_OF_MEMORY";
			case 0x0600:
				return "GL_2D";
			case 0x0601:
				return "GL_3D";
			case 0x0602:
				return "GL_3D_COLOR";
			case 0x0603:
				return "GL_3D_COLOR_TEXTURE";
			case 0x0604:
				return "GL_4D_COLOR_TEXTURE";
			case 0x0700:
				return "GL_PASS_THROUGH_TOKEN";
			case 0x0701:
				return "GL_POINT_TOKEN";
			case 0x0702:
				return "GL_LINE_TOKEN";
			case 0x0703:
				return "GL_POLYGON_TOKEN";
			case 0x0704:
				return "GL_BITMAP_TOKEN";
			case 0x0705:
				return "GL_DRAW_PIXEL_TOKEN";
			case 0x0706:
				return "GL_COPY_PIXEL_TOKEN";
			case 0x0707:
				return "GL_LINE_RESET_TOKEN";
			case 0x0800:
				return "GL_EXP";
			case 0x0801:
				return "GL_EXP2";
			case 0x0900:
				return "GL_CW";
			case 0x0901:
				return "GL_CCW";
			case 0x0A00:
				return "GL_COEFF";
			case 0x0A01:
				return "GL_ORDER";
			case 0x0A02:
				return "GL_DOMAIN";
			case 0x0B00:
				return "GL_CURRENT_COLOR";
			case 0x0B01:
				return "GL_CURRENT_INDEX";
			case 0x0B02:
				return "GL_CURRENT_NORMAL";
			case 0x0B03:
				return "GL_CURRENT_TEXTURE_COORDS";
			case 0x0B04:
				return "GL_CURRENT_RASTER_COLOR";
			case 0x0B05:
				return "GL_CURRENT_RASTER_INDEX";
			case 0x0B06:
				return "GL_CURRENT_RASTER_TEXTURE_COORDS";
			case 0x0B07:
				return "GL_CURRENT_RASTER_POSITION";
			case 0x0B08:
				return "GL_CURRENT_RASTER_POSITION_VALID";
			case 0x0B09:
				return "GL_CURRENT_RASTER_DISTANCE";
			case 0x0B10:
				return "GL_POINT_SMOOTH";
			case 0x0B11:
				return "GL_POINT_SIZE";
			case 0x0B12:
				return "GL_POINT_SIZE_RANGE";
			case 0x0B13:
				return "GL_POINT_SIZE_GRANULARITY";
			case 0x0B20:
				return "GL_LINE_SMOOTH";
			case 0x0B21:
				return "GL_LINE_WIDTH";
			case 0x0B22:
				return "GL_LINE_WIDTH_RANGE";
			case 0x0B23:
				return "GL_LINE_WIDTH_GRANULARITY";
			case 0x0B24:
				return "GL_LINE_STIPPLE";
			case 0x0B25:
				return "GL_LINE_STIPPLE_PATTERN";
			case 0x0B26:
				return "GL_LINE_STIPPLE_REPEAT";
			case 0x0B30:
				return "GL_LIST_MODE";
			case 0x0B31:
				return "GL_MAX_LIST_NESTING";
			case 0x0B32:
				return "GL_LIST_BASE";
			case 0x0B33:
				return "GL_LIST_INDEX";
			case 0x0B40:
				return "GL_POLYGON_MODE";
			case 0x0B41:
				return "GL_POLYGON_SMOOTH";
			case 0x0B42:
				return "GL_POLYGON_STIPPLE";
			case 0x0B43:
				return "GL_EDGE_FLAG";
			case 0x0B44:
				return "GL_CULL_FACE";
			case 0x0B45:
				return "GL_CULL_FACE_MODE";
			case 0x0B46:
				return "GL_FRONT_FACE";
			case 0x0B50:
				return "GL_LIGHTING";
			case 0x0B51:
				return "GL_LIGHT_MODEL_LOCAL_VIEWER";
			case 0x0B52:
				return "GL_LIGHT_MODEL_TWO_SIDE";
			case 0x0B53:
				return "GL_LIGHT_MODEL_AMBIENT";
			case 0x0B54:
				return "GL_SHADE_MODEL";
			case 0x0B55:
				return "GL_COLOR_MATERIAL_FACE";
			case 0x0B56:
				return "GL_COLOR_MATERIAL_PARAMETER";
			case 0x0B57:
				return "GL_COLOR_MATERIAL";
			case 0x0B60:
				return "GL_FOG";
			case 0x0B61:
				return "GL_FOG_INDEX";
			case 0x0B62:
				return "GL_FOG_DENSITY";
			case 0x0B63:
				return "GL_FOG_START";
			case 0x0B64:
				return "GL_FOG_END";
			case 0x0B65:
				return "GL_FOG_MODE";
			case 0x0B66:
				return "GL_FOG_COLOR";
			case 0x0B70:
				return "GL_DEPTH_RANGE";
			case 0x0B71:
				return "GL_DEPTH_TEST";
			case 0x0B72:
				return "GL_DEPTH_WRITEMASK";
			case 0x0B73:
				return "GL_DEPTH_CLEAR_VALUE";
			case 0x0B74:
				return "GL_DEPTH_FUNC";
			case 0x0B80:
				return "GL_ACCUM_CLEAR_VALUE";
			case 0x0B90:
				return "GL_STENCIL_TEST";
			case 0x0B91:
				return "GL_STENCIL_CLEAR_VALUE";
			case 0x0B92:
				return "GL_STENCIL_FUNC";
			case 0x0B93:
				return "GL_STENCIL_VALUE_MASK";
			case 0x0B94:
				return "GL_STENCIL_FAIL";
			case 0x0B95:
				return "GL_STENCIL_PASS_DEPTH_FAIL";
			case 0x0B96:
				return "GL_STENCIL_PASS_DEPTH_PASS";
			case 0x0B97:
				return "GL_STENCIL_REF";
			case 0x0B98:
				return "GL_STENCIL_WRITEMASK";
			case 0x0BA0:
				return "GL_MATRIX_MODE";
			case 0x0BA1:
				return "GL_NORMALIZE";
			case 0x0BA2:
				return "GL_VIEWPORT";
			case 0x0BA3:
				return "GL_MODELVIEW_STACK_DEPTH";
			case 0x0BA4:
				return "GL_PROJECTION_STACK_DEPTH";
			case 0x0BA5:
				return "GL_TEXTURE_STACK_DEPTH";
			case 0x0BA6:
				return "GL_MODELVIEW_MATRIX";
			case 0x0BA7:
				return "GL_PROJECTION_MATRIX";
			case 0x0BA8:
				return "GL_TEXTURE_MATRIX";
			case 0x0BB0:
				return "GL_ATTRIB_STACK_DEPTH";
			case 0x0BB1:
				return "GL_CLIENT_ATTRIB_STACK_DEPTH";
			case 0x0BC0:
				return "GL_ALPHA_TEST";
			case 0x0BC1:
				return "GL_ALPHA_TEST_FUNC";
			case 0x0BC2:
				return "GL_ALPHA_TEST_REF";
			case 0x0BD0:
				return "GL_DITHER";
			case 0x0BE0:
				return "GL_BLEND_DST";
			case 0x0BE1:
				return "GL_BLEND_SRC";
			case 0x0BE2:
				return "GL_BLEND";
			case 0x0BF0:
				return "GL_LOGIC_OP_MODE";
			case 0x0BF1:
				return "GL_INDEX_LOGIC_OP";
			case 0x0BF2:
				return "GL_COLOR_LOGIC_OP";
			case 0x0C00:
				return "GL_AUX_BUFFERS";
			case 0x0C01:
				return "GL_DRAW_BUFFER";
			case 0x0C02:
				return "GL_READ_BUFFER";
			case 0x0C10:
				return "GL_SCISSOR_BOX";
			case 0x0C11:
				return "GL_SCISSOR_TEST";
			case 0x0C20:
				return "GL_INDEX_CLEAR_VALUE";
			case 0x0C21:
				return "GL_INDEX_WRITEMASK";
			case 0x0C22:
				return "GL_COLOR_CLEAR_VALUE";
			case 0x0C23:
				return "GL_COLOR_WRITEMASK";
			case 0x0C30:
				return "GL_INDEX_MODE";
			case 0x0C31:
				return "GL_RGBA_MODE";
			case 0x0C32:
				return "GL_DOUBLEBUFFER";
			case 0x0C33:
				return "GL_STEREO";
			case 0x0C40:
				return "GL_RENDER_MODE";
			case 0x0C50:
				return "GL_PERSPECTIVE_CORRECTION_HINT";
			case 0x0C51:
				return "GL_POINT_SMOOTH_HINT";
			case 0x0C52:
				return "GL_LINE_SMOOTH_HINT";
			case 0x0C53:
				return "GL_POLYGON_SMOOTH_HINT";
			case 0x0C54:
				return "GL_FOG_HINT";
			case 0x0C60:
				return "GL_TEXTURE_GEN_S";
			case 0x0C61:
				return "GL_TEXTURE_GEN_T";
			case 0x0C62:
				return "GL_TEXTURE_GEN_R";
			case 0x0C63:
				return "GL_TEXTURE_GEN_Q";
			case 0x0C70:
				return "GL_PIXEL_MAP_I_TO_I";
			case 0x0C71:
				return "GL_PIXEL_MAP_S_TO_S";
			case 0x0C72:
				return "GL_PIXEL_MAP_I_TO_R";
			case 0x0C73:
				return "GL_PIXEL_MAP_I_TO_G";
			case 0x0C74:
				return "GL_PIXEL_MAP_I_TO_B";
			case 0x0C75:
				return "GL_PIXEL_MAP_I_TO_A";
			case 0x0C76:
				return "GL_PIXEL_MAP_R_TO_R";
			case 0x0C77:
				return "GL_PIXEL_MAP_G_TO_G";
			case 0x0C78:
				return "GL_PIXEL_MAP_B_TO_B";
			case 0x0C79:
				return "GL_PIXEL_MAP_A_TO_A";
			case 0x0CB0:
				return "GL_PIXEL_MAP_I_TO_I_SIZE";
			case 0x0CB1:
				return "GL_PIXEL_MAP_S_TO_S_SIZE";
			case 0x0CB2:
				return "GL_PIXEL_MAP_I_TO_R_SIZE";
			case 0x0CB3:
				return "GL_PIXEL_MAP_I_TO_G_SIZE";
			case 0x0CB4:
				return "GL_PIXEL_MAP_I_TO_B_SIZE";
			case 0x0CB5:
				return "GL_PIXEL_MAP_I_TO_A_SIZE";
			case 0x0CB6:
				return "GL_PIXEL_MAP_R_TO_R_SIZE";
			case 0x0CB7:
				return "GL_PIXEL_MAP_G_TO_G_SIZE";
			case 0x0CB8:
				return "GL_PIXEL_MAP_B_TO_B_SIZE";
			case 0x0CB9:
				return "GL_PIXEL_MAP_A_TO_A_SIZE";
			case 0x0CF0:
				return "GL_UNPACK_SWAP_BYTES";
			case 0x0CF1:
				return "GL_UNPACK_LSB_FIRST";
			case 0x0CF2:
				return "GL_UNPACK_ROW_LENGTH";
			case 0x0CF3:
				return "GL_UNPACK_SKIP_ROWS";
			case 0x0CF4:
				return "GL_UNPACK_SKIP_PIXELS";
			case 0x0CF5:
				return "GL_UNPACK_ALIGNMENT";
			case 0x0D00:
				return "GL_PACK_SWAP_BYTES";
			case 0x0D01:
				return "GL_PACK_LSB_FIRST";
			case 0x0D02:
				return "GL_PACK_ROW_LENGTH";
			case 0x0D03:
				return "GL_PACK_SKIP_ROWS";
			case 0x0D04:
				return "GL_PACK_SKIP_PIXELS";
			case 0x0D05:
				return "GL_PACK_ALIGNMENT";
			case 0x0D10:
				return "GL_MAP_COLOR";
			case 0x0D11:
				return "GL_MAP_STENCIL";
			case 0x0D12:
				return "GL_INDEX_SHIFT";
			case 0x0D13:
				return "GL_INDEX_OFFSET";
			case 0x0D14:
				return "GL_RED_SCALE";
			case 0x0D15:
				return "GL_RED_BIAS";
			case 0x0D16:
				return "GL_ZOOM_X";
			case 0x0D17:
				return "GL_ZOOM_Y";
			case 0x0D18:
				return "GL_GREEN_SCALE";
			case 0x0D19:
				return "GL_GREEN_BIAS";
			case 0x0D1A:
				return "GL_BLUE_SCALE";
			case 0x0D1B:
				return "GL_BLUE_BIAS";
			case 0x0D1C:
				return "GL_ALPHA_SCALE";
			case 0x0D1D:
				return "GL_ALPHA_BIAS";
			case 0x0D1E:
				return "GL_DEPTH_SCALE";
			case 0x0D1F:
				return "GL_DEPTH_BIAS";
			case 0x0D30:
				return "GL_MAX_EVAL_ORDER";
			case 0x0D31:
				return "GL_MAX_LIGHTS";
			case 0x0D32:
				return "GL_MAX_CLIP_PLANES";
			case 0x0D33:
				return "GL_MAX_TEXTURE_SIZE";
			case 0x0D34:
				return "GL_MAX_PIXEL_MAP_TABLE";
			case 0x0D35:
				return "GL_MAX_ATTRIB_STACK_DEPTH";
			case 0x0D36:
				return "GL_MAX_MODELVIEW_STACK_DEPTH";
			case 0x0D37:
				return "GL_MAX_NAME_STACK_DEPTH";
			case 0x0D38:
				return "GL_MAX_PROJECTION_STACK_DEPTH";
			case 0x0D39:
				return "GL_MAX_TEXTURE_STACK_DEPTH";
			case 0x0D3A:
				return "GL_MAX_VIEWPORT_DIMS";
			case 0x0D3B:
				return "GL_MAX_CLIENT_ATTRIB_STACK_DEPTH";
			case 0x0D50:
				return "GL_SUBPIXEL_BITS";
			case 0x0D51:
				return "GL_INDEX_BITS";
			case 0x0D52:
				return "GL_RED_BITS";
			case 0x0D53:
				return "GL_GREEN_BITS";
			case 0x0D54:
				return "GL_BLUE_BITS";
			case 0x0D55:
				return "GL_ALPHA_BITS";
			case 0x0D56:
				return "GL_DEPTH_BITS";
			case 0x0D57:
				return "GL_STENCIL_BITS";
			case 0x0D58:
				return "GL_ACCUM_RED_BITS";
			case 0x0D59:
				return "GL_ACCUM_GREEN_BITS";
			case 0x0D5A:
				return "GL_ACCUM_BLUE_BITS";
			case 0x0D5B:
				return "GL_ACCUM_ALPHA_BITS";
			case 0x0D70:
				return "GL_NAME_STACK_DEPTH";
			case 0x0D80:
				return "GL_AUTO_NORMAL";
			case 0x0D90:
				return "GL_MAP1_COLOR_4";
			case 0x0D91:
				return "GL_MAP1_INDEX";
			case 0x0D92:
				return "GL_MAP1_NORMAL";
			case 0x0D93:
				return "GL_MAP1_TEXTURE_COORD_1";
			case 0x0D94:
				return "GL_MAP1_TEXTURE_COORD_2";
			case 0x0D95:
				return "GL_MAP1_TEXTURE_COORD_3";
			case 0x0D96:
				return "GL_MAP1_TEXTURE_COORD_4";
			case 0x0D97:
				return "GL_MAP1_VERTEX_3";
			case 0x0D98:
				return "GL_MAP1_VERTEX_4";
			case 0x0DB0:
				return "GL_MAP2_COLOR_4";
			case 0x0DB1:
				return "GL_MAP2_INDEX";
			case 0x0DB2:
				return "GL_MAP2_NORMAL";
			case 0x0DB3:
				return "GL_MAP2_TEXTURE_COORD_1";
			case 0x0DB4:
				return "GL_MAP2_TEXTURE_COORD_2";
			case 0x0DB5:
				return "GL_MAP2_TEXTURE_COORD_3";
			case 0x0DB6:
				return "GL_MAP2_TEXTURE_COORD_4";
			case 0x0DB7:
				return "GL_MAP2_VERTEX_3";
			case 0x0DB8:
				return "GL_MAP2_VERTEX_4";
			case 0x0DD0:
				return "GL_MAP1_GRID_DOMAIN";
			case 0x0DD1:
				return "GL_MAP1_GRID_SEGMENTS";
			case 0x0DD2:
				return "GL_MAP2_GRID_DOMAIN";
			case 0x0DD3:
				return "GL_MAP2_GRID_SEGMENTS";
			case 0x0DE0:
				return "GL_TEXTURE_1D";
			case 0x0DE1:
				return "GL_TEXTURE_2D";
			case 0x0DF0:
				return "GL_FEEDBACK_BUFFER_POINTER";
			case 0x0DF1:
				return "GL_FEEDBACK_BUFFER_SIZE";
			case 0x0DF2:
				return "GL_FEEDBACK_BUFFER_TYPE";
			case 0x0DF3:
				return "GL_SELECTION_BUFFER_POINTER";
			case 0x0DF4:
				return "GL_SELECTION_BUFFER_SIZE";
			case 0x1000:
				return "GL_TEXTURE_WIDTH";
			case 0x1001:
				return "GL_TEXTURE_HEIGHT";
			case 0x1003:
				return "GL_TEXTURE_INTERNAL_FORMAT";
			case 0x1004:
				return "GL_TEXTURE_BORDER_COLOR";
			case 0x1005:
				return "GL_TEXTURE_BORDER";
			case 0x1100:
				return "GL_DONT_CARE";
			case 0x1101:
				return "GL_FASTEST";
			case 0x1102:
				return "GL_NICEST";
			case 0x1200:
				return "GL_AMBIENT";
			case 0x1201:
				return "GL_DIFFUSE";
			case 0x1202:
				return "GL_SPECULAR";
			case 0x1203:
				return "GL_POSITION";
			case 0x1204:
				return "GL_SPOT_DIRECTION";
			case 0x1205:
				return "GL_SPOT_EXPONENT";
			case 0x1206:
				return "GL_SPOT_CUTOFF";
			case 0x1207:
				return "GL_CONSTANT_ATTENUATION";
			case 0x1208:
				return "GL_LINEAR_ATTENUATION";
			case 0x1209:
				return "GL_QUADRATIC_ATTENUATION";
			case 0x1300:
				return "GL_COMPILE";
			case 0x1301:
				return "GL_COMPILE_AND_EXECUTE";
			case 0x1400:
				return "GL_BYTE";
			case 0x1401:
				return "GL_UNSIGNED_BYTE";
			case 0x1402:
				return "GL_SHORT";
			case 0x1403:
				return "GL_UNSIGNED_SHORT";
			case 0x1404:
				return "GL_INT";
			case 0x1405:
				return "GL_UNSIGNED_INT";
			case 0x1406:
				return "GL_FLOAT";
			case 0x1407:
				return "GL_2_BYTES";
			case 0x1408:
				return "GL_3_BYTES";
			case 0x1409:
				return "GL_4_BYTES";
			case 0x140A:
				return "GL_DOUBLE";
			case 0x1500:
				return "GL_CLEAR";
			case 0x1501:
				return "GL_AND";
			case 0x1502:
				return "GL_AND_REVERSE";
			case 0x1503:
				return "GL_COPY";
			case 0x1504:
				return "GL_AND_INVERTED";
			case 0x1505:
				return "GL_NOOP";
			case 0x1506:
				return "GL_XOR";
			case 0x1507:
				return "GL_OR";
			case 0x1508:
				return "GL_NOR";
			case 0x1509:
				return "GL_EQUIV";
			case 0x150A:
				return "GL_INVERT";
			case 0x150B:
				return "GL_OR_REVERSE";
			case 0x150C:
				return "GL_COPY_INVERTED";
			case 0x150D:
				return "GL_OR_INVERTED";
			case 0x150E:
				return "GL_NAND";
			case 0x150F:
				return "GL_SET";
			case 0x1600:
				return "GL_EMISSION";
			case 0x1601:
				return "GL_SHININESS";
			case 0x1602:
				return "GL_AMBIENT_AND_DIFFUSE";
			case 0x1603:
				return "GL_COLOR_INDEXES";
			case 0x1700:
				return "GL_MODELVIEW";
			case 0x1701:
				return "GL_PROJECTION";
			case 0x1702:
				return "GL_TEXTURE";
			case 0x1800:
				return "GL_COLOR";
			case 0x1801:
				return "GL_DEPTH";
			case 0x1802:
				return "GL_STENCIL";
			case 0x1900:
				return "GL_COLOR_INDEX";
			case 0x1901:
				return "GL_STENCIL_INDEX";
			case 0x1902:
				return "GL_DEPTH_COMPONENT";
			case 0x1903:
				return "GL_RED";
			case 0x1904:
				return "GL_GREEN";
			case 0x1905:
				return "GL_BLUE";
			case 0x1906:
				return "GL_ALPHA";
			case 0x1907:
				return "GL_RGB";
			case 0x1908:
				return "GL_RGBA";
			case 0x1909:
				return "GL_LUMINANCE";
			case 0x190A:
				return "GL_LUMINANCE_ALPHA";
			case 0x1A00:
				return "GL_BITMAP";
			case 0x1B00:
				return "GL_POINT";
			case 0x1B01:
				return "GL_LINE";
			case 0x1B02:
				return "GL_FILL";
			case 0x1C00:
				return "GL_RENDER";
			case 0x1C01:
				return "GL_FEEDBACK";
			case 0x1C02:
				return "GL_SELECT";
			case 0x1D00:
				return "GL_FLAT";
			case 0x1D01:
				return "GL_SMOOTH";
			case 0x1E00:
				return "GL_KEEP";
			case 0x1E01:
				return "GL_REPLACE";
			case 0x1E02:
				return "GL_INCR";
			case 0x1E03:
				return "GL_DECR";
			case 0x1F00:
				return "GL_VENDOR";
			case 0x1F01:
				return "GL_RENDERER";
			case 0x1F02:
				return "GL_VERSION";
			case 0x1F03:
				return "GL_EXTENSIONS";
			case 0x2000:
				return "GL_S";
			case 0x2001:
				return "GL_T";
			case 0x2002:
				return "GL_R";
			case 0x2003:
				return "GL_Q";
			case 0x2100:
				return "GL_MODULATE";
			case 0x2101:
				return "GL_DECAL";
			case 0x2200:
				return "GL_TEXTURE_ENV_MODE";
			case 0x2201:
				return "GL_TEXTURE_ENV_COLOR";
			case 0x2300:
				return "GL_TEXTURE_ENV";
			case 0x2400:
				return "GL_EYE_LINEAR";
			case 0x2401:
				return "GL_OBJECT_LINEAR";
			case 0x2402:
				return "GL_SPHERE_MAP";
			case 0x2500:
				return "GL_TEXTURE_GEN_MODE";
			case 0x2501:
				return "GL_OBJECT_PLANE";
			case 0x2502:
				return "GL_EYE_PLANE";
			case 0x2600:
				return "GL_NEAREST";
			case 0x2601:
				return "GL_LINEAR";
			case 0x2700:
				return "GL_NEAREST_MIPMAP_NEAREST";
			case 0x2701:
				return "GL_LINEAR_MIPMAP_NEAREST";
			case 0x2702:
				return "GL_NEAREST_MIPMAP_LINEAR";
			case 0x2703:
				return "GL_LINEAR_MIPMAP_LINEAR";
			case 0x2800:
				return "GL_TEXTURE_MAG_FILTER";
			case 0x2801:
				return "GL_TEXTURE_MIN_FILTER";
			case 0x2802:
				return "GL_TEXTURE_WRAP_S";
			case 0x2803:
				return "GL_TEXTURE_WRAP_T";
			case 0x2900:
				return "GL_CLAMP";
			case 0x2901:
				return "GL_REPEAT";
			case 0x2A00:
				return "GL_POLYGON_OFFSET_UNITS";
			case 0x2A01:
				return "GL_POLYGON_OFFSET_POINT";
			case 0x2A02:
				return "GL_POLYGON_OFFSET_LINE";
			case 0x2A10:
				return "GL_R3_G3_B2";
			case 0x2A20:
				return "GL_V2F";
			case 0x2A21:
				return "GL_V3F";
			case 0x2A22:
				return "GL_C4UB_V2F";
			case 0x2A23:
				return "GL_C4UB_V3F";
			case 0x2A24:
				return "GL_C3F_V3F";
			case 0x2A25:
				return "GL_N3F_V3F";
			case 0x2A26:
				return "GL_C4F_N3F_V3F";
			case 0x2A27:
				return "GL_T2F_V3F";
			case 0x2A28:
				return "GL_T4F_V4F";
			case 0x2A29:
				return "GL_T2F_C4UB_V3F";
			case 0x2A2A:
				return "GL_T2F_C3F_V3F";
			case 0x2A2B:
				return "GL_T2F_N3F_V3F";
			case 0x2A2C:
				return "GL_T2F_C4F_N3F_V3F";
			case 0x2A2D:
				return "GL_T4F_C4F_N3F_V4F";
			case 0x3000:
				return "GL_CLIP_PLANE0";
			case 0x3001:
				return "GL_CLIP_PLANE1";
			case 0x3002:
				return "GL_CLIP_PLANE2";
			case 0x3003:
				return "GL_CLIP_PLANE3";
			case 0x3004:
				return "GL_CLIP_PLANE4";
			case 0x3005:
				return "GL_CLIP_PLANE5";
			case 0x4000:
				return "GL_LIGHT0";
			case 0x4001:
				return "GL_LIGHT1";
			case 0x4002:
				return "GL_LIGHT2";
			case 0x4003:
				return "GL_LIGHT3";
			case 0x4004:
				return "GL_LIGHT4";
			case 0x4005:
				return "GL_LIGHT5";
			case 0x4006:
				return "GL_LIGHT6";
			case 0x4007:
				return "GL_LIGHT7";
			case 0x8037:
				return "GL_POLYGON_OFFSET_FILL";
			case 0x8038:
				return "GL_POLYGON_OFFSET_FACTOR";
			case 0x803B:
				return "GL_ALPHA4";
			case 0x803C:
				return "GL_ALPHA8";
			case 0x803D:
				return "GL_ALPHA12";
			case 0x803E:
				return "GL_ALPHA16";
			case 0x803F:
				return "GL_LUMINANCE4";
			case 0x8040:
				return "GL_LUMINANCE8";
			case 0x8041:
				return "GL_LUMINANCE12";
			case 0x8042:
				return "GL_LUMINANCE16";
			case 0x8043:
				return "GL_LUMINANCE4_ALPHA4";
			case 0x8044:
				return "GL_LUMINANCE6_ALPHA2";
			case 0x8045:
				return "GL_LUMINANCE8_ALPHA8";
			case 0x8046:
				return "GL_LUMINANCE12_ALPHA4";
			case 0x8047:
				return "GL_LUMINANCE12_ALPHA12";
			case 0x8048:
				return "GL_LUMINANCE16_ALPHA16";
			case 0x8049:
				return "GL_INTENSITY";
			case 0x804A:
				return "GL_INTENSITY4";
			case 0x804B:
				return "GL_INTENSITY8";
			case 0x804C:
				return "GL_INTENSITY12";
			case 0x804D:
				return "GL_INTENSITY16";
			case 0x804F:
				return "GL_RGB4";
			case 0x8050:
				return "GL_RGB5";
			case 0x8051:
				return "GL_RGB8";
			case 0x8052:
				return "GL_RGB10";
			case 0x8053:
				return "GL_RGB12";
			case 0x8054:
				return "GL_RGB16";
			case 0x8055:
				return "GL_RGBA2";
			case 0x8056:
				return "GL_RGBA4";
			case 0x8057:
				return "GL_RGB5_A1";
			case 0x8058:
				return "GL_RGBA8";
			case 0x8059:
				return "GL_RGB10_A2";
			case 0x805A:
				return "GL_RGBA12";
			case 0x805B:
				return "GL_RGBA16";
			case 0x805C:
				return "GL_TEXTURE_RED_SIZE";
			case 0x805D:
				return "GL_TEXTURE_GREEN_SIZE";
			case 0x805E:
				return "GL_TEXTURE_BLUE_SIZE";
			case 0x805F:
				return "GL_TEXTURE_ALPHA_SIZE";
			case 0x8060:
				return "GL_TEXTURE_LUMINANCE_SIZE";
			case 0x8061:
				return "GL_TEXTURE_INTENSITY_SIZE";
			case 0x8063:
				return "GL_PROXY_TEXTURE_1D";
			case 0x8064:
				return "GL_PROXY_TEXTURE_2D";
			case 0x8066:
				return "GL_TEXTURE_PRIORITY";
			case 0x8067:
				return "GL_TEXTURE_RESIDENT";
			case 0x8068:
				return "GL_TEXTURE_BINDING_1D";
			case 0x8069:
				return "GL_TEXTURE_BINDING_2D";
			case 0x8074:
				return "GL_VERTEX_ARRAY";
			case 0x8075:
				return "GL_NORMAL_ARRAY";
			case 0x8076:
				return "GL_COLOR_ARRAY";
			case 0x8077:
				return "GL_INDEX_ARRAY";
			case 0x8078:
				return "GL_TEXTURE_COORD_ARRAY";
			case 0x8079:
				return "GL_EDGE_FLAG_ARRAY";
			case 0x807A:
				return "GL_VERTEX_ARRAY_SIZE";
			case 0x807B:
				return "GL_VERTEX_ARRAY_TYPE";
			case 0x807C:
				return "GL_VERTEX_ARRAY_STRIDE";
			case 0x807E:
				return "GL_NORMAL_ARRAY_TYPE";
			case 0x807F:
				return "GL_NORMAL_ARRAY_STRIDE";
			case 0x8081:
				return "GL_COLOR_ARRAY_SIZE";
			case 0x8082:
				return "GL_COLOR_ARRAY_TYPE";
			case 0x8083:
				return "GL_COLOR_ARRAY_STRIDE";
			case 0x8085:
				return "GL_INDEX_ARRAY_TYPE";
			case 0x8086:
				return "GL_INDEX_ARRAY_STRIDE";
			case 0x8088:
				return "GL_TEXTURE_COORD_ARRAY_SIZE";
			case 0x8089:
				return "GL_TEXTURE_COORD_ARRAY_TYPE";
			case 0x808A:
				return "GL_TEXTURE_COORD_ARRAY_STRIDE";
			case 0x808C:
				return "GL_EDGE_FLAG_ARRAY_STRIDE";
			case 0x808E:
				return "GL_VERTEX_ARRAY_POINTER";
			case 0x808F:
				return "GL_NORMAL_ARRAY_POINTER";
			case 0x8090:
				return "GL_COLOR_ARRAY_POINTER";
			case 0x8091:
				return "GL_INDEX_ARRAY_POINTER";
			case 0x8092:
				return "GL_TEXTURE_COORD_ARRAY_POINTER";
			case 0x8093:
				return "GL_EDGE_FLAG_ARRAY_POINTER";
			case 0x80D8:
				return "GL_COLOR_TABLE_FORMAT_EXT";
			case 0x80D9:
				return "GL_COLOR_TABLE_WIDTH_EXT";
			case 0x80DA:
				return "GL_COLOR_TABLE_RED_SIZE_EXT";
			case 0x80DB:
				return "GL_COLOR_TABLE_GREEN_SIZE_EXT";
			case 0x80DC:
				return "GL_COLOR_TABLE_BLUE_SIZE_EXT";
			case 0x80DD:
				return "GL_COLOR_TABLE_ALPHA_SIZE_EXT";
			case 0x80DE:
				return "GL_COLOR_TABLE_LUMINANCE_SIZE_EXT";
			case 0x80DF:
				return "GL_COLOR_TABLE_INTENSITY_SIZE_EXT";
			case 0x80E0:
				return "GL_BGR_EXT";
			case 0x80E1:
				return "GL_BGRA_EXT";
			case 0x80E2:
				return "GL_COLOR_INDEX1_EXT";
			case 0x80E3:
				return "GL_COLOR_INDEX2_EXT";
			case 0x80E4:
				return "GL_COLOR_INDEX4_EXT";
			case 0x80E5:
				return "GL_COLOR_INDEX8_EXT";
			case 0x80E6:
				return "GL_COLOR_INDEX12_EXT";
			case 0x80E7:
				return "GL_COLOR_INDEX16_EXT";
			default:
				return "*** Unknown OpenGL enum!! ***";
		}
}

} // namespace

namespace sdpgl
{

namespace implementation_private
{

void trace(const std::string Text)
{
	raw_trace("//*** " + Text + "\n");
}

void set_stream(std::ostream& Stream)
{
	g_output_stream = &Stream;
}

void push_trace(bool State)
{
	if(State)
		trace("SDPGL tracing enabled ...");
	else
		trace("SDPGL tracing disabled ...");

	g_output_stack.push(State);
}

void pop_trace()
{
	bool current = g_output_stack.top();

	if(current)
		trace("SDPGL tracing disabled ...");
	else
		trace("SDPGL tracing re-enabled ...");

	g_output_stack.pop();
}

void Accum(GLenum op,GLfloat value)
{
	begin_function_trace("glAccum", sdpGLenum(op) + "," + sdpGLfloat(value));
	glAccum(op,value);
	end_function_trace();
}

void AlphaFunc(GLenum func,GLclampf ref)
{
	begin_function_trace("glAlphaFunc", sdpGLenum(func) + "," + sdpGLclampf(ref));
	glAlphaFunc(func,ref);
	end_function_trace();
}

GLboolean AreTexturesResident(GLsizei n,const GLuint *textures,GLboolean *residences)
{
	begin_function_trace("glAreTexturesResident", sdpGLsizei(n));//,const GLuint *textures,GLboolean *residenc);
	GLboolean result = glAreTexturesResident(n,textures,residences);
	result_trace(sdpGLboolean(result));
	end_function_trace();
	return result;
}

void ArrayElement(GLint i)
{
	begin_function_trace("glArrayElement", sdpGLint(i));
	glArrayElement(i);
	end_function_trace();
}

void Begin(GLenum mode)
{
	begin_function_trace("glBegin", sdpGLenum(mode));
	glBegin(mode);
	end_function_trace();
}

void BindTexture(GLenum target,GLuint texture)
{
	begin_function_trace("glBindTexture", sdpGLenum(target) + "," + sdpGLuint(texture));
	glBindTexture(target,texture);
	end_function_trace();
}

void Bitmap(GLsizei width,GLsizei height,GLfloat xorig,GLfloat yorig,GLfloat xmove,GLfloat ymove,const GLubyte *bitmap)
{
	begin_function_trace("glBitmap", sdpGLsizei(width) + "," + sdpGLsizei(height) + "," + sdpGLfloat(xorig) + "," + sdpGLfloat(yorig) + "," + sdpGLfloat(xmove) + "," + sdpGLfloat(ymove));//,const GLubyte *bitmap",);
	glBitmap(width,height,xorig,yorig,xmove,ymove,bitmap);
	end_function_trace();
}

void BlendFunc(GLenum sfactor,GLenum dfactor)
{
	begin_function_trace("glBlendFunc", sdpGLenum(sfactor) + "," + sdpGLenum(dfactor));
	glBlendFunc(sfactor,dfactor);
	end_function_trace();
}

void CallList(GLuint list)
{
	begin_function_trace("glCallList", sdpGLuint(list));
	glCallList(list);
	end_function_trace();
}

void CallLists(GLsizei n,GLenum type,const GLvoid *lists)
{
	begin_function_trace("glCallLists", sdpGLsizei(n) + "," + sdpGLenum(type) + "," + sdpGLvoid(lists));
	glCallLists(n,type,lists);
	end_function_trace();
}

void Clear(GLbitfield mask)
{
	begin_function_trace("glClear", sdpGLbitfield(mask));
	glClear(mask);
	end_function_trace();
}

void ClearAccum(GLfloat red,GLfloat green,GLfloat blue,GLfloat alpha)
{
	begin_function_trace("glClearAccum", sdpGLfloat(red) + "," + sdpGLfloat(green) + "," + sdpGLfloat(blue) + "," + sdpGLfloat(alpha));
	glClearAccum(red,green,blue,alpha);
	end_function_trace();
}

void ClearColor(GLclampf red,GLclampf green,GLclampf blue,GLclampf alpha)
{
	begin_function_trace("glClearColor", sdpGLclampf(red) + "," + sdpGLclampf(green) + "," + sdpGLclampf(blue) + "," + sdpGLclampf(alpha));
	glClearColor(red,green,blue,alpha);
	end_function_trace();
}

void ClearDepth(GLclampd depth)
{
	begin_function_trace("glClearDepth", sdpGLclampd(depth));
	glClearDepth(depth);
	end_function_trace();
}

void ClearIndex(GLfloat c)
{
	begin_function_trace("glClearIndex", sdpGLfloat(c));
	glClearIndex(c);
	end_function_trace();
}

void ClearStencil(GLint s)
{
	begin_function_trace("glClearStencil", sdpGLint(s));
	glClearStencil(s);
	end_function_trace();
}

void ClipPlane(GLenum plane,const GLdouble *equation)
{
	begin_function_trace("glClipPlane", sdpGLenum(plane));
	glClipPlane(plane,equation);
	end_function_trace();
}

void Color3b(GLbyte red,GLbyte green,GLbyte blue)
{
	begin_function_trace("glColor3b", sdpGLbyte(red) + "," + sdpGLbyte(green) + "," + sdpGLbyte(blue));
	glColor3b(red,green,blue);
	end_function_trace();
}

void Color3bv(const GLbyte *v)
{
	begin_function_trace("glColor3bv", sdpGLbyte(v[0]) + "," + sdpGLbyte(v[1]) + "," + sdpGLbyte(v[2]));
	glColor3bv(v);
	end_function_trace();
}

void Color3d(GLdouble red,GLdouble green,GLdouble blue)
{
	begin_function_trace("glColor3d", sdpGLdouble(red) + "," + sdpGLdouble(green) + "," + sdpGLdouble(blue));
	glColor3d(red,green,blue);
	end_function_trace();
}

void Color3dv(const GLdouble *v)
{
	begin_function_trace("glColor3dv", sdpGLdouble(v[0]) + "," + sdpGLdouble(v[1]) + "," + sdpGLdouble(v[2]));
	glColor3dv(v);
	end_function_trace();
}

void Color3f(GLfloat red,GLfloat green,GLfloat blue)
{
	begin_function_trace("glColor3f", sdpGLfloat(red) + "," + sdpGLfloat(green) + "," + sdpGLfloat(blue));
	glColor3f(red,green,blue);
	end_function_trace();
}

void Color3fv(const GLfloat *v)
{
	begin_function_trace("glColor3fv", sdpGLfloat(v[0]) + "," + sdpGLfloat(v[2]) + "," + sdpGLfloat(v[3]));
	glColor3fv(v);
	end_function_trace();
}

void Color3i(GLint red,GLint green,GLint blue)
{
	begin_function_trace("glColor3i", sdpGLint(red) + "," + sdpGLint(green) + "," + sdpGLint(blue));
	glColor3i(red,green,blue);
	end_function_trace();
}

void Color3iv(const GLint *v)
{
	begin_function_trace("glColor3iv", ""); //,const GLint );
	glColor3iv(v);
	end_function_trace();
}

void Color3s(GLshort red,GLshort green,GLshort blue)
{
	begin_function_trace("glColor3s", sdpGLshort(red) + "," + sdpGLshort(green) + "," + sdpGLshort(blue));
	glColor3s(red,green,blue);
	end_function_trace();
}

void Color3sv(const GLshort *v)
{
	begin_function_trace("glColor3sv", ""); //,const GLshort );
	glColor3sv(v);
	end_function_trace();
}

void Color3ub(GLubyte red,GLubyte green,GLubyte blue)
{
	begin_function_trace("glColor3ub", sdpGLubyte(red) + "," + sdpGLubyte(green) + "," + sdpGLubyte(blue));
	glColor3ub(red,green,blue);
	end_function_trace();
}

void Color3ubv(const GLubyte *v)
{
	begin_function_trace("glColor3ubv", ""); //,const GLubyte );
	glColor3ubv(v);
	end_function_trace();
}

void Color3ui(GLuint red,GLuint green,GLuint blue)
{
	begin_function_trace("glColor3ui", sdpGLuint(red) + "," + sdpGLuint(green) + "," + sdpGLuint(blue));
	glColor3ui(red,green,blue);
	end_function_trace();
}

void Color3uiv(const GLuint *v)
{
	begin_function_trace("glColor3uiv", ""); //,const GLuint );
	glColor3uiv(v);
	end_function_trace();
}

void Color3us(GLushort red,GLushort green,GLushort blue)
{
	begin_function_trace("glColor3us", sdpGLushort(red) + "," + sdpGLushort(green) + "," + sdpGLushort(blue));
	glColor3us(red,green,blue);
	end_function_trace();
}

void Color3usv(const GLushort *v)
{
	begin_function_trace("glColor3usv", ""); //,const GLushort );
	glColor3usv(v);
	end_function_trace();
}

void Color4b(GLbyte red,GLbyte green,GLbyte blue,GLbyte alpha)
{
	begin_function_trace("glColor4b", sdpGLbyte(red) + "," + sdpGLbyte(green) + "," + sdpGLbyte(blue) + "," + sdpGLbyte(alpha));
	glColor4b(red,green,blue,alpha);
	end_function_trace();
}

void Color4bv(const GLbyte *v)
{
	begin_function_trace("glColor4bv", ""); //,const GLbyte );
	glColor4bv(v);
	end_function_trace();
}

void Color4d(GLdouble red,GLdouble green,GLdouble blue,GLdouble alpha)
{
	begin_function_trace("glColor4d", sdpGLdouble(red) + "," + sdpGLdouble(green) + "," + sdpGLdouble(blue) + "," + sdpGLdouble(alpha));
	glColor4d(red,green,blue,alpha);
	end_function_trace();
}

void Color4dv(const GLdouble *v)
{
	begin_function_trace("glColor4dv", sdpGLdouble(v[0]) + "," + sdpGLdouble(v[1]) + "," + sdpGLdouble(v[2]) + "," + sdpGLdouble(v[3]));
	glColor4dv(v);
	end_function_trace();
}

void Color4f(GLfloat red,GLfloat green,GLfloat blue,GLfloat alpha)
{
	begin_function_trace("glColor4f", sdpGLfloat(red) + "," + sdpGLfloat(green) + "," + sdpGLfloat(blue) + "," + sdpGLfloat(alpha));
	glColor4f(red,green,blue,alpha);
	end_function_trace();
}

void Color4fv(const GLfloat *v)
{
	begin_function_trace("glColor4fv", sdpGLfloat(v[0]) + "," + sdpGLfloat(v[1]) + "," + sdpGLfloat(v[2]) + "," + sdpGLfloat(v[3]));
	glColor4fv(v);
	end_function_trace();
}

void Color4i(GLint red,GLint green,GLint blue,GLint alpha)
{
	begin_function_trace("glColor4i", sdpGLint(red) + "," + sdpGLint(green) + "," + sdpGLint(blue) + "," + sdpGLint(alpha));
	glColor4i(red,green,blue,alpha);
	end_function_trace();
}

void Color4iv(const GLint *v)
{
	begin_function_trace("glColor4iv", ""); //,const GLint );
	glColor4iv(v);
	end_function_trace();
}

void Color4s(GLshort red,GLshort green,GLshort blue,GLshort alpha)
{
	begin_function_trace("glColor4s", sdpGLshort(red) + "," + sdpGLshort(green) + "," + sdpGLshort(blue) + "," + sdpGLshort(alpha));
	glColor4s(red,green,blue,alpha);
	end_function_trace();
}

void Color4sv(const GLshort *v)
{
	begin_function_trace("glColor4sv", ""); //,const GLshort );
	glColor4sv(v);
	end_function_trace();
}

void Color4ub(GLubyte red,GLubyte green,GLubyte blue,GLubyte alpha)
{
	begin_function_trace("glColor4ub", ""); //,GLubyte red,GLubyte green,GLubyte blue,GLubyte alpha);
	glColor4ub(red,green,blue,alpha);
	end_function_trace();
}

void Color4ubv(const GLubyte *v)
{
	begin_function_trace("glColor4ubv", ""); //,const GLubyte );
	glColor4ubv(v);
	end_function_trace();
}

void Color4ui(GLuint red,GLuint green,GLuint blue,GLuint alpha)
{
	begin_function_trace("glColor4ui", sdpGLuint(red) + "," + sdpGLuint(green) + "," + sdpGLuint(blue) + "," + sdpGLuint(alpha));
	glColor4ui(red,green,blue,alpha);
	end_function_trace();
}

void Color4uiv(const GLuint *v)
{
	begin_function_trace("glColor4uiv", ""); //,const GLuint );
	glColor4uiv(v);
	end_function_trace();
}

void Color4us(GLushort red,GLushort green,GLushort blue,GLushort alpha)
{
	begin_function_trace("glColor4us", ""); //,GLushort red,GLushort green,GLushort blue,GLushort alpha);
	glColor4us(red,green,blue,alpha);
	end_function_trace();
}

void Color4usv(const GLushort *v)
{
	begin_function_trace("glColor4usv", ""); //,const GLushort );
	glColor4usv(v);
	end_function_trace();
}

void ColorMask(GLboolean red,GLboolean green,GLboolean blue,GLboolean alpha)
{
	begin_function_trace("glColorMask", sdpGLboolean(red) + sdpGLboolean(green) + sdpGLboolean(blue) + sdpGLboolean(alpha));//);
	glColorMask(red,green,blue,alpha);
	end_function_trace();
}

void ColorMaterial(GLenum face,GLenum mode)
{
	begin_function_trace("glColorMaterial", sdpGLenum(face) + "," + sdpGLenum(mode));
	glColorMaterial(face,mode);
	end_function_trace();
}

void ColorPointer(GLint size,GLenum type,GLsizei stride,const GLvoid *pointer)
{
	begin_function_trace("glColorPointer", sdpGLint(size) + "," + sdpGLenum(type) + "," + sdpGLsizei(stride));//,const GLvoid *pointer");
	glColorPointer(size,type,stride,pointer);
	end_function_trace();
}

void CopyPixels(GLint x,GLint y,GLsizei width,GLsizei height,GLenum type)
{
	begin_function_trace("glCopyPixels", sdpGLint(x) + "," + sdpGLint(y) + "," + sdpGLsizei(width) + "," + sdpGLsizei(height) + "," + sdpGLenum(type));
	glCopyPixels(x,y,width,height,type);
	end_function_trace();
}

void CopyTexImage1D(GLenum target,GLint level,GLenum internalFormat,GLint x,GLint y,GLsizei width,GLint border)
{
	begin_function_trace("glCopyTexImage1D", sdpGLenum(target) + "," + sdpGLint(level) + "," + sdpGLenum(internalFormat) + "," + sdpGLint(x) + "," + sdpGLint(y));//,GLsizei width,GLint border",&);
	glCopyTexImage1D(target,level,internalFormat,x,y,width,border);
	end_function_trace();
}

void CopyTexImage2D(GLenum target,GLint level,GLenum internalFormat,GLint x,GLint y,GLsizei width,GLsizei height,GLint border)
{
	begin_function_trace("glCopyTexImage2D", sdpGLenum(target) + "," + sdpGLint(level) + "," + sdpGLenum(internalFormat) + "," + sdpGLint(x) + "," + sdpGLint(y));//,GLsizei width,GLsizei height,GLint border",&);
	glCopyTexImage2D(target,level,internalFormat,x,y,width,height,border);
	end_function_trace();
}

void CopyTexSubImage1D(GLenum target,GLint level,GLint xoffset,GLint x,GLint y,GLsizei width)
{
	begin_function_trace("glCopyTexSubImage1D", sdpGLenum(target) + "," + sdpGLint(level) + "," + sdpGLint(xoffset) + "," + sdpGLint(x) + "," + sdpGLint(y) + "," + sdpGLsizei(width));
	glCopyTexSubImage1D(target,level,xoffset,x,y,width);
	end_function_trace();
}

void CopyTexSubImage2D(GLenum target,GLint level,GLint xoffset,GLint yoffset,GLint x,GLint y,GLsizei width,GLsizei height)
{
	begin_function_trace("glCopyTexSubImage2D", sdpGLenum(target) + "," + sdpGLint(level) + "," + sdpGLint(xoffset) + "," + sdpGLint(yoffset) + "," + sdpGLint(x) + "," + sdpGLint(y) + "," + sdpGLsizei(width) + "," + sdpGLsizei(height));
	glCopyTexSubImage2D(target,level,xoffset,yoffset,x,y,width,height);
	end_function_trace();
}

void CullFace(GLenum mode)
{
	begin_function_trace("glCullFace", sdpGLenum(mode));
	glCullFace(mode);
	end_function_trace();
}

void DeleteLists(GLuint list,GLsizei range)
{
	begin_function_trace("glDeleteLists", sdpGLuint(list) + "," + sdpGLsizei(range));
	glDeleteLists(list,range);
	end_function_trace();
}

void DeleteTextures(GLsizei n,const GLuint *textures)
{
	begin_function_trace("glDeleteTextures", sdpGLsizei(n));//,const GLuint *textur);
	glDeleteTextures(n,textures);
	end_function_trace();
}

void DepthFunc(GLenum func)
{
	begin_function_trace("glDepthFunc", sdpGLenum(func));
	glDepthFunc(func);
	end_function_trace();
}

void DepthMask(GLboolean flag)
{
	begin_function_trace("glDepthMask", sdpGLboolean(flag));//");
	glDepthMask(flag);
	end_function_trace();
}

void DepthRange(GLclampd zNear,GLclampd zFar)
{
	begin_function_trace("glDepthRange", ""); //,GLclampd zNear,GLclampd zFar",);
	glDepthRange(zNear,zFar);
	end_function_trace();
}

void Disable(GLenum cap)
{
	begin_function_trace("glDisable", sdpGLenum(cap));
	glDisable(cap);
	end_function_trace();
}

void DisableClientState(GLenum array)
{
	begin_function_trace("glDisableClientState", sdpGLenum(array));
	glDisableClientState(array);
	end_function_trace();
}

void DrawArrays(GLenum mode,GLint first,GLsizei count)
{
	begin_function_trace("glDrawArrays", sdpGLenum(mode) + "," + sdpGLint(first) + "," + sdpGLsizei(count));
	glDrawArrays(mode,first,count);
	end_function_trace();
}

void DrawBuffer(GLenum mode)
{
	begin_function_trace("glDrawBuffer", sdpGLenum(mode));
	glDrawBuffer(mode);
	end_function_trace();
}

void DrawElements(GLenum mode,GLsizei count,GLenum type,const GLvoid *indices)
{
	begin_function_trace("glDrawElements", sdpGLenum(mode) + "," + sdpGLsizei(count) + "," + sdpGLenum(type));//,const GLvoid *indices");
	glDrawElements(mode,count,type,indices);
	end_function_trace();
}

void DrawPixels(GLsizei width,GLsizei height,GLenum format,GLenum type,const GLvoid *pixels)
{
	begin_function_trace("glDrawPixels", sdpGLsizei(width) + "," + sdpGLsizei(height) + "," + sdpGLenum(format) + "," + sdpGLenum(type));//,const GLvoid *pixels",);
	glDrawPixels(width,height,format,type,pixels);
	end_function_trace();
}

void EdgeFlag(GLboolean flag)
{
	begin_function_trace("glEdgeFlag", sdpGLboolean(flag));
	glEdgeFlag(flag);
	end_function_trace();
}

void EdgeFlagPointer(GLsizei stride, const GLboolean *pointer)
{
	begin_function_trace("glEdgeFlagPointer", sdpGLsizei(stride));//,const GLvoid *pointer",&);
	glEdgeFlagPointer(stride, pointer);
	end_function_trace();
}

void EdgeFlagv(const GLboolean *flag)
{
	begin_function_trace("glEdgeFlagv", ""); //,const GLboolean *flag");
	glEdgeFlagv(flag);
	end_function_trace();
}

void Enable(GLenum cap)
{
	begin_function_trace("glEnable", sdpGLenum(cap));
	glEnable(cap);
	end_function_trace();
}

void EnableClientState(GLenum array)
{
	begin_function_trace("glEnableClientState", sdpGLenum(array));
	glEnableClientState(array);
	end_function_trace();
}

void End()
{
	begin_function_trace("glEnd", "");
	glEnd();
	end_function_trace();
}

void EndList()
{
	begin_function_trace("glEndList", "");
	glEndList();
	end_function_trace();
}

void EvalCoord1d(GLdouble u)
{
	begin_function_trace("glEvalCoord1d", sdpGLdouble(u));
	glEvalCoord1d(u);
	end_function_trace();
}

void EvalCoord1dv(const GLdouble *u)
{
	begin_function_trace("glEvalCoord1dv", ""); //,const GLdouble );
	glEvalCoord1dv(u);
	end_function_trace();
}

void EvalCoord1f(GLfloat u)
{
	begin_function_trace("glEvalCoord1f", sdpGLfloat(u));
	glEvalCoord1f(u);
	end_function_trace();
}

void EvalCoord1fv(const GLfloat *u)
{
	begin_function_trace("glEvalCoord1fv", ""); //,const GLfloat );
	glEvalCoord1fv(u);
	end_function_trace();
}

void EvalCoord2d(GLdouble u,GLdouble v)
{
	begin_function_trace("glEvalCoord2d", sdpGLdouble(u) + "," + sdpGLdouble(v));
	glEvalCoord2d(u,v);
	end_function_trace();
}

void EvalCoord2dv(const GLdouble *u)
{
	begin_function_trace("glEvalCoord2dv", ""); //,const GLdouble );
	glEvalCoord2dv(u);
	end_function_trace();
}

void EvalCoord2f(GLfloat u,GLfloat v)
{
	begin_function_trace("glEvalCoord2f", sdpGLfloat(u) + "," + sdpGLfloat(v));
	glEvalCoord2f(u,v);
	end_function_trace();
}

void EvalCoord2fv(const GLfloat *u)
{
	begin_function_trace("glEvalCoord2fv", ""); //,const GLfloat );
	glEvalCoord2fv(u);
	end_function_trace();
}

void EvalMesh1(GLenum mode,GLint i1,GLint i2)
{
	begin_function_trace("glEvalMesh1", sdpGLenum(mode) + "," + sdpGLint(i1) + "," + sdpGLint(i2));
	glEvalMesh1(mode,i1,i2);
	end_function_trace();
}

void EvalMesh2(GLenum mode,GLint i1,GLint i2,GLint j1,GLint j2)
{
	begin_function_trace("glEvalMesh2", sdpGLenum(mode) + "," + sdpGLint(i1) + "," + sdpGLint(i2) + "," + sdpGLint(j1) + "," + sdpGLint(j2));
	glEvalMesh2(mode,i1,i2,j1,j2);
	end_function_trace();
}

void EvalPoint1(GLint i)
{
	begin_function_trace("glEvalPoint1", sdpGLint(i));
	glEvalPoint1(i);
	end_function_trace();
}

void EvalPoint2(GLint i,GLint j)
{
	begin_function_trace("glEvalPoint2", sdpGLint(i) + "," + sdpGLint(j));
	glEvalPoint2(i,j);
	end_function_trace();
}

void FeedbackBuffer(GLsizei size,GLenum type,GLfloat *buffer)
{
	begin_function_trace("glFeedbackBuffer", sdpGLsizei(size) + "," + sdpGLenum(type));//,GLfloat *buffer");
	glFeedbackBuffer(size,type,buffer);
	end_function_trace();
}

void Finish()
{
	begin_function_trace("glFinish", "");
	glFinish();
	end_function_trace();
}

void Flush()
{
	begin_function_trace("glFlush", "");
	glFlush();
	end_function_trace();
}

void Fogf(GLenum pname,GLfloat param)
{
	begin_function_trace("glFogf", sdpGLenum(pname) + "," + sdpGLfloat(param));
	glFogf(pname,param);
	end_function_trace();
}

void Fogfv(GLenum pname,const GLfloat *params)
{
	begin_function_trace("glFogfv", sdpGLenum(pname));//,const GLfloat *params",);
	glFogfv(pname,params);
	end_function_trace();
}

void Fogi(GLenum pname,GLint param)
{
	begin_function_trace("glFogi", sdpGLenum(pname) + "," + sdpGLint(param));
	glFogi(pname,param);
	end_function_trace();
}

void Fogiv(GLenum pname,const GLint *params)
{
	begin_function_trace("glFogiv", sdpGLenum(pname));//,const GLint *params",);
	glFogiv(pname,params);
	end_function_trace();
}

void FrontFace(GLenum mode)
{
	begin_function_trace("glFrontFace", sdpGLenum(mode));
	glFrontFace(mode);
	end_function_trace();
}

void Frustum(GLdouble left,GLdouble right,GLdouble bottom,GLdouble top,GLdouble zNear,GLdouble zFar)
{
	begin_function_trace("glFrustum", sdpGLdouble(left) + "," + sdpGLdouble(right) + "," + sdpGLdouble(bottom) + "," + sdpGLdouble(top) + "," + sdpGLdouble(zNear) + "," + sdpGLdouble(zFar));
	glFrustum(left,right,bottom,top,zNear,zFar);
	end_function_trace();
}

GLuint GenLists(GLsizei range)
{
	begin_function_trace("glGenLists", sdpGLsizei(range));
	GLuint result = glGenLists(range);
	result_trace(sdpGLuint(result));
	end_function_trace();
	return result;
}

void GenTextures(GLsizei n,GLuint *textures)
{
	begin_function_trace("glGenTextures", sdpGLsizei(n));//,GLuint *textur);
	glGenTextures(n,textures);
	end_function_trace();
}

void GetBooleanv(GLenum pname,GLboolean *params)
{
	begin_function_trace("glGetBooleanv", sdpGLenum(pname));
	glGetBooleanv(pname,params);
	end_function_trace();
}

void GetClipPlane(GLenum plane,GLdouble *equation)
{
	begin_function_trace("glGetClipPlane", sdpGLenum(plane));
	glGetClipPlane(plane,equation);
	end_function_trace();
}

void GetDoublev(GLenum pname,GLdouble *params)
{
	begin_function_trace("glGetDoublev", sdpGLenum(pname));
	glGetDoublev(pname,params);
	end_function_trace();
}

GLenum GetError()
{
	begin_function_trace("glGetError", "");
	GLenum result = glGetError();
	result_trace(sdpGLenum(result));
	end_function_trace();
	return result;
}

void GetFloatv(GLenum pname,GLfloat *params)
{
	begin_function_trace("glGetFloatv", sdpGLenum(pname));
	glGetFloatv(pname,params);
	end_function_trace();
}

void GetIntegerv(GLenum pname,GLint *params)
{
	begin_function_trace("glGetIntegerv", sdpGLenum(pname));
	glGetIntegerv(pname,params);
	end_function_trace();
}

void GetLightfv(GLenum light,GLenum pname,GLfloat *params)
{
	begin_function_trace("glGetLightfv", sdpGLenum(light) + "," + sdpGLenum(pname));
	glGetLightfv(light,pname,params);
	end_function_trace();
}

void GetLightiv(GLenum light,GLenum pname,GLint *params)
{
	begin_function_trace("glGetLightiv", sdpGLenum(light) + "," + sdpGLenum(pname));
	glGetLightiv(light,pname,params);
	end_function_trace();
}

void GetMapdv(GLenum target,GLenum query,GLdouble *v)
{
	begin_function_trace("glGetMapdv", sdpGLenum(target) + "," + sdpGLenum(query));
	glGetMapdv(target,query,v);
	end_function_trace();
}

void GetMapfv(GLenum target,GLenum query,GLfloat *v)
{
	begin_function_trace("glGetMapfv", sdpGLenum(target) + "," + sdpGLenum(query));
	glGetMapfv(target,query,v);
	end_function_trace();
}

void GetMapiv(GLenum target,GLenum query,GLint *v)
{
	begin_function_trace("glGetMapiv", sdpGLenum(target) + "," + sdpGLenum(query));
	glGetMapiv(target,query,v);
	end_function_trace();
}

void GetMaterialfv(GLenum face,GLenum pname,GLfloat *params)
{
	begin_function_trace("glGetMaterialfv", sdpGLenum(face) + "," + sdpGLenum(pname));
	glGetMaterialfv(face,pname,params);
	end_function_trace();
}

void GetMaterialiv(GLenum face,GLenum pname,GLint *params)
{
	begin_function_trace("glGetMaterialiv", sdpGLenum(face) + "," + sdpGLenum(pname));
	glGetMaterialiv(face,pname,params);
	end_function_trace();
}

void GetPixelMapfv(GLenum map,GLfloat *values)
{
	begin_function_trace("glGetPixelMapfv", sdpGLenum(map));
	glGetPixelMapfv(map,values);
	end_function_trace();
}

void GetPixelMapuiv(GLenum map,GLuint *values)
{
	begin_function_trace("glGetPixelMapuiv", sdpGLenum(map));
	glGetPixelMapuiv(map,values);
	end_function_trace();
}

void GetPixelMapusv(GLenum map,GLushort *values)
{
	begin_function_trace("glGetPixelMapusv", sdpGLenum(map));
	glGetPixelMapusv(map,values);
	end_function_trace();
}

void GetPointerv(GLenum pname,GLvoid* *params)
{
	begin_function_trace("glGetPointerv", sdpGLenum(pname));
	glGetPointerv(pname,params);
	end_function_trace();
}

void GetPolygonStipple(GLubyte *mask)
{
	begin_function_trace("glGetPolygonStipple", "");
	glGetPolygonStipple(mask);
	end_function_trace();
}

const GLubyte* GetString(GLenum name)
{
	begin_function_trace("glGetString", sdpGLenum(name));
	const GLubyte* result = glGetString(name);
	result_trace(reinterpret_cast<const char*>(result));
	end_function_trace();
	return result;
}

void GetTexEnvfv(GLenum target,GLenum pname,GLfloat *params)
{
	begin_function_trace("glGetTexEnvfv", sdpGLenum(target) + "," + sdpGLenum(pname));
	glGetTexEnvfv(target,pname,params);
	end_function_trace();
}

void GetTexEnviv(GLenum target,GLenum pname,GLint *params)
{
	begin_function_trace("glGetTexEnviv", sdpGLenum(target) + "," + sdpGLenum(pname));
	glGetTexEnviv(target,pname,params);
	end_function_trace();
}

void GetTexGendv(GLenum coord,GLenum pname,GLdouble *params)
{
	begin_function_trace("glGetTexGendv", sdpGLenum(coord) + "," + sdpGLenum(pname));
	glGetTexGendv(coord,pname,params);
	end_function_trace();
}

void GetTexGenfv(GLenum coord,GLenum pname,GLfloat *params)
{
	begin_function_trace("glGetTexGenfv", sdpGLenum(coord) + "," + sdpGLenum(pname));
	glGetTexGenfv(coord,pname,params);
	end_function_trace();
}

void GetTexGeniv(GLenum coord,GLenum pname,GLint *params)
{
	begin_function_trace("glGetTexGeniv", sdpGLenum(coord) + "," + sdpGLenum(pname));
	glGetTexGeniv(coord,pname,params);
	end_function_trace();
}

void GetTexImage(GLenum target,GLint level,GLenum format,GLenum type,GLvoid *pixels)
{
	begin_function_trace("glGetTexImage", sdpGLenum(target) + "," + sdpGLint(level) + "," + sdpGLenum(format) + "," + sdpGLenum(type));
	glGetTexImage(target,level,format,type,pixels);
	end_function_trace();
}

void GetTexLevelParameterfv(GLenum target,GLint level,GLenum pname,GLfloat *params)
{
	begin_function_trace("glGetTexLevelParameterfv", sdpGLenum(target) + "," + sdpGLint(level) + "," + sdpGLenum(pname));
	glGetTexLevelParameterfv(target,level,pname,params);
	end_function_trace();
}

void GetTexLevelParameteriv(GLenum target,GLint level,GLenum pname,GLint *params)
{
	begin_function_trace("glGetTexLevelParameteriv", sdpGLenum(target) + "," + sdpGLint(level) + "," + sdpGLenum(pname));
	glGetTexLevelParameteriv(target,level,pname,params);
	end_function_trace();
}

void GetTexParameterfv(GLenum target,GLenum pname,GLfloat *params)
{
	begin_function_trace("glGetTexParameterfv", sdpGLenum(target) + "," + sdpGLenum(pname));
	glGetTexParameterfv(target,pname,params);
	end_function_trace();
}

void GetTexParameteriv(GLenum target,GLenum pname,GLint *params)
{
	begin_function_trace("glGetTexParameteriv", sdpGLenum(target) + "," + sdpGLenum(pname));
	glGetTexParameteriv(target,pname,params);
	end_function_trace();
}

void Hint(GLenum target,GLenum mode)
{
	begin_function_trace("glHint", sdpGLenum(target) + "," + sdpGLenum(mode));
	glHint(target,mode);
	end_function_trace();
}

void IndexMask(GLuint mask)
{
	begin_function_trace("glIndexMask", sdpGLuint(mask));
	glIndexMask(mask);
	end_function_trace();
}

void IndexPointer(GLenum type,GLsizei stride,const GLvoid *pointer)
{
	begin_function_trace("glIndexPointer", sdpGLenum(type) + "," + sdpGLsizei(stride));//,const GLvoid *pointer");
	glIndexPointer(type,stride,pointer);
	end_function_trace();
}

void Indexd(GLdouble c)
{
	begin_function_trace("glIndexd", sdpGLdouble(c));
	glIndexd(c);
	end_function_trace();
}

void Indexdv(const GLdouble *c)
{
	begin_function_trace("glIndexdv", ""); //,const GLdouble );
	glIndexdv(c);
	end_function_trace();
}

void Indexf(GLfloat c)
{
	begin_function_trace("glIndexf", sdpGLfloat(c));
	glIndexf(c);
	end_function_trace();
}

void Indexfv(const GLfloat *c)
{
	begin_function_trace("glIndexfv", ""); //,const GLfloat );
	glIndexfv(c);
	end_function_trace();
}

void Indexi(GLint c)
{
	begin_function_trace("glIndexi", sdpGLint(c));
	glIndexi(c);
	end_function_trace();
}

void Indexiv(const GLint *c)
{
	begin_function_trace("glIndexiv", ""); //,const GLint );
	glIndexiv(c);
	end_function_trace();
}

void Indexs(GLshort c)
{
	begin_function_trace("glIndexs", sdpGLshort(c));
	glIndexs(c);
	end_function_trace();
}

void Indexsv(const GLshort *c)
{
	begin_function_trace("glIndexsv", ""); //,const GLshort );
	glIndexsv(c);
	end_function_trace();
}

void Indexub(GLubyte c)
{
	begin_function_trace("glIndexub", ""); //,GLubyte);
	glIndexub(c);
	end_function_trace();
}

void Indexubv(const GLubyte *c)
{
	begin_function_trace("glIndexubv", ""); //,const GLubyte );
	glIndexubv(c);
	end_function_trace();
}

void InitNames()
{
	begin_function_trace("glInitNames", "");
	glInitNames();
	end_function_trace();
}

void InterleavedArrays(GLenum format,GLsizei stride,const GLvoid *pointer)
{
	begin_function_trace("glInterleavedArrays", sdpGLenum(format) + "," + sdpGLsizei(stride));//,const GLvoid *pointer",&);
	glInterleavedArrays(format,stride,pointer);
	end_function_trace();
}

GLboolean IsEnabled(GLenum cap)
{
	begin_function_trace("glIsEnabled", sdpGLenum(cap));
	GLboolean result = glIsEnabled(cap);
	result_trace(sdpGLboolean(result));
	end_function_trace();
	return result;
}

GLboolean IsList(GLuint list)
{
	begin_function_trace("glIsList", sdpGLuint(list));
	GLboolean result = glIsList(list);
	result_trace(sdpGLboolean(result));
	end_function_trace();
	return result;
}

GLboolean IsTexture(GLuint texture)
{
	begin_function_trace("glIsTexture", sdpGLuint(texture));
	GLboolean result = glIsTexture(texture);
	result_trace(sdpGLboolean(result));
	end_function_trace();
	return result;
}

void LightModelf(GLenum pname,GLfloat param)
{
	begin_function_trace("glLightModelf", sdpGLenum(pname) + "," + sdpGLfloat(param));
	glLightModelf(pname,param);
	end_function_trace();
}

void LightModelfv(GLenum pname,const GLfloat *params)
{
	begin_function_trace("glLightModelfv", sdpGLenum(pname));//,const GLfloat *params",);
	glLightModelfv(pname,params);
	end_function_trace();
}

void LightModeli(GLenum pname,GLint param)
{
	begin_function_trace("glLightModeli", sdpGLenum(pname) + "," + sdpGLint(param));
	glLightModeli(pname,param);
	end_function_trace();
}

void LightModeliv(GLenum pname,const GLint *params)
{
	begin_function_trace("glLightModeliv", sdpGLenum(pname));//,const GLint *params",);
	glLightModeliv(pname,params);
	end_function_trace();
}

void Lightf(GLenum light,GLenum pname,GLfloat param)
{
	begin_function_trace("glLightf", sdpGLenum(light) + "," + sdpGLenum(pname) + "," + sdpGLfloat(param));
	glLightf(light,pname,param);
	end_function_trace();
}

void Lightfv(GLenum light,GLenum pname,const GLfloat *params)
{
	begin_function_trace("glLightfv", sdpGLenum(light) + "," + sdpGLenum(pname) + ",");
	if(pname == GL_AMBIENT)
		raw_trace(sdpGLfloat(params[0]) + "," + sdpGLfloat(params[1]) + "," + sdpGLfloat(params[2]));
	else if(pname == GL_DIFFUSE)
		raw_trace(sdpGLfloat(params[0]) + "," + sdpGLfloat(params[1]) + "," + sdpGLfloat(params[2]));
	else if(pname == GL_SPECULAR)
		raw_trace(sdpGLfloat(params[0]) + "," + sdpGLfloat(params[1]) + "," + sdpGLfloat(params[2]));
	else if(pname == GL_POSITION)
		raw_trace(sdpGLfloat(params[0]) + "," + sdpGLfloat(params[1]) + "," + sdpGLfloat(params[2]));
	else
		sdpGLvoid((void*)params);

	glLightfv(light,pname,params);
	end_function_trace();
}

void Lighti(GLenum light,GLenum pname,GLint param)
{
	begin_function_trace("glLighti", sdpGLenum(light) + "," + sdpGLenum(pname) + "," + sdpGLint(param));
	glLighti(light,pname,param);
	end_function_trace();
}

void Lightiv(GLenum light,GLenum pname,const GLint *params)
{
	begin_function_trace("glLightiv", sdpGLenum(light) + "," + sdpGLenum(pname));//,const GLint *params",);
	glLightiv(light,pname,params);
	end_function_trace();
}

void LineStipple(GLint factor,GLushort pattern)
{
	begin_function_trace("glLineStipple", sdpGLint(factor) + "," + sdpGLushort(pattern));
	glLineStipple(factor,pattern);
	end_function_trace();
}

void LineWidth(GLfloat width)
{
	begin_function_trace("glLineWidth", sdpGLfloat(width));
	glLineWidth(width);
	end_function_trace();
}

void ListBase(GLuint base)
{
	begin_function_trace("glListBase", sdpGLuint(base));
	glListBase(base);
	end_function_trace();
}

void LoadIdentity()
{
	begin_function_trace("glLoadIdentity", "");
	glLoadIdentity();
	end_function_trace();
}

void LoadMatrixd(const GLdouble *m)
{
	begin_function_trace("glLoadMatrixd", ""); //,const GLdouble );
	glLoadMatrixd(m);
	end_function_trace();
}

void LoadMatrixf(const GLfloat *m)
{
	begin_function_trace("glLoadMatrixf", ""); //,const GLfloat );
	glLoadMatrixf(m);
	end_function_trace();
}

void LoadName(GLuint name)
{
	begin_function_trace("glLoadName", sdpGLuint(name));
	glLoadName(name);
	end_function_trace();
}

void LogicOp(GLenum opcode)
{
	begin_function_trace("glLogicOp", sdpGLenum(opcode));
	glLogicOp(opcode);
	end_function_trace();
}

void Map1d(GLenum target,GLdouble u1,GLdouble u2,GLint stride,GLint order,const GLdouble *points)
{
	begin_function_trace("glMap1d", sdpGLenum(target) + "," + sdpGLdouble(u1) + "," + sdpGLdouble(u2) + "," + sdpGLint(stride) + "," + sdpGLint(order));//,const GLdouble *points",&);
	glMap1d(target,u1,u2,stride,order,points);
	end_function_trace();
}

void Map1f(GLenum target,GLfloat u1,GLfloat u2,GLint stride,GLint order,const GLfloat *points)
{
	begin_function_trace("glMap1f", sdpGLenum(target) + "," + sdpGLfloat(u1) + "," + sdpGLfloat(u2) + "," + sdpGLint(stride) + "," + sdpGLint(order));//,const GLfloat *points",&);
	glMap1f(target,u1,u2,stride,order,points);
	end_function_trace();
}

void Map2d(GLenum target,GLdouble u1,GLdouble u2,GLint ustride,GLint uorder,GLdouble v1,GLdouble v2,GLint vstride,GLint vorder,const GLdouble *points)
{
	begin_function_trace("glMap2d", sdpGLenum(target) + "," + sdpGLdouble(u1) + "," + sdpGLdouble(u2) + "," + sdpGLint(ustride) + "," + sdpGLint(uorder) + "," + sdpGLdouble(v1) + "," + sdpGLdouble(v2));//,GLint vstride,GLint vorder,const GLdouble *points",&);
	glMap2d(target,u1,u2,ustride,uorder,v1,v2,vstride,vorder,points);
	end_function_trace();
}

void Map2f(GLenum target,GLfloat u1,GLfloat u2,GLint ustride,GLint uorder,GLfloat v1,GLfloat v2,GLint vstride,GLint vorder,const GLfloat *points)
{
	begin_function_trace("glMap2f", sdpGLenum(target) + "," + sdpGLfloat(u1) + "," + sdpGLfloat(u2) + "," + sdpGLint(ustride) + "," + sdpGLint(uorder) + "," + sdpGLfloat(v1) + "," + sdpGLfloat(v2));//,GLint vstride,GLint vorder,const GLfloat *points",&);
	glMap2f(target,u1,u2,ustride,uorder,v1,v2,vstride,vorder,points);
	end_function_trace();
}

void MapGrid1d(GLint un,GLdouble u1,GLdouble u2)
{
	begin_function_trace("glMapGrid1d", sdpGLint(un) + "," + sdpGLdouble(u1) + "," + sdpGLdouble(u2));
	glMapGrid1d(un,u1,u2);
	end_function_trace();
}

void MapGrid1f(GLint un,GLfloat u1,GLfloat u2)
{
	begin_function_trace("glMapGrid1f", sdpGLint(un) + "," + sdpGLfloat(u1) + "," + sdpGLfloat(u2));
	glMapGrid1f(un,u1,u2);
	end_function_trace();
}

void MapGrid2d(GLint un,GLdouble u1,GLdouble u2,GLint vn,GLdouble v1,GLdouble v2)
{
	begin_function_trace("glMapGrid2d", sdpGLint(un) + "," + sdpGLdouble(u1) + "," + sdpGLdouble(u2));//,GLint vn,GLdouble v1,GLdouble v);
	glMapGrid2d(un,u1,u2,vn,v1,v2);
	end_function_trace();
}

void MapGrid2f(GLint un,GLfloat u1,GLfloat u2,GLint vn,GLfloat v1,GLfloat v2)
{
	begin_function_trace("glMapGrid2f", sdpGLint(un) + "," + sdpGLfloat(u1) + "," + sdpGLfloat(u2));//,GLint vn,GLfloat v1,GLfloat v);
	glMapGrid2f(un,u1,u2,vn,v1,v2);
	end_function_trace();
}

void Materialf(GLenum face,GLenum pname,GLfloat param)
{
	begin_function_trace("glMaterialf", sdpGLenum(face) + "," + sdpGLenum(pname) + "," + sdpGLfloat(param));
	glMaterialf(face,pname,param);
	end_function_trace();
}

void Materialfv(GLenum face,GLenum pname,const GLfloat *params)
{
	begin_function_trace("glMaterialfv", sdpGLenum(face) + "," + sdpGLenum(pname) + ",");
	if(pname == GL_AMBIENT)
		raw_trace(sdpGLfloat(params[0]) + "," + sdpGLfloat(params[1]) + "," + sdpGLfloat(params[2]));
	else if(pname == GL_DIFFUSE)
		raw_trace(sdpGLfloat(params[0]) + "," + sdpGLfloat(params[1]) + "," + sdpGLfloat(params[2]));
	else if(pname == GL_SPECULAR)
		raw_trace(sdpGLfloat(params[0]) + "," + sdpGLfloat(params[1]) + "," + sdpGLfloat(params[2]));
	else if(pname == GL_EMISSION)
		raw_trace(sdpGLfloat(params[0]) + "," + sdpGLfloat(params[1]) + "," + sdpGLfloat(params[2]));
	else
		sdpGLvoid((void*)params);

	glMaterialfv(face,pname,params);
	end_function_trace();
}

void Materiali(GLenum face,GLenum pname,GLint param)
{
	begin_function_trace("glMateriali", sdpGLenum(face) + "," + sdpGLenum(pname) + "," + sdpGLint(param));
	glMateriali(face,pname,param);
	end_function_trace();
}

void Materialiv(GLenum face,GLenum pname,const GLint *params)
{
	begin_function_trace("glMaterialiv", sdpGLenum(face) + "," + sdpGLenum(pname));//,const GLint *params");
	glMaterialiv(face,pname,params);
	end_function_trace();
}

void MatrixMode(GLenum mode)
{
	begin_function_trace("glMatrixMode", sdpGLenum(mode));
	glMatrixMode(mode);
	end_function_trace();
}

void MultMatrixd(const GLdouble *m)
{
	begin_function_trace("glMultMatrixd", sdpGLMatrixd(m));
	glMultMatrixd(m);
	end_function_trace();
}

void MultMatrixf(const GLfloat *m)
{
	begin_function_trace("glMultMatrixf", ""); //,const GLfloat );
	glMultMatrixf(m);
	end_function_trace();
}

void NewList(GLuint list,GLenum mode)
{
	begin_function_trace("glNewList", sdpGLuint(list) + "," + sdpGLenum(mode));
	glNewList(list,mode);
	end_function_trace();
}

void Normal3b(GLbyte nx,GLbyte ny,GLbyte nz)
{
	begin_function_trace("glNormal3b", sdpGLbyte(nx) + "," + sdpGLbyte(ny) + "," + sdpGLbyte(nz));
	glNormal3b(nx,ny,nz);
	end_function_trace();
}

void Normal3bv(const GLbyte *v)
{
	begin_function_trace("glNormal3bv", ""); //,const GLbyte );
	glNormal3bv(v);
	end_function_trace();
}

void Normal3d(GLdouble nx,GLdouble ny,GLdouble nz)
{
	begin_function_trace("glNormal3d", sdpGLdouble(nx) + "," + sdpGLdouble(ny) + "," + sdpGLdouble(nz));
	glNormal3d(nx,ny,nz);
	end_function_trace();
}

void Normal3dv(const GLdouble *v)
{
	begin_function_trace("glNormal3dv", sdpGLdouble(v[0]) + "," + sdpGLdouble(v[1]) + "," + sdpGLdouble(v[2]));
	glNormal3dv(v);
	end_function_trace();
}

void Normal3f(GLfloat nx,GLfloat ny,GLfloat nz)
{
	begin_function_trace("glNormal3f", sdpGLfloat(nx) + "," + sdpGLfloat(ny) + "," + sdpGLfloat(nz));
	glNormal3f(nx,ny,nz);
	end_function_trace();
}

void Normal3fv(const GLfloat *v)
{
	begin_function_trace("glNormal3fv", sdpGLfloat(v[0]) + "," + sdpGLfloat(v[1]) + "," + sdpGLfloat(v[2]) + ",");
	glNormal3fv(v);
	end_function_trace();
}

void Normal3i(GLint nx,GLint ny,GLint nz)
{
	begin_function_trace("glNormal3i", sdpGLint(nx) + "," + sdpGLint(ny) + "," + sdpGLint(nz));
	glNormal3i(nx,ny,nz);
	end_function_trace();
}

void Normal3iv(const GLint *v)
{
	begin_function_trace("glNormal3iv", ""); //,const GLint );
	glNormal3iv(v);
	end_function_trace();
}

void Normal3s(GLshort nx,GLshort ny,GLshort nz)
{
	begin_function_trace("glNormal3s", sdpGLshort(nx) + "," + sdpGLshort(ny) + "," + sdpGLshort(nz));
	glNormal3s(nx,ny,nz);
	end_function_trace();
}

void Normal3sv(const GLshort *v)
{
	begin_function_trace("glNormal3sv", ""); //,const GLshort );
	glNormal3sv(v);
	end_function_trace();
}

void NormalPointer(GLenum type,GLsizei stride,const GLvoid *pointer)
{
	begin_function_trace("glNormalPointer", sdpGLenum(type) + "," + sdpGLsizei(stride));//,const GLvoid *pointer");
	glNormalPointer(type,stride,pointer);
	end_function_trace();
}

void Ortho(GLdouble left,GLdouble right,GLdouble bottom,GLdouble top,GLdouble zNear,GLdouble zFar)
{
	begin_function_trace("glOrtho", sdpGLdouble(left) + "," + sdpGLdouble(right) + "," + sdpGLdouble(bottom) + "," + sdpGLdouble(top) + "," + sdpGLdouble(zNear) + "," + sdpGLdouble(zFar));
	glOrtho(left,right,bottom,top,zNear,zFar);
	end_function_trace();
}

void PassThrough(GLfloat token)
{
	begin_function_trace("glPassThrough", sdpGLfloat(token));
	glPassThrough(token);
	end_function_trace();
}

void PixelMapfv(GLenum map,GLsizei mapsize,const GLfloat *values)
{
	begin_function_trace("glPixelMapfv", sdpGLenum(map) + "," + sdpGLsizei(mapsize));//,const GLfloat *values);
	glPixelMapfv(map,mapsize,values);
	end_function_trace();
}

void PixelMapuiv(GLenum map,GLsizei mapsize,const GLuint *values)
{
	begin_function_trace("glPixelMapuiv", sdpGLenum(map) + "," + sdpGLsizei(mapsize));//,const GLuint *values);
	glPixelMapuiv(map,mapsize,values);
	end_function_trace();
}

void PixelMapusv(GLenum map,GLsizei mapsize,const GLushort *values)
{
	begin_function_trace("glPixelMapusv", sdpGLenum(map) + "," + sdpGLsizei(mapsize));//,const GLushort *values);
	glPixelMapusv(map,mapsize,values);
	end_function_trace();
}

void PixelStoref(GLenum pname,GLfloat param)
{
	begin_function_trace("glPixelStoref", sdpGLenum(pname) + "," + sdpGLfloat(param));
	glPixelStoref(pname,param);
	end_function_trace();
}

void PixelStorei(GLenum pname,GLint param)
{
	begin_function_trace("glPixelStorei", sdpGLenum(pname) + "," + sdpGLint(param));
	glPixelStorei(pname,param);
	end_function_trace();
}

void PixelTransferf(GLenum pname,GLfloat param)
{
	begin_function_trace("glPixelTransferf", sdpGLenum(pname) + "," + sdpGLfloat(param));
	glPixelTransferf(pname,param);
	end_function_trace();
}

void PixelTransferi(GLenum pname,GLint param)
{
	begin_function_trace("glPixelTransferi", sdpGLenum(pname) + "," + sdpGLint(param));
	glPixelTransferi(pname,param);
	end_function_trace();
}

void PixelZoom(GLfloat xfactor,GLfloat yfactor)
{
	begin_function_trace("glPixelZoom", sdpGLfloat(xfactor) + "," + sdpGLfloat(yfactor));
	glPixelZoom(xfactor,yfactor);
	end_function_trace();
}

void PointSize(GLfloat size)
{
	begin_function_trace("glPointSize", sdpGLfloat(size));
	glPointSize(size);
	end_function_trace();
}

void PolygonMode(GLenum face,GLenum mode)
{
	begin_function_trace("glPolygonMode", sdpGLenum(face) + "," + sdpGLenum(mode));
	glPolygonMode(face,mode);
	end_function_trace();
}

void PolygonOffset(GLfloat factor,GLfloat units)
{
	begin_function_trace("glPolygonOffset", sdpGLfloat(factor) + "," + sdpGLfloat(units));
	glPolygonOffset(factor,units);
	end_function_trace();
}

void PolygonStipple(const GLubyte *mask)
{
	begin_function_trace("glPolygonStipple", ""); //,const GLubyte *mask");
	glPolygonStipple(mask);
	end_function_trace();
}

void PopAttrib()
{
	begin_function_trace("glPopAttrib", "");
	glPopAttrib();
	end_function_trace();
}

void PopClientAttrib()
{
	begin_function_trace("glPopClientAttrib", "");
	glPopClientAttrib();
	end_function_trace();
}

void PopMatrix()
{
	begin_function_trace("glPopMatrix", "");
	glPopMatrix();
	end_function_trace();
}

void PopName()
{
	begin_function_trace("glPopName", "");
	glPopName();
	end_function_trace();
}

void PrioritizeTextures(GLsizei n,const GLuint *textures,const GLclampf *priorities)
{
	begin_function_trace("glPrioritizeTextures", sdpGLsizei(n));//,const GLuint *textures,const GLclampf *prioriti);
	glPrioritizeTextures(n,textures,priorities);
	end_function_trace();
}

void PushAttrib(GLbitfield mask)
{
	begin_function_trace("glPushAttrib", sdpGLbitfield(mask));
	glPushAttrib(mask);
	end_function_trace();
}

void PushClientAttrib(GLbitfield mask)
{
	begin_function_trace("glPushClientAttrib", sdpGLbitfield(mask));
	glPushClientAttrib(mask);
	end_function_trace();
}

void PushMatrix()
{
	begin_function_trace("glPushMatrix", "");
	glPushMatrix();
	end_function_trace();
}

void PushName(GLuint name)
{
	begin_function_trace("glPushName", sdpGLuint(name));
	glPushName(name);
	end_function_trace();
}

void RasterPos2d(GLdouble x,GLdouble y)
{
	begin_function_trace("glRasterPos2d", sdpGLdouble(x) + "," + sdpGLdouble(y));
	glRasterPos2d(x,y);
	end_function_trace();
}

void RasterPos2dv(const GLdouble *v)
{
	begin_function_trace("glRasterPos2dv", ""); //,const GLdouble );
	glRasterPos2dv(v);
	end_function_trace();
}

void RasterPos2f(GLfloat x,GLfloat y)
{
	begin_function_trace("glRasterPos2f", sdpGLfloat(x) + "," + sdpGLfloat(y));
	glRasterPos2f(x,y);
	end_function_trace();
}

void RasterPos2fv(const GLfloat *v)
{
	begin_function_trace("glRasterPos2fv", sdpGLfloat(v[0]) + "," + sdpGLfloat(v[1]));
	glRasterPos2fv(v);
	end_function_trace();
}

void RasterPos2i(GLint x,GLint y)
{
	begin_function_trace("glRasterPos2i", sdpGLint(x) + "," + sdpGLint(y));
	glRasterPos2i(x,y);
	end_function_trace();
}

void RasterPos2iv(const GLint *v)
{
	begin_function_trace("glRasterPos2iv", ""); //,const GLint );
	glRasterPos2iv(v);
	end_function_trace();
}

void RasterPos2s(GLshort x,GLshort y)
{
	begin_function_trace("glRasterPos2s", sdpGLshort(x) + "," + sdpGLshort(y));
	glRasterPos2s(x,y);
	end_function_trace();
}

void RasterPos2sv(const GLshort *v)
{
	begin_function_trace("glRasterPos2sv", ""); //,const GLshort );
	glRasterPos2sv(v);
	end_function_trace();
}

void RasterPos3d(GLdouble x,GLdouble y,GLdouble z)
{
	begin_function_trace("glRasterPos3d", sdpGLdouble(x) + "," + sdpGLdouble(y) + "," + sdpGLdouble(z));
	glRasterPos3d(x,y,z);
	end_function_trace();
}

void RasterPos3dv(const GLdouble *v)
{
	begin_function_trace("glRasterPos3dv", sdpGLdouble(v[0]) + "," + sdpGLdouble(v[1]) + "," + sdpGLdouble(v[2]));
	glRasterPos3dv(v);
	end_function_trace();
}

void RasterPos3f(GLfloat x,GLfloat y,GLfloat z)
{
	begin_function_trace("glRasterPos3f", sdpGLfloat(x) + "," + sdpGLfloat(y) + "," + sdpGLfloat(z));
	glRasterPos3f(x,y,z);
	end_function_trace();
}

void RasterPos3fv(const GLfloat *v)
{
	begin_function_trace("glRasterPos3fv", sdpGLfloat(v[0]) + "," + sdpGLfloat(v[1]) + "," + sdpGLfloat(v[2]));
	glRasterPos3fv(v);
	end_function_trace();
}

void RasterPos3i(GLint x,GLint y,GLint z)
{
	begin_function_trace("glRasterPos3i", sdpGLint(x) + "," + sdpGLint(y) + "," + sdpGLint(z));
	glRasterPos3i(x,y,z);
	end_function_trace();
}

void RasterPos3iv(const GLint *v)
{
	begin_function_trace("glRasterPos3iv", ""); //,const GLint );
	glRasterPos3iv(v);
	end_function_trace();
}

void RasterPos3s(GLshort x,GLshort y,GLshort z)
{
	begin_function_trace("glRasterPos3s", sdpGLshort(x) + "," + sdpGLshort(y) + "," + sdpGLshort(z));
	glRasterPos3s(x,y,z);
	end_function_trace();
}

void RasterPos3sv(const GLshort *v)
{
	begin_function_trace("glRasterPos3sv", ""); //,const GLshort );
	glRasterPos3sv(v);
	end_function_trace();
}

void RasterPos4d(GLdouble x,GLdouble y,GLdouble z,GLdouble w)
{
	begin_function_trace("glRasterPos4d", sdpGLdouble(x) + "," + sdpGLdouble(y) + "," + sdpGLdouble(z) + "," + sdpGLdouble(w));
	glRasterPos4d(x,y,z,w);
	end_function_trace();
}

void RasterPos4dv(const GLdouble *v)
{
	begin_function_trace("glRasterPos4dv", sdpGLdouble(v[0]) + "," + sdpGLdouble(v[1]) + "," + sdpGLdouble(v[2]) + "," + sdpGLdouble(v[3]));
	glRasterPos4dv(v);
	end_function_trace();
}

void RasterPos4f(GLfloat x,GLfloat y,GLfloat z,GLfloat w)
{
	begin_function_trace("glRasterPos4f", sdpGLfloat(x) + "," + sdpGLfloat(y) + "," + sdpGLfloat(z) + "," + sdpGLfloat(w));
	glRasterPos4f(x,y,z,w);
	end_function_trace();
}

void RasterPos4fv(const GLfloat *v)
{
	begin_function_trace("glRasterPos4fv", sdpGLfloat(v[0]) + "," + sdpGLfloat(v[1]) + "," + sdpGLfloat(v[2]) + "," + sdpGLfloat(v[3]));
	glRasterPos4fv(v);
	end_function_trace();
}

void RasterPos4i(GLint x,GLint y,GLint z,GLint w)
{
	begin_function_trace("glRasterPos4i", sdpGLint(x) + "," + sdpGLint(y) + "," + sdpGLint(z) + "," + sdpGLint(w));
	glRasterPos4i(x,y,z,w);
	end_function_trace();
}

void RasterPos4iv(const GLint *v)
{
	begin_function_trace("glRasterPos4iv", ""); //,const GLint );
	glRasterPos4iv(v);
	end_function_trace();
}

void RasterPos4s(GLshort x,GLshort y,GLshort z,GLshort w)
{
	begin_function_trace("glRasterPos4s", sdpGLshort(x) + "," + sdpGLshort(y) + "," + sdpGLshort(z) + "," + sdpGLshort(w));
	glRasterPos4s(x,y,z,w);
	end_function_trace();
}

void RasterPos4sv(const GLshort *v)
{
	begin_function_trace("glRasterPos4sv", ""); //,const GLshort );
	glRasterPos4sv(v);
	end_function_trace();
}

void ReadBuffer(GLenum mode)
{
	begin_function_trace("glReadBuffer", sdpGLenum(mode));
	glReadBuffer(mode);
	end_function_trace();
}

void ReadPixels(GLint x,GLint y,GLsizei width,GLsizei height,GLenum format,GLenum type,GLvoid *pixels)
{
	begin_function_trace("glReadPixels", sdpGLint(x) + "," + sdpGLint(y) + "," + sdpGLsizei(width) + "," + sdpGLsizei(height) + "," + sdpGLenum(format) + "," + sdpGLenum(type));//,GLvoid *pixe);
	glReadPixels(x,y,width,height,format,type,pixels);
	end_function_trace();
}

void Rectd(GLdouble x1,GLdouble y1,GLdouble x2,GLdouble y2)
{
	begin_function_trace("glRectd", sdpGLdouble(x1) + "," + sdpGLdouble(y1) + "," + sdpGLdouble(x2) + "," + sdpGLdouble(y2));
	glRectd(x1,y1,x2,y2);
	end_function_trace();
}

void Rectdv(const GLdouble *v1,const GLdouble *v2)
{
	begin_function_trace("glRectdv", ""); //,const GLdouble *v1,const GLdouble *v);
	glRectdv(v1,v2);
	end_function_trace();
}

void Rectf(GLfloat x1,GLfloat y1,GLfloat x2,GLfloat y2)
{
	begin_function_trace("glRectf", sdpGLfloat(x1) + "," + sdpGLfloat(y1) + "," + sdpGLfloat(x2) + "," + sdpGLfloat(y2));
	glRectf(x1,y1,x2,y2);
	end_function_trace();
}

void Rectfv(const GLfloat *v1,const GLfloat *v2)
{
	begin_function_trace("glRectfv", ""); //,const GLfloat *v1,const GLfloat *v);
	glRectfv(v1,v2);
	end_function_trace();
}

void Recti(GLint x1,GLint y1,GLint x2,GLint y2)
{
	begin_function_trace("glRecti", sdpGLint(x1) + "," + sdpGLint(y1) + "," + sdpGLint(x2) + "," + sdpGLint(y2));
	glRecti(x1,y1,x2,y2);
	end_function_trace();
}

void Rectiv(const GLint *v1,const GLint *v2)
{
	begin_function_trace("glRectiv", ""); //,const GLint *v1,const GLint *v);
	glRectiv(v1,v2);
	end_function_trace();
}

void Rects(GLshort x1,GLshort y1,GLshort x2,GLshort y2)
{
	begin_function_trace("glRects", sdpGLshort(x1) + "," + sdpGLshort(y1) + "," + sdpGLshort(x2) + "," + sdpGLshort(y2));
	glRects(x1,y1,x2,y2);
	end_function_trace();
}

void Rectsv(const GLshort *v1,const GLshort *v2)
{
	begin_function_trace("glRectsv", ""); //,const GLshort *v1,const GLshort *v);
	glRectsv(v1,v2);
	end_function_trace();
}

GLint RenderMode(GLenum mode)
{
	begin_function_trace("glRenderMode", sdpGLenum(mode));
	GLint result = glRenderMode(mode);
	result_trace(sdpGLint(result));
	end_function_trace();
	return result;
}

void Rotated(GLdouble angle,GLdouble x,GLdouble y,GLdouble z)
{
	begin_function_trace("glRotated", sdpGLdouble(angle) + "," + sdpGLdouble(x) + "," + sdpGLdouble(y) + "," + sdpGLdouble(z));
	glRotated(angle,x,y,z);
	end_function_trace();
}

void Rotatef(GLfloat angle,GLfloat x,GLfloat y,GLfloat z)
{
	begin_function_trace("glRotatef", sdpGLfloat(angle) + "," + sdpGLfloat(x) + "," + sdpGLfloat(y) + "," + sdpGLfloat(z));
	glRotatef(angle,x,y,z);
	end_function_trace();
}

void Scaled(GLdouble x,GLdouble y,GLdouble z)
{
	begin_function_trace("glScaled", sdpGLdouble(x) + "," + sdpGLdouble(y) + "," + sdpGLdouble(z));
	glScaled(x,y,z);
	end_function_trace();
}

void Scalef(GLfloat x,GLfloat y,GLfloat z)
{
	begin_function_trace("glScalef", sdpGLfloat(x) + "," + sdpGLfloat(y) + "," + sdpGLfloat(z));
	glScalef(x,y,z);
	end_function_trace();
}

void Scissor(GLint x,GLint y,GLsizei width,GLsizei height)
{
	begin_function_trace("glScissor", sdpGLint(x) + "," + sdpGLint(y) + "," + sdpGLsizei(width) + "," + sdpGLsizei(height));
	glScissor(x,y,width,height);
	end_function_trace();
}

void SelectBuffer(GLsizei size,GLuint *buffer)
{
	begin_function_trace("glSelectBuffer", sdpGLsizei(size));//,GLuint *buffer");
	glSelectBuffer(size,buffer);
	end_function_trace();
}

void ShadeModel(GLenum mode)
{
	begin_function_trace("glShadeModel", sdpGLenum(mode));
	glShadeModel(mode);
	end_function_trace();
}

void StencilFunc(GLenum func,GLint ref,GLuint mask)
{
	begin_function_trace("glStencilFunc", sdpGLenum(func) + "," + sdpGLint(ref) + "," + sdpGLuint(mask));
	glStencilFunc(func,ref,mask);
	end_function_trace();
}

void StencilMask(GLuint mask)
{
	begin_function_trace("glStencilMask", sdpGLuint(mask));
	glStencilMask(mask);
	end_function_trace();
}

void StencilOp(GLenum fail,GLenum zfail,GLenum zpass)
{
	begin_function_trace("glStencilOp", sdpGLenum(fail) + "," + sdpGLenum(zfail) + "," + sdpGLenum(zpass));
	glStencilOp(fail,zfail,zpass);
	end_function_trace();
}

void TexCoord1d(GLdouble s)
{
	begin_function_trace("glTexCoord1d", sdpGLdouble(s));
	glTexCoord1d(s);
	end_function_trace();
}

void TexCoord1dv(const GLdouble *v)
{
	begin_function_trace("glTexCoord1dv", ""); //,const GLdouble );
	glTexCoord1dv(v);
	end_function_trace();
}

void TexCoord1f(GLfloat s)
{
	begin_function_trace("glTexCoord1f", sdpGLfloat(s));
	glTexCoord1f(s);
	end_function_trace();
}

void TexCoord1fv(const GLfloat *v)
{
	begin_function_trace("glTexCoord1fv", sdpGLfloat(v[0]));
	glTexCoord1fv(v);
	end_function_trace();
}

void TexCoord1i(GLint s)
{
	begin_function_trace("glTexCoord1i", sdpGLint(s));
	glTexCoord1i(s);
	end_function_trace();
}

void TexCoord1iv(const GLint *v)
{
	begin_function_trace("glTexCoord1iv", ""); //,const GLint );
	glTexCoord1iv(v);
	end_function_trace();
}

void TexCoord1s(GLshort s)
{
	begin_function_trace("glTexCoord1s", sdpGLshort(s));
	glTexCoord1s(s);
	end_function_trace();
}

void TexCoord1sv(const GLshort *v)
{
	begin_function_trace("glTexCoord1sv", ""); //,const GLshort );
	glTexCoord1sv(v);
	end_function_trace();
}

void TexCoord2d(GLdouble s,GLdouble t)
{
	begin_function_trace("glTexCoord2d", sdpGLdouble(s) + "," + sdpGLdouble(t));
	glTexCoord2d(s,t);
	end_function_trace();
}

void TexCoord2dv(const GLdouble *v)
{
	begin_function_trace("glTexCoord2dv", ""); //,const GLdouble );
	glTexCoord2dv(v);
	end_function_trace();
}

void TexCoord2f(GLfloat s,GLfloat t)
{
	begin_function_trace("glTexCoord2f", sdpGLfloat(s) + "," + sdpGLfloat(t));
	glTexCoord2f(s,t);
	end_function_trace();
}

void TexCoord2fv(const GLfloat *v)
{
	begin_function_trace("glTexCoord2fv", sdpGLfloat(v[0]) + "," + sdpGLfloat(v[1]));
	glTexCoord2fv(v);
	end_function_trace();
}

void TexCoord2i(GLint s,GLint t)
{
	begin_function_trace("glTexCoord2i", sdpGLint(s) + "," + sdpGLint(t));
	glTexCoord2i(s,t);
	end_function_trace();
}

void TexCoord2iv(const GLint *v)
{
	begin_function_trace("glTexCoord2iv", ""); //,const GLint );
	glTexCoord2iv(v);
	end_function_trace();
}

void TexCoord2s(GLshort s,GLshort t)
{
	begin_function_trace("glTexCoord2s", sdpGLshort(s) + "," + sdpGLshort(t));
	glTexCoord2s(s,t);
	end_function_trace();
}

void TexCoord2sv(const GLshort *v)
{
	begin_function_trace("glTexCoord2sv", ""); //,const GLshort );
	glTexCoord2sv(v);
	end_function_trace();
}

void TexCoord3d(GLdouble s,GLdouble t,GLdouble r)
{
	begin_function_trace("glTexCoord3d", sdpGLdouble(s) + "," + sdpGLdouble(t) + "," + sdpGLdouble(r));
	glTexCoord3d(s,t,r);
	end_function_trace();
}

void TexCoord3dv(const GLdouble *v)
{
	begin_function_trace("glTexCoord3dv", sdpGLdouble(v[0]) + "," + sdpGLdouble(v[1]) + "," + sdpGLdouble(v[2]));
	glTexCoord3dv(v);
	end_function_trace();
}

void TexCoord3f(GLfloat s,GLfloat t,GLfloat r)
{
	begin_function_trace("glTexCoord3f", sdpGLfloat(s) + "," + sdpGLfloat(t) + "," + sdpGLfloat(r));
	glTexCoord3f(s,t,r);
	end_function_trace();
}

void TexCoord3fv(const GLfloat *v)
{
	begin_function_trace("glTexCoord3fv", sdpGLfloat(v[0]) + "," + sdpGLfloat(v[1]) + "," + sdpGLfloat(v[2]));
	glTexCoord3fv(v);
	end_function_trace();
}

void TexCoord3i(GLint s,GLint t,GLint r)
{
	begin_function_trace("glTexCoord3i", sdpGLint(s) + "," + sdpGLint(t) + "," + sdpGLint(r));
	glTexCoord3i(s,t,r);
	end_function_trace();
}

void TexCoord3iv(const GLint *v)
{
	begin_function_trace("glTexCoord3iv", ""); //,const GLint );
	glTexCoord3iv(v);
	end_function_trace();
}

void TexCoord3s(GLshort s,GLshort t,GLshort r)
{
	begin_function_trace("glTexCoord3s", sdpGLshort(s) + "," + sdpGLshort(t) + "," + sdpGLshort(r));
	glTexCoord3s(s,t,r);
	end_function_trace();
}

void TexCoord3sv(const GLshort *v)
{
	begin_function_trace("glTexCoord3sv", ""); //,const GLshort );
	glTexCoord3sv(v);
	end_function_trace();
}

void TexCoord4d(GLdouble s,GLdouble t,GLdouble r,GLdouble q)
{
	begin_function_trace("glTexCoord4d", sdpGLdouble(s) + "," + sdpGLdouble(t) + "," + sdpGLdouble(r) + "," + sdpGLdouble(q));
	glTexCoord4d(s,t,r,q);
	end_function_trace();
}

void TexCoord4dv(const GLdouble *v)
{
	begin_function_trace("glTexCoord4dv", sdpGLdouble(v[0]) + "," + sdpGLdouble(v[1]) + "," + sdpGLdouble(v[2]) + "," + sdpGLdouble(v[3]));
	glTexCoord4dv(v);
	end_function_trace();
}

void TexCoord4f(GLfloat s,GLfloat t,GLfloat r,GLfloat q)
{
	begin_function_trace("glTexCoord4f", sdpGLfloat(s) + "," + sdpGLfloat(t) + "," + sdpGLfloat(r) + "," + sdpGLfloat(q));
	glTexCoord4f(s,t,r,q);
	end_function_trace();
}

void TexCoord4fv(const GLfloat *v)
{
	begin_function_trace("glTexCoord4fv", sdpGLfloat(v[0]) + "," + sdpGLfloat(v[1]) + "," + sdpGLfloat(v[2]) + "," + sdpGLfloat(v[3]));
	glTexCoord4fv(v);
	end_function_trace();
}

void TexCoord4i(GLint s,GLint t,GLint r,GLint q)
{
	begin_function_trace("glTexCoord4i", sdpGLint(s) + "," + sdpGLint(t) + "," + sdpGLint(r) + "," + sdpGLint(q));
	glTexCoord4i(s,t,r,q);
	end_function_trace();
}

void TexCoord4iv(const GLint *v)
{
	begin_function_trace("glTexCoord4iv", ""); //,const GLint );
	glTexCoord4iv(v);
	end_function_trace();
}

void TexCoord4s(GLshort s,GLshort t,GLshort r,GLshort q)
{
	begin_function_trace("glTexCoord4s", sdpGLshort(s) + "," + sdpGLshort(t) + "," + sdpGLshort(r) + "," + sdpGLshort(q));
	glTexCoord4s(s,t,r,q);
	end_function_trace();
}

void TexCoord4sv(const GLshort *v)
{
	begin_function_trace("glTexCoord4sv", ""); //,const GLshort );
	glTexCoord4sv(v);
	end_function_trace();
}

void TexCoordPointer(GLint size,GLenum type,GLsizei stride,const GLvoid *pointer)
{
	begin_function_trace("glTexCoordPointer", sdpGLint(size) + "," + sdpGLenum(type) + "," + sdpGLsizei(stride));//,const GLvoid *pointer");
	glTexCoordPointer(size,type,stride,pointer);
	end_function_trace();
}

void TexEnvf(GLenum target,GLenum pname,GLfloat param)
{
	begin_function_trace("glTexEnvf", sdpGLenum(target) + "," + sdpGLenum(pname) + "," + sdpGLfloat(param));
	glTexEnvf(target,pname,param);
	end_function_trace();
}

void TexEnvfv(GLenum target,GLenum pname,const GLfloat *params)
{
	begin_function_trace("glTexEnvfv", sdpGLenum(target) + "," + sdpGLenum(pname));//,const GLfloat *params",&);
	glTexEnvfv(target,pname,params);
	end_function_trace();
}

void TexEnvi(GLenum target,GLenum pname,GLint param)
{
	begin_function_trace("glTexEnvi", sdpGLenum(target) + "," + sdpGLenum(pname) + "," + sdpGLint(param));
	glTexEnvi(target,pname,param);
	end_function_trace();
}

void TexEnviv(GLenum target,GLenum pname,const GLint *params)
{
	begin_function_trace("glTexEnviv", sdpGLenum(target) + "," + sdpGLenum(pname));//,const GLint *params",&);
	glTexEnviv(target,pname,params);
	end_function_trace();
}

void TexGend(GLenum coord,GLenum pname,GLdouble param)
{
	begin_function_trace("glTexGend", sdpGLenum(coord) + "," + sdpGLenum(pname) + "," + sdpGLdouble(param));
	glTexGend(coord,pname,param);
	end_function_trace();
}

void TexGendv(GLenum coord,GLenum pname,const GLdouble *params)
{
	begin_function_trace("glTexGendv", sdpGLenum(coord) + "," + sdpGLenum(pname));//,const GLdouble *params",);
	glTexGendv(coord,pname,params);
	end_function_trace();
}

void TexGenf(GLenum coord,GLenum pname,GLfloat param)
{
	begin_function_trace("glTexGenf", sdpGLenum(coord) + "," + sdpGLenum(pname) + "," + sdpGLfloat(param));
	glTexGenf(coord,pname,param);
	end_function_trace();
}

void TexGenfv(GLenum coord,GLenum pname,const GLfloat *params)
{
	begin_function_trace("glTexGenfv", sdpGLenum(coord) + "," + sdpGLenum(pname));//,const GLfloat *params",);
	glTexGenfv(coord,pname,params);
	end_function_trace();
}

void TexGeni(GLenum coord,GLenum pname,GLint param)
{
	begin_function_trace("glTexGeni", sdpGLenum(coord) + "," + sdpGLenum(pname) + "," + sdpGLint(param));
	glTexGeni(coord,pname,param);
	end_function_trace();
}

void TexGeniv(GLenum coord,GLenum pname,const GLint *params)
{
	begin_function_trace("glTexGeniv", sdpGLenum(coord) + "," + sdpGLenum(pname));//,const GLint *params",);
	glTexGeniv(coord,pname,params);
	end_function_trace();
}

void TexImage1D(GLenum target,GLint level,GLint internalformat,GLsizei width,GLint border,GLenum format,GLenum type,const GLvoid *pixels)
{
	begin_function_trace("glTexImage1D", sdpGLenum(target) + "," + sdpGLint(level) + "," + sdpGLint(internalformat) + "," + sdpGLsizei(width) + "," + sdpGLint(border) + "," + sdpGLenum(format) + "," + sdpGLenum(type));//,const GLvoid *pixels",&);
	glTexImage1D(target,level,internalformat,width,border,format,type,pixels);
	end_function_trace();
}

void TexImage2D(GLenum target,GLint level,GLint internalformat,GLsizei width,GLsizei height,GLint border,GLenum format,GLenum type,const GLvoid *pixels)
{
	begin_function_trace("glTexImage2D", sdpGLenum(target) + "," + sdpGLint(level) + "," + sdpGLint(internalformat) + "," + sdpGLsizei(width) + "," + sdpGLsizei(height) + "," + sdpGLint(border) + "," + sdpGLenum(format) + "," + sdpGLenum(type));//,const GLvoid *pixels",&);
	glTexImage2D(target,level,internalformat,width,height,border,format,type,pixels);
	end_function_trace();
}

void TexParameterf(GLenum target,GLenum pname,GLfloat param)
{
	begin_function_trace("glTexParameterf", sdpGLenum(target) + "," + sdpGLenum(pname) + "," + sdpGLfloat(param));
	glTexParameterf(target,pname,param);
	end_function_trace();
}

void TexParameterfv(GLenum target,GLenum pname,const GLfloat *params)
{
	begin_function_trace("glTexParameterfv", sdpGLenum(target) + "," + sdpGLenum(pname));//,const GLfloat *params",&);
	glTexParameterfv(target,pname,params);
	end_function_trace();
}

void TexParameteri(GLenum target,GLenum pname,GLint param)
{
	begin_function_trace("glTexParameteri", sdpGLenum(target) + "," + sdpGLenum(pname) + "," + sdpGLint(param));
	glTexParameteri(target,pname,param);
	end_function_trace();
}

void TexParameteriv(GLenum target,GLenum pname,const GLint *params)
{
	begin_function_trace("glTexParameteriv", sdpGLenum(target) + "," + sdpGLenum(pname));//,const GLint *params",&);
	glTexParameteriv(target,pname,params);
	end_function_trace();
}

void TexSubImage1D(GLenum target,GLint level,GLint xoffset,GLsizei width,GLenum format,GLenum type,const GLvoid *pixels)
{
	begin_function_trace("glTexSubImage1D", sdpGLenum(target) + "," + sdpGLint(level) + "," + sdpGLint(xoffset) + "," + sdpGLsizei(width) + "," + sdpGLsizei(format) + "," + sdpGLenum(type));//,const GLvoid *pixels",&);
	glTexSubImage1D(target,level,xoffset,width,format,type,pixels);
	end_function_trace();
}

void TexSubImage2D(GLenum target,GLint level,GLint xoffset,GLint yoffset,GLsizei width,GLsizei height,GLenum format,GLenum type,const GLvoid *pixels)
{
	begin_function_trace("glTexSubImage2D", sdpGLenum(target) + "," + sdpGLint(level) + "," + sdpGLint(xoffset) + "," + sdpGLint(yoffset) + "," + sdpGLsizei(width) + "," + sdpGLsizei(height) + "," + sdpGLenum(format) + "," + sdpGLenum(type));//,const GLvoid *pixels",&);
	glTexSubImage2D(target,level,xoffset,yoffset,width,height,format,type,pixels);
	end_function_trace();
}

void Translated(GLdouble x,GLdouble y,GLdouble z)
{
	begin_function_trace("glTranslated", sdpGLdouble(x) + "," + sdpGLdouble(y) + "," + sdpGLdouble(z));
	glTranslated(x,y,z);
	end_function_trace();
}

void Translatef(GLfloat x,GLfloat y,GLfloat z)
{
	begin_function_trace("glTranslatef", sdpGLfloat(x) + "," + sdpGLfloat(y) + "," + sdpGLfloat(z));
	glTranslatef(x,y,z);
	end_function_trace();
}

void Vertex2d(GLdouble x,GLdouble y)
{
	begin_function_trace("glVertex2d", sdpGLdouble(x) + "," + sdpGLdouble(y));
	glVertex2d(x,y);
	end_function_trace();
}

void Vertex2dv(const GLdouble *v)
{
	begin_function_trace("glVertex2dv", ""); //,const GLdouble );
	glVertex2dv(v);
	end_function_trace();
}

void Vertex2f(GLfloat x,GLfloat y)
{
	begin_function_trace("glVertex2f", sdpGLfloat(x) + "," + sdpGLfloat(y));
	glVertex2f(x,y);
	end_function_trace();
}

void Vertex2fv(const GLfloat *v)
{
	begin_function_trace("glVertex2fv", sdpGLfloat(v[0]) + "," + sdpGLfloat(v[1]));
	glVertex2fv(v);
	end_function_trace();
}

void Vertex2i(GLint x,GLint y)
{
	begin_function_trace("glVertex2i", sdpGLint(x) + "," + sdpGLint(y));
	glVertex2i(x,y);
	end_function_trace();
}

void Vertex2iv(const GLint *v)
{
	begin_function_trace("glVertex2iv", ""); //,const GLint );
	glVertex2iv(v);
	end_function_trace();
}

void Vertex2s(GLshort x,GLshort y)
{
	begin_function_trace("glVertex2s", sdpGLshort(x) + "," + sdpGLshort(y));
	glVertex2s(x,y);
	end_function_trace();
}

void Vertex2sv(const GLshort *v)
{
	begin_function_trace("glVertex2sv", ""); //,const GLshort );
	glVertex2sv(v);
	end_function_trace();
}

void Vertex3d(GLdouble x,GLdouble y,GLdouble z)
{
	begin_function_trace("glVertex3d", sdpGLdouble(x) + "," + sdpGLdouble(y) + "," + sdpGLdouble(z));
	glVertex3d(x,y,z);
	end_function_trace();
}

void Vertex3dv(const GLdouble *v)
{
	begin_function_trace("glVertex3dv", sdpGLdouble(v[0]) + "," + sdpGLdouble(v[1]) + "," + sdpGLdouble(v[2]));
	glVertex3dv(v);
	end_function_trace();
}

void Vertex3f(GLfloat x,GLfloat y,GLfloat z)
{
	begin_function_trace("glVertex3f", sdpGLfloat(x) + "," + sdpGLfloat(y) + "," + sdpGLfloat(z));
	glVertex3f(x,y,z);
	end_function_trace();
}

void Vertex3fv(const GLfloat *v)
{
	begin_function_trace("glVertex3fv", sdpGLfloat(v[0]) + "," + sdpGLfloat(v[1]) + "," + sdpGLfloat(v[2]));
	glVertex3fv(v);
	end_function_trace();
}

void Vertex3i(GLint x,GLint y,GLint z)
{
	begin_function_trace("glVertex3i", sdpGLint(x) + "," + sdpGLint(y) + "," + sdpGLint(z));
	glVertex3i(x,y,z);
	end_function_trace();
}

void Vertex3iv(const GLint *v)
{
	begin_function_trace("glVertex3iv", ""); //,const GLint );
	glVertex3iv(v);
	end_function_trace();
}

void Vertex3s(GLshort x,GLshort y,GLshort z)
{
	begin_function_trace("glVertex3s", sdpGLshort(x) + "," + sdpGLshort(y) + "," + sdpGLshort(z));
	glVertex3s(x,y,z);
	end_function_trace();
}

void Vertex3sv(const GLshort *v)
{
	begin_function_trace("glVertex3sv", ""); //,const GLshort );
	glVertex3sv(v);
	end_function_trace();
}

void Vertex4d(GLdouble x,GLdouble y,GLdouble z,GLdouble w)
{
	begin_function_trace("glVertex4d", sdpGLdouble(x) + "," + sdpGLdouble(y) + "," + sdpGLdouble(z) + "," + sdpGLdouble(w));
	glVertex4d(x,y,z,w);
	end_function_trace();
}

void Vertex4dv(const GLdouble *v)
{
	begin_function_trace("glVertex4dv", sdpGLdouble(v[0]) + "," + sdpGLdouble(v[1]) + "," + sdpGLdouble(v[2]) + "," + sdpGLdouble(v[3]));
	glVertex4dv(v);
	end_function_trace();
}

void Vertex4f(GLfloat x,GLfloat y,GLfloat z,GLfloat w)
{
	begin_function_trace("glVertex4f", sdpGLfloat(x) + "," + sdpGLfloat(y) + "," + sdpGLfloat(z) + "," + sdpGLfloat(w));
	glVertex4f(x,y,z,w);
	end_function_trace();
}

void Vertex4fv(const GLfloat *v)
{
	begin_function_trace("glVertex4fv", sdpGLfloat(v[0]) + "," + sdpGLfloat(v[1]) + "," + sdpGLfloat(v[2]) + "," + sdpGLfloat(v[3]));
	glVertex4fv(v);
	end_function_trace();
}

void Vertex4i(GLint x,GLint y,GLint z,GLint w)
{
	begin_function_trace("glVertex4i", sdpGLint(x) + "," + sdpGLint(y) + "," + sdpGLint(z) + "," + sdpGLint(w));
	glVertex4i(x,y,z,w);
	end_function_trace();
}

void Vertex4iv(const GLint *v)
{
	begin_function_trace("glVertex4iv", ""); //,const GLint );
	glVertex4iv(v);
	end_function_trace();
}

void Vertex4s(GLshort x,GLshort y,GLshort z,GLshort w)
{
	begin_function_trace("glVertex4s", sdpGLshort(x) + "," + sdpGLshort(y) + "," + sdpGLshort(z) + "," + sdpGLshort(w));
	glVertex4s(x,y,z,w);
	end_function_trace();
}

void Vertex4sv(const GLshort *v)
{
	begin_function_trace("glVertex4sv", ""); //,const GLshort );
	glVertex4sv(v);
	end_function_trace();
}

void VertexPointer(GLint size,GLenum type,GLsizei stride,const GLvoid *pointer)
{
	begin_function_trace("glVertexPointer", sdpGLint(size) + "," + sdpGLenum(type) + "," + sdpGLsizei(stride));//,const GLvoid *pointer");
	glVertexPointer(size,type,stride,pointer);
	end_function_trace();
}

void Viewport(GLint x,GLint y,GLsizei width,GLsizei height)
{
	begin_function_trace("glViewport", sdpGLint(x) + "," + sdpGLint(y) + "," + sdpGLsizei(width) + "," + sdpGLsizei(height));
	glViewport(x,y,width,height);
	end_function_trace();
}

} // namespace implementation_private

} // namespace sdpgl


