// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
		\author Wladyslaw Strugala (fizws@julia.univ.gda.pl)
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/file_helpers.h>
#include <k3dsdk/i18n.h>
#include <k3dsdk/imaterial.h>
#include <k3dsdk/material.h>
#include <k3dsdk/mesh_sink.h>
#include <k3dsdk/module.h>
#include <k3dsdk/node.h>
#include <k3dsdk/persistent.h>
#include <k3dsdk/version.h>

using namespace k3d::xml;

#include <k3dsdk/fstream.h>

namespace libk3dgeometry
{

/////////////////////////////////////////////////////////////////////////////
// obj_mesh_output_implementation

class obj_mesh_output_implementation :
	public k3d::mesh_sink<k3d::persistent<k3d::node> >
{
	typedef k3d::mesh_sink<k3d::persistent<k3d::node> > base;

public:
	obj_mesh_output_implementation(k3d::idocument& Document) :
		base(Document),
		m_file(init_owner(*this) + init_name("file") + init_label(_("File")) + init_description(_("Output file")) + init_value(k3d::filesystem::path()) + init_path_mode(k3d::ipath_property::WRITE) + init_path_type("obj_files"))
	{
		m_file.changed_signal().connect(sigc::mem_fun(*this, &obj_mesh_output_implementation::on_write_file));
		m_input_mesh.changed_signal().connect(sigc::mem_fun(*this, &obj_mesh_output_implementation::on_write_file));
	}

	void on_write_file(k3d::iunknown*)
	{
		const k3d::filesystem::path path = m_file.value();
		k3d::mesh* const mesh = m_input_mesh.value();

		if(!mesh || path.empty())
			return;

		k3d::log() << info << "Writing " << path.native_console_string() << " with " << factory().name() << std::endl;

		k3d::filesystem::ofstream file(path);
		if(!file)
		{
			k3d::log() << error << k3d_file_reference << ": error opening [" << path.native_console_string() << "]" << std::endl;
			return;
		}

		file << "# Written by K-3D " << K3D_VERSION << "\n\n";

		// Create a one-based index of points ...
		std::map<k3d::point*, unsigned long> point_map;
		point_map[0] = 0;
		for(k3d::mesh::points_t::const_iterator point = mesh->points.begin(); point != mesh->points.end(); ++point)
			point_map.insert(std::make_pair(*point, point_map.size()));

		// Store points ...
		for(k3d::mesh::points_t::const_iterator point = mesh->points.begin(); point != mesh->points.end(); ++point)
			file << "v " << (*point)->position[0] << " " << (*point)->position[1] << " " << (*point)->position[2] << "\n";

		// Store polyhedra ...
		for(k3d::mesh::polyhedra_t::const_iterator polyhedron = mesh->polyhedra.begin(); polyhedron != mesh->polyhedra.end(); ++polyhedron)
		{
			for(k3d::polyhedron::faces_t::const_iterator face = (*polyhedron)->faces.begin(); face != (*polyhedron)->faces.end(); ++face)
			{
				file << "f";
				for(k3d::split_edge* edge = (*face)->first_edge; edge; edge = edge->face_clockwise)
				{
					file << " " << point_map[edge->vertex];
					if(edge->face_clockwise == (*face)->first_edge)
						break;
				}
				file << "\n";
			}
		}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<obj_mesh_output_implementation>, k3d::interface_list<k3d::imesh_sink > > factory(
			k3d::uuid(0x32120889, 0x85964fd3, 0x8dac7deb, 0xe3fc9676),
			"OBJMeshOutput",
			_("Mesh sink that saves external Wavefront (.obj) files"),
			"GeometryWriter",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(k3d::filesystem::path, immutable_name, change_signal, with_undo, local_storage, no_constraint, path_property, path_serialization) m_file;
};

k3d::iplugin_factory& obj_mesh_output_factory()
{
	return obj_mesh_output_implementation::get_factory();
}

} // namespace libk3dgeometry


