#ifndef KADU_SOUND_H
#define KADU_SOUND_H

#include <QtCore/QObject>
#include <QtCore/QString>
#include <QtCore/QDateTime>
#include <QtCore/QStringList>
#include <QtCore/QMap>
#include <QtCore/QThread>
#include <QtCore/QMutex>
#include <QtCore/QSemaphore>

#include "../notify/notify.h"

#include "config_file.h"
#include "gadu.h"
#include "main_configuration_window.h"
#include "misc.h"
#include "message_box.h"
#include "modules.h"
#include "themes.h"
#include "usergroup.h"
#include "userlist.h"

#include "sound_exports.h"

/**
 * @defgroup sound Sound
 * The sound module.
 * @{
 */

/**
	Uoglniony deskryptor urzdzenia dwikowego.
**/
typedef void* SoundDevice;

/**
**/
enum SoundDeviceType {RECORD_ONLY, PLAY_ONLY, PLAY_AND_RECORD};

/**
	To jest klasa uywana wewntrznie przez klas SoundManager
	i nie powiniene mie potrzeby jej uywania.
**/
class SamplePlayThread : public QThread
{
	Q_OBJECT

	private:
		SoundDevice Device;
		const int16_t* Sample;
		int SampleLen;
		bool Stopped;
		QSemaphore PlayingSemaphore;
		QSemaphore SampleSemaphore;

	protected:
		virtual void run();
		virtual bool event(QEvent* event);

	public:
		SamplePlayThread(SoundDevice device);
		void playSample(const int16_t* data, int length);
		void stop();

	signals:
		void samplePlayed(SoundDevice device);
};

/**
	To jest klasa uywana wewntrznie przez klas SoundManager
	i nie powiniene mie potrzeby jej uywania.
**/
class SampleRecordThread : public QThread
{
	Q_OBJECT

	private:
		SoundDevice Device;
		int16_t* Sample;
		int SampleLen;
		bool Stopped;
		QSemaphore RecordingSemaphore;
		QSemaphore SampleSemaphore;

	protected:
		virtual void run();
		virtual bool event(QEvent* event);

	public:
		SampleRecordThread(SoundDevice device);
		void recordSample(int16_t* data, int length);
		void stop();

	signals:
		void sampleRecorded(SoundDevice device);
};

/**
	To jest klasa uywana wewntrznie przez klas SoundManager
	i nie powiniene mie potrzeby jej uywania.
**/
class SndParams
{
	public:
		SndParams(QString fm = QString::null, bool volCntrl = false, float vol = 1);
		SndParams(const SndParams &p);

		QString filename;
		bool volumeControl;
		float volume;
};

/**
	To jest klasa uywana wewntrznie przez klas SoundManager
	i nie powiniene mie potrzeby jej uywania.
**/
class SoundPlayThread : public QThread
{
	public:
		SoundPlayThread();
		~SoundPlayThread();
		void run();
		void tryPlay(const char *path, bool volCntrl=false, float volume=1.0);
		void endThread();

	private:
		static bool play(const char *path, bool volCntrl=false, float volume=1.0);
		QMutex mutex;
		QSemaphore *semaphore;
		bool end;
		QList<SndParams> list;
};

class SoundConfigurationWidget;

class SOUNDAPI SoundManager : public Notifier, public ConfigurationUiHandler
{
    Q_OBJECT
	private:
		Themes *themes;
		ConfigComboBox *themesComboBox;
		PathListEdit *themesPaths;
		SoundConfigurationWidget *configurationWidget;

		friend class SamplePlayThread;
		friend class SampleRecordThread;
		QTime lastsoundtime;
		bool mute;
		QMap<SoundDevice, SamplePlayThread*> PlayingThreads;
		QMap<SoundDevice, SampleRecordThread*> RecordingThreads;
		SoundPlayThread *play_thread;

		int simple_player_count;
		virtual void connectNotify(const char *signal);
		virtual void disconnectNotify(const char *signal);

		void playSound(const QString &soundName);

		void copyConfiguration(const QString &fromEvent, const QString &toEvent) {}

		void applyTheme(const QString &themeName);
		void import_0_5_0_configuration();
		void createDefaultConfiguration();

	private slots:
		void setSoundThemes();
		void configurationWindowApplied();

	public slots:
		void play(const QString &path, bool force=false);
		void play(const QString &path, bool volCntrl, double vol);
		void setMute(const bool& enable);
		void stop();

	public:

		SoundManager(bool firstLoad, const QString& name, const QString& configname);
		~SoundManager();

		virtual void mainConfigurationWindowCreated(MainConfigurationWindow *mainConfigurationWindow);
		virtual NotifierConfigurationWidget *createConfigurationWidget(QWidget *parent = 0, char *name = 0);

		virtual void notify(Notification *notification);

		Themes *theme();

		bool isMuted() const;
		int timeAfterLastSound() const;
		/**
			Otwiera urzdzenie dwikowe do operacji
			odtwarzania i nagrywania sampli.
			Niektre implementacje pozwalaj na otwarcie
			wielu niezalenie dziaajcych "pocze"
			z urzdzeniami. Wystarczy wtedy kilkukrotnie
			wywoa t metod.
			Emituje sygna openDeviceImpl() w celu
			przekazania dania do konkrentego moduu
			dwikowego.
			@param type
			@param sample_rate sample rate - np. 8000 lub 48000
			@param channels ilo kanaw: 1 - mono, 2 - stereo
			@return uoglniony deskryptor urzdzenia lub NULL jeli otwarcie si nie powiodo.
		**/
		SoundDevice openDevice(SoundDeviceType type, int sample_rate, int channels = 1);
		/**
			Zamyka urzdzenie dwikowe otwarte za pomoc
			metody openDevice().
			Niektre implementacje pozwalaj na otwarcie
			wielu niezalenie dziaajcych "pocze"
			z urzdzeniami. Kade otwarte poczenie naley
			zamkn za pomoc tej metody.
			Jeli wczylimy operacje nieblokujce to
			metoda ta czeka na zakoczenie trwajcych operacji
			i koczy dziaanie wtkw.
			Emituje sygna closeDeviceImpl() w celu
			przekazania dania do konkrentego moduu
			dwikowego.
			@param device uoglniony deskryptor urzdzenia.
		**/
		void closeDevice(SoundDevice device);
		/**
			Powouje do ycia wtki zajmujce si odtwarzaniem
			i nagrywaniem prbek dla danego poczenia z
			urzdzeniem dwikowym.
			Od tej chwili playSample() i recordSample()
			bd operacjami nieblokujcymi.
			@param device uoglniony deskryptor urzdzenia.
		**/
		void enableThreading(SoundDevice device);
		/**
			Standardowo po przekazaniu odtwarzanego sampla
			do sterownika dwikowego program (w trybie blokujcym)
			lub wtek odtwarzajcy (w trybie nieblokujcym) czeka
			na zakoczenie emitowania dwiku przez sterownik, aby
			zagwarantowa, e wyjcie z funkcji playSample() (w trybie
			blokujcym) lub wyemitowanie sygnau samplePlayed() (w
			trybie nieblokujcym) nastpi dopiero po fizycznym
			odegraniu dwiku. Inaczej mogo by to nastpi dopiero
			przy wywoaniu metody closeDevice() co ma rne nieprzyjemne
			efekty uboczne.
			Czasem przy odtwarzaniu kilku maych sampli jeden po drugim
			powoduje to powstanie przerw midzy nimi. Aby tego unikn
			moemy uy setFlushingEnabled(device, false) umoliwiajc
			w ten sposb pynne odtwarzanie kilku sampli bezporednio
			po sobie.
		**/
		void setFlushingEnabled(SoundDevice device, bool enabled);
		/**
			Odtwarza prbk dwikow. Standardowo jest to
			operacja blokujca. Moe by wywoana z innego
			wtku (a nawet powinna).
			Emituje sygna playSampleImpl() w celu
			przekazania dania do konkrentego moduu
			dwikowego.
			Po uprzednim wywoaniu enableThreading() dziaanie
			metoda jest nieblokujca i przekazuje jedynie polecenie
			odtwarzania do wtku.
			W takim wypadku naley uwaa, aby nie zwolni pamici
			zajmowanej przez dane sampla zanim odtwarzanie si nie
			zakoczy.
			@param device uoglniony deskryptor urzdzenia
			@param data wskanik do danych sampla
			@param length dugo danych sampla (w bajtach)
			@return true jeli odtwarzanie zakoczyo si powodzeniem.
		**/
		bool playSample(SoundDevice device, const int16_t* data, int length);
		/**
			Nagrywa prbk dwikow. Standardowo jest to
			operacja blokujca. Moe by wywoana z innego
			wtku (a nawet powinna).
			Emituje sygna recordSampleImpl() w celu
			przekazania dania do konkrentego moduu
			dwikowego.
			Po uprzednim wywoaniu enableThreading() dziaanie
			metoda jest nieblokujca i przekazuje jedynie polecenie
			nagrywania do wtku.
			W takim wypadku naley uwaa, aby nie zwolni pamici
			bufora na dane sampla zanim nagrywanie si nie
			zakoczy.
			@param device uoglniony deskryptor urzdzenia
			@param data wskanik na bufor dla danych sampla
			@param length dugo sampla do nagrania (wielko bufora w bajtach)
			@return true jeli nagrywanie zakoczyo si powodzeniem.
		**/
		bool recordSample(SoundDevice device, int16_t* data, int length);

	signals:
		void playSound(const QString &sound, bool volCntrl, double vol);
		/**
			Sygna emitowany gdy odtwarzanie sampla si
			zakoczyo (odnosi si tylko do sytuacji gdy
			wczone s operacje nieblokujce).
		**/
		void samplePlayed(SoundDevice device);
		/**
			Sygna emitowany gdy nagrywanie sampla si
			zakoczyo (odnosi si tylko do sytuacji gdy
			wczone s operacje nieblokujce).
		**/
		void sampleRecorded(SoundDevice device);
		/**
			Pod ten sygna powinien podpi si modu
			dwikowy jeli obsuguje funkcj odtwarzania
			prbki dwikowej.
			Wyemitowanie sygnau oznacza danie
			otwarcia urzdzenia dwikowego do operacji
			odtwarzania i nagrywania sampli.
			@param type
			@param sample_rate sample rate - np. 8000 lub 48000
			@param channels ilo kanaw: 1 - mono, 2 - stereo
			@device zwrcony uoglniony deskryptor urzdzenia lub NULL jeli otwarcie si nie powiodo.
			@param mutex - mutex to be unlock after setting the device
		**/
		void openDeviceImpl(SoundDeviceType type, int sample_rate, int channels, SoundDevice* device);
		/**
			Pod ten sygna powinien podpi si modu
			dwikowy jeli obsuguje funkcj odtwarzania
			prbki dwikowej.
			Wyemitowanie sygnau oznacza danie
			Zamknicia urzdzenia dwikowego otwartegp za pomoc
			metody openDevice().
			@param device uoglniony deskryptor urzdzenia.
		**/
		void closeDeviceImpl(SoundDevice device);
		/**
			Pod ten sygna powinien podpi si modu
			dwikowy jeli obsuguje funkcj odtwarzania
			prbki dwikowej.
			Wyemitowanie sygnau oznacza danie
			odtworzenia prbki dwikowej.
			Modu powinien odda sterowanie dopiero po
			odtworzeniu prbki.
			Sygna zazwyczaj bdzie emitowany z innego
			wtku i slot musi by do tego przystosowany.
			@param device uoglniony deskryptor urzdzenia
			@param data wskanik do danych sampla
			@param length dugo danych sampla (w bajtach)
			@param result zwrcony rezultat operacji - true jeli odtwarzanie zakoczyo si powodzeniem.
		**/
		void playSampleImpl(SoundDevice device, const int16_t* data, int length, bool *result);
		/**
			Pod ten sygna powinien podpi si modu
			dwikowy jeli obsuguje funkcj odtwarzania
			prbki dwikowej.
			Wyemitowanie sygnau oznacza danie
			nagrania prbki dwikowej.
			Modu powinien odda sterowanie dopiero po
			nagraniu prbki.
			Sygna zazwyczaj bdzie emitowany z innego
			wtku i slot musi by do tego przystosowany.
			@param device uoglniony deskryptor urzdzenia
			@param data wskanik na bufor dla danych sampla
			@param length dugo sampla do nagrania (wielko bufora w bajtach)
			@param result zwrcony rezultat operacji - true jeli nagrywanie zakoczyo si powodzeniem.
		**/
		void recordSampleImpl(SoundDevice device, int16_t* data, int length, bool *result);
		/**
		**/
		void setFlushingEnabledImpl(SoundDevice device, bool enabled);
};

extern SOUNDAPI SoundManager* sound_manager;
/** @} */

#endif
