/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

#include "KayaAPI.h"
#include "Heap.h"
#include "Array.h"
#include "VM.h"
#include "VMState.h"

#include <stdarg.h>

KValue newKayaValue()
{
    return new Value(NULL,KVT_NULL);
}

KValue KayaInt(kint x)
{
    KValue v = newKayaValue();
    KayaSetInt(v,x);
    return v;
}

KValue KayaChar(wchar_t c)
{
    KValue v = newKayaValue();
    KayaSetChar(v,c);
    return v;
}

KValue KayaFloat(double f)
{
    KValue v = newKayaValue();
    KayaSetFloat(v,f);
    return v;
}

KValue KayaString(wchar_t* str)
{
    KValue v = newKayaValue();
    KayaSetString(v,str);
    return v;
}

KValue KayaCharString(char* str)
{
    KValue v = newKayaValue();
    KayaSetCharString(v,str);
    return v;
}

kint KayaGetInt(KValue v)
{
    return v->getInt();
}

wchar_t KayaGetChar(KValue v)
{
    return (wchar_t)(v->getInt());
}

double KayaGetFloat(KValue v)
{
    return v->getReal();
}

wchar_t* KayaGetString(KValue v)
{
    return v->getString()->getVal();
}

void KayaSetInt(KValue v,kint i)
{
    v->setInt(i);
}

void KayaSetChar(KValue v,wchar_t c)
{
    v->setInt((int)c);
}

void KayaSetFloat(KValue v,double f)
{
    v->setReal(f);
}

void KayaSetString(KValue v,wchar_t* s)
{
    v->setString(new String(s));
}

void KayaSetCharString(KValue v,char* s)
{
    v->setString(new String(strtowc(s)));
}


KArray newKayaArray(int size)
{
    return new Array(size);
}

KArray KayaGetArray(KValue v)
{
    return v->getArray();
}

void KayaSetArray(KValue v, KArray a)
{
    v->setArray(a);
}


void KayaArrayPush(KArray a,KValue v)
{
    a->push_back(v);
}

void KayaArraySet(KArray a, int index, KValue v)
{
    Value* vloc = a->lookup(index);
    vloc->setPtr(v);
}

KValue KayaArrayLookup(KArray a, int index)
{
    return a->lookup(index);
}

int KayaArraySize(KArray a)
{
    return a->size();
}

KValue KayaCall(KValue fn, int args, ...)
{
    VMState* vm = initstack();
//    Value* arg;
    Value** arglist = new Value*[args];
    va_list argp;
    va_start(argp,args);
    int i;
    // Gah! Need to push backwards!
    for(i=0;i<args;i++) {
	arglist[args-i-1] = va_arg(argp,Value*);
    }
    va_end(argp);
    for(i=0;i<args;i++) {
	PUSH(arglist[i]);
    }
    delete arglist;
    TRY(cbexcept);
    CALLFUN(fn);
    if (vm->emptyStack()) {
	return NULL;
    }
    else
	return vm->doPop();

    LABEL(cbexcept); // There was an exception in the callback. This is fatal.
    cerr << "Exception in callback: ";
    Exception* except = vm->doPop()->getExcept();
    except->show();
    exit(1);
}

KValue KayaUnion(int tag, int args)
{
    Union* u = new Union(NULL,tag,args,false);
    return new Value((void*)u,KVT_UNION);
}

KValue KayaArrayVal(KArray ar)
{
    KValue v = newKayaValue();
    KayaSetArray(v,ar);
    return v;
}

int KayaUnionGetTag(KValue v)
{
    Union* u = v->getUnion();
    return U_TAG(u);
}

KValue KayaUnionGetArg(KValue v,int i)
{
    Union* u = v->getUnion();
    return u->args[i];
}

void KayaUnionSetArg(KValue v,int i,KValue val)
{
    Union* u = v->getUnion();
    u->args[i] = val;
}

void* KayaAlloc(int size)
{
    // Just use libgc...
    return GC_MALLOC_ATOMIC(size);
}

int orArray(KArray a, int(*func)(KValue))
{
    int val = 0;
    int size = KayaArraySize(a);
    for(int i=0;i<size;i++) {
	KValue v = KayaArrayLookup(a,i);
	val = val | func(v);
    }
    return val;
}
