/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Entry Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Entry Public License for more details.
 *
 *  You should have received a copy of the GNU Entry Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "prefs_entry.h"

#include <glib/gi18n.h>
#include "kazehakase.h"
#include "utils/gtk-utils.h"

#define DATA_KEY "KzPrefsEntry::info"


static GtkWidget     *prefs_entry_create          (void);
static void           prefs_entry_response        (GtkWidget *widget,
						   gint       response);


static KzPrefsWinPageEntry prefs_entry =
{
	path:            N_("/Entry"),
	priority_hint:   0,
	ui_level:        ~KZ_UI_LEVEL_BEGINNER,
	create:          prefs_entry_create,
	response:        prefs_entry_response,
};


typedef struct _KzPrefsEntry
{
	GtkWidget     *main_vbox;
	GtkWidget     *open_in_new_tab;
	GtkWidget     *auto_completion;
	GtkWidget     *new_tab_mask;
	GtkWidget     *inline_completion;
	gboolean       changed;
} KzPrefsEntry;


static void
prefs_entry_destroy (gpointer data)
{
	KzPrefsEntry *prefsui = data;

	g_free(prefsui);
}

static void
cb_changed (GtkWidget *widget, KzPrefsEntry *prefsui)
{
	prefsui->changed = TRUE;
}

static void
cb_inline_changed (GtkWidget *widget, KzPrefsEntry *prefsui)
{
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)))
	{
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON(prefsui->auto_completion), TRUE);
	}
	
	cb_changed(widget, prefsui);
}

static GtkWidget *
prefs_entry_create (void)
{
	KzPrefsEntry *prefsui = g_new0(KzPrefsEntry, 1);
	GtkWidget *main_vbox, *vbox, *hbox, *frame;
	GtkWidget *label, *check, *combo;
	GtkListStore *store;
	GtkTreeIter iter;
	GtkCellRenderer *renderer;
	gboolean newtab = FALSE, autocompletion = FALSE;
	guint mask = GDK_CONTROL_MASK;
	gboolean inline_completion = FALSE;

	main_vbox = gtk_vbox_new(FALSE, 0);
	prefsui->main_vbox = main_vbox;
	g_object_set_data_full(G_OBJECT(main_vbox), DATA_KEY,
			       prefsui, prefs_entry_destroy);

	label = kz_prefs_ui_utils_create_title(_("URL and keyword Entry"));
	gtk_box_pack_start(GTK_BOX(main_vbox), label,
			   FALSE, FALSE, 0);
	gtk_widget_show(label);

	frame = gtk_frame_new(_("How to open"));
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, FALSE, 2);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	gtk_widget_show(vbox);

	/*
	 *  Entry Box behaviour
	 */
	check = gtk_check_button_new_with_mnemonic(_("Open in _new tab by default"));
	prefsui->open_in_new_tab = check;
	gtk_box_pack_start(GTK_BOX(vbox), check, FALSE, FALSE, 2);
	KZ_CONF_GET("Global", "entry_open_in_new_tab", newtab, BOOL);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), newtab);
	g_signal_connect(check, "toggled",
			 G_CALLBACK(cb_changed), prefsui);
	gtk_widget_show(check);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);
	gtk_widget_show(hbox);

	label = gtk_label_new_with_mnemonic(_("_Switch the behavior from default to inverse with"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
	gtk_widget_show(label);

	KZ_CONF_GET("Tab", "new_tab_mask", mask, INT);
	store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, 0, "Shift Key", 1, GDK_SHIFT_MASK, -1);
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, 0, "Control Key", 1, GDK_CONTROL_MASK, -1);
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, 0, "Alt Key", 1, GDK_MOD1_MASK, -1);
	prefsui->new_tab_mask = combo = gtk_combo_box_new_with_model(GTK_TREE_MODEL(store));
	renderer = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(combo), renderer, TRUE);
	gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(combo), renderer, "text", 0, NULL);

	switch (mask)
	{
	case GDK_SHIFT_MASK:
		mask = 0;
		break;
	case GDK_CONTROL_MASK:
		mask = 1;
		break;
	case GDK_MOD1_MASK:
		mask = 2;
		break;
	default:
		mask = 1;
		break;		
	}
	gtk_combo_box_set_active(GTK_COMBO_BOX(combo), mask);

	gtk_widget_show(combo);
	gtk_box_pack_start (GTK_BOX(hbox), combo, TRUE, TRUE, 0);
	g_signal_connect(combo, "changed",
			 G_CALLBACK(cb_changed), prefsui);
	/* auto-completion */
	frame = gtk_frame_new(_("Behavior on inputting"));
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, FALSE, 2);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	gtk_widget_show(vbox);

	check = gtk_check_button_new_with_mnemonic(_("Use _auto-completion in URL entry box"));
	prefsui->auto_completion = check;
	gtk_box_pack_start(GTK_BOX(vbox), check, FALSE, FALSE, 2);
	KZ_CONF_GET("Global", "use_auto_completion", autocompletion, BOOL);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), autocompletion);
	g_signal_connect(check, "toggled",
			 G_CALLBACK(cb_changed), prefsui);
	gtk_widget_show(check);

	/* inline-completion */
	check = gtk_check_button_new_with_mnemonic(_("Use _inline-completion in URL entry box"));
	prefsui->inline_completion = check;
	gtk_box_pack_start(GTK_BOX(vbox), check, FALSE, FALSE, 2);
	KZ_CONF_GET("Global", "use_inline_completion", inline_completion, BOOL);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check),
				     inline_completion);
	g_signal_connect(check, "toggled",
			 G_CALLBACK(cb_inline_changed), prefsui);
	gtk_widget_show(check);

	prefsui->changed = FALSE;

	return main_vbox;
}



static void
prefs_entry_response (GtkWidget *widget, gint response)
{
	KzPrefsEntry *prefsui = g_object_get_data(G_OBJECT(widget), DATA_KEY);

	g_return_if_fail(prefsui);

	switch (response) {
	case GTK_RESPONSE_ACCEPT:
	case GTK_RESPONSE_APPLY:
	{
		gboolean newtab, autocompletion, inline_completion;
		gint mask;
		GtkTreeModel *model;
		GtkTreeIter iter;

		if (!prefsui->changed)
			break;
	
                /* entry box behaviour */
		newtab = gtk_toggle_button_get_active
			(GTK_TOGGLE_BUTTON(prefsui->open_in_new_tab));
		/* Should change the section name? */
                KZ_CONF_SET("Global", "entry_open_in_new_tab", newtab, BOOL);

		model = gtk_combo_box_get_model(GTK_COMBO_BOX(prefsui->new_tab_mask));
		gtk_combo_box_get_active_iter(GTK_COMBO_BOX(prefsui->new_tab_mask), &iter);
		gtk_tree_model_get(model, &iter,
				   1, &mask, -1);
		
		KZ_CONF_SET("Tab", "new_tab_mask", mask, INT);

		/* auto-completion */
		autocompletion = gtk_toggle_button_get_active
			(GTK_TOGGLE_BUTTON(prefsui->auto_completion));
		/* Should change the section name? */
		KZ_CONF_SET("Global", "use_auto_completion", autocompletion, BOOL);

		/* inline-completion */
		inline_completion = gtk_toggle_button_get_active
			(GTK_TOGGLE_BUTTON(prefsui->inline_completion));
		/* Should change the section name? */
		KZ_CONF_SET("Global", "use_inline_completion",
			    inline_completion, BOOL);

                prefsui->changed = FALSE;

		break;
	}
	case GTK_RESPONSE_REJECT:
		break;
	case KZ_RESPONSE_UI_LEVEL_BEGINNER:
	{
		break;
	}
	case KZ_RESPONSE_UI_LEVEL_MEDIUM:
	{
		break;
	}
	case KZ_RESPONSE_UI_LEVEL_EXPERT:
	{
		break;
	}
	case KZ_RESPONSE_UI_LEVEL_CUSTOM:
		break;
	default:
		break;
	}
}


KzPrefsWinPageEntry *
prefs_entry_get_entry (gint idx)
{
	if (idx == 0)
		return &prefs_entry;
	else
		return NULL;
}
