/***************************************************************************
*   Copyright (C) 2004-2006 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <qregexp.h>
#include <qstring.h>
#include <qstringlist.h>

#include "value.h"

namespace BibTeX
{
    ValueItem::ValueItem( const QString& text, bool isStringKey )
            : m_isStringKey( isStringKey )
    {
        setText( text );
        setIsStringKey( isStringKey );
    }

    ValueItem::ValueItem( ValueItem *other )
    {
        setText( other->text() );
        setIsStringKey( other->isStringKey() );
    }

    QString ValueItem::text() const
    {
        return m_text;
    }

    bool ValueItem::isStringKey() const
    {
        return m_isStringKey;
    }

    void ValueItem::setText( const QString &text )
    {
        m_text = text;
    }

    void ValueItem::setIsStringKey( bool isStringKey )
    {
        m_isStringKey = isStringKey;
    }

    bool ValueItem::containsPattern( const QString & pattern, bool caseSensitive ) const
    {
        return m_text.contains( pattern, caseSensitive );
    }

    Value::Value( )
    {
        // nothing
    }

    Value::Value( const Value* other )
    {
        for ( QValueList<ValueItem*>::const_iterator it = other->m_valueItems.begin(); it != other->m_valueItems.end(); ++it )
            m_valueItems.append( new ValueItem( *it ) );
    }

    Value::~Value( )
    {
        for ( QValueList<ValueItem*>::ConstIterator it = m_valueItems.begin(); it != m_valueItems.end(); ++it )
            delete *it;
    }

    void Value::add ( ValueItem* valueItem )
    {
        m_valueItems.append( valueItem );
    }

    void Value::clear()
    {
        m_valueItems.clear();
    }

    BibTeX::ValueItem *Value::first()
    {
        if ( m_valueItems.isEmpty() )
            return NULL;
        else
            return m_valueItems.first();
    }

    bool Value::isEmpty() const
    {
        return m_valueItems.isEmpty();
    }

    int Value::count() const
    {
        return m_valueItems.count();
    }

    bool Value::containsPattern( const QString & pattern, bool caseSensitive ) const
    {
        bool result = FALSE;
        for ( QValueList<ValueItem*>::const_iterator it = m_valueItems.begin(); !result && it != m_valueItems.end(); ++it )
            result |= ( *it ) ->containsPattern( pattern, caseSensitive );
        return result;
    }

    bool Value::checkIsStringKey( const QString &text )
    {
        bool charsOK = TRUE;
        static QString validChars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKMNOPQRSTUVWXYZ0123456789";

        for ( unsigned int i = 0; charsOK && ( i < text.length() ); i++ )
            charsOK &= validChars.contains( text.at( i ) );

        return charsOK;
    }

    QValueList<BibTeX::ValueItem*>::const_iterator Value::begin() const
    {
        return m_valueItems.begin();
    }

    QValueList<BibTeX::ValueItem*>::const_iterator Value::end() const
    {
        return m_valueItems.end();
    }

    QString Value::plainString() const
    {
        QString result;

        for ( QValueList<ValueItem*>::const_iterator it = m_valueItems.begin(); it != m_valueItems.end(); ++it )
        {
            if ( !result.isEmpty() )
                result.append( " " );
            result.append( ( *it ) ->text() );
        }

        result.replace( QRegExp( "\\\\[A-Za-z0-9]+" ), "" ).replace( '{', "" ).replace( '}', "" );

        return result;
    }

    QString Value::debugString() const
    {
        QString result;

        for ( QValueList<ValueItem*>::const_iterator it = m_valueItems.begin(); it != m_valueItems.end(); ++it )
        {
            const ValueItem *item = *it;
            if ( !result.isEmpty() )
                result.append( " " );
            result.append( "[" );
            result.append( item->text() );
            if ( item->isStringKey() )
                result.append( "|KEY" );
            result.append( "]" );
        }

        result.replace( '{', "" ).replace( '}', "" );

        return result;
    }

    ValuePersons::ValuePersons() : Value()
    {
        // nothing
    }

    ValuePersons::ValuePersons( const BibTeX::ValuePersons* other ) : Value( other )
    {
        for ( QValueList<Person*>::const_iterator it = other->m_persons.begin(); it != other->m_persons.end(); ++it )
            m_persons.append( new Person( *it ) );
    }

    ValuePersons::~ValuePersons()
    {
        for ( QValueList<Person*>::ConstIterator it = m_persons.begin(); it != m_persons.end(); ++it )
            delete *it;
    }

    void ValuePersons::add( ValueItem* valueItem )
    {
        Value::add( valueItem );

        if ( !valueItem->isStringKey() )
        {
            QStringList names = QStringList::split( QRegExp( "\\s+and\\s+" ), valueItem->text() );
            for ( QStringList::Iterator it = names.begin(); it != names.end(); ++it )
                m_persons.append( new Person( *it ) );
        }
    }

    void ValuePersons::clear()
    {
        Value::clear();
        m_persons.clear();
    }

    bool ValuePersons::containsPattern( const QString & pattern, bool caseSensitive ) const
    {
        bool result = FALSE;

        for ( QValueList<Person*>::const_iterator it = m_persons.begin();!result && it != m_persons.end(); ++it )
            result |= ( *it ) ->containsPattern( pattern, caseSensitive );

        return result;
    }

    QValueList<Person*> ValuePersons::persons() const
    {
        return m_persons;
    }

    Person::Person( const QString& text )
    {
        parseText( text );
    }

    Person::Person( const Person* other )
    {
        m_lastName = other->m_lastName;
        m_firstName = other->m_firstName;
    }

    void Person::setText( const QString &text )
    {
        parseText( text );
    }

    QString Person::text() const
    {
        return QString( "%1, %2" ).arg( m_lastName ).arg( m_firstName );
    }

    QString Person::lastName() const
    {
        return m_lastName;
    }

    QString Person::firstName() const
    {
        return m_firstName;
    }

    bool Person::containsPattern( const QString & pattern, bool caseSensitive ) const
    {
        return m_lastName.contains( pattern, caseSensitive ) || m_firstName.contains( pattern, caseSensitive ) || text().contains( pattern, caseSensitive );
    }

    void Person::parseText( const QString &text )
    {
        QString ltext = QString( text );
        ltext = ltext.replace( "{", "" ).replace( "}", "" );

        if ( !ltext.contains( ',' ) )
        {
            QStringList splitName = QStringList::split( QRegExp( "\\s+" ), ltext );
            //             for ( QStringList::Iterator it = --splitName.end(); it != --splitName.begin(); --it )
            //                 qDebug( "splitName= %s", ( *it ).latin1() );

            QString lastItem = *splitName.at( splitName.count() - 1 );
            if ( splitName.count() == 2 && lastItem.compare( lastItem.upper() ) == 0 )
            {
                // this is a special case for names from PubMed, which are formatted like "Fischer T"
                m_firstName = lastItem;
                m_lastName = *splitName.begin();
            }
            else if ( splitName.count() > 0 )
            {
                int from = splitName.count() - 1;
                m_lastName = *splitName.at( from );
                while ( from > 0 )
                {
                    if ( ( *splitName.at( from - 1 ) ).compare( ( *splitName.at( from - 1 ) ).lower() ) != 0 )
                        break;
                    --from;
                    m_lastName.prepend( " " );
                    m_lastName.prepend( *splitName.at( from ) );
                }

                if ( from > 0 )
                {
                    m_firstName = *splitName.begin();
                    for ( QStringList::Iterator it = ++splitName.begin(); from > 1; ++it, --from )
                    {
                        m_firstName.append( " " );
                        m_firstName.append( *it );
                    }
                }
            }
        }
        else
        {
            QStringList splitName = QStringList::split( QRegExp( ",\\s+" ), ltext );
            if ( splitName.count() > 0 )
            {
                m_firstName = ( *splitName.at( splitName.count() - 1 ) ).stripWhiteSpace();
                if ( splitName.count() > 1 )
                    m_lastName = ( *splitName.begin() ).stripWhiteSpace();
            }
        }
    }

}
