/*
    This file is part of kio_obex.

    Copyright (c) 2003 Mathias Froehlich <Mathias.Froehlich@web.de>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include <qwidget.h>
#include <qheader.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qpushbutton.h>

#include <kdebug.h>
#include <klocale.h>
#include <kconfig.h>
#include <kgenericfactory.h>
#include <kcmodule.h>

#include "wizard.h"
#include "edit.h"

#include "obex.h"

typedef KGenericFactory<KObexConfigWidget, QWidget> KObexConfigFactory;
K_EXPORT_COMPONENT_FACTORY( kcm_obex, KObexConfigFactory( "kcmobex" ) )

extern "C"
{
  KCModule *create_kcm_obex( QWidget *parent, const char *name )
  {
    return new KObexConfigWidget(parent, name);
  };
}

KObexConfigWidget::KObexConfigWidget( QWidget *parent, const char *name, const QStringList& )
  : KCModule( parent, name ) {

  // FIXME does not really work ...
  setButtons( Help | Ok | Cancel | Apply | Reset );

  mTopWidgetLayout = new QGridLayout( this, 1, 1, 5, 6, "mTopWidgetLayout");

  mListView = new QListView( this, "mListView" );
  mListView->addColumn( i18n( "Alias" ) );

  mTopWidgetLayout->addMultiCellWidget( mListView, 2, 5, 0, 0 );

  mAddButton = new QPushButton( this, "mAddButton" );
  mTopWidgetLayout->addWidget( mAddButton, 2, 1 );

  mEditButton = new QPushButton( this, "mEditButton" );
  mEditButton->setEnabled( FALSE );
  mTopWidgetLayout->addWidget( mEditButton, 3, 1 );

  mRemoveButton = new QPushButton( this, "mRemoveButton" );
  mRemoveButton->setEnabled( FALSE );
  mTopWidgetLayout->addWidget( mRemoveButton, 4, 1 );

  QSpacerItem* spacer = new QSpacerItem( 16, 16, QSizePolicy::Minimum, QSizePolicy::Expanding );
  mTopWidgetLayout->addItem( spacer, 5, 1 );

  lblIntro = new QLabel( this, "lblIntro" );
  lblIntro->setTextFormat( QLabel::RichText );
  lblIntro->setAlignment( int( QLabel::WordBreak | QLabel::AlignTop ) );
  mTopWidgetLayout->addMultiCellWidget( lblIntro, 0, 0, 0, 1 );

  setCaption( i18n( "Form2" ) );
  mListView->header()->setLabel( 0, i18n( "Alias" ) );
//   mListView->header()->setLabel( 1, i18n( "Description" ) );
  mRemoveButton->setText( i18n( "&Remove" ) );
  mAddButton->setText( i18n( "&Add..." ) );
  mEditButton->setText( i18n( "&Edit..." ) );
  lblIntro->setText( i18n( "<b><u>OBEX Connection Aliases</u></b><br>\n"
			   "\n"
			   "OBEX connections can be established using different transports.\n"
			   "Use this to define connections and their parameters.<p>\n"
			   "\n"
			   "To use such a connection simply use the alias name as hostname in the url." ) );

  resize( QSize(429, 303).expandedTo(minimumSizeHint()) );
  clearWState( WState_Polished );

  // tab order
  setTabOrder( mListView, mAddButton );
  setTabOrder( mAddButton, mEditButton );
  setTabOrder( mEditButton, mRemoveButton );

  // signals and slots connections
  // Do something sensible.
  connect( mAddButton, SIGNAL(clicked()), SLOT(slotAdd()) );
  connect( mRemoveButton, SIGNAL(clicked()), SLOT(slotRemove()) );
  connect( mEditButton, SIGNAL(clicked()), SLOT(slotEdit()) );
  connect( mListView, SIGNAL(doubleClicked(QListViewItem*)), SLOT(slotEdit()) );

  // Activate edit and delete only if we have a selected item.
  connect( mListView, SIGNAL(selectionChanged()), SLOT(slotSelectionChanged()) );

  mAboutData
    = new KAboutData( "kcmobex", I18N_NOOP("KDE OBEX Configuration Control Module"), "1.0",
		      I18N_NOOP("Configure OBEX connections."), KAboutData::License_GPL_V2 );
  mAboutData->addAuthor( "Mathias Fr\xC3\xB6hlich", I18N_NOOP("Current maintainer"),
			 "Mathias.Froehlich@web.de" );

  // Load and evaluate the stored configuration.
  mObexConfig = new KConfig( "obexrc" );
  load();
}

KObexConfigWidget::~KObexConfigWidget() {
  mObexConfig->rollback();
  delete mObexConfig;
  delete mAboutData;
}

void KObexConfigWidget::load() {
  // clear listview
  mObexConfig->rollback();
  mObexConfig->reparseConfiguration();
  while ( QListViewItem* item = mListView->lastItem() ) {
    mListView->takeItem( item );
    delete item;
  }

  // Reread config file.
  QStringList groupList = mObexConfig->groupList();
  QStringList::Iterator it;
  for ( it = groupList.begin(); it != groupList.end(); ++it ) {
    mObexConfig->setGroup( *it );
    if ( mObexConfig->readEntry( "transport", QString::null ) != QString::null ) {
      QListViewItem *item = new QListViewItem( mListView );
      item->setText( 0, *it );
//       item->setText( 1, mObexConfig->readEntry( "description" ) );
      mListView->insertItem( item );
    }
  }
}

void KObexConfigWidget::save() {
  // write config file
  mObexConfig->sync();
}

QString KObexConfigWidget::quickHelp() const {
  return i18n("<h1>OBEX</h1> This module allows you to configure OBEX connection parameters.");
}

const KAboutData* KObexConfigWidget::aboutData() const {
  return mAboutData;
}

void KObexConfigWidget::slotRemove()
{
  QListViewItem *item = mListView->selectedItem();
  if ( !item )
    return;

  mListView->takeItem( item );
  mObexConfig->deleteGroup( item->text( 0 ) );
  delete item;

  // tell kcm that something has happened.
  emit changed( true );
}

void KObexConfigWidget::slotAdd()
{
  // Create the wizard
  Wizard wizard( this, "Wizard", true );

  // Run it ...
  if ( wizard.exec() ) {
    // ... and evaluate the result.
    QString name = wizard.name();
    mObexConfig->setGroup( name );
    QString transport = wizard.transport();
    mObexConfig->writeEntry( "transport", transport );
//     mObexConfig->writeEntry( "description", ha.description );
//     mObexConfig->writeEntry( "uuid", ha.uuid );
//     mObexConfig->writeEntry( "overridePerms", ha.overridePerms );
    mObexConfig->writeEntry( "disconnectTimeout", 1 );
    if ( transport == "ip" ) {
      mObexConfig->writeEntry( "peer", wizard.peer() );
      mObexConfig->writeEntry( "ipPort", wizard.ipPort() );
    }
    else if ( transport == "serial" || transport == "serialsiemens" || transport == "serialericsson" ) {
      mObexConfig->writeEntry( "device", wizard.device() );
      mObexConfig->writeEntry( "speed", wizard.speed() );
    }
    else if ( transport == "irda" || transport == "bluetooth" ) {
      mObexConfig->writeEntry( "peer", wizard.peer() );
    }

    QListViewItem *item = new QListViewItem( mListView );
    item->setText( 0, name );
//     item->setText( 1, "FIXME" );
    mListView->insertItem( item );
    emit changed( true );
  }
}

void KObexConfigWidget::slotEdit()
{
  QListViewItem *item = mListView->selectedItem();
  if ( !item )
    return;

  QString name = item->text( 0 );
  mObexConfig->setGroup( name );

  Edit edit( this, "Edit", true );
  edit.setName( name );
  QString transport = mObexConfig->readEntry( "transport" );
  edit.setTransport( transport );
  edit.setOverridePerms( mObexConfig->readNumEntry( "overridePerms" ) );
  edit.setDisconnectTimeout( mObexConfig->readNumEntry( "disconnectTimeout", 1 ) );

  if ( transport == "ip" ) {
    edit.setPeer( mObexConfig->readEntry( "peer" ) );
    edit.setIpPort( mObexConfig->readNumEntry( "ipPort" ) );
  }
  else if ( transport == "irda" || transport == "bluetooth" ) {
    edit.setPeer( mObexConfig->readEntry( "peer" ) );
  }
  else if ( transport == "serial" ||
	    transport == "serialsiemens" ||
	    transport == "serialericsson" ) {
    edit.setDevice( mObexConfig->readEntry( "device" ) );
    edit.setSpeed( mObexConfig->readNumEntry( "speed" ) );
  }

  if ( edit.exec() ) {
    if ( name != edit.name() ) {
      mObexConfig->deleteGroup( name );
      name = edit.name();
      item->setText( 0, name );
      mObexConfig->setGroup( name );
    }
    transport = edit.transport();
    mObexConfig->writeEntry( "transport", transport );
    mObexConfig->writeEntry( "overridePerms", edit.overridePerms() );
    mObexConfig->writeEntry( "disconnectTimeout", edit.disconnectTimeout() );
    if ( transport == "ip" ) {
      mObexConfig->writeEntry( "peer", edit.peer() );
      mObexConfig->writeEntry( "ipPort", edit.ipPort() );
    }
    else if ( transport == "serial" ||
	      transport == "serialsiemens" ||
	      transport == "serialericsson" ) {
      mObexConfig->writeEntry( "device", edit.device() );
      mObexConfig->writeEntry( "speed", edit.speed() );
    }
    else if ( transport == "irda" || transport == "bluetooth" ) {
      mObexConfig->writeEntry( "peer", edit.peer() );
    }

    // tell kcm that something has happened.
    emit changed( true );
  }
}

void KObexConfigWidget::slotSelectionChanged()
{
  QListViewItem *item = mListView->selectedItem();
  mRemoveButton->setEnabled( item );
  mEditButton->setEnabled( item );
}

#include "obex.moc"
