/*
 *  linux/drivers/char/hilkbd.c
 *
 *  Copyright (C) 1998 Philip Blundell <philb@gnu.org>
 *  Copyright (C) 1999 Matthew Wilcox <willy@bofh.ai>
 *  Copyright (C) 1999-2001 Helge Deller <deller@gmx.de>
 *
 *  HP Human Interface Loop driver.  This handles the keyboard and mouse
 *  on the HP300 series and some of the HP700 series machines.
 */

#include <linux/errno.h>
#include <linux/types.h>
#include <linux/stddef.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/init.h>
#include <linux/keyboard.h>
#include <linux/kbd_ll.h>
#include <linux/ioport.h>
#include <linux/config.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/interrupt.h>

#include <asm/io.h>
#include <asm/ptrace.h>
#include <asm/irq.h>
#include <asm/system.h>
#include <asm/gsc.h>
#include <asm/hardware.h>

#if defined(CONFIG_PARISC)

#include <asm/hil.h>
#include <asm/keyboard.h>

unsigned long hil_base;	/* HPA for the HIL device */
unsigned int hil_irq;

#elif defined(CONFIG_HP300)

#define HILBASE			0xf0428000	/* HP300 (m86k) specific */
#define HIL_DATA		0x1
#define HIL_CMD			0x3
#define HIL_IRQ			2
#define hil_busy()		(readb(HILBASE + HIL_CMD) & HIL_BUSY)
#define hil_data_available()	(readb(HILBASE + HIL_CMD) & HIL_DATA_RDY)
#define hil_status()		(readb(HILBASE + HIL_CMD))
#define hil_command(x)		do { writeb((x), HILBASE + HIL_CMD); } while (0)
#define hil_read_data()		(readb(HILBASE + HIL_DATA))
#define hil_write_data(x)	do { writeb((x), HILBASE + HIL_DATA); } while (0)

#endif


#define	HIL_BUSY		0x02
#define	HIL_DATA_RDY		0x01

#define	HIL_SETARD		0xA0		/* set auto-repeat delay */
#define	HIL_SETARR		0xA2		/* set auto-repeat rate */
#define	HIL_SETTONE		0xA3		/* set tone generator */
#define	HIL_CNMT		0xB2		/* clear nmi */
#define	HIL_INTON		0x5C		/* Turn on interrupts. */
#define	HIL_INTOFF		0x5D		/* Turn off interrupts. */
#define	HIL_TRIGGER		0xC5		/* trigger command */
#define	HIL_STARTCMD		0xE0		/* start loop command */
#define	HIL_TIMEOUT		0xFE		/* timeout */
#define	HIL_READTIME		0x13		/* Read real time register */

#define	HIL_READBUSY		0x02		/* internal "busy" register */
#define	HIL_READKBDLANG		0x12		/* read keyboard language code */
#define	HIL_READKBDSADR	 	0xF9
#define	HIL_WRITEKBDSADR 	0xE9
#define	HIL_READLPSTAT  	0xFA
#define	HIL_WRITELPSTAT 	0xEA
#define	HIL_READLPCTRL  	0xFB
#define	HIL_WRITELPCTRL 	0xEB

#define plain_map		__initdata hp_plain_map
#define shift_map		__initdata hp_shift_map
#define altgr_map		__initdata hp_altgr_map
#define ctrl_map		__initdata hp_ctrl_map
#define shift_ctrl_map		__initdata hp_shift_ctrl_map
#define alt_map			__initdata hp_alt_map
#define ctrl_alt_map		__initdata hp_ctrl_alt_map

/* 
 * US standard HIL keymap,
 * generated with "loadkeys --mktable key.map > key.c" 
 */

u_short plain_map[NR_KEYS] = {
	0xf200,	0xf200,	0xf701,	0xf703,	0xf700,	0xf700,	0xf702,	0xf205,
	0xf304,	0xf308,	0xf305,	0xf309,	0xf306,	0xf307,	0xf30f,	0xf311,
	0xf301,	0xf30d,	0xf302,	0xf30a,	0xf303,	0xf30c,	0xf300,	0xf30b,
	0xfb62,	0xfb76,	0xfb63,	0xfb78,	0xfb7a,	0xf200,	0xf200,	0xf01b,
	0xf200,	0xf109,	0xf200,	0xf10a,	0xf02e,	0xf108,	0xf30e,	0xf10b,
	0xfb68,	0xfb67,	0xfb66,	0xfb64,	0xfb73,	0xfb61,	0xf200,	0xf207,
	0xfb75,	0xfb79,	0xfb74,	0xfb72,	0xfb65,	0xfb77,	0xfb71,	0xf009,
	0xf037,	0xf036,	0xf035,	0xf034,	0xf033,	0xf032,	0xf031,	0xf060,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,
	0xf200,	0xf103,	0xf102,	0xf101,	0xf100,	0xf200,	0xf209,	0xf200,
	0xf208,	0xf104,	0xf105,	0xf106,	0xf107,	0xf200,	0xf200,	0xf200,
	0xf038,	0xf039,	0xf030,	0xf02d,	0xf03d,	0xf07f,	0xf200,	0xf200,
	0xfb69,	0xfb6f,	0xfb70,	0xf05b,	0xf05d,	0xf05c,	0xf115,	0xf116,
	0xfb6a,	0xfb6b,	0xfb6c,	0xf03b,	0xf027,	0xf201,	0xf114,	0xf118,
	0xfb6d,	0xf02c,	0xf02e,	0xf02f,	0xf200,	0xf117,	0xf200,	0xf119,
	0xfb6e,	0xf020,	0xf200,	0xf200,	0xf601,	0xf600,	0xf603,	0xf602,
};

static u_short shift_map[NR_KEYS] = {
	0xf200,	0xf200,	0xf701,	0xf703,	0xf700,	0xf700,	0xf702,	0xf003,
	0xf304,	0xf308,	0xf305,	0xf309,	0xf306,	0xf307,	0xf02e,	0xf311,
	0xf301,	0xf30d,	0xf302,	0xf30a,	0xf303,	0xf208,	0xf300,	0xf30b,
	0xfb42,	0xfb56,	0xfb43,	0xfb58,	0xfb5a,	0xf200,	0xf200,	0xf87f,
	0xf200,	0xf11f,	0xf200,	0xf120,	0xf30f,	0xf11e,	0xf809,	0xf121,
	0xfb48,	0xfb47,	0xfb46,	0xfb44,	0xfb53,	0xfb41,	0xf200,	0xf207,
	0xfb55,	0xfb59,	0xfb54,	0xfb52,	0xfb45,	0xfb57,	0xfb51,	0xf009,
	0xf026,	0xf05e,	0xf025,	0xf024,	0xf023,	0xf040,	0xf021,	0xf07e,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,
	0xf200,	0xf10f,	0xf10e,	0xf10d,	0xf10c,	0xf200,	0xf209,	0xf200,
	0xf204,	0xf110,	0xf111,	0xf112,	0xf113,	0xf200,	0xf200,	0xf200,
	0xf02a,	0xf028,	0xf029,	0xf05f,	0xf02b,	0xf07f,	0xf200,	0xf200,
	0xfb49,	0xfb4f,	0xfb50,	0xf07b,	0xf07d,	0xf07c,	0xf115,	0xf116,
	0xfb4a,	0xfb4b,	0xfb4c,	0xf03a,	0xf022,	0xf201,	0xf114,	0xf118,
	0xfb4d,	0xf03c,	0xf03e,	0xf03f,	0xf200,	0xf117,	0xf200,	0xf119,
	0xfb4e,	0xf020,	0xf200,	0xf200,	0xf601,	0xf600,	0xf603,	0xf602,
};

static u_short altgr_map[NR_KEYS] = {
	0xf200,	0xf200,	0xf701,	0xf703,	0xf700,	0xf700,	0xf702,	0xf205,
	0xf90e,	0xf912,	0xf90f,	0xf913,	0xf910,	0xf911,	0xf200,	0xf311,
	0xf90b,	0xf915,	0xf90c,	0xf30a,	0xf90d,	0xf30c,	0xf90a,	0xf917,
	0xf915,	0xf200,	0xf916,	0xf200,	0xf200,	0xf702,	0xf914,	0xf01b,
	0xf917,	0xf515,	0xfb67,	0xf516,	0xf02e,	0xf514,	0xf919,	0xf517,
	0xf200,	0xf200,	0xf919,	0xf917,	0xfb7a,	0xf914,	0xf916,	0xf207,
	0xf915,	0xfb6e,	0xfb6d,	0xf200,	0xf918,	0xf200,	0xf700,	0xf009,
	0xf07b,	0xf200,	0xf200,	0xf024,	0xf200,	0xf040,	0xf0b9,	0xf200,
	0xf511,	0xf512,	0xf513,	0xf514,	0xf515,	0xf200,	0xf202,	0xf200,
	0xf200,	0xf50f,	0xf50e,	0xf50d,	0xf50c,	0xf910,	0xf209,	0xf90b,
	0xf202,	0xf510,	0xf511,	0xf512,	0xf513,	0xf200,	0xf200,	0xf200,
	0xf05b,	0xf05d,	0xf07d,	0xf05c,	0xf200,	0xf200,	0xf200,	0xf200,
	0xf200,	0xf702,	0xf200,	0xf200,	0xf07e,	0xf200,	0xf115,	0xf116,
	0xf118,	0xf601,	0xf602,	0xf117,	0xf200,	0xf201,	0xf114,	0xf118,
	0xf11a,	0xf200,	0xf200,	0xf200,	0xf200,	0xf117,	0xf311,	0xf119,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf601,	0xf600,	0xf603,	0xf602,
};

static u_short ctrl_map[NR_KEYS] = {
	0xf200,	0xf200,	0xf701,	0xf703,	0xf700,	0xf700,	0xf702,	0xf205,
	0xf304,	0xf308,	0xf305,	0xf309,	0xf306,	0xf307,	0xf200,	0xf311,
	0xf301,	0xf30d,	0xf302,	0xf30a,	0xf303,	0xf213,	0xf300,	0xf30b,
	0xf200,	0xf200,	0xf003,	0xf200,	0xf200,	0xf200,	0xf200,	0xf01b,
	0xf200,	0xf12b,	0xf200,	0xf12c,	0xf02e,	0xf12a,	0xf009,	0xf12d,
	0xf008,	0xf000,	0xf200,	0xf004,	0xf013,	0xf001,	0xf200,	0xf207,
	0xf015,	0xf019,	0xf014,	0xf012,	0xf005,	0xf017,	0xf011,	0xf009,
	0xf01f,	0xf01e,	0xf01d,	0xf01c,	0xf01b,	0xf200,	0xf0a1,	0xf200,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,
	0xf200,	0xf125,	0xf124,	0xf123,	0xf122,	0xf200,	0xf209,	0xf200,
	0xf203,	0xf126,	0xf127,	0xf128,	0xf129,	0xf200,	0xf200,	0xf200,
	0xf07f,	0xf200,	0xf200,	0xf01f,	0xf200,	0xf008,	0xf200,	0xf200,
	0xf200,	0xf00f,	0xf200,	0xf01b,	0xf01d,	0xf01c,	0xf115,	0xf116,
	0xf00a,	0xf00b,	0xf00c,	0xf200,	0xf007,	0xf201,	0xf114,	0xf118,
	0xf00d,	0xf200,	0xf20e,	0xf07f,	0xf200,	0xf117,	0xf200,	0xf119,
	0xf00e,	0xf000,	0xf200,	0xf200,	0xf601,	0xf600,	0xf603,	0xf602,
};

static u_short shift_ctrl_map[NR_KEYS] = {
	0xf200,	0xf200,	0xf701,	0xf703,	0xf700,	0xf700,	0xf702,	0xf205,
	0xf304,	0xf308,	0xf305,	0xf309,	0xf306,	0xf307,	0xf200,	0xf311,
	0xf301,	0xf30d,	0xf302,	0xf30a,	0xf303,	0xf30c,	0xf300,	0xf30b,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf702,	0xf001,	0xf01b,
	0xf004,	0xf109,	0xf007,	0xf10a,	0xf02e,	0xf108,	0xf30e,	0xf10b,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf01a,	0xf018,	0xf003,	0xf207,
	0xf002,	0xf00e,	0xf00d,	0xf200,	0xf200,	0xf200,	0xf700,	0xf009,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,
	0xf200,	0xf103,	0xf102,	0xf101,	0xf100,	0xf306,	0xf209,	0xf301,
	0xf208,	0xf104,	0xf105,	0xf106,	0xf107,	0xf200,	0xf200,	0xf200,
	0xf200,	0xf200,	0xf200,	0xf01f,	0xf200,	0xf200,	0xf200,	0xf200,
	0xf200,	0xf702,	0xf200,	0xf200,	0xf200,	0xf200,	0xf115,	0xf116,
	0xf118,	0xf601,	0xf602,	0xf117,	0xf200,	0xf201,	0xf114,	0xf118,
	0xf11a,	0xf200,	0xf200,	0xf200,	0xf200,	0xf117,	0xf311,	0xf119,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf601,	0xf600,	0xf603,	0xf602,
};

static u_short alt_map[NR_KEYS] = {
	0xf200,	0xf81b,	0xf701,	0xf703,	0xf700,	0xf700,	0xf702,	0xf205,
	0xf904,	0xf908,	0xf905,	0xf909,	0xf906,	0xf907,	0xf82c,	0xf311,
	0xf901,	0xf30d,	0xf902,	0xf30a,	0xf903,	0xf208,	0xf900,	0xf30b,
	0xf200,	0xf200,	0xf85b,	0xf85d,	0xf200,	0xf702,	0xf861,	0xf81b,
	0xf864,	0xf509,	0xf867,	0xf50a,	0xf82c,	0xf508,	0xf30e,	0xf50b,
	0xf827,	0xf860,	0xf200,	0xf85c,	0xf87a,	0xf878,	0xf863,	0xf207,
	0xf862,	0xf86e,	0xf86d,	0xf82c,	0xf82e,	0xf82f,	0xf700,	0xf809,
	0xf837,	0xf836,	0xf835,	0xf834,	0xf833,	0xf832,	0xf831,	0xf200,
	0xf505,	0xf506,	0xf507,	0xf508,	0xf509,	0xf200,	0xf209,	0xf200,
	0xf200,	0xf503,	0xf502,	0xf501,	0xf500,	0xf906,	0xf209,	0xf901,
	0xf202,	0xf504,	0xf505,	0xf506,	0xf507,	0xf200,	0xf200,	0xf200,
	0xf838,	0xf839,	0xf830,	0xf82d,	0xf83d,	0xf87f,	0xf200,	0xf200,
	0xf200,	0xf702,	0xf200,	0xf85b,	0xf85d,	0xf85c,	0xf115,	0xf116,
	0xf118,	0xf210,	0xf211,	0xf117,	0xf827,	0xf80d,	0xf114,	0xf20b,
	0xf11a,	0xf82c,	0xf82e,	0xf82f,	0xf200,	0xf117,	0xf311,	0xf20a,
	0xf200,	0xf820,	0xf200,	0xf200,	0xf210,	0xf600,	0xf212,	0xf211,
};

static u_short ctrl_alt_map[NR_KEYS] = {
	0xf200,	0xf200,	0xf701,	0xf703,	0xf700,	0xf700,	0xf702,	0xf205,
	0xf304,	0xf308,	0xf305,	0xf309,	0xf306,	0xf307,	0xf200,	0xf311,
	0xf301,	0xf30d,	0xf302,	0xf30a,	0xf303,	0xf30c,	0xf300,	0xf30b,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf702,	0xf801,	0xf01b,
	0xf804,	0xf509,	0xf807,	0xf50a,	0xf02e,	0xf508,	0xf30e,	0xf50b,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf81a,	0xf818,	0xf803,	0xf207,
	0xf802,	0xf80e,	0xf80d,	0xf200,	0xf200,	0xf200,	0xf700,	0xf009,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,
	0xf505,	0xf506,	0xf507,	0xf508,	0xf509,	0xf200,	0xf200,	0xf200,
	0xf200,	0xf503,	0xf502,	0xf501,	0xf500,	0xf306,	0xf209,	0xf301,
	0xf208,	0xf504,	0xf505,	0xf506,	0xf507,	0xf200,	0xf200,	0xf200,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,	0xf200,
	0xf200,	0xf702,	0xf200,	0xf200,	0xf200,	0xf200,	0xf115,	0xf20c,
	0xf118,	0xf601,	0xf602,	0xf117,	0xf200,	0xf201,	0xf114,	0xf118,
	0xf11a,	0xf200,	0xf200,	0xf200,	0xf200,	0xf117,	0xf311,	0xf119,
	0xf200,	0xf200,	0xf200,	0xf200,	0xf601,	0xf600,	0xf603,	0xf602,
};

#undef plain_map
#undef shift_map
#undef altgr_map
#undef ctrl_map
#undef shift_ctrl_map
#undef alt_map
#undef ctrl_alt_map


/* standard HP specific HIL keymaps */
static ushort * __initdata hp_key_maps[MAX_NR_KEYMAPS] = {
	hp_plain_map, hp_shift_map, hp_altgr_map, 0,
	hp_ctrl_map, hp_shift_ctrl_map, 0, 0,
	hp_alt_map, 0, 0, 0,
	hp_ctrl_alt_map, 0, 0, 0
};


struct {
	unsigned char s, c;
	int valid;
} hil_last;

#define hil_getlast(s,c)  do { s = hil_last.s; c = hil_last.c; hil_last.valid = 0; } while (0)

struct {
	unsigned char data[16];
	unsigned int ptr;
} poll;

unsigned char curdev;

static void poll_finished(void)
{
	switch (poll.data[0])
	{
	case 0x40:
		{
			int down = (poll.data[1] & 1) == 0;
			unsigned char scode = poll.data[1] >> 1;
#if 0
			if (down)
				printk(KERN_DEBUG "[%d]", scode);
#endif
			handle_scancode(scode, down);
		}
		break;
	}
	curdev = 0;
}

static inline void handle_status(unsigned char s, unsigned char c)
{
	if (c & 0x8) {
		/* End of block */
		if (c & 0x10)
			poll_finished();
	} else {
		if (c & 0x10) {
			if (curdev)
				poll_finished();		/* just in case */
			curdev = c & 7;
			poll.ptr = 0;
		}
	}
}

static inline void handle_data(unsigned char s, unsigned char c)
{
	if (curdev)
		poll.data[poll.ptr++] = c;
}

/* 
 * Handle HIL interrupts.
 */

static void hil_interrupt(int irq, void *handle, struct pt_regs *regs)
{
	unsigned char s, c;
	s = hil_status(); c = hil_read_data();
	switch (s >> 4) {
	case 0x5:
		handle_status(s, c);
		break;
	case 0x6:
		handle_data(s, c);
		break;
	case 0x4:
		hil_last.s = s;
		hil_last.c = c;
		mb();
		hil_last.valid = 1;
		break;
	}
}

/*
 * Send a command to the HIL
 */

static void hil_do(unsigned char cmd, unsigned char *data, unsigned int len)
{
	unsigned long flags;
	save_flags(flags); cli();
	while (hil_busy());
	hil_command(cmd);
	while (len--) {
		while (hil_busy());
		hil_write_data(*(data++));
	}
	restore_flags(flags);
}


#if defined(CONFIG_PARISC)

extern int pckbd_setkeycode(unsigned int scancode, unsigned int keycode);
extern int pckbd_getkeycode(unsigned int scancode);
extern int pckbd_pretranslate(unsigned char scancode, char raw_mode);
extern int pckbd_translate(unsigned char scancode, unsigned char *keycode, char raw_mode);
extern char pckbd_unexpected_up(unsigned char keycode);

static int def_translate(unsigned char scancode, unsigned char *keycode, char raw)
{	
	*keycode = scancode;
	return 1;
}

static char def_unexpected_up(unsigned char c)
{
	return 128;
}

static void def_leds(unsigned char leds)
{
}

static void def_init_hw(void)
{
}


static struct kbd_ops hil_kbd_ops = {
	setkeycode:	pckbd_setkeycode,
	getkeycode:	pckbd_getkeycode,
	translate:	def_translate,	// pckbd_translate,	/* ?? */
	unexpected_up:	def_unexpected_up, // pckbd_unexpected_up,
	leds:		def_leds,	/* do we have LED's on HIL ? - me not ! */
	init_hw:	def_init_hw,

#ifdef CONFIG_MAGIC_SYSRQ
	sysrq_key:	0x48,		/* "Menu" key */
	sysrq_xlate:	hp_ps2kbd_sysrq_xlate,
#endif /* CONFIG_MAGIC_SYSRQ */
};

#endif	/* CONFIG_PARISC */

/*
 * Initialise HIL. 
 */

int __init hil_keyb_init(void)
{
	unsigned char s, c, kbid;
	unsigned int n = 0;

#if defined(CONFIG_HP300)
	if (!hwreg_present((void *)(HILBASE + HIL_DATA)))
		return 1;		/* maybe this can happen */
#endif /* CONFIG_HP300 */

	request_irq(HIL_IRQ, hil_interrupt, 0, "hil", (void*) HILBASE);

	/* Turn on interrupts */
	hil_do(HIL_INTON, NULL, 0);

	/* Look for keyboards */
	hil_getlast(s, c);	/* clear any pending data */
	hil_do(HIL_READKBDSADR, NULL, 0);
	while (!hil_last.valid) {
		if (n++ > 100000) {
			printk(KERN_DEBUG "HIL: timed out, assuming no keyboard present.\n");
			return 1;
		}
		mb();
	}

	hil_getlast(s, c);
	if (c == 0) {
		printk(KERN_WARNING "HIL: no keyboard present.\n");
		return 1;
	}

	kbid = ffz(~c);
	printk(KERN_INFO "HIL: keyboard found at id %d\n", kbid);

	for (n=0; n<MAX_NR_KEYMAPS; n++)
		if (key_maps[n] && hp_key_maps[n])
			memcpy(key_maps[n], hp_key_maps[n], sizeof(plain_map));
	printk(KERN_INFO "HIL: keymap loaded.\n");

#if defined(CONFIG_PARISC)
	register_kbd_ops(&hil_kbd_ops);
#endif

	request_region(HILBASE+HIL_DATA, 2, "hil");

	/* set it to raw mode */
	c = 0;
	hil_do(HIL_WRITEKBDSADR, &c, 1);
	
	return 0;
}

static int __init
hil_init_chip(struct parisc_device *dev)
{
	if (!dev->irq) {
		printk(KERN_WARNING __FILE__ ": IRQ not found for HIL at 0x%lx\n", dev->hpa);
		return -ENODEV;
	}

	hil_base = dev->hpa;
	hil_irq  = dev->irq;
	
	printk(KERN_INFO "Found HIL at 0x%lx, IRQ %d\n", hil_base, hil_irq);

	return hil_keyb_init();
}


static struct parisc_device_id hil_tbl[] = {
	{ HPHW_FIO, HVERSION_REV_ANY_ID, HVERSION_ANY_ID, 0x00073 },
	{ 0, }
};

MODULE_DEVICE_TABLE(parisc, hil_tbl);

static struct parisc_driver hil_driver = {
	name:		"HIL",
	id_table:	hil_tbl,
	probe:		hil_init_chip,
};

static int __init hil_init(void)
{
	return register_parisc_driver(&hil_driver);
}

module_init(hil_init);
