\ tpf.4th
\
\  Demonstrate TPF control of a chaotic piecewise-linear Rossler circuit.
\
\  Copyright (c) 2003  Krishna Myneni
\
\  Transit-time Pulse-width modulation Feedback (TPF) is a technique
\  for controlling unstable periodic orbits (UPOs) within the dynamics
\  of a chaotic system. Natural trajectories of the system can include
\  period-1 (one loop), period-2 (two loops), and higher period orbits;
\  however, these orbits happen to be unstable and the system will wander
\  away from these orbits if left alone. In real systems, noise makes the
\  system wander off the UPOs while in a computer simulation, finite 
\  precision in the calculations acts like noise. TPF makes use of occasional 
\  feedback to keep the system on one of these periodic orbits. When the 
\  system's trajectory passes through a window, feedback is applied to 
\  one of the circuit parameters. The feedback is a rectangular pulse 
\  with a width equal to the time for the system to transit through the 
\  window. Therefore, the strength of the occasional perturbation is 
\  proportional to the transit time. If a square window is placed such 
\  that one of its corners touches the trajectory of an unstable periodic 
\  orbit, as the system attempts to move off the UPO, the strength of 
\  the feedback will increase and push the system back onto the UPO. It 
\  is possible to use one window to accomplish control of a UPO if the 
\  chaotic system has a "simply folded band" attractor, such as the circuit
\  simulated below. The sign of the feedback signal, its amplitude,
\  and the placement of the window, must all be chosen so that the
\  feedback will push the system sufficiently back towards the UPO. For
\  ideal settings, the control signal power required to stabilize a UPO
\  will be very small --- just enough to counter the effects of noise 
\  in the system. For more information on the TPF control technique,
\  see, "New Method for the Control of Fast Chaotic Oscillations",
\  by K. Myneni, T. A. Barr, N. J. Corron, and S. D. Pethel, Physical
\  Review Letters, vol. 83, pp. 2175--2178, September 1999. 
\
\  In this demonstration, the dynamics of a chaotic electronic circuit
\  will be animated on the console. While the "attractor" has three
\  dimensions, we will look "down" onto the trajectory in the x-y plane.
\  You can move about a square window in the x-y plane using
\  the keypad. When the trajectory enters the window, the feedback
\  signal is applied to the circuit. In this simulation, we apply the
\  feedback to the "x-dot" or dx/dt equation. The effect of the feedback
\  on the subsequent dynamics of the circuit may be observed. You
\  may clear the screen to erase the display of the past trajectory
\  and observe the new trajectory. By moving the window to different
\  locations in the x-y plane, you can find positions where different
\  UPOs are stabilized. A display of the center coordinates of the
\  window are shown at the top left. In addition to moving the window,
\  in coarse or fine steps, you may also toggle the control on or off, 
\  zoom in on the region near the window, and record the trajectory 
\  to a data file. It should be possible for you to find period-1, 
\  period-2, period-4, and even period-6 orbits.
\
\  Once you think you have found a periodic orbit, use the "zoom view" 
\  option to take a closer look at the trajectory near the window. If 
\  the dynamics is periodic, there should only be one path intersecting 
\  the window. You may also "record" the data to a file called tpf.dat 
\  and use a plotting program, such as xyplot, to view the trajectory
\  in high resolution. The deflection of the trajectory by the window 
\  will be noticeable as a small kink in the trajectory. Try to move the 
\  window to minimize the disturbance to the trajectory while maintaining 
\  a sharp periodic orbit.
\
\  The differential equations for the PLR chaotic circuit are from 
\  T.L. Carroll, "A simple circuit for demonstrating regular and 
\  synchronized chaos," American Journal of Physics, vol. 63(4), 
\  pp. 377--379, April 1995. This circuit can be built from
\  inexpensive op-amps, resistors, capacitors, and a diode.
\
\ The PLR circuit is described by the following eqns:
\
\	C1 dx/dt =      -(1/R1)*x                 -(1/R4)*y -(1/R5)*z
\	C2 dy/dt = (R7/(R8*R9))*x + (R7/(R6*R9) - (1/R2))*y
\	C3 dz/dt = (R12/(R10*R13))*H(x-V_c)*(x-V_c)         -(1/R3)*z
\
\  where x, y, and z are the three output state variables (voltages),
\  H(x-V_c) is the Heaviside function, i.e. H = 0 if x <= V_c,
\  H = 1 if x > V_c,  and V_c is given by:
\
\	V_c = (R10 + R11)*V_d/R11 + R10*V/R11
\
\  where V_d is the diode forward-bias turn on voltage, and V is the
\  supply voltage for the circuit.
\
\
\ Notes:
\
\ 1) Under Windows, replace the statement "include files"
\    with "include filesw".
\
\ 2) This program runs in a console window and uses text graphics
\    for ANSI terminals. Under Windows, the Command Prompt window
\    should be resized to have at least 50 lines. In Windows NT
\    and later, the Command Prompt window may be resized just like
\    any other window. Under Windows 98, the MS-DOS Prompt window
\    can also be set for 50 lines, but it requires editing the
\    "Properties" of the MS-DOS Prompt window. The font size should
\    also be adjusted (a font of 8x12 works well for a screen
\    resolution of 1024x768).
\
\ Revisions:
\
\	2003-05-16  first version  km
\	2003-05-17  revised comments; modified CLEANUP  km
\ 
include ans-words
include fsl-util
include dynmem
include runge4
include matrix
include strings
include files		\ filesw for Windows
include ansi

\ A variant of the routine rk4qc_step in runge4.4th is needed here,
\   because the fractional error needs to be computed with respect
\   to fixed quantities rather than the current state variables
\   (the z value can become exceedingly small).

: rk4qc_step2 ( step t -- step' t' flag )
	tstart F!  step F!  ( scale'm)
	uorig{ u1{ dim }fcopy	\ we need a fresh start after a shrink
	uorig{ u2{ dim }fcopy
   BEGIN	
	tstart F@ step F@ F2/ uorig{ 2 runge_kutta4_integrate() FDROP
	tstart F@ step F@ u1{    1     runge_kutta4_integrate() (  -- t' )
	FDUP tstart F@ 0.0E0 F~ IF 0.0E0 FSWAP FALSE EXIT THEN
	shrink?			\ maximum difference between these two tries
   WHILE			\ too large, shrink step size
	FLN pshrink F* FEXP step F@ F* safety F* step F!  FDROP
	u2{ uorig{ dim }fcopy	\ a fresh start after a shrink...
	u2{ u1{    dim }fcopy

  REPEAT			\ ok, grow step size for next time
	FDUP errcon F< IF  FDROP step F@ 4e F* 
		     ELSE  FLN pgrow F* FEXP step F@ F* safety F*
		     THEN 
	maxstep F@ FMIN		\ but don't grow excessively!
	FSWAP TRUE 4th->5th ;



\ Describe the PLR Circuit

2.2e6  fconstant  R1
4.7e6  fconstant  R2
1.0e5  fconstant  R3
1.8e5  fconstant  R4
1.0e5  fconstant  R5
5.0e5  fconstant  R6  
1.0e5  fconstant  R7
1.0e5  fconstant  R8
1.0e5  fconstant  R9
1.0e5  fconstant  R10
2.2e6  fconstant  R11
4.7e5  fconstant  R12
1.0e5  fconstant  R13

0.82e-9  fconstant  C1
0.82e-9  fconstant  C2
0.82e-9  fconstant  C3

0.7e  fconstant  V_d		\ turn on voltage of diode
15.e  fconstant  V		\ supply voltage
R10 R11 f+ V_d f* R11 f/ R10 V f* R11 f/ f+  fconstant V_c

3 3 fmatrix a

-1e R1 C1 f* f/		1 1 a fmat!
-1e R4 C1 f* f/		1 2 a fmat!
-1e R5 C1 f* f/		1 3 a fmat!
R7 R8 R9 C2 f* f* f/	2 1 a fmat!
R7 R6 R9 f* f/ 1e R2 f/ f- C2 f/   2 2 a fmat!
0e			2 3 a fmat!
0e			3 1 a fmat!  ( nonlinear element is calculated)
0e			3 2 a fmat!
-1e R3 f/ C3 f/		3 3 a fmat!

R12 R10 R13 C3 f* f* f/ fconstant DZXC


fvariable delta-p

: derivs() ( ft 'u 'dudt -- )

	2SWAP FDROP	\ does not explicitly use t
	  
	\ Multiply the state vector by the matrix a to obtain
	\   rates of change

	3 0 DO		\ 'u 'dudt
	  OVER	0e	\ 'u 'dudt 'u 0e
	  3 0 DO
	    2 PICK I } F@  
	    J 1+  I 1+ a fmat@  F* F+
	  LOOP
	  ROT DROP
	  2 PICK I } F!
	LOOP
		
		\ 'u 'dudt

	DUP 0 } F@  delta-p F@ F+ 2 PICK 0 } F!  \ add perturbation to dx/dt

	\ compute H(x-Vc) term and add to dz/dt term

	OVER 0 } F@  FDUP V_c  F<= 
	IF  FDROP 0e  ELSE  V_c  F-  DZXC F*  THEN	\ 'u 'dudt f
	2 PICK  2 } F@  F+ 
	ROT 2 } F!
	DROP ;


75 constant MAX_COLS
40 constant MAX_ROWS

MAX_COLS s>f fconstant MAX_COLSF
MAX_ROWS s>f fconstant MAX_ROWSF

fvariable xmin
fvariable xmax
fvariable ymin
fvariable ymax
fvariable xdel
fvariable ydel

-3e xmin  F!
 3e xmax  F!
-3e ymin  F!
 3e ymax  F!

xmax F@  xmin F@  F-  xdel  F!
ymax F@  ymin F@  F-  ydel  F!

fvariable wx		\ x of window
fvariable wy		\ y of window
fvariable wsize		\ window width and height (square window)
fvariable gamma		\ perturbation strength for window

-2e wx    F!		\ starting window position
 2e wy    F!
.15e wsize F!		\ This gives a 300 mV square window, which works
			\   well for the default circuit parameters

.05e FCONSTANT COARSE-STEP	\ window movement coarse step
.005e FCONSTANT FINE-STEP	\ window movement fine step
7000e gamma  F!

5e-6 FCONSTANT RK4-MAXSTEP1     \ maxstep for adaptive step-size RK4 routine
1e-7 FCONSTANT RK4-MAXSTEP2	\ fifty times smaller near the window

: crbounds ( col row -- col' row' | enforce screen bounds )
	0 MAX  MAX_ROWS MIN  SWAP 0 MAX  MAX_COLS MIN SWAP ;

: xy>cr ( fx fy -- col row | transform coords to screen coords )
	FSWAP  xmin F@  F-  xdel F@ F/  MAX_COLSF  F*
	FSWAP  ymin F@  F-  ydel F@ F/  MAX_ROWSF  F* 
	FROUND>S  MAX_ROWS  SWAP - >R
	FROUND>S  R> ;
 

: set-pixel ( col row -- )  crbounds AT-XY [CHAR] * EMIT ;


TRUE value control-on?


: w-up    ( fstep -- )  wy F@  F+        ymax F@  FMIN  wy F! ;
: w-down  ( fstep -- )  wy F@  FSWAP F-  ymin F@  FMAX  wy F! ;
: w-left  ( fstep -- )  wx F@  FSWAP F-  xmin F@  FMAX  wx F! ;
: w-right ( fstep -- )  wx F@  F+        ymax F@  FMIN  wx F! ;


: show-tpf-window ( -- )
	CYAN foreground
	control-on? IF
	  wx F@ wy F@ xy>cr  AT-XY  [CHAR] W EMIT 
	   1 1 AT-XY ." wx: " wx F@ F. 2 SPACES
	  12 1 AT-XY ." wy: " wy F@ F. 2 SPACES
	ELSE
	  wx F@ wy F@ xy>cr  AT-XY SPACE
	  1 1 AT-XY 25 SPACES
	  1 1 AT-XY ." CONTROL IS OFF" 
	THEN ;

: in-window? ( fx fy -- flag )
	wy F@  wsize F@  F~  >R
	wx F@  wsize F@  F~  R> AND ;	

: near-window? ( fx fy -- flag )
	wy F@  wsize F@ 2e F* FDUP 2>R F~ 
	-ROT 
	wx F@  2R> F~ AND ; 

: set-max-step-size ( fx fy -- )
	near-window? control-on? AND
	IF    RK4-MAXSTEP2 
	ELSE  RK4-MAXSTEP1
	THEN maxstep F! ;

: set-perturbation ( fx fy -- )
	in-window?  control-on? AND
	IF    RED foreground  gamma F@
	ELSE  GREEN foreground 0e
	THEN  delta-p F! ;

3 float array x{

variable last-col
variable last-row

: last? ( col row -- flag | true if col and row are same as last-col and last-row)
	last-row @ = SWAP last-col @ = AND ;

: !last ( col row -- ) last-row ! last-col ! ;

FALSE value recording?
10000 constant MAXREC	\ maximum number of points to record
variable nrec 		\ number of points recorded
variable rec_file

: start-recording ( -- )
	TRUE to recording?
	0 nrec !
	s" tpf.dat" W/O O_TRUNC or create-file  \ O_TRUNC is needed in 
	ABORT" Unable to open output file!"     \   case file already exists
	rec_file ! ;

: stop-recording ( -- )
	rec_file @ close-file drop
	FALSE to recording? ;
	
: write-current-state ( -- )
	x{ 0 } F@ 8 f>string COUNT s"   " strcat
	x{ 1 } F@ 8 f>string COUNT strcat s"   " strcat
	x{ 2 } F@ 8 f>string COUNT strcat
	rec_file @ write-line DROP 
	1 nrec +! ;

: key-help
	WHITE foreground
	 5 42 AT-XY  ." KEY COMMANDS: "
	YELLOW foreground
	 6 44 AT-XY  [CHAR] i EMIT	32 44 AT-XY  [CHAR] I EMIT
	 6 45 AT-XY  [CHAR] k EMIT	32 45 AT-XY  [CHAR] K EMIT
	 6 46 AT-XY  [CHAR] j EMIT	32 46 AT-XY  [CHAR] J EMIT
	 6 47 AT-XY  [CHAR] l EMIT	32 47 AT-XY  [CHAR] L EMIT
	 6 48 AT-XY  [CHAR] z EMIT	32 48 AT-XY  [CHAR] Z EMIT
	55 44 AT-XY  [CHAR] C EMIT
	55 45 AT-XY  [CHAR] X EMIT
	55 46 AT-XY  [CHAR] R EMIT
	55 47 AT-XY  ." Esc"
	WHITE foreground
	10 44 AT-XY  ." up    [fine]"
	10 45 AT-XY  ." down  [fine]"
	10 46 AT-XY  ." left  [fine]"
	10 47 AT-XY  ." right [fine]"
	10 48 AT-XY  ." normal view"
	36 44 AT-XY  ." UP    [coarse]"
	36 45 AT-XY  ." DOWN  [coarse]"
	36 46 AT-XY  ." LEFT  [coarse]"
	36 47 AT-XY  ." RIGHT [coarse]"
	36 48 AT-XY  ." zoom view"
	59 44 AT-XY  ." clear screen"   
	59 45 AT-XY  ." toggle control ON/OFF"
	59 46 AT-XY  recording? IF
		     ." recording ...  "
	ELSE         ." record to tpf.dat" THEN
	59 47 AT-XY  ." halt simulation" ;

: refresh ( -- | refresh the display)  PAGE  key-help  show-tpf-window ;

: cleanup ( -- | perform cleanup on exit)
	recording? IF stop-recording THEN
	rk4qc_done
	1 1 AT-XY ;

: zoom-in ( -- )
	wsize F@ 3e F* 2>R 
	wx F@  2R@  F-  xmin F!
	wx F@  2R@  F+  xmax F!
	wy F@  2R@  F-  ymin F!
	wy F@  2R>  F+  ymax F!
	xmax F@  xmin F@  F-  xdel F!
	ymax F@  ymin F@  F-  ydel F! ;

: zoom-out ( -- )
	-3e xmin  F!
 	 3e xmax  F!
	-3e ymin  F!
	 3e ymax  F!
	xmax F@  xmin F@  F-  xdel  F!
	ymax F@  ymin F@  F-  ydel  F! ;
	 
  
: go  ( -- )

	RK4-MAXSTEP1 1e-3 use( derivs() 3 x{  )rk4qc_init
	2e uscal{ 0 } F!  2e uscal{ 1 } F!  2e uscal{ 2 } F!
	

	BLACK background
	PAGE  key-help
	show-tpf-window
	RK4-MAXSTEP1 0e	  \ max-step start-time

	BEGIN
	  rk4qc_step2 DROP
	  x{ 0 } F@  x{ 1 } F@  set-max-step-size
	  x{ 0 } F@  x{ 1 } F@  set-perturbation
	  x{ 0 } F@ x{ 1 } F@ xy>cr 
	  2DUP last? NOT IF 2DUP set-pixel !last 1 MS
	  ELSE 2DROP THEN
	 
	  recording? IF 
	    write-current-state
	    nrec @ MAXREC = IF stop-recording 7 emit refresh THEN
	  THEN  
	  \ x{ 0 } F@ F. 2 SPACES x{ 1 } F@ F. 2 SPACES x{ 2 } F@ F. CR

	  KEY? IF KEY 
	    CASE
	      27        OF  FDROP FDROP  cleanup  EXIT            ENDOF
	      [CHAR] C  OF  refresh                               ENDOF
	      [CHAR] i  OF  FINE-STEP   w-up     show-tpf-window  ENDOF
	      [CHAR] k  OF  FINE-STEP   w-down   show-tpf-window  ENDOF
	      [CHAR] j  OF  FINE-STEP   w-left   show-tpf-window  ENDOF
	      [CHAR] l  OF  FINE-STEP   w-right  show-tpf-window  ENDOF
	      [CHAR] I  OF  COARSE-STEP w-up     show-tpf-window  ENDOF
	      [CHAR] K  OF  COARSE-STEP w-down   show-tpf-window  ENDOF
	      [CHAR] J  OF  COARSE-STEP w-left   show-tpf-window  ENDOF
	      [CHAR] L  OF  COARSE-STEP w-right  show-tpf-window  ENDOF
	      [CHAR] X  OF  control-on? NOT TO control-on? 
				show-tpf-window                   ENDOF
	      [CHAR] R  OF  start-recording  refresh              ENDOF
	      [CHAR] Z  OF  zoom-in  refresh                      ENDOF
	      [CHAR] z  OF  zoom-out refresh                      ENDOF
	    ENDCASE
	  THEN
	AGAIN ; 

\ initial values of the state variables
0.1e x{ 0 } F!  0.1e x{ 1 } F!  0.1e x{ 2 } F!

PAGE
CR CR
.( Type 'go' to start the simulation. ) CR CR


	
