/*
  Copyright Mission Critical Linux, 2000

  Kimberlite is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  Kimberlite is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Kimberlite; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
 *
 * Author: Gregory P. Myrdal <Myrdal@MissionCriticalLinux.Com>
 *         Xudong Tang       <Tang@MissionCriticalLinux.Com>
 *
 * svcDB.c
 *
 * This file contains the functions to manipulate the service
 * section of the cluster configuration database.  Functions
 * here provide support to get, set and check values for service
 * entries.
 */

/*
 * Version string that is filled in by CVS
 */
static const char *version __attribute__ ((unused)) = "$Revision: 1.36 $";

/*
 * This module contains functions that deal with writing and reading
 * from the service section of the cluster database.
 *
 */

/*
 * System includes
 */
#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <string.h>
#include <errno.h> 
#include <netdb.h>  
#include <pwd.h>
#include <grp.h>
#include <regex.h>
#include <sys/stat.h>         
#include <sys/ioctl.h> 
#include <sys/socket.h>  
#include <sys/param.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <net/if.h>    
#include <ctype.h>


/*
 * Cluster includes
 */
#include <clusterdefs.h>
#include <parseconf.h>
#include <svcmgr.h>
#include <logger.h>

/*****************************************************************************
 *
 * The following functions are general utility functions for the service 
 * section of the cluster configuration database.
 *
 ****************************************************************************/

/*
 * getDatabaseToken
 *
 * Given a token string, get its associated value in the database.
 *
 * Return values:
 *	SUCCESS		- found value, value is in 'value'
 *	NOT_FOUND	- did not find a value
 *	FAIL		- failure
 */
int
getDatabaseToken(char *token, char **value)
{
	int retVal;

	switch (retVal=CFG_Get(token, (char *)NULL, value))
	  {
	  case CFG_DEFAULT:		// token not defined
	    return(NOT_FOUND);
	    break;
	  case CFG_OK:			// found it
	    return(SUCCESS);
	    break;
	  default:			// Uh oh, we failed in CFG_Get()
	    clulog(LOG_ERR, 
"Cannot get %s from database; CFG_Get() failed, err=%d\n", token, retVal);
	    return(FAIL);
	  }     
}

/*
 * setDatabaseToken
 *
 * Given a token string, set its associated value in the database.
 *
 * Note: the database is not written out to disk.  This is the responsibility
 * of the caller.
 *
 * Return values:
 *	SUCCESS		- set value in database
 *	FAIL		- failure
 */
int
setDatabaseToken(char *token, char *value)
{
	int retVal=0;

	if ((retVal=CFG_Set(token, value)) != CFG_OK)
	  {
	    clulog(LOG_ERR, "Cannot set %s = %s; error = %d\n", 
	           token, value, retVal);
	    return(FAIL);
	  }
	else
	    return(SUCCESS);
}

/*
 * findNextFreeEntry
 *
 * Given a token string for the database, find the next available
 * entry in the database assuming the last field is enumerate by
 * whole numbers (i.e. 0 1 2 3 4 ...).  The path to the unused entry
 * is passed back in freeEntry.
 *
 * Return values:
 *	SUCCESS		- free entry found, value returned in freeEntry
 *	FAIL		- failure
 */
int
findNextFreeEntry(char *tokenPath, int *freeEntry)
{
	char *value;
	char token[MAX_TOKEN_LEN];
	int N=MIN_TOKEN_ID-1;

	while (1)
	  {
	    N=N+1;
	    sprintf(token, "%s%c%d", tokenPath, CLU_CONFIG_SEPARATOR, N);

	    switch (getDatabaseToken(token, &value))
	      {
	        case NOT_FOUND:		// found free entry, return it
	          *freeEntry=N;
	          return(SUCCESS);
	        case FAIL:		// got an error
	          *freeEntry=-1;
	          return(FAIL);
	        default:		// must be in use
	          continue;
	      }
	  }
}

/*****************************************************************************
 *
 * The following functions get the token string for a service in the
 * cluster database.  These token strings are defined by metaconfig.
 *
 ****************************************************************************/


/*
 * getSvcNameTokenStr
 *
 * Given a service ID return the metaconfig token string for its name
 * entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcNameTokenStr(int svcID, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service name token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%name
	sprintf(tokenStr, "%s%c%s%d%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR, SVC_NAME_STR);
	return(SUCCESS);
}

/*
 * getSvcScriptTokenStr
 *
 * Given a service ID return the metaconfig token string for its user script
 * entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcScriptTokenStr(int svcID, char *tokenStr)
{

	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service script token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%userScript
	sprintf(tokenStr, "%s%c%s%d%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR,
	        SVC_USER_SCRIPT_STR);
	return(SUCCESS);
}

/*
 * getSvcRelocateOnPreferredNodeBootTokenStr
 *
 * Given a service ID return the metaconfig token string for its relocate
 * on prefferred node boot entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 * 
 * Special Note: Brian oughta like this name .... can't say it in
 * one breath ... he he he ...
 */
int
getSvcRelocateOnPreferredNodeBootTokenStr(int svcID, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service relocate token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%relocateOnPreferredMemberBoot
	sprintf(tokenStr, "%s%c%s%d%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR,
	        SVC_RELOCATE_ON_BOOT_STR);
	return(SUCCESS);
}

/*
 * getSvcPreferredNodeNameTokenStr
 *
 * Given a service ID return the metaconfig token string for its preferred
 * node name entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcPreferredNodeNameTokenStr(int svcID, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service preferred member token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%preferredNode
	sprintf(tokenStr, "%s%c%s%d%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR, 
	        SVC_PREFERRED_NODE_STR);
	return(SUCCESS);
}

/*
 * getSvcDisabledTokenStr
 *
 * Given a service ID return the metaconfig token string for its disabled
 * entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcDisabledTokenStr(int svcID, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service disabled token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%disabled
	sprintf(tokenStr, "%s%c%s%d%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR, 
	        SVC_DISABLED_STR);
	return(SUCCESS);
}

/*
 * getSvcIPaddressTokenStr
 *
 * Given a service ID return the metaconfig token string for its IP address
 * entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcIPaddressTokenStr(int svcID, int N, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service IP address token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%network0%ipAddress
	sprintf(tokenStr, "%s%c%s%d%c%s%d%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR,
	        SVC_NETWORK_STR, N, CLU_CONFIG_SEPARATOR,
		SVC_IP_ADDR_STR);
	return(SUCCESS);
}

/*
 * getSvcNetmaskTokenStr
 *
 * Given a service ID return the metaconfig token string for its netmask
 * address entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcNetmaskTokenStr(int svcID, int N, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service IP netmask token string, token string is NULL\n");
	    return(FAIL);
	  }
	
	// services%service0%network0%netmask
	sprintf(tokenStr, "%s%c%s%d%c%s%d%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR,
	        SVC_NETWORK_STR, N, CLU_CONFIG_SEPARATOR,
		SVC_NETMASK_STR);
	return(SUCCESS);
}

/*
 * getSvcBroadcastTokenStr
 *
 * Given a service ID return the metaconfig token string for its broadcast
 * address entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcBroadcastTokenStr(int svcID, int N, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service IP broadcast token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%network0%broadcast
	sprintf(tokenStr, "%s%c%s%d%c%s%d%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR,
	        SVC_NETWORK_STR, N, CLU_CONFIG_SEPARATOR,
		SVC_BROADCAST_STR);
	return(SUCCESS);
}

/*
 * getSvcDeviceTokenStr
 *
 * Given a service ID return the metaconfig token string for the Nth
 * device entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcDeviceTokenStr(int svcID, int N, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service device token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%device0%name
	sprintf(tokenStr, "%s%c%s%d%c%s%d%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR,
	        SVC_DEVICE_STR, N, CLU_CONFIG_SEPARATOR,
		SVC_DEVICE_NAME_STR);
	return(SUCCESS);
}

/*
 * getSvcMountPointTokenStr
 *
 * Given a service ID return the metaconfig token string for the Nth
 * mount point entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcMountPointTokenStr(int svcID, int N, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service mount point token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%device0%mount%name
	sprintf(tokenStr, "%s%c%s%d%c%s%d%c%s%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR,
	        SVC_DEVICE_STR, N, CLU_CONFIG_SEPARATOR,
	        SVC_MOUNT_STR, CLU_CONFIG_SEPARATOR,
		SVC_MOUNT_NAME_STR);
	return(SUCCESS);
}

/*
 * getSvcMountOptionsTokenStr
 *
 * Given a service ID return the metaconfig token string for the Nth
 * mount option entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcMountOptionsTokenStr(int svcID, int N, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service mount options token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%device0%mount%options
	sprintf(tokenStr, "%s%c%s%d%c%s%d%c%s%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR,
		SVC_DEVICE_STR, N, CLU_CONFIG_SEPARATOR,
		SVC_MOUNT_STR, CLU_CONFIG_SEPARATOR,
	        SVC_MOUNT_OPTIONS_STR);
	return(SUCCESS);
}

/*
 * getSvcMountFstypeTokenStr
 *
 * Given a service ID return the metaconfig token string for the Nth
 * mount fstype entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcMountFstypeTokenStr(int svcID, int N, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service mount fstype token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%device0%mount%fstype
	sprintf(tokenStr, "%s%c%s%d%c%s%d%c%s%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR,
		SVC_DEVICE_STR, N, CLU_CONFIG_SEPARATOR,
		SVC_MOUNT_STR, CLU_CONFIG_SEPARATOR,
	        SVC_MOUNT_FSTYPE_STR);
	return(SUCCESS);
}

/*
 * getSvcForceUnmountTokenStr
 *
 * Given a service ID return the metaconfig token string for the Nth
 * force unmount entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcForceUnmountTokenStr(int svcID, int N, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service forced unmount token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%device0%mount%forceUmount
	sprintf(tokenStr, "%s%c%s%d%c%s%d%c%s%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR,
		SVC_DEVICE_STR, N, CLU_CONFIG_SEPARATOR,
		SVC_MOUNT_STR, CLU_CONFIG_SEPARATOR,
	        SVC_FORCE_UNMOUNT_STR);
	return(SUCCESS);
}

/*
 * getSvcDeviceOwnerTokenStr
 *
 * Given a service ID return the metaconfig token string for the Nth
 * device owner entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcDeviceOwnerTokenStr(int svcID, int N, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service device owner token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%device0%owner
	sprintf(tokenStr, "%s%c%s%d%c%s%d%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR,
		SVC_DEVICE_STR, N, CLU_CONFIG_SEPARATOR,
	        SVC_DEVICE_OWNER_STR);
	return(SUCCESS);
}

/*
 * getSvcDeviceGroupTokenStr
 *
 * Given a service ID return the metaconfig token string for the Nth
 * device group entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 */
int
getSvcDeviceGroupTokenStr(int svcID, int N, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service device group token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%device0%group
	sprintf(tokenStr, "%s%c%s%d%c%s%d%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR,
		SVC_DEVICE_STR, N, CLU_CONFIG_SEPARATOR,
	        SVC_DEVICE_GROUP_STR);
	return(SUCCESS);
}

/*
 * getSvcDeviceModeTokenStr
 *
 * Given a service ID return the metaconfig token string for the Nth
 * device mode entry in tokenStr.
 *
 * Return values:
 *	SUCCESS		- tokenStr was filled with token string 
 *	FAIL		- on failure
 */
int
getSvcDeviceModeTokenStr(int svcID, int N, char *tokenStr)
{
	if (tokenStr == NULL)
	  {
	    clulog(LOG_ERR,
"Cannot get service device mode token string, token string is NULL\n");
	    return(FAIL);
	  }

	// services%service0%device0%mode
	sprintf(tokenStr, "%s%c%s%d%c%s%d%c%s", 
	        SVC_SERVICES_LIST_STR, CLU_CONFIG_SEPARATOR,
	        SVC_SERVICE_STR, svcID, CLU_CONFIG_SEPARATOR,
		SVC_DEVICE_STR, N, CLU_CONFIG_SEPARATOR,
	        SVC_DEVICE_MODE_STR);
	return(SUCCESS);
}



/*****************************************************************************
 *
 * The following functions are used to check the service attributes values and
 * format. 
 *	FAIL	-  data format error or the value is not valid. 
 *                 (such as the IP address is not valid or a service has 
 *                  been used by other member in the cluster)
 *	SUCCESS	-  the value is valid.
 *	WARNING	-  the value can be set as you wanted, but you should 
 *                 take some further actions (such as a mount point is not 
 *                 created yet). 
 *      TRY_MORE-  try run the test again, because of system problems (such as 
 *                 run out of memory, the record in a system database is 
 *                 locked, etc) 
 *
 ****************************************************************************/


/*
 * ckSvcName
 *
 * To check (1) if the name is a valid string
 *          (2) then check if the service name 
 *              is unique in the cluster configuration(except the current record under modification)  
 *
 * Return values:
 *	FAIL	- svcName format error or have been
 *	SUCCESS	- svcName is not used by other member of the cluster
 */ 
int
ckSvcName(int current_svcID, char *svcName, char *errMsg)
{       
        int svcID;
        int l;
        int i;
        char *s;

        /* below is added for string format check */
        l=strlen(svcName);
        if(l<=0)
	{ 
          strcpy(errMsg, "Service name is empty\n");
          return(FAIL);  // return with error
        }

        for(i=0;i<l;i++)
	  {   
	    if((!isalnum(svcName[i]))&&(svcName[i]!='.') &&
               (svcName[i]!='-') && (svcName[i]!='_') && (svcName[i]!='~') &&
               (svcName[i]!='/'))
	    {
             sprintf(errMsg,
"Service name, '%s', is not a valid string\n", svcName);
             return(FAIL);
            }
        }
	
       /* now, check the service name is unique in the cluster database */
        for (svcID=MIN_SERVICE; svcID < MAX_SERVICES; svcID++)
        {
	   if(svcID==current_svcID)
	      continue;  // skip the current record which is under modification          
	   if (serviceExists(svcID) != YES)
	      continue;       // not in database             
           getSvcName(svcID, &s);
	   if(strcmp(s,svcName)==0) // Service name is in use
	   {
              sprintf(errMsg,
"Service name, '%s', is already defined in the cluster database\n", svcName);
              return(FAIL); 
	   }
        }
        return(SUCCESS);
}



/*
 * ckSvcScript
 *
 * Get the user service script for the service associated with the given
 * service ID. 
 *
 * Return values:
 *	SUCCESS		- a service script was found and executable
 *	FAIL	        - a service script was not found
 *      WARNING         - the file not executable, administrator may need to 
                          set it executable 
 *		   
 */
int
ckSvcScript(char *svcScript, char *errMsg)
{     
	int flag;
        struct stat BUF;

 	flag=stat(svcScript,&BUF);
	if(flag<0)
	{ 
           sprintf(errMsg,"User script file %s: %s\n",
	           svcScript, strerror(errno));
           return(WARNING);
        }
        else
	{ 
	  /* first, to test if it is an ordinary file, not a link or directory.
             then, to test if current user can run it
             NOTE:if you set the script file is executable for other person 
	          not your own, then we still think it is not executable
	  */ 
   
	  if((S_ISREG(BUF.st_mode)) && ((S_IEXEC) & BUF.st_mode))
	    ;
          else
	    {
	       sprintf(errMsg,"User script file %s: not executable or not a file\n",
                                 svcScript);
               return(WARNING); // the file not executable, administrator 
                                // may need to set it executable   
	    }
        }       
        return(SUCCESS);
}



/*
 * ckSvcRelocateOnPreferredNodeBoot
 * Check if the format for relocate field is right or not
 * currently we only check if the string in char *relocate is
 * yes or no.
 *
 * Return values:
 *	SUCCESS		- data in char *relocate passed check  
 *	FAIL	        - data format error
 */
int
ckSvcRelocateOnPreferredNodeBoot(char *relocate, char *errMsg)
{
        static char *tab[]={ "yes","no",0};

        /* check if it is yes/no values. It is case sesentive */
        if( checkIsInDataset(relocate,tab, errMsg, 0)==FAIL)
	{
	    sprintf(errMsg,"Relocate policy can only be 'yes' or 'no'\n");
            return(FAIL);
        }
        return(SUCCESS);
}



/*
 * ckSvcPreferredNodeName
 *
 * Check if Preferred Node is used in the cluster configuration file
 * already.
 *
 * Return values
 *	SUCCESS		- the preferred node name is in Clu_Cfg database
 *	FAIL		- the node under the name is not in  Clu_Cfg database
 *      WARNING         - cannot read Clu_Cfg struct for some unknow reasons
 */
int
ckSvcPreferredNodeName( char *nodeName, char *errMsg)
{
	int nodeID;
	char *value;

	/* check if Preferred Node is in Clu_Cfg database */        
  	for (nodeID=0; nodeID < MAX_NODES; nodeID++)
	{
       	  if (getNodeName(nodeID, &value)==FAIL)
	    continue; // not found, try next nodeID   
	  if (strcmp(value,nodeName) == 0)
	     return(SUCCESS);
       }
       sprintf(errMsg, "Node name %s was not found in the cluster database\n", 
	       nodeName);
       return(FAIL);
}



/*
 *  ckSvcPreferredNodeID( int nodeID, char *errMsg)
 *
 *  Currently, the function does nothing.
 *
 * Return values
 *	SUCCESS	        - always return this value  
 */
int
ckSvcPreferredNodeID( int nodeID  __attribute__ ((unused)), char *errMsg __attribute__ ((unused)) )
{
        /* do nothing */
        return(SUCCESS);
}


/*
 * ckSvcDisabled
 * Check if the format for disable field is valid.
 * Currently we only check if the string in char *disabled is
 * yes or no.
 *
 * Return values:
 *	SUCCESS		- data in char *disabled is correct  
 *	FAIL	        - data format error
 * 
 */
int
ckSvcDisabled(char *disabled, char *errMsg)
{          
        static char *tab[]={ "yes","no",0};
 
        /* lookup the tab to see if char *disabled is in it */
        if( checkIsInDataset(disabled,tab, errMsg, 0)==FAIL)

        {
	    sprintf(errMsg,
                    "Service disabled value can only be 'yes' or 'no'\n");
	    return(FAIL);
        }
        return(SUCCESS);
}


/*
 * ckSvcIPaddress
 * To check the IP address if it is valid or not(format and content).
 * From the network interface, we  get host IPs and MASKs, then decide  if    
 * it is in the subnet. 
 * We further check if the IP is in use by somebody already. If yes, give
 * warning. That is if we ping the IP, and the IP site is up, we give Warning 
 * message. But this part will be done in the future.  
 *
 * Return values
 *      FAIL:  given IP (IPaddr) is not in the subnet
 *      SUCCESS: if given IP (IPaddr) is in the subnet, given IP is not used.
 *      WARNING: if given IP (IPaddr) is in the subnet, but is in use by other
 *                or we couldn't read network interface . 
 */
int
ckSvcIPaddress(int current_svcID, int current_ipN, char *IPaddr, char *errMsg)
{

	int svcID;
        int i,total_interface;
        int sockfd, len, lastlen;
        long int Mask[MAX_INTERFACE],Network_IP[MAX_INTERFACE],Temp_IP;        
	char *svcName, *value;
        char *ptr, *buf;
        struct ifconf ifc;
        struct ifreq  *ifr, ifr1,ifr2;
        struct sockaddr_in  *my_sin;

	/*
	 * Check if the IP address is already defined in this service
	 * or another service.
	 */
        for (svcID=MIN_SERVICE; svcID < MAX_SERVICES; svcID++)
        {
             if (serviceExists(svcID) != YES)
                continue;       // not in database
             i=0; 
             for(;;)
             {     
               if((svcID==current_svcID)&&(i==current_ipN))
               {
                 i++;
		 continue; // skip current entry point
               }
               if((getSvcIPaddress(svcID, i, &value)==NOT_FOUND))
		  break;
	       i++; 
	       if(strcmp(value,IPaddr)==0)
	       {
		  getSvcName(svcID,&svcName); 
                  sprintf(errMsg,
	                  "IP address '%s' is already defined in service %s\n",
	                  IPaddr, svcName);
                  return(FAIL);    // return with error 
	       }
             }
	}           

        if(checkFormatOfIP(IPaddr, errMsg)==FAIL)
        return(FAIL);  /* meet with data format error, return with the 
                          error in errMsg */
        
	// IPaddr is a valid IP string, now we can convert it to a long number
        inet_aton(IPaddr, (struct in_addr*) &Temp_IP);

        /* Above, we ensure IP string is correct. Now let's get netmask and IP
           address from network interface. Get the interface head first */
   
        sockfd = socket(AF_INET, SOCK_DGRAM, 0); 
        if (sockfd < 0) 
        {
           sprintf(errMsg,"Network interface read error:%s\n",
	           strerror(errno));
	   return(WARNING);
        }
 
        lastlen = 0;
        len = 100 * sizeof(struct ifreq);
    
        for (;;) 
        {
	   buf = (char *) malloc(len);
	   ifc.ifc_len = len;
	   ifc.ifc_buf = buf;
           /* SIOCGIFCONF request is used to retrieve interface configuration 
              for machine (useful for programs which must know all networks 
              accessible).  
	   */  
           if (ioctl(sockfd,SIOCGIFCONF,  &ifc) < 0) 
           {
              if (errno != EINVAL || lastlen != 0)
              {
		 free(buf);
		 close(sockfd);
               }
  	       sprintf(errMsg,"Cannot get network configuration: %s\n",
	               strerror(errno));
               return(WARNING);
	   } 
           else 
           {
	           if (ifc.ifc_len == lastlen)
		      break; /* success, len has not changed */
	           lastlen = ifc.ifc_len;
	   }
	   len += 10 * sizeof(struct ifreq);
	   free(buf);
        }   

        /* get Net MASK */
        i=0;
        for (ptr = buf; ptr < buf + ifc.ifc_len; ptr += sizeof(struct ifreq)) 
        {
            ifr = (struct ifreq *) ptr;
            strcpy(ifr1.ifr_name,ifr->ifr_name); 
	    if (ioctl(sockfd,SIOCGIFNETMASK ,&ifr1) < 0) 
            {
	       sprintf(errMsg, "Netmask read error: %s\n", strerror(errno));	
               return(WARNING);   
            } 
            my_sin = (struct sockaddr_in *) &ifr1.ifr_netmask;
            inet_aton(inet_ntoa(my_sin->sin_addr),(struct in_addr *) &Mask[i]);
            i++;
	}
        total_interface=i;

        /* get IP  address */
        i=0;
        for (ptr = buf; ptr < buf + ifc.ifc_len; ptr += sizeof(struct ifreq)) 
        {
            ifr = (struct ifreq *) ptr;
            strcpy(ifr2.ifr_name,ifr->ifr_name); 
	    if (ioctl(sockfd, SIOCGIFADDR ,&ifr2) < 0) 
            {
	       sprintf(errMsg,"Network interface read error: %s",strerror(errno));
               return(WARNING);   
            }
            my_sin = (struct sockaddr_in *) &ifr2.ifr_addr;
            inet_aton(inet_ntoa(my_sin->sin_addr),
                      (struct in_addr *)&Network_IP[i]);
            i++;
        }    

        for (i=0;i<total_interface;i++)
        {
            if((Mask[i] & Temp_IP)==(Mask[i] & Network_IP[i]))
	    {
               free(buf);
               close(sockfd);
	       return(SUCCESS);
            }
        }
        free(buf);
        close(sockfd);
        sprintf(errMsg,
"IP address '%s' is not in a subnet of any network interface!\n", IPaddr);
        return(FAIL);
}



/*
 * ckSvcDevice
 *
 * To check if the device exists and is valid(block or character facility).
 *
 * Return values
 *	SUCCESS		- device exists and is valid.
 *	FAIL	        - device doesn't exist or is not a block or character 
 *                        special file
 */
int
ckSvcDevice(int current_svcID, int current_devN, char *device, char *errMsg)
{     
	int flag, i, svcID;
        char *value, *svcName;
        struct stat BUF;
      
 	flag=stat(device,&BUF);
	if(flag<0)
	{ 
            sprintf(errMsg,"%s: %s\n", device, strerror(errno));
            return(FAIL);
        }
        else
	{  /* Block or character device */ 
	   if((!S_ISBLK(BUF.st_mode))&&(!S_ISCHR(BUF.st_mode)))
	
	    /***** NOTE: a device to be mounted could be a charater
                   special file (a device like a terminal)
                   so (!S_ISCHR(BUF.st_mod) test is added here.
	    ******/
	    {
	       sprintf(errMsg,"Device %s: not a block or character device\n",
                       device);
               return(FAIL);  
	    }
        }       

	/*
	 * Check if the device is already defined in this service or 
	 * another service.
	 */
        for (svcID=MIN_SERVICE; svcID < MAX_SERVICES; svcID++)
        {
             if (serviceExists(svcID) != YES)
                continue;       // not in database
             i=0; 
             for(;;)
             {     
               if((svcID==current_svcID)&&(i==current_devN))
               {
                 i++;
		 continue; // skip current entry point
               }
               if((getSvcDevice(svcID, i, &value)==NOT_FOUND))
		  break;
	       i++; 
	       if(strcmp(value,device)==0)
	       {
		  getSvcName(svcID,&svcName); 
                  sprintf(errMsg,
	                  "Device '%s' is already defined in service %s\n",
	                  device, svcName);
                  return(FAIL);    // return with error 
	       }
             }
	}           
        return(SUCCESS);
}


/*
 * ckSvcMountPoint
 *
 * To check 
 *	   (1) If the mount point exists & is  a directory 
 *            NOTE:Greg thinks that is not necessary. Because user can
 *            create a mount point later. This check is turned off.
 *         (2) If somebody has use the mount point in the 
 *             CLU_CFG structure.
 *
 * Return values
 *	SUCCESS		- Mount point is valid.
 *	FAIL	        - Mount point doesn't exist or is not a directory
 */
int
ckSvcMountPoint(int current_svcID, int current_devN, char *mountPoint, char *errMsg)
{      
        int svcID;
        char *svcName;
	int i;
        char *value;
      
	/* to check if the mount point is used by somebody already
	   in the cluster's configuration database */
        for (svcID=MIN_SERVICE; svcID < MAX_SERVICES; svcID++)
        {
             if (serviceExists(svcID) != YES)
                continue;       // not in database
             i=0; 
             for(;;)
             {     
               if((svcID==current_svcID)&&(i==current_devN))
               {
                 i++;
		 continue; // skip current entry point
               }
               if((getSvcMountPoint(svcID, i, &value)==NOT_FOUND))
		  break;
	       i++; 
	       if(strcmp(value,mountPoint)==0)
                 // find the mount Point has been used
	       {
		  getSvcName(svcID,&svcName); 
                  sprintf(errMsg,
	                  "Mount point '%s' is already defined in service%s\n",
	                  mountPoint, svcName);
                  return(FAIL);    // return with error 
	       }
             }
	}           
        return(SUCCESS);
}

/*
 * ckSvcMountOptions
 * To check if the MountOption is a valid string for ordinary mount options
 * NOTE: MountOption is a string, something like "rw,suid,exec" which is 
 *       seperated by ',' or ' '(a blank space).     
 *
 * Return values:
 *	SUCCESS		- mount options are valid  
 *	WARNING	        - mounht options are not valid 
 *
 */
int
ckSvcMountOptions(char *mountOptions, char *errMsg)
{
  static char *tab[]={  /* ISO9660 file system */
                       "rw","ro","suid", "dev", "nodev","exec", "noexec",
                       "sync","async", "atime","noatime","auto","noauto",
                       "nosuid","user", "nouser","remount", "defaults",
                       /* ext2 file system */
                       "bsddf","minixdf","check","check=none", "check=nocheck",
                       "check=normal","check=strict","debug","errors=continue",
                       "errors=remount-ro","errors=panic","grpid","bsdgroups",
                       "nogrpid", "sysvgroups","resgid=n","resuid=n","sb=n",
                       "grpquota","noquota","quota", "usrquota",
                       0 };
        int first;
        char *s;
        const char delim[4]=", ";
        char *p;

         //below is for option string check
        s=(char *)malloc(MAX_LINELEN);
        strcpy(s,mountOptions);

        first=0;
        for(;;)
	{             
            if(first==0)
	    {
               p=strtok(s,delim);
               first=1;
            } 
            else p=strtok(NULL,delim);
            if(p==NULL)
              break;

            if( checkIsInDataset(p,tab, errMsg, 0)==FAIL)
	    {
	        sprintf(errMsg,"Mount option, %s, is not valid\n",p);
                free(s);
	        return(WARNING);  /* User may use some new options to mount
                                   * , so it is not necessary be an error
	                           * if we cannot find it in the option set.
                                   */
            }
        }
        free(s); 
        return(SUCCESS);
}


/*
 * ckSvcMountFstype
 * To check if the MountFstype is a valid string.
 *
 * NOTE: MountFstype is a string, that represents a valid filesystem type
 *       such as 'ext2', 'iso9660', 'minix'
 * 
 *       This code looks through /proc/filesystems and looks for
 *       filesystems that are configured in the kernel.
 *
 * $ cat /proc/filesystems
 * # flag     \ttype\n
 * 
 * 		ext2
 * nodev	proc
 * 		iso9660
 * nodev	devpts
 * nodev	nfs
 *
 * Return values:
 *	SUCCESS		- mount options are valid  
 *	WARNING	        - mounht options are not valid 
 *
 */
int
ckSvcMountFstype(char *mountFstype, char *errMsg)
{
	static char *proc_filename="/proc/filesystems";
	FILE * filesystems;
	char fs_line[80];
        char *p;

        if(strlen(mountFstype)==0)
	    return(SUCCESS); // That mountFstype is empty is OK.
	
	// Read in the entries from /proc/filesystems
	filesystems = fopen (proc_filename, "r");
	if (!filesystems) {
	  sprintf(errMsg,"Can't open %s to determine fstype\n",
		  proc_filename);
	  return(WARNING); 
	}

	// Loop through the file to read all the filesystem types the
	// kernel knows about.
	while (fgets(fs_line, sizeof(fs_line),filesystems)) {
	  p = strchr(fs_line, '\t');
	  if (p && *p) {
	    // skip tab
	    p++;
	    // remove newline
	    p[strlen(p)-1]=0;
	    //
	    if (!strcmp (mountFstype, p)) {
	    clulog(LOG_DEBUG,
			     "Matched specified fstype %s to %s\n",
			     p, mountFstype);
	      fclose(filesystems);
	      return(SUCCESS);
	    }
	  }
	}
	fclose(filesystems);

	sprintf(errMsg, "Filesystem %s not confingured in kernel through %s\n",
		mountFstype,
		proc_filename);	
        return(FAIL);
}

/*
 * ckSvcForceUnmount
 * Check if the data in char *forceUnmount is correct or not.
 *
 * Return values:
 *	SUCCESS		- data is in correct format  
 *	FAIL	        - data format error.  
 */
int
ckSvcForceUnmount(char *forceUnmount, char *errMsg)
{
        static char *tab[]={ "yes","no",0};
 
        /* lookup the tab to see if forceUnmount is in it */
        if( checkIsInDataset(forceUnmount,tab, errMsg, 0)==FAIL)
	{
	    sprintf(errMsg,"Forced unmount value can only be 'yes'or 'no'\n");
            return(FAIL);
        }
        return(SUCCESS);
}



/*
 * ckSvcDeviceOwner
 * Check if the device owner is valid in /etc/passwd file and 
 * distributed file, yppasswd
 *
 * Return values:    
 *	WARNING		- no, the owner is not in passwd/yppasswd file.
 *                        the administrator of cluster may want to create 
 *                        a valid owner. So "not found" is WARNING, not FAIL. 
 *      SUCCESS         - yes, find the owner is in passwd/yppasswd file
 */
int
ckSvcDeviceOwner(char *deviceOwner, char *errMsg)
{ 
        struct passwd *pw;
        
        pw=getpwnam (deviceOwner);
        if(pw==NULL)
	{
	   sprintf(errMsg, "User '%s' was  not found in the password file\n", 
	           deviceOwner);
            return(WARNING); 
           // may be user want to create a new user account for the device
        }
        return(SUCCESS);
}



/*
 * ckSvcDeviceGroup
 *
 * Check if the device group is valid in /etc/group file and 
 * distributed group file
 *
 * Return values:
 *	SUCCESS		- device group found
 *	WARNING 	- device group not found
 */
int
ckSvcDeviceGroup(char *deviceGroup, char *errMsg)
{
        struct group *gr;
    
        gr=getgrnam (deviceGroup);
        if(gr==NULL)
	{
	  sprintf(errMsg, "Group '%s' was  not found in the group file\n",
                  deviceGroup);
          return(WARNING);
        }
	return(SUCCESS);
}



/*
 * ckSvcDeviceMode
 * To check if a deviceMode string is a valid format like "4777","4770",
 * and so on.
 *
 * Return values
 *	SUCCESS		- device Mode is in a valid format
 *	FAIL	        - device Mode is not in a valid format 
 */
int
ckSvcDeviceMode(char *deviceMode, char *errMsg)
{	
        int l;
        int i;

        l=strlen(deviceMode);
        if(l>4)
	{ 
          strcpy(errMsg, "Device mode too long\n");
          return(FAIL);
        }
       
        for(i=0;i<l;i++)
        {
         
	  if((deviceMode[i]<'0') || (deviceMode[i]>'7')) 
	  {
               sprintf(errMsg,
"Invalid character '%c' in device permissions\n", deviceMode[i]);
               return(FAIL);
	  }
        }     
	return(SUCCESS);
}

/*
 * ckSvcBroadcast()
 * To check if a string is in a valid broadcast address format
 *
 *
 * Return values
 *	SUCCESS		- char *Broadcast is in a valid format
 *	FAIL	        - char *Broadcast is not in a valid format 
 */
int
ckSvcBroadcast(char *Broadcast, char *errMsg)
{
        return(checkFormatOfIP(Broadcast,errMsg));
}


/*
 * ckSvcNetmask()
 * To check if a netmask string is in a valid netmask format.
 *
 * Return values
 *	SUCCESS		- char *Netmask is in a valid format
 *	FAIL	        - char *Netmask is not in a valid format 
 */
int
ckSvcNetmask(char *Netmask, char *errMsg)
{
        return(checkFormatOfIP(Netmask,errMsg));
}

/*
 * checkFormatOfIP()
 * To check if a string is in a valid IPv4 address format
 * Like 10.2.8.28, 18.62.2.1, 255.255.255.0, etc. Two kinds of check 
 * are done: (1) it is strictly in A.B.C.D format
 *           (2) A,B,C,D are number and in the range of 0 to 255.
 *           
 * Return values
 *	SUCCESS		- char *String is in a correct IPv4 format
 *	FAIL	        - char *String is not in a valid format 
 */
int
checkFormatOfIP(char *String, char *errMsg)
{
       int l,range,i;       
       char temp[MAX_LINE_LEN];
       int start, dot_num;

       start=0;

       /* first, let get rid of blanks in the String */
       clean_string(String,0);
       l=strlen(String);

       if (l <= 0)
       { 
          strcpy(errMsg,"IP address is empty\n");
          return(FAIL);
       }

       if(l >= 16)
       { 
          sprintf(errMsg,"IP address '%s' is too long\n", String);
          return(FAIL);
       }
        
        
        start=0; dot_num=0;
        for(i=0;i<l;i++)
        {
	  if((!isdigit(String[i])) && (String[i]!='.'))
          {
            sprintf(errMsg,"IP address '%s' contains invalid characters\n",
	            String);
            return(FAIL);
          }
          else if((String[i]=='.')||(i==l-1))
          {
            if(String[i]=='.')
	    {
              dot_num++;
              if ((i==0) || (i==l-1))
	      {
                sprintf(errMsg,"IP address '%s' is not in a valid format\n",
	               String);
                return(FAIL);
              }
	    }           
            if(i==l-1) 
	    /* meet string's end, then scan the digital to temp 
               string for IP field's data
            */
            {
	       temp[start]=String[i];
               start++;
            }

            if(start==0)   // wrong format like a dot followed by a dot
	    {	           // such as 192..23.1 will be checked as an error
                sprintf(errMsg, "IP address '%s' is not in a valid format\n",
	               String);
                return(FAIL);
	    }

            temp[start]='\0';      
            start=0;
            range=atoi(temp);
            if((range>255)||(range<0))
	    { 
               sprintf(errMsg,
"A field of the IP address '%s' is not in the range 0..255\n", String);
               return(FAIL);
            }             
          }  
          else if (isdigit(String[i]))
          {
	       temp[start]=String[i];
               start++;
          }
        }
       if(dot_num!=3)
       { 
	 sprintf(errMsg, "IP address '%s' is not in a valid format\n", 
	         String);
         return(FAIL); 
       }     
       return(SUCCESS);
}

/*
 * checkIsInDataset
 * Check if the data is in the data set or not.
 * if flag==0, match is case sensetive, flag==1 the match is not case sensetive
 * if flag>=2, then match is case sensetive and only match partly of data string which is before '='. 
 * Return values:
 *	SUCCESS		- data is in the data set  
 *	FAIL	        - data is not in the data set.  
 */
int 
checkIsInDataset(char *data,char **dataset, char *errMsg, int flag)
{
        int i=0;
        unsigned int len;

        len=strlen(data); 
        for(;;)
        { 
          if(dataset[i]==NULL)
             break;
          if(flag==0)
	  {
            if(strcmp(data,dataset[i])==0) 
            return(SUCCESS);
          }
          else if(flag==1)
	  {
             if(strcasecmp(data,dataset[i])==0) 
             return(SUCCESS);
          }
          else // if flag >= 2
	  {   if(strcmp(data,dataset[i])==0) 
                  return(SUCCESS);
	      if(len>=1)
	      { 
		if((strncmp(data,dataset[i],(len>strlen(dataset[i]) ? strlen(dataset[i]) : len))==0) && (dataset[i][strlen(dataset[i])-1]=='='))
		   return(SUCCESS); 
              }   
          } 
	  i++;
        }
        sprintf(errMsg, "data '%s' is not in option set\n", data); 
        return(FAIL);
}

/*
 * ckIsMemberIP
 * Check if the IP is used by a member of cluster already 
 * Return values:
 *	NO		- IP is not in use by the cluster members  
 *	YES	        - IP is in use by one of the cluster members
 */
int 
ckIsMemberIP(char *IP, char *errMsg)
{
  struct hostent * host_p;
  struct sockaddr_in  my_sin;
  int i;
  char *nodeName,temp_IP[20];
  int nodeID;

  clean_string(IP,0);      
  for (nodeID=0; nodeID < MAX_NODES; nodeID++)
  {
     if (getNodeName(nodeID, &nodeName)==FAIL)
	 continue; // not found, try next nodeID   
     i=0;
     host_p=gethostbyname(nodeName);
     if(host_p) // find the host with nodeName
     { 
         my_sin.sin_family = AF_INET;
         while(host_p->h_addr_list[i])
         {
            my_sin.sin_addr = *(struct in_addr *) host_p->h_addr_list[i];
            strcpy(temp_IP, inet_ntoa(my_sin.sin_addr));
            if(strcmp(temp_IP,IP)==0)
	    {
	        sprintf(errMsg, "IP '%s' is in use by the member '%s'\n",IP,nodeName);   
                return(YES); //Opps, the IP is in use by one of cluster members
            } 
            i++;
         }
      }
  }
  return(NO);
}

/****************************************************************************
 *
 * The following functions set the different service entries in the database.
 *
 ***************************************************************************/

/*
 * setSvcName
 *
 * Given a service ID set its service name to svcName.
 */
int
setSvcName(int svcID, char *svcName, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcName(svcID,svcName,errMsg)==FAIL)
	{
           clulog(LOG_ERR, "%s", errMsg);
           return(FAIL);
        } 
	getSvcNameTokenStr(svcID, token);
	return(setDatabaseToken(token, svcName));
}

/*
 * setSvcScript
 *
 * Given a service ID assign a new pathname for its user script as defined
 * in svcScript.
 */
int
setSvcScript(int svcID, char *svcScript, char *errMsg)
{
	char token[MAX_TOKEN_LEN];
        int check; 
       
        sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        check=ckSvcScript(svcScript,errMsg);
        if(check!=SUCCESS)
            clulog(LOG_ERR, "%s", errMsg);
        if (check==FAIL)
	    return(FAIL);
        getSvcScriptTokenStr(svcID, token);
	return(setDatabaseToken(token, svcScript));
}

/*
 * setSvcRelocateOnPreferredNodeBoot
 *
 * Given a service ID set the "relocate service on preferred node boot"
 * setting to YES_STR or NO_STR.
 */
int
setSvcRelocateOnPreferredNodeBoot(int svcID, char *relocate, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcRelocateOnPreferredNodeBoot(relocate,errMsg)==FAIL)
	{ 
          clulog(LOG_ERR, "%s", errMsg);
          return(FAIL);
        } 
	getSvcRelocateOnPreferredNodeBootTokenStr(svcID, token);
	return(setDatabaseToken(token, relocate));
}

/*
 * setSvcPreferredNodeName
 *
 * Given a service ID set the preferred node name to nodeName.
 */
int
setSvcPreferredNodeName(int svcID, char *nodeName, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcPreferredNodeName(nodeName,errMsg)==FAIL)
	{
            clulog(LOG_ERR, "%s", errMsg);
            return(FAIL);
        } 
	getSvcPreferredNodeNameTokenStr(svcID, token);
	return(setDatabaseToken(token, nodeName));
}

/*
 * setSvcPreferredNodeID
 *
 * Given a service ID set the preferred node ID to nodeID.
 */
int
setSvcPreferredNodeID(int svcID, int nodeID, char *errMsg)
{
	char *nodeName;

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcPreferredNodeID(nodeID,errMsg)==FAIL)
	{
            clulog(LOG_ERR, "%s", errMsg);
            return(FAIL);
        }  
	if (getNodeName(nodeID, &nodeName) == FAIL)
	    return(FAIL);

	return(setSvcPreferredNodeName(svcID, nodeName, errMsg));
}

/*
 * setSvcDisabled
 *
 * Given a service ID set a service to be disabled or enabled.
 */
int
setSvcDisabled(int svcID, char *disabled, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcDisabled(disabled,errMsg)==FAIL)
	{
            clulog(LOG_ERR, "%s", errMsg);
            return(FAIL);
        }
	getSvcDisabledTokenStr(svcID, token);
	return(setDatabaseToken(token, disabled));
}

/*
 * setSvcIPaddress
 *
 * Given a service ID assign the Nth IP address to IPaddr.
 */
int
setSvcIPaddress(int svcID, int N, char *IPaddr, char *errMsg)
{
	char token[MAX_TOKEN_LEN];
        int check;

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        
        check=ckSvcIPaddress(svcID,N,IPaddr,errMsg);
        if(check!=SUCCESS)
            clulog(LOG_ERR, "%s", errMsg);
        if(check==FAIL)
            return(FAIL);
        if(ckIsMemberIP(IPaddr, errMsg)==YES) 
        //check further to see if IP is in use by the cluster members
	{   
            clulog(LOG_ERR, "%s", errMsg);
            return(FAIL);
        }    
 	getSvcIPaddressTokenStr(svcID, N, token);
	return(setDatabaseToken(token, IPaddr));
}

/*
 * setSvcNetmask
 *
 * Given a service ID assign the Nth netmask address to netmask.
 */
int
setSvcNetmask(int svcID, int N, char *netmask, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcNetmask(netmask,errMsg)==FAIL)
	{
            clulog(LOG_ERR, "%s", errMsg);
            return(FAIL);
        } 
	getSvcNetmaskTokenStr(svcID, N, token);
	return(setDatabaseToken(token, netmask));
}

/*
 * setSvcBroadcast
 *
 * Given a service ID assign the Nth broadcast address to broadcast.
 */
int
setSvcBroadcast(int svcID, int N, char *broadcast, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcBroadcast(broadcast,errMsg)==FAIL)
	{
            clulog(LOG_ERR, "%s", errMsg);
            return(FAIL);
        }
	getSvcBroadcastTokenStr(svcID, N, token);
	return(setDatabaseToken(token, broadcast));
}

/*
 * setSvcDevice
 *
 * Given a service ID assign the Nth device name to device.
 */
int
setSvcDevice(int svcID, int N, char *device, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcDevice(svcID, N, device,errMsg)==FAIL)
        {           
            clulog(LOG_ERR, "%s", errMsg); 
            return(FAIL);
        } 
	getSvcDeviceTokenStr(svcID, N, token);
	return(setDatabaseToken(token, device));
}

/*
 * setSvcMountPoint
 *
 * Given a service ID assign the Nth mount point to mountPoint.
 */
int
setSvcMountPoint(int svcID, int N, char *mountPoint, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if( ckSvcMountPoint(svcID,N, mountPoint,errMsg)==WARNING)
            clulog(LOG_ERR, "%s", errMsg); 
	getSvcMountPointTokenStr(svcID, N, token);
	return(setDatabaseToken(token, mountPoint));
}

/*
 * setSvcMountOptions
 *
 * Given a service ID assign the Nth mount point to mountOptions.
 */
int
setSvcMountOptions(int svcID, int N, char *mountOptions, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcMountOptions(mountOptions,errMsg)==FAIL)
        {           
            clulog(LOG_ERR, "%s", errMsg); 
            return(FAIL);
        } 
	getSvcMountOptionsTokenStr(svcID, N, token);
	return(setDatabaseToken(token, mountOptions));
}

/*
 * setSvcMountFstype
 *
 * Given a service ID assign the Nth mount point to mountFstype.
 */
int
setSvcMountFstype(int svcID, int N, char *mountFstype, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcMountFstype(mountFstype,errMsg)==FAIL)
        {           
            clulog(LOG_ERR, "%s", errMsg); 
            return(FAIL);
        } 
	getSvcMountFstypeTokenStr(svcID, N, token);
	return(setDatabaseToken(token, mountFstype));
}

/*
 * setSvcForceUnmount
 *
 * Given a service ID assign the Nth mount point to mountOptions.
 */
int
setSvcForceUnmount(int svcID, int N, char *forceUnmount, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcForceUnmount(forceUnmount,errMsg)==FAIL)
        {           
            clulog(LOG_ERR, "%s", errMsg); 
            return(FAIL);
        } 
	getSvcForceUnmountTokenStr(svcID, N, token);
	return(setDatabaseToken(token, forceUnmount));
}

/*
 * setSvcDeviceOwner
 *
 * Given a service ID assign the Nth device owner deviceOwner.
 */
int
setSvcDeviceOwner(int svcID, int N, char *deviceOwner, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcDeviceOwner(deviceOwner,errMsg)==WARNING)
            clulog(LOG_ERR, "%s", errMsg); 
	getSvcDeviceOwnerTokenStr(svcID, N, token);
	return(setDatabaseToken(token, deviceOwner));
}

/*
 * setSvcDeviceGroup
 *
 * Given a service ID assign the Nth device owner deviceGroup.
 */
int
setSvcDeviceGroup(int svcID, int N, char *deviceGroup, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcDeviceGroup(deviceGroup,errMsg)==WARNING) // group not found
	    clulog(LOG_ERR, "%s", errMsg); 
        /* only write warning message, but continue set what you want */
	getSvcDeviceGroupTokenStr(svcID, N, token);
	return(setDatabaseToken(token, deviceGroup));
}

/*
 * setSvcDeviceMode
 *
 * Given a service ID assign the Nth device mode deviceGroup.
 */
int
setSvcDeviceMode(int svcID, int N, char *deviceMode, char *errMsg)
{
	char token[MAX_TOKEN_LEN];

	sprintf(errMsg, "Failed\n");	// temp fix for compiler errors
        if(ckSvcDeviceMode(deviceMode,errMsg)==FAIL)
        {    
             clulog(LOG_ERR, "%s", errMsg); 
             return(FAIL);
        } 
	getSvcDeviceModeTokenStr(svcID, N, token);
	return(setDatabaseToken(token, deviceMode));
}

/****************************************************************************
 *
 * The following functions get the different service entries in the database.
 *
 ***************************************************************************/

/*
 * getSvcID
 *
 * Given a service name return the service ID.  
 *
 * NOTE: there is no setSvcID() as service ID's are persistent and assigned
 * at service create time.  A service ID is created by finding the next 
 * available ID and creating the services%service1%name entry.
 *
 * Return values:
 *	SUCCESS		- service ID found, value stored in svcID
 *	NOT_FOUND	- service ID not found
 *	FAIL		- failure
 */
int
getSvcID(char *svcName, int *svcID)
{
	
	int serviceID;
	char *value=(char *)NULL;

	for (serviceID=0; serviceID < MAX_SERVICES; serviceID++)
	  {
	    if (serviceExists(serviceID) != YES)
	        continue;

	    if (getSvcName(serviceID, &value) != SUCCESS)
	      continue;

	    /*
	     * See if this is the service we are looking for
	     */
	    if (strcmp(value, svcName) == 0)
	      {
	        *svcID = serviceID;
	        return(SUCCESS);
	      }
	  }

	clulog(LOG_ERR, 
"Cannot get service ID for service %s from database: not found\n", svcName);

	*svcID=SVC_ID_NONE;
	return(NOT_FOUND);
}

/*
 * getSvcName
 *
 * Given a service ID return the service name in svcName.
 *
 * Return values:
 *	SUCCESS		- a service name was found and stored in svcName
 *	FAIL		- failure
 */
int
getSvcName(int svcID, char **svcName)
{
	char token[MAX_TOKEN_LEN];
	int retVal;
	static char nameBuf[MAX_SERVICE_NAMELEN];

	getSvcNameTokenStr(svcID, token);
	/*
	 * If we do not find the service name, return the reason why
	 * but send back a string of the service ID so we can get
	 * useful printf's to the log.
	 */
	if ((retVal=getDatabaseToken(token, svcName)) != SUCCESS)
	  {
	    clulog(LOG_WARNING, 
	               "Cannot get service name for service #%d\n", svcID);
	    sprintf(nameBuf, "%d", svcID);
	    *svcName=(char *)&nameBuf;
	    return(retVal);
	  }

	return(SUCCESS);
}

/*
 * getSvcScript
 *
 * Get the user service script for the service associated with the given
 * service ID. 
 *
 * Return values:
 *	SUCCESS		- a service script was found and stored in svcScript
 *	NOT_FOUND	- a service script was not found, svcName==(char *)NULL
 *	FAIL		- failure
 */
int
getSvcScript(int svcID, char **svcScript)
{
	char token[MAX_TOKEN_LEN];

	getSvcScriptTokenStr(svcID, token);
	return(getDatabaseToken(token, svcScript));
}

/*
 * getSvcRelocateOnPreferredNodeBoot
 *
 * Given a service ID return the setting for "relocate service on preferred
 * node boot".
 *
 * Return values
 *	SUCCESS		- service relocate setting found, value in relocate
 *	NOT_FOUND	- service reloate setting not found
 *	FAIL		- failure
 */
int
getSvcRelocateOnPreferredNodeBoot(int svcID, char **relocate)
{
	char token[MAX_TOKEN_LEN];

	getSvcRelocateOnPreferredNodeBootTokenStr(svcID, token);
	return(getDatabaseToken(token, relocate));
}

/*
 * getSvcPreferredNodeName
 *
 * Given a service number get the preferred node name.  If it is not defined 
 * return NODE_ID_NONE.
 *
 * Return values
 *	SUCCESS		- preferred node name found, value in nodeName
 *	NOT_FOUND	- there is no preferred node
 *	FAIL		- failure
 */
int
getSvcPreferredNodeName(int svcID, char **nodeName) 
{
	char token[MAX_TOKEN_LEN];

	getSvcPreferredNodeNameTokenStr(svcID, token);
	return (getDatabaseToken(token, nodeName));
}

/*
 * getSvcPreferredNodeID
 *
 * Given a service number get the preferred node ID.  If it is not defined 
 * return NODE_ID_NONE.
 *
 * Return values
 *	SUCCESS		- preferred node ID found, value in nodeID
 *	NOT_FOUND	- there is no preferred node
 *	FAIL		- failure
 */
int
getSvcPreferredNodeID(int svcID, int *nodeID)
{
	char *nodeName;

	*nodeID = NODE_ID_NONE;

	switch (getSvcPreferredNodeName(svcID, &nodeName))
	  {
	    case FAIL:
	      return(FAIL);

	    case NOT_FOUND:
	      return(NOT_FOUND);
	  }

	if (nodeName == (char *)NULL)	// no preferred node defined
	    return(NOT_FOUND);

	/*
	 * Convert the node name string into a node ID.
	 */
	if ((*nodeID=getNodeID(nodeName)) == FAIL)
	    return(FAIL);

	return(SUCCESS);
}

/*
 * getSvcDisabled
 *
 * Given a service ID return settings which defines if it is disabled.
 *
 * Return values
 *	SUCCESS		- service disabled setting found result in disabled
 *	NOT_FOUND	- service disable setting not found
 *	FAIL		- failure
 */
int
getSvcDisabled(int svcID, char **disabled)
{
	char token[MAX_TOKEN_LEN];

	getSvcDisabledTokenStr(svcID, token);
	return(getDatabaseToken(token, disabled));
}

/*
 * getSvcIPaddress
 *
 * Given a service ID find the Nth IP address entry.
 *
 * Return values
 *	SUCCESS		- IP address found, value returned in IPaddr
 *	NOT_FOUND	- IP address not found
 *	FAIL		- failure
 */
int
getSvcIPaddress(int svcID, int N, char **IPaddr)
{
	char token[MAX_TOKEN_LEN];

	getSvcIPaddressTokenStr(svcID, N, token);
	return(getDatabaseToken(token, IPaddr));
}

/*
 * getSvcNetmask
 *
 * Given a service ID find the Nth network mask address entry.
 *
 * Return values
 *	SUCCESS		- Netmask found, value returned in netmask
 *	NOT_FOUND	- Netmask not found
 *	FAIL		- failure
 */
int
getSvcNetmask(int svcID, int N, char **netmask)
{
	char token[MAX_TOKEN_LEN];

	getSvcNetmaskTokenStr(svcID, N, token);
	return(getDatabaseToken(token, netmask));
}

/*
 * getSvcBroadcast
 *
 * Given a service ID find the Nth broadcast address entry.
 *
 * Return values
 *	SUCCESS		- Broadcast found, value returned in broadcast
 *	NOT_FOUND	- Broadcast not found
 *	FAIL		- failure
 */
int
getSvcBroadcast(int svcID, int N, char **broadcast)
{
	char token[MAX_TOKEN_LEN];

	getSvcBroadcastTokenStr(svcID, N, token);
	return(getDatabaseToken(token, broadcast));
}

/*
 * getSvcDevice
 *
 * Given a service ID find the Nth device address entry.
 *
 * Return values
 *	SUCCESS		- device found, value returned in device
 *	NOT_FOUND	- device not found
 *	FAIL		- failure
 */
int
getSvcDevice(int svcID, int N, char **device)
{
	char token[MAX_TOKEN_LEN];

	getSvcDeviceTokenStr(svcID, N, token);
	return(getDatabaseToken(token, device));
}

/*
 * getSvcMountPoint
 *
 * Given a service ID find the Nth mount point entry.
 *
 * Return values
 *	SUCCESS		- mount point found, value returned in mountPoint
 *	NOT_FOUND	- mount point not found
 *	FAIL		- failure
 */
int
getSvcMountPoint(int svcID, int N, char **mountPoint)
{
	char token[MAX_TOKEN_LEN];

	getSvcMountPointTokenStr(svcID, N, token);
	return(getDatabaseToken(token, mountPoint));
}


/*
 * getSvcMountFstype
 *
 * Given a service ID find the Nth mount fstype entry.  The Nth entry
 * corresponds to the Nth mount point entry for a service.
 *
 * Return values
 *	SUCCESS		- mount fstype found, value returned in mountFstype
 *	NOT_FOUND	- mount fstype not found
 *	FAIL		- failure
 */
int
getSvcMountFstype(int svcID, int N, char **mountFstype)
{
	char token[MAX_TOKEN_LEN];

	getSvcMountFstypeTokenStr(svcID, N, token);
	return(getDatabaseToken(token, mountFstype));
}

/*
 * getSvcMountOptions
 *
 * Given a service ID find the Nth mount options entry.  The Nth entry
 * corresponds to the Nth mount point entry for a service.
 *
 * Return values
 *	SUCCESS		- mount options found, value returned in mountOptions
 *	NOT_FOUND	- mount options not found
 *	FAIL		- failure
 */
int
getSvcMountOptions(int svcID, int N, char **mountOptions)
{
	char token[MAX_TOKEN_LEN];

	getSvcMountOptionsTokenStr(svcID, N, token);
	return(getDatabaseToken(token, mountOptions));
}

/*
 * getSvcForceUnmount
 *
 * Given a service ID find the forced unmount setting for mount point N. 
 * The Nth entry corresponds to the Nth mount point entry for a service.  
 * If the forced unmount setting is turned on YES is returned.  If is
 * not set, NO is returned.  If there is a failure, FAIL will be returned.
 * Given a service ID find the Nth device address entry.
 *
 * Return values
 *	SUCCESS		- forced unmount setting found, value in forceUnmount
 *	NOT_FOUND	- forced unmount setting not found
 *	FAIL		- failure
 */
int
getSvcForceUnmount(int svcID, int N, char **forceUnmount)
{
	char token[MAX_TOKEN_LEN];

	getSvcForceUnmountTokenStr(svcID, N, token);
	return(getDatabaseToken(token, forceUnmount));
}

/*
 * getSvcDeviceOwner
 *
 * Given a service ID find the device owner setting for mount point N. 
 * The Nth entry corresponds to the Nth mount point entry for a service.  
 *
 * Return values
 *	SUCCESS		- device owner found, value returned in deviceOwner
 *	NOT_FOUND	- device owner not found
 *	FAIL		- failure
 */
int
getSvcDeviceOwner(int svcID, int N, char **deviceOwner)
{
	char token[MAX_TOKEN_LEN];

	getSvcDeviceOwnerTokenStr(svcID, N, token);
	return(getDatabaseToken(token, deviceOwner));
}

/*
 * getSvcDeviceGroup
 *
 * Given a service ID find the device group setting for mount point N. 
 * The Nth entry corresponds to the Nth mount point entry for a service.  
 *
 * Return values
 *	SUCCESS		- device group found, value returned in device
 *	NOT_FOUND	- device group not found
 *	FAIL		- failure
 */
int
getSvcDeviceGroup(int svcID, int N, char **deviceGroup)
{
	char token[MAX_TOKEN_LEN];

	getSvcDeviceGroupTokenStr(svcID, N, token);
	return(getDatabaseToken(token, deviceGroup));
}

/*
 * getSvcDeviceMode
 *
 * Given a service ID find the device mode setting for mount point N. 
 * The Nth entry corresponds to the Nth mount point entry for a service.  
 *
 * Return values
 *	SUCCESS		- device found, value returned in device
 *	NOT_FOUND	- device not found
 *	FAIL		- failure
 */
int
getSvcDeviceMode(int svcID, int N, char **deviceMode)
{
	char token[MAX_TOKEN_LEN];

	getSvcDeviceModeTokenStr(svcID, N, token);
	return(getDatabaseToken(token, deviceMode));
}


/*****************************************************************************
 *
 * The following functions are used to find the next entry for resource
 * types that can have more than one entry in the service section of the
 * cluster configuration database.
 *
 ****************************************************************************/

/*
 * getNextFreeSvcID
 *
 * Return the next available service ID from the database.  If there
 * are no more service entries available return FAIL.
 *
 * Note: This does not create the service and the caller should lock the 
 * database when running this function if it is going to create a service.
 */
int
getNextFreeSvcID(void)
{
	int svcID;

	for (svcID=MIN_SERVICE; svcID < MAX_SERVICES; svcID++)
	  {
	    if (serviceExists(svcID) == YES)
	        continue;
	    else
	        break;
	  }
	
	if (svcID >= MAX_SERVICES)
	  {
	    clulog(LOG_ERR, "Reached maximum limit of services, %d\n", 
	           MAX_SERVICES);
	    return(FAIL);
	  }

	return(svcID);
}

/*
 * getNextSvcIPaddress
 *
 * Given a service ID find the next free IP address entry.  Nth entry
 * that is free is returned in N. 
 *
 * Return values
 *	SUCCESS		- Next free entry found
 *	FAIL		- failure
 */
int
getNextFreeSvcIPaddress(int svcID, int *N)
{
	char token[MAX_TOKEN_LEN];
	char *value;

	*N=MIN_TOKEN_ID;
	while (1)
	  {
	    getSvcIPaddressTokenStr(svcID, *N, token);
	    switch (getDatabaseToken(token, &value))
	      {
	        case NOT_FOUND:		// found free entry, return it
	          return(SUCCESS);
	        case FAIL:		// got an error
	          *N=SERVICE_ID_NONE;
	          return(FAIL);
	      }				// else, must be in use
	    *N=*N+1;
	  }

	return(SUCCESS);		// should never get here
}

/*
 * getNextFreeSvcDevice
 *
 * Given a service ID find the next free device address entry.  Nth entry
 * that is free is return in N.
 *
 * Return values
 *	SUCCESS		- next free entry found
 *	FAIL		- failure
 */
int
getNextFreeSvcDevice(int svcID, int *N)
{
	char token[MAX_TOKEN_LEN];
	char *value;

	*N=MIN_TOKEN_ID;
	while (1)
	  {
	    getSvcDeviceTokenStr(svcID, *N, token);
	    switch (getDatabaseToken(token, &value))
	      {
	        case NOT_FOUND:		// found free entry, return it
	          return(SUCCESS);
	        case FAIL:		// got an error
	          *N=SERVICE_ID_NONE;
	          return(FAIL);
	      }				// else, must be in use
	    *N=*N+1;
	  }

	return(SUCCESS);		// should never get here
}

/*****************************************************************************
 *
 * The following functions are used to check the "uniqueness" of service
 * attributes values.
 *
 ****************************************************************************/


/*****************************************************************************
 *
 * Following are functions that query the database and return YES NO values
 *
 *****************************************************************************/

/*
 * serviceExists
 *
 * Given a service ID determine if the service exists by checking to
 * see if its ID is in the database.  An easy way to do this is to check
 * for the service name since this is the only required attribute of a
 * service.  If the service exists, YES is returned.  If it does not
 * exist, NO is returned.
 */
int
serviceExists(int svcID)
{
	char *svcName;
	char token[MAX_TOKEN_LEN];

/* XXX add later after callers update for FAIL return
	if ((svcID < MIN_SERVICE) || (svcID > MAX_SERVICES))
	  {
	    clulog(LOG_ERR, 
	           "Invalid service ID %d passed to serviceExists()\n",
	           svcID);
	    return(FAIL);
	  }
*/

	getSvcNameTokenStr(svcID, token);

	if (getDatabaseToken(token, &svcName) != SUCCESS)
	    return(NO);
	else
	    return(YES);
}

/*
 * isServiceDisabled
 *
 * Given a service ID looks up its state setting in the database and 
 * determine if it is disabled.  If it is return YES, otherwise return
 * NO.  On failure return FAIL.
 */
int
isServiceDisabled(int svcID)
{
	char *disabled;

	switch (getSvcDisabled(svcID, &disabled))
	  {
	    case FAIL:
	      return(FAIL);

	    case NOT_FOUND:
	      return(NO);
	  }

	if (strcmp(disabled, YES_STR) == 0)
	    return(YES);

	return(NO);
}

/*
 * doesServiceRelocateOnPreferredNodeBoot
 *
 * Given a service ID return if the service "relocate's on preferred
 * node boot".
 *
 * Return values
 *	YES	- service does relocate on preferred node boot
 *	NO	- service does not ""          ""          ""
 *	FAIL	- failure
 */
int
doesServiceRelocateOnPreferredNodeBoot(int svcID)
{
	char *relocate;
	char *svcName;

	switch (getSvcRelocateOnPreferredNodeBoot(svcID, &relocate))
	  {
	    case FAIL:		return(FAIL);
	    case NOT_FOUND:	return(NO);
	  }

	
	getSvcName(svcID, &svcName);

	if (strcmp(relocate, YES_STR) == 0)
	  {
	    clulog(LOG_DEBUG,
"Service %s is set to relocate on preferred node boot\n", svcName);
	    return(YES);
	  }

	clulog(LOG_DEBUG, 
"Service %s is not set to relocate on preferred node boot\n", svcName);
	return(NO);
}

/*
 * unmountsForcefully
 *
 * Given a service ID find the forced unmount setting for mount point N. 
 * The Nth entry corresponds to the Nth mount point entry for a service.  
 * If the forced unmount setting is turned on YES is returned.  If is
 * not set, NO is returned.  If there is a failure, FAIL will be returned.
 * Given a service ID find the Nth device address entry.
 *
 * Return values
 *	YES		- forced unmount is turned on for this mount point
 *	NO		- forced unmount is not turned on ""    ""     ""
 *	FAIL		- failure
 */
int
unmountsForcefully(int svcID, int N)
{
	char *force;

	switch (getSvcForceUnmount(svcID, N, &force))
	  {
	    case FAIL:
	      return(FAIL);

	    case NOT_FOUND:
	      return(NO);
	  }

	if (strcmp(force, YES_STR) == 0)
	    return(YES);

	return(NO);
}

/*****************************************************************************
 *
 * The following might be better placed in another API library
 *
 *****************************************************************************/

/*
 * reReadDatabase
 *
 * Re-read the cluster configuration database.
 *
 * NOTE: this is a general database tool that could live in a different
 * library.
 */
int
reReadDatabase(char *database)
{
	if (database == NULL)
	  {
	    clulog(LOG_ERR, "Cannot re-read a NULL database\n");
	    return(FAIL);
	  }

	clulog(LOG_INFO, "Re-reading the cluster database\n");

	CFG_Destroy();

	return(SUCCESS);
}

/*
 * getSvcMgrLogLevel
 *
 * Get the log level for the Service Manager from the database.
 */
int
getSvcMgrLogLevel(int *logLevel)
{
	char token[MAX_TOKEN_LEN];
	char *value;
	int retVal;

	sprintf(token, "%s%c%s", 
	        SVCMGR_STR, CLU_CONFIG_SEPARATOR, SVCMGR_LOGLEVEL_STR);

	switch (retVal=CFG_Get(token, NULL, &value))
	  {
	  case CFG_DEFAULT:		// no log level defined, do not error
	    *logLevel=DEFAULT_LOGLEVEL;
	    break;
	  case CFG_OK:			// found it
	    *logLevel=atoi(value);
	    break;
	  default:			// Uh oh, we failed in CFG_Get()
	    clulog(LOG_ERR, 
"Cannot get log level for Service Manager from database; CFG_Get() failed, err=%d\n");
	    return(FAIL);
	  }     

	return(SUCCESS);
}

/*
 * getNodeName
 *
 * Given a node ID return its node name in nodeName.  If one is not found
 * return (char *)NULL.
 */
int
getNodeName(int nodeID, char **nodeName)
{
	char token[MAX_TOKEN_LEN];
	char *value;
	int retVal;
	static char nameBuf[MAXHOSTNAMELEN];

	sprintf(token, "%s%c%s%d%c%s", 
	        NODE_LIST_STR, CLU_CONFIG_SEPARATOR,
	        NODE_STR, nodeID, CLU_CONFIG_SEPARATOR,
	        NODE_NAME_STR);

	/*
	 * If we do not find the service name, return the reason why
	 * but send back a string of the service ID so we can get
	 * useful printf's to the log.
	 */
	if ((retVal=getDatabaseToken(token, &value)) != SUCCESS)
	  {
	    clulog(LOG_WARNING, 
	           "Cannot get node name for node #%d, err=%d\n", 
	           nodeID, retVal);
	    sprintf(nameBuf, "%d", nodeID);
	    *nodeName=(char *)&nameBuf;
	  }

	switch (retVal)
	  {
	  case SUCCESS:			// found it
	    strcpy(nameBuf, value);
	    *nodeName = (char*)&nameBuf;
	    break;
	  case NOT_FOUND:		// no node name defined
	  case FAIL:			// error
	    *nodeName=(char *)NULL;
	    return(retVal);
	  }     

	return(SUCCESS);
}

/*
 * getNodeID
 *
 * Given a nodeName, return the corresponding node ID from the cluster
 * database.
 */
int
getNodeID(char *nodeName)
{
	char token[MAX_TOKEN_LEN];
	int nodeID;
	int retVal;
	char *value;

	for (nodeID=MIN_NODE; nodeID < MAX_NODES; nodeID++)
	  {
	    sprintf(token, "%s%c%s%d%c%s", 
	            NODE_LIST_STR, CLU_CONFIG_SEPARATOR,  
	            NODE_STR, nodeID, CLU_CONFIG_SEPARATOR,
	            NODE_NAME_STR);
	    switch (retVal=CFG_Get(token, NULL, &value))
	      {
	        case CFG_OK:		// Found it
	          break;

	        case CFG_DEFAULT:	// Not found, try next
	          continue;

	        default:
	          clulog(LOG_ERR,
"Cannot get node ID for %s from database; CFG_Get() failed, err=%d.\n", 
	              nodeName, retVal);
	          return(FAIL);
	      }

	    if (strcmp(value,nodeName) == 0)
	      {
	        return(nodeID);
	      }
	  }

	clulog(LOG_ERR,
"Cannot get node ID for %s from database; not found\n", nodeName);
	return(FAIL);
}


/* clean_string
 *
 * clean space in the string then return cleaned string back in the same address 
 * flag==0, clean spaces at head and tails
 * flag==1, clean all the space in the string pointed by the pointer p
 */

void clean_string(char *p, int flag)
{
       int l,i;       
       char temp[MAX_LINE_LEN];
       int start;

       l=strlen(p);
       if(l<=0)
          return;
       start=0;

       if(flag==0)
       {   
	 if(p[l-1]==' ')
	 {
	    for(i=l-1;i>=0;i--)
	    { 
              if(p[i]==' ')
	         p[i]='\0';
              else break;
                
            }
            l=strlen(p);
         }
         for(i=0;i<l;i++)
          { 
	     if(p[i]!=' ')
	        break;
          }
          strcpy(temp,&p[i]);
          strcpy(p,temp); // a clean string is saved back to p
       }
       else if(flag==1)
       {
          for(i=0;i<l;i++)
          { 
	     if(p[i]!=' ')
	     {
	        temp[start]=p[i];
                start++;
             }
          }
          temp[start]='\0';
          strcpy(p,temp); // a clean string is saved back to p
       } 
}

/* The following are a collection of routines that are used to compact
   the database after some portion of a service has been removed.

   No error codes are returned, or error messages spewn anywhere,
   because (provided that the database was ok before we begin) nothing
   can possibly go wrong.  The entry points do return an integer to
   allow for the later addition of return codes when the impossible is
   found eventually to occur.

   Author: Ron Lawrence <lawrence@missioncriticallinux.com> 
*/

#include <math.h>
#define COMPACTOR_BUFFER_SIZE 1024

/* Compact one set of elements in the database.  The prefix gives the
   part of the entries to be compacted that is shared, i.e., the part
   before the actual index that is to be altered. 

   The following code works by inserting new entries in the database
   under new index that correspond to each of the entries under old
   index.  Then all of the entries under old index are deleted.  This
   should work fine, provided that we always work from the bottom up,
   filling entries by moving entries downward.  Thus, the sorting of
   the indices is extremely important.  */
static void compact1 (int oldindex, 
                      int newindex, 
                      char* prefix) {
  struct CFG_Iter *iter;
  char buffer[COMPACTOR_BUFFER_SIZE];
  char newbuffer[COMPACTOR_BUFFER_SIZE];
  char *k, *v;

  if(oldindex == newindex) {
    /* Nothing needs to be moved. */
    return;
  }
  /* Create a glob iterator that will iterate over all of the entries
     with respect to old index. */
  snprintf(buffer,COMPACTOR_BUFFER_SIZE,
           "%s%d%%*", prefix, oldindex);
  CFG_CreateGlobIterator(buffer, &iter);

  /* For each item in the glob iterator: */
  while(CFG_IteratorMore(iter)) {
    /* Create a corresponding entry under the new index. */
    CFG_IteratorNext(iter,&k,&v);

    /* Each found element has as a prefix exactly the prefix in
       buffer, without the star. So... */
    if(k) {
      int len;
      /* Set newbuffer to be the prefix, but without the star. */
      snprintf(newbuffer,COMPACTOR_BUFFER_SIZE,
               "%s%d%%", prefix, newindex);
      /* Append the part of k that is different. */
      len = strlen(newbuffer);
      strncat(newbuffer, 
              k + len, strlen(k) - len);
      
      /* Altering the database additively is ok, but keys can't be
         deleted until we are done with the iterator. */
      CFG_Set(newbuffer, v);
    }
  }
  /* Remove match on the cfg database for entries corresponding to the
     oldindex. */
  CFG_RemoveMatch(buffer);
}

static int cmp_indices(const void *i1, const void *i2) {
  return (*(int*)i1 - *(int*)i2);
}

/* Iterate over indices, which is treated as a list of database
   element id's, moving the associated entries in such a way that
   after the relocations the indices of the resulting elements are
   consecutive, starting at zero. */
static void compact_indices(int *indices, 
                            char *common_prefix, 
                            int count) {
  int i;
  /* Sort the indices. */
  qsort(indices, count, sizeof(int), cmp_indices);

  /* Call compact1 on each index in indices. */
  for(i=0; i<count; i++) {
    compact1(indices[i],i,common_prefix);
  }
}

static regex_t match_regex;
static int match_element(char *key, char *value __attribute__((unused))) {
  int rc;
  regmatch_t match[3];
  rc = regexec(&match_regex, key, 3, match, 0);
  return (REG_NOMATCH != rc);
}

/* Count the elements that share the common prefix, and that differ
   from it only by a trailing number. */
static void count_elements(int *count, 
                           char *common_prefix,
                           char *postfix_pattern) {
  struct CFG_Iter *iter;
  char *pattern_buffer = NULL;
  int buffer_size = 0;

  /* Make a regexp pattern */
  buffer_size = strlen(common_prefix) + strlen(postfix_pattern) + 1;
  pattern_buffer = alloca(buffer_size * sizeof(char));
  snprintf(pattern_buffer, buffer_size, "%s%s", common_prefix, postfix_pattern);
  pattern_buffer[buffer_size] = '\0';

  /* Compile it */
  regcomp(&match_regex, pattern_buffer, REG_EXTENDED);

  /* Create a filter iterator on the regexp pattern */
  CFG_CreateFilterIterator(match_element, &iter);

  CFG_IteratorCount(iter, count);
  
  regfree(&match_regex);
  CFG_DestroyIterator(iter);
  return;
}

/* Fill the passed array, list, with the id's of count elements that
   share a common prefix, and that differ from it only by a number in
   the postfix_pattern. */
static void list_elements(int *list, 
                          int count, 
                          char *common_prefix, 
                          char *postfix_pattern) {
  struct CFG_Iter *iter;
  char *pattern_buffer = NULL;
  int buffer_size = 0;
  char *k, *v;
  int i=0;
  int rc;
  regmatch_t match[3];

  /* Make a regexp pattern */
  buffer_size = strlen(common_prefix) + strlen(postfix_pattern) + 1;
  pattern_buffer = alloca(buffer_size * sizeof(char));
  snprintf(pattern_buffer, buffer_size, "%s%s", common_prefix, postfix_pattern);
  pattern_buffer[buffer_size] = '\0';
  regcomp(&match_regex, pattern_buffer, REG_EXTENDED);

  /* Create a filter iterator on the regexp pattern */
  CFG_CreateFilterIterator(match_element, &iter);

  while(CFG_IteratorMore(iter) && i < count) {
    CFG_IteratorNext(iter, &k, &v);

    /* Run the match again, so we can get the substring containing the
       interesting digits.  FIXME This wastes some small amount of
       time; revisit as time permits. */
    rc = regexec(&match_regex, k, 3, match, 0);
    if(REG_NOMATCH != rc)
      /* If we matched, put the integer that corresponds to the found
         digits into the list */
      sscanf(k + match[1].rm_so, "%d", &(list[i++]));
  }
  regfree(&match_regex);
  CFG_DestroyIterator(iter);
  return;
}

/* Compact the elements of the service which has the given
   serviceId.

   It would be much nicer if the strings used in the following were
   taken from the getToken routines that appear above.  The strings
   need to be hacked upon, though, so that is laborious.  Should be
   done, though.  Left as an enhancement.  FIXME */
int compact_service(int serviceId) {
  int count;
  int *lst;
  char buffer[COMPACTOR_BUFFER_SIZE];
  
  /* Make an ip address string. */
  snprintf(buffer, COMPACTOR_BUFFER_SIZE, 
           "services%%service%d%%network", 
           serviceId);

  /* get a list of the ip addresses. */
  count_elements(&count, buffer, "([0-9]+)%ipAddress");
  lst = alloca(count * sizeof(int));
  list_elements(lst, count, buffer, "([0-9]+)%ipAddress");

  /* compact the ip addresses */
  compact_indices(lst, buffer, count); 

  /* Make a device string */
  snprintf(buffer, COMPACTOR_BUFFER_SIZE, 
           "services%%service%d%%device", 
           serviceId);

  /* get a list of the device elements */
  count_elements(&count, buffer, "([0-9]+)%name");
  lst = alloca(count * sizeof(int));
  list_elements(lst, count, buffer, "([0-9]+)%name");

  /* compact the devices */
  compact_indices(lst, buffer, count); 

  {
    int device_count = count;
    int i;
    /* loop over devices: */
    for(i=0; i<device_count; ++i) {
      /* make an export string */
      snprintf(buffer, COMPACTOR_BUFFER_SIZE, 
               "services%%service%d%%device%d%%mount%%NFSexports%%directory", 
               serviceId, i);

      /* get a list of exports */
      count_elements(&count, buffer, "([0-9]+)%name");
      if(count>0) {
        lst = alloca(count * sizeof(int));
        list_elements(lst, count, buffer, "([0-9]+)%name");    

        /* compact the exports string */
        compact_indices(lst, buffer, count); 
      }
    }
    {
      int export_count = count;
      int j;
      /* loop over devices, then exports */
      for(i=0; i< device_count; ++i) {
        for(j=0; j<export_count; ++j) {
          /* make a client string */
          snprintf(buffer, COMPACTOR_BUFFER_SIZE, 
                   "services%%service%d%%device%d%%mount%%NFSexports%%directory%d%%client",
                   serviceId, i, j);
          /* get a list of clients */
          count_elements(&count, buffer, "([0-9]+)%name");
          if(count>0) {
            lst = alloca(count * sizeof(int));
            list_elements(lst, count, buffer, "([0-9]+)%name");    

            /* compact the client string */
            compact_indices(lst, buffer, count); 
          }
        }
      }
    }
  }
  return 0;
}

/* Compact all of the services in the database.  After this operation,
   all of the service components, e.g., all of the devices, will have
   id's that are consecutive, starting at zero.  Returns zero on
   success, some non-zero value otherwise. */
int compact_all_services(void) {
  int svcID;
  /* Services can have gaps between them */
  for (svcID=MIN_SERVICE; svcID < MAX_SERVICES; svcID++)
  {
    if (serviceExists(svcID) != YES)
      continue;       /* not in database */
    compact_service(svcID);
  }
  return 0;
}

