#ifndef __PACKETRACKER_H__
#define __PACKETRACKER_H__

#include <stdio.h>
#include <time.h>
#include <list.h>
#include <map.h>
#include <vector.h>
#include <algo.h>
#include <string>

#include "config.h"
#include "gpsd.h"
#include "packet.h"

#define MAC_STR_LEN (MAC_LEN * 2) + 6

const string NOSSID("<no ssid>");

enum wireless_network_type {
    network_ap,
    network_adhoc,
    network_probe,
    network_data
};

enum address_type {
    address_none,
    address_udp,
    address_arp,
    address_dhcp
};
// A network
typedef struct {
    wireless_network_type type;

    string ssid;

    // Packet counts
    int llc_packets;
    int data_packets;
    int crypt_packets;
    int interesting_packets;

    // info extracted from packets
    //uint8_t bssid[MAC_LEN];
    int channel;
    int wep;
    string bssid;

    // Last time we saw a packet
    time_t last_time;
    // First packet
    time_t first_time;

    // Are we in the list?
    int listed;

    address_type atype;

    // How many address octets have we matched?
    int octets;

    // What IP range are we using?
    uint8_t range_ip[4];
    // What netmask are we using?
    uint8_t mask[4];
    // What's the gateway?
    uint8_t gate_ip[4];

    map<string, cdp_packet> cisco_equip;

    float gps_lat, gps_lon, gps_alt, gps_spd;
    int gps_mode;

    float first_lat, first_lon, first_alt, first_spd;
    int first_mode;

} wireless_network;

// Wireless network comparator
class WirelessFirstTimeLT {
public:
    bool operator() (const wireless_network *x, const wireless_network *y) const {
        if (x->first_time < y->first_time)
            return 1;
        return 0;
    }
};

class WirelessLastTimeLT {
public:
    bool operator() (const wireless_network *x, const wireless_network *y) const {
        if (x->last_time < y->last_time)
            return 1;
        return 0;
    }
};

class WirelessLastTimeGT {
public:
    bool operator() (const wireless_network *x, const wireless_network *y) const {
        if (x->last_time > y->last_time)
            return 1;
        return 0;
    }
};

class WirelessTimeCombo {
public:
    bool operator() (const wireless_network *x, const wireless_network *y) const {
        if ((x->first_time < y->first_time) && (x->last_time < y->last_time))
            return 1;

        return 0;
    }
};


class Packetracker {
public:
    Packetracker();
//    ~Packetracker();

    // Get the error
    char *FetchError() { return errstr; };

    // Set up the gps
    void AddGPS(GPSD *in_gps);

    // Packet tracker stuff
    int ProcessPacket(packet_info info, char *in_status);

    // Get the most recently touched n-th networks
    vector<wireless_network *> FetchNthRecent(unsigned int n);

    int WriteNetworks(FILE *in_file);
    int WriteCisco(FILE *in_file);

    // Convert the MAC
    static string Mac2String(uint8_t *mac, char seperator);

    int FetchNumNetworks() { return num_networks; };
    int FetchNumPackets() { return num_packets; };
    int FetchNumDropped() { return num_dropped; };
    int FetchNumNoise() { return num_noise; };
    int FetchNumCrypt() { return num_crypt; };
    int FetchNumInteresting() { return num_interesting; };
    int FetchNumCisco() { return num_cisco; };

    // Utility to find strings that are empty or contain all spaces
    static bool IsBlank(const char *s);

protected:
    char errstr[1024];
    GPSD *gps;

    int num_networks, num_packets, num_dropped, num_noise,
        num_crypt, num_interesting, num_cisco;

    // all the networks
    vector<wireless_network *> network_list;

    // Several maps to refer to networks
    map<string, wireless_network *> ssid_map;
    map<string, wireless_network *> bssid_map;
    map<string, wireless_network *> probe_map;
};

#endif

