//                       -*- mode: C++ -*-
//
// Copyright(C) 2007,2008 Stefan Siegl <stesie@brokenpipe.de>
//
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <kdebug.h>
#include <assert.h>

#include "silcbuddycliententry.h"
#include "silcchannelcontact.h"
#include "silcbuddycontact.h"
#include "silcaccount.h"

SilcBuddyClientEntry::SilcBuddyClientEntry(SilcBuddyContact *bc, bool temp)
  : _active(-1), _buddy(bc), _isTemporary(temp)
{
}

SilcBuddyClientEntry::~SilcBuddyClientEntry()
{
}

SilcTK::SilcClientEntry
SilcBuddyClientEntry::operator() (void) const
{
  if(! size())
    return NULL;

  int i = active();

  if (active() < 0 || active() >= size())
    i = chooseActive(); // we cannot mark one CE active (we are const)

  else if (operator[](i)->mode & (SILC_UMODE_GONE
				  | SILC_UMODE_BUSY
				  | SILC_UMODE_INDISPOSED
				  | SILC_UMODE_DETACHED))
    // temporarily (try to) choose a better client entry
    // i.e. one that isn't detached, gone or something else
    i = chooseActive();

  kDebug() << "SilcBuddyClientEntry: returning active ce " << i << endl;
  assert(i >= 0 && i < size());
  return operator[] (i);
}


SilcTK::SilcClientEntry
SilcBuddyClientEntry::operator() (void)
{
  if(! size())
    return NULL;

  if(active() < 0 || active() >= size())
    setActive(chooseActive());

  else if (operator[](active ())->mode & (SILC_UMODE_GONE
					  | SILC_UMODE_BUSY
					  | SILC_UMODE_INDISPOSED
					  | SILC_UMODE_DETACHED))
    // temporarily (try to) choose a better client entry
    // i.e. one that isn't detached, gone or something else
    return operator[] (chooseActive());

  return operator[] (active());
}


void
SilcBuddyClientEntry::replace(SilcTK::SilcClientEntry oldce,
			      SilcTK::SilcClientEntry newce)
{
  int i = indexOf(oldce);

  if(i < 0) {
    kDebug() << "old SilcClientEntry not found in "
	     << "SilcBuddyClientEntry::replace" << endl;

    append(newce); // is this right?
    return;        // complain more?
  }

  newce = SilcTK::silc_client_ref_client(buddy()->account()->client(),
					 buddy()->account()->conn(), newce);
  SilcTK::silc_client_unref_client(buddy()->account()->client(),
                                   buddy()->account()->conn(), oldce);

  QList<SilcTK::SilcClientEntry>::replace(i, newce);
}


void
SilcBuddyClientEntry::setActive(int i)
{
  if(i < 0 || i >= size())
    return; // invalid argument
 
  kDebug() << "SilcBuddyClientEntry::setActive ce " << i
	   << " of " << buddy()->nickName() << endl;

  _active = i;

  QString newNick = QString::fromUtf8(at(i)->nickname);

  if(! _isTemporary && buddy()->nickName() != newNick)
    buddy()->setNickName(newNick);
}


void
SilcBuddyClientEntry::setActive(SilcTK::SilcClientEntry ce)
{
  int i = indexOf(ce);
  setActive(i);
}

int
SilcBuddyClientEntry::chooseActive(void) const
{
  for(int i = 0; i < size(); i ++) {
    if(operator[](i)->mode & (SILC_UMODE_GONE
			      | SILC_UMODE_BUSY
			      | SILC_UMODE_INDISPOSED
			      | SILC_UMODE_DETACHED))
      continue; // prefer CEs that are just online or marked hyper active ..
    return i;
  }

  return size() ? 0 : -1;
}

SilcBuddyClientEntry
SilcBuddyClientEntry::joinedTo(SilcChannelContact *ch)
{
  SilcBuddyClientEntry newlist = SilcBuddyClientEntry(buddy(), true);

  SilcTK::SilcChannelEntry che = ch->channelEntry();
  if(! che) return newlist; // empty set

  foreach(SilcTK::SilcClientEntry ce, *this) {
    SilcTK::SilcChannelUser cu = SilcTK::silc_client_on_channel(che, ce);
    if(cu)
      newlist.append(ce);
  }

  kDebug() << "joinedTo for " << ch->nickName() << " found "
	   << newlist.size() << " suitable entries" << endl;
  return newlist;
}

SilcTK::SilcClientEntry
SilcBuddyClientEntry::operator() (SilcChannelContact *ch)
{
  SilcBuddyClientEntry joined = joinedTo(ch);
  return joined();
}


void
SilcBuddyClientEntry::append(SilcTK::SilcClientEntry ce)
{
  assert(ce);

  ce = SilcTK::silc_client_ref_client(buddy()->account()->client(),
				      buddy()->account()->conn(), ce);

  QList<SilcTK::SilcClientEntry>::append(ce);
  ce->context = buddy();
}


int
SilcBuddyClientEntry::removeAll(SilcTK::SilcClientEntry ce)
{
  bool removedActive = false;
  int removedEntries = 0;

  for(int i = 0; i < size();) {
    if(at(i) == ce) {
      if(i == active())
	removedActive = true;
      removeAt(i);
      removedEntries ++;

      SilcTK::silc_client_unref_client(buddy()->account()->client(),
				       buddy()->account()->conn(), ce);
    }
    else
      i ++;
  }

  if(removedEntries > 1)
    kDebug() << "SilcBuddyClientEntry::removeAll found same CE twice!" << endl;

  if(removedActive && size())
    setActive(chooseActive());

  return removedEntries;
}


void
SilcBuddyClientEntry::clear(void)
{
  foreach(SilcTK::SilcClientEntry ce, *this) {
    ce->context = NULL;

    SilcTK::silc_client_unref_client(buddy()->account()->client(),
				     buddy()->account()->conn(), ce);
  }

  QList<SilcTK::SilcClientEntry>::clear();
}

//#include "silcbuddycliententry.moc"
