//              -*- mode: C++ -*-
// Copyright(C) 2005,2007 Stefan Siegl <stesie@brokenpipe.de>
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <iostream>

#include <klocale.h>
#include <kmessagebox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <kconfigbase.h>
#include <knuminput.h>
#include <ktextedit.h>
#include <kdebug.h>

#include "silcprotocol.h"
#include "silceditaccountwidget.h"
#include "silcaccount.h"

#include "ui_silceditaccount.h" 

SilcEditAccountWidget::SilcEditAccountWidget(SilcProtocol *,
					     SilcAccount *account,
					     QWidget *parent)
  : QWidget(parent), KopeteEditAccountWidget(account),
    serverKeysToDelete()
{
  _dlg = new Ui::SilcEditAccount();
  _dlg->setupUi(this);

  if(account) {
    // editing an already existing SilcAccount, therefore show it's current
    // settings in our SilcEditAccountWidget
    _dlg->NickName->setText(account->myself()->nickName());
    _dlg->UserName->setText(account->userName());
    _dlg->RealName->setText(account->realName());
    _dlg->RemoteHost->setText(account->hostName());
    _dlg->ExcludeConnect->setChecked(account->excludeConnect());

    // set message sign flags ...
    _dlg->chkSignChannelMessages->setChecked(account->signChannelMessages());
    _dlg->chkSignChannelActions->setChecked(account->signChannelActions());
    _dlg->chkSignChannelNotices->setChecked(account->signChannelNotices());
    _dlg->chkSignPrivateMessages->setChecked(account->signPrivateMessages());
    _dlg->chkSignPrivateActions->setChecked(account->signPrivateActions());
    _dlg->chkSignPrivateNotices->setChecked(account->signPrivateNotices());
    _dlg->chkDisplayImagesInline->setChecked(account->displayImagesInline());

    // set File Transfer settings ...
    _dlg->chkUseSilcMime->setChecked(account->useSilcMime());
    _dlg->chkFtAutoAccept->setChecked(account->ftAutoAccept());
    _dlg->bindSelection->setCurrentIndex(account->getFtBind());

    // set Default Messages settings ...
    _dlg->QuitMessage->setText(account->quitMessage());

    // list available server fingerprints ...
    QVariant r = account->configGroup()->readEntry("serverkey_hostlist");
    QStringList list = r.toString().split(QChar::fromAscii(','));

    for(QStringList::Iterator it = list.begin(); it != list.end(); it ++) {
      const int i = (* it).lastIndexOf(QChar::fromAscii(':'));
      if(i < 0) {
	kDebug() << "invalid hostlist entry " << (* it) << endl;
	continue;
      }

      const QString port = (* it).mid(i + 1);
      const int j = (* it).lastIndexOf(QChar::fromAscii('_'), i);
      if(j < 0) {
	kDebug() << "invalid hostlist entry " << (* it) << endl;
	continue;
      }

      const QString ip = (* it).mid(j + 1, i - j - 1);
      const QString host = (* it).left(j);
      const QString hostent = QString("serverkey_%2").arg(* it);
      const QString finger = account->configGroup()->readEntry
	(hostent.toUtf8().constData());

      QStringList row;
      row << host << ip << port << finger;

      QTreeWidgetItem *item = new QTreeWidgetItem(row);
      if(! item) {
	kDebug() << "unable to allocate QListViewItem" << endl;
	continue;
      }

      _dlg->listServerFingerprints->addTopLevelItem(item);
    }

    // set moods
    setAttributeMood((SilcTK::SilcAttributeMood) account->getAttributeMood());
    // set contact mediums
    setAttributeContact
      ((SilcTK::SilcAttributeContact) account->getAttributeContact());

    // set other attributes
    _dlg->attrTimezone->setChecked(account->getAttributeTimezone());
    _dlg->chkAttr->setChecked(account->getAttributesAllowed());
    _dlg->attrLanguage->setText(account->getAttributeLanguage());
    _dlg->attrStatus->setText(account->getAttributeMessage());

    _dlg->chkAttrGeo->setChecked(account->getGeoAllowed());
    _dlg->attrGeoLong->setValue(account->getGeoLongitude());
    _dlg->attrGeoLat->setValue(account->getGeoLatitude());
    _dlg->attrGeoAlt->setValue(account->getGeoAltitude());
  
    slotGeoAllowToggle(account->getGeoAllowed());
    slotAllowToggle(account->getAttributesAllowed());
  }

  QObject::connect((QObject *) _dlg->cmdRemove, SIGNAL(clicked()), 
		   this, SLOT(slotRemoveServerKey()));
  QObject::connect((QObject *) _dlg->chkAttrGeo, SIGNAL(toggled(bool)), 
		   this, SLOT(slotGeoAllowToggle(bool)));
  QObject::connect((QObject *) _dlg->chkAttr, SIGNAL(toggled(bool)), 
		   this, SLOT(slotAllowToggle(bool)));
}


SilcEditAccountWidget::~SilcEditAccountWidget()
{
  delete _dlg;
}


bool
SilcEditAccountWidget::validateData(void)
{
  if(_dlg->NickName->text().isEmpty())
    KMessageBox::sorry(this, i18n("<qt>You must enter a nickname.</qt>"));
  else if(_dlg->NickName->text().contains("@")
	  || _dlg->NickName->text().contains(" ")) {
    // @bug this is a hack, do it properly
    KMessageBox::sorry(this, 
    i18n("<qt>Please remove any @ or space from the nickname</qt>"));
  }
  else if(_dlg->RemoteHost->text().isEmpty())
    KMessageBox::sorry(this, i18n("<qt>You have to specify a "
				  "remote host.</qt>"));
  else
    return true; // all the stuff is valid

  return false;
}

Kopete::Account *
SilcEditAccountWidget::apply(void)
{
  
  const QString nickName = _dlg->NickName->text();
  const QString remoteHost = _dlg->RemoteHost->text();

  SilcAccount *a;

  if(! this->account()) {
    a = new SilcAccount(SilcProtocol::protocol(),
			QString("%1@%2").arg(nickName).arg(remoteHost));
    if(! a) return NULL;
    this->setAccount(a);
  }
  else {
    a = static_cast<SilcAccount *>(this->account());
  }

  if(! validateData())
    return a;

  a->setNickName(nickName);

  a->setRealName(_dlg->RealName->text());
  a->setUserName(_dlg->UserName->text());
  a->setHostName(remoteHost);
  a->setExcludeConnect(_dlg->ExcludeConnect->isChecked());

  a->setSignChannelMessages(_dlg->chkSignChannelMessages->isChecked());
  a->setSignChannelActions(_dlg->chkSignChannelActions->isChecked());
  a->setSignChannelNotices(_dlg->chkSignChannelNotices->isChecked());

  a->setSignPrivateMessages(_dlg->chkSignPrivateMessages->isChecked());
  a->setSignPrivateActions(_dlg->chkSignPrivateActions->isChecked());
  a->setSignPrivateNotices(_dlg->chkSignPrivateNotices->isChecked());

  a->setDisplayImagesInline(_dlg->chkDisplayImagesInline->isChecked());

  a->setUseSilcMime(_dlg->chkUseSilcMime->isChecked());
  a->setFtAutoAccept(_dlg->chkFtAutoAccept->isChecked());
  a->setFtBind(_dlg->bindSelection->currentIndex());

  a->setQuitMessage(_dlg->QuitMessage->text());
  a->setAttributeMood(getAttributeMood());
  a->setAttributeContact(getAttributeContact());

  // delete the keys, queued for deletion
  QVariant r = a->configGroup()->readEntry("serverkey_hostlist");
  QStringList list = r.toString().split(QChar::fromAscii(','));
  for(QStringList::Iterator it = serverKeysToDelete.begin(); 
      it != serverKeysToDelete.end(); it ++) {
    a->configGroup()->deleteEntry(QString("serverkey_%1").arg(* it));
    list.erase(it);
  }
  a->configGroup()->writeEntry("serverkey_hostlist", list);

  a->setGeoInformations(_dlg->chkAttrGeo->isChecked(),
			_dlg->attrGeoLong->value(),
                        _dlg->attrGeoLat->value(),
			_dlg->attrGeoAlt->value());
  a->setAttributeLanguage(_dlg->attrLanguage->text());
  a->setAttributeMessage(_dlg->attrStatus->toPlainText());
  a->setAttributesAllowed(_dlg->chkAttr->isChecked());
  a->setAttributeTimezone(_dlg->attrTimezone->isChecked());

  a->updateAttributes();

  return a;
}



void
SilcEditAccountWidget::slotRemoveServerKey(void)
{
  QList<QTreeWidgetItem *> sel = _dlg->listServerFingerprints->selectedItems();

  for(QList<QTreeWidgetItem *>::Iterator it = sel.begin();
      it != sel.end(); it ++) {
    QString hostent = 
      QString("%1_%2:%3").arg((* it)->text(0)).arg((* it)->text(1))
      .arg((* it)->text(2));

    serverKeysToDelete += hostent; // queue for real deletion
    delete (* it);                 // remove item from list
  }
}


void
SilcEditAccountWidget::slotGeoAllowToggle(bool value)
{
  _dlg->attrGeoLong->setEnabled(value);
  _dlg->attrGeoLat->setEnabled(value);
  _dlg->attrGeoAlt->setEnabled(value);
}

void 
SilcEditAccountWidget::slotAllowToggle(bool value)
{
  _dlg->attrTimezone->setEnabled(value);
  _dlg->attrLanguage->setEnabled(value);
  _dlg->attrMoodHappy->setEnabled(value);
  _dlg->attrMoodSad->setEnabled(value);
  _dlg->attrMoodAngry->setEnabled(value);
  _dlg->attrMoodJealous->setEnabled(value);
  _dlg->attrMoodAshamed->setEnabled(value);
  _dlg->attrMoodInvincible->setEnabled(value);
  _dlg->attrMoodInLove->setEnabled(value);
  _dlg->attrMoodSleepy->setEnabled(value);
  _dlg->attrMoodBored->setEnabled(value);
  _dlg->attrMoodExcited->setEnabled(value);
  _dlg->attrMoodAnxious->setEnabled(value);
  _dlg->attrContactEmail->setEnabled(value);
  _dlg->attrContactPhone->setEnabled(value);
  _dlg->attrContactPaging->setEnabled(value);
  _dlg->attrContactSMS->setEnabled(value);
  _dlg->attrContactMMS->setEnabled(value);
  _dlg->attrContactChat->setEnabled(value);
  _dlg->attrContactVideo->setEnabled(value);
  _dlg->attrStatus->setEnabled(value);  
  _dlg->chkAttrGeo->setEnabled(value);

  slotGeoAllowToggle(value && _dlg->chkAttrGeo->isChecked());
}

void
SilcEditAccountWidget::setAttributeMood(SilcTK::SilcAttributeMood mood)
{
  _dlg->attrMoodHappy->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_HAPPY);
  _dlg->attrMoodSad->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_SAD);
  _dlg->attrMoodAngry->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_ANGRY);
  _dlg->attrMoodJealous->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_JEALOUS);
  _dlg->attrMoodAshamed->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_ASHAMED);
  _dlg->attrMoodInvincible->setChecked
    (mood & SilcTK::SILC_ATTRIBUTE_MOOD_INVINCIBLE);
  _dlg->attrMoodInLove->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_INLOVE);
  _dlg->attrMoodSleepy->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_SLEEPY);
  _dlg->attrMoodBored->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_BORED);
  _dlg->attrMoodExcited->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_EXCITED);
  _dlg->attrMoodAnxious->setChecked(mood & SilcTK::SILC_ATTRIBUTE_MOOD_ANXIOUS);
}

SilcTK::SilcAttributeMood 
SilcEditAccountWidget::getAttributeMood(void)
{
  SilcTK::SilcUInt32 mood = 0;
  
  if(_dlg->attrMoodHappy->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_HAPPY;
  if(_dlg->attrMoodSad->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_SAD;
  if(_dlg->attrMoodAngry->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_ANGRY;
  if(_dlg->attrMoodJealous->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_JEALOUS;
  if(_dlg->attrMoodAshamed->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_ASHAMED;
  if(_dlg->attrMoodInvincible->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_INVINCIBLE;
  if(_dlg->attrMoodInLove->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_INLOVE;
  if(_dlg->attrMoodSleepy->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_SLEEPY;
  if(_dlg->attrMoodBored->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_BORED;
  if(_dlg->attrMoodExcited->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_EXCITED;
  if(_dlg->attrMoodAnxious->isChecked())
    mood |= SilcTK::SILC_ATTRIBUTE_MOOD_ANXIOUS;
  
  return (SilcTK::SilcAttributeMood) mood;  
}

void 
SilcEditAccountWidget::setAttributeContact(SilcTK::SilcAttributeContact contact)
{
  _dlg->attrContactEmail->setChecked
    (contact & SilcTK::SILC_ATTRIBUTE_CONTACT_EMAIL);
  _dlg->attrContactPhone->setChecked
    (contact & SilcTK::SILC_ATTRIBUTE_CONTACT_CALL);
  _dlg->attrContactPaging->setChecked
    (contact & SilcTK::SILC_ATTRIBUTE_CONTACT_PAGE);
  _dlg->attrContactSMS->setChecked(contact & SilcTK::SILC_ATTRIBUTE_CONTACT_SMS);
  _dlg->attrContactMMS->setChecked(contact & SilcTK::SILC_ATTRIBUTE_CONTACT_MMS);
  _dlg->attrContactChat->setChecked
    (contact & SilcTK::SILC_ATTRIBUTE_CONTACT_CHAT);
  _dlg->attrContactVideo->setChecked
    (contact & SilcTK::SILC_ATTRIBUTE_CONTACT_VIDEO);
}

SilcTK::SilcAttributeContact 
SilcEditAccountWidget::getAttributeContact(void)
{
  SilcTK::SilcUInt32 contact = 0;
  if(_dlg->attrContactEmail->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_EMAIL;
  if(_dlg->attrContactPhone->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_CALL;
  if(_dlg->attrContactPaging->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_PAGE;
  if(_dlg->attrContactSMS->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_SMS;
  if(_dlg->attrContactMMS->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_MMS;
  if(_dlg->attrContactChat->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_CHAT;
  if(_dlg->attrContactVideo->isChecked())
    contact |= SilcTK::SILC_ATTRIBUTE_CONTACT_VIDEO;

  return (SilcTK::SilcAttributeContact) contact;
}

#include "silceditaccountwidget.moc"
