#ifndef SERVICEMANAGER_H
#define SERVICEMANAGER_H

#include <kore/kore.h>
#include <kore/serviceprovider.h>
#ifdef WIN32
#include <map>
#else
#include <hash_map>
#endif
#include <set>

namespace kore
{
/**
 * class kore::ServiceManager - provides Kore Service management.
 * When creating a new ServiceProvider instance, you have to register the services
 * it provides to the ServiceManager, as well as unregister them right before
 * the instance will get destroyed.
 * The ServiceManager can be queried (registered...(...)) for the existing
 * registered Services and ServiceProviders.
 */
class KORE_API ServiceManager : public ServiceProvider
{
public:
    /**
     * Default constructor. Creates a new ServiceManager instance.
     */
    ServiceManager();
    /**
     * Destructor.
     */
    virtual ~ServiceManager();

    /**
     * Registers all the services provided by the given ServiceProvider.
     * @param provider - the ServiceProvider whos services to be registered.
     */
    virtual void registerProvider(ServiceProvider* provider);
    /**
     * Unregisters all the services provided by the given ServiceProvider.
     * @param provider - the ServiceProvider whos services to be unregistered.
     */
    virtual void unregisterProvider(ServiceProvider* provider);
    /**
     * Registers all the services provided by the given ServiceProviders.
     * @param providers - NULL-teminated array with ServiceProviders whos
     * services to be registered.
     */
    virtual void registerProviders(ServiceProvider** providers);
    /**
     * Unregisters ALL the Services registered to this ServiceManager.
     */
    virtual void unregisterProviders();
    /**
     * Registers the given Service to this ServiceManager.
     * @param service - the Service to be registered.
     */
    // TO DO: do not register services that are already registered
    virtual void registerService(const Service* service);
    /**
     * Unregisters the given Service from this ServiceManager.
     * @param service - the Service to be unregistered.
     */
    virtual void unregisterService(const Service* service);
    /**
     * Registers a list of Services to this ServiceManager.
     * @param srvs - NULL-terminated array of Services to be registered.
     */
    virtual void registerServices(const Service** srvs);
    /**
     * Unregisters ALL the Services registered to this ServiceManager.
     */
    virtual void unregisterServices();

    /**
     * Gets the list of Services registered to this ServiceManager.
     * @return - NULL-terminated array containing all the Services
     * registered to this ServiceManager. Please note that it's the user's
     * responsability to delete[] the returned array when it's no longer needed.
     */
    virtual const Service** registeredServices() const;
    /**
     * Gets the list of Services provided by the given ServiceProvider
     * AND registered to this ServiceManager.
     * @param provider - the ServiceProvider whos services to be returned.
     * @return - NULL-terminated array containing the Services.
     * Please note that it's the user's responsability to delete[]
     * the returned array when it's no longer needed.
     */
    virtual const Service** registeredServices(ServiceProvider* provider) const;
    /**
     * Gets the list of Services registered to this ServiceManager
     * AND matching the given Service.
     * @param service - the service whos name is used for matching.
     * @return - NULL-terminated array containing the Services.
     * Please note that it's the user's responsability to delete[]
     * the returned array when it's no longer needed.
     */
    // TODO: define Service name mathcing rules.
    virtual const Service** registeredServices(const Service* service) const;
    /**
     * Gets the list of Services registered to this ServiceManager
     * AND matching the given Service name.
     * @param service - the service name to be used for matching.
     * @return - NULL-terminated array containing the Services.
     * Please note that it's the user's responsability to delete[]
     * the returned array when it's no longer needed.
     */
    virtual const Service** registeredServices(const char* service) const;
    /**
     * Gets the first Service best matching the given Service.
     * @param service - the service whos name is used for matching.
     * @return - the first best matching Service.
     */
    // TODO: define Service name matching priorities.
    virtual const Service* registeredService(const Service* service) const;
    /**
     * Gets the first Service best matching the given Service name.
     * @param service - the service name to be used for matching.
     * @return - the first best matching Service.
     */
    virtual const Service* registeredService(const char* service) const;

    /**
     * Gets the list of ServiceProviders which have Services registered
     * to this ServiceManager.
     * @return - NULL-terminated array containing the ServicesProviders.
     * Please note that it's the user's responsability to delete[]
     * the returned array when it's no longer needed.
     */
    virtual ServiceProvider** registeredProviders() const;
    /**
     * Gets the list of ServiceProviders which have Services registered
     * to this ServiceManager AND matching the given Service.
     * @param service - the service whos name is used for matching.
     * @return - NULL-terminated array containing the ServicesProviders.
     * Please note that it's the user's responsability to delete[]
     * the returned array when it's no longer needed.
     */
    virtual ServiceProvider** registeredProviders(const Service* service) const;
    /**
     * Gets the list of ServiceProviders which have Services registered
     * to this ServiceManager AND matching the given Service name.
     * @param service - the service name to be used for matching.
     * @return - NULL-terminated array containing the ServicesProviders.
     * Please note that it's the user's responsability to delete[]
     * the returned array when it's no longer needed.
     */
    virtual ServiceProvider** registeredProviders(const char* service) const;
    /**
     * Gets the first ServiceProvider which has a Service best matching the
     * given Service.
     * @param service - the service whos name is used for matching.
     * @return - the first best matching ServiceProvider.
     */
    virtual ServiceProvider* registeredProvider(const Service* service) const;
    /**
     * Gets the first ServiceProvider which has a Service best matching the
     * given Service name.
     * @param service - the service name to be used for matching.
     * @return - the first best matching ServiceProvider.
     */
    virtual ServiceProvider* registeredProvider(const char* service) const;

protected:
private:
    // current SM version
    const Version* _smVersion;
    // Kernel API version required by SM
    const Version* _smAPIVersion;
    // SM info
    const Info* _smInfo;
    // basic SM service
    const Service* _smService;

#ifdef WIN32
    struct ltstr
    {
        bool operator()(const char* s1, const char* s2) const
        {
            return strcmp(s1,s2) == -1;
        }
    };
    // Windows STL doesn't have stl::hash_multimap
    typedef multimap<const char*,const Service*, ltstr> srv_hash_type;
#else
    struct eqstr
    {
        bool operator()(const char* s1, const char* s2) const
        {
            return strcmp(s1,s2) == 0;
        }
    };
    typedef hash_multimap<const char*,const Service*, hash<const char*>, eqstr> srv_hash_type;
#endif
    struct ltptr
    {
        bool operator()(ServiceProvider* sp1, ServiceProvider* sp2) const
        {
            return sp1 < sp2;
        }
    };
    typedef set<ServiceProvider* , ltptr> sp_set_type;

    // the list (multimap<key: service name, value: Service>) of Services
    // registered to this ServiceManager
    srv_hash_type _services;
};
};

#endif
