// KreateCD - CD recording software for the K desktop environment
//
// 2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "AudioFileOp.h"
#include "AudioFileInfo.h"

#include "ProcessInterface.h"

#include <stdlib.h>
#include <stdio.h>
#include <strings.h>
#include <unistd.h>
#include <fcntl.h>

#include <sys/stat.h>
#include <sys/ioctl.h>

#include <qtimer.h>
#include <qobject.h>

#include "AudioFileOp.moc"

AudioFileOp::AudioFileOp(AudioFileInfo *afil) {
  AudioFD=-1;
  ImageFD=-1;
  audioInfo=afil;
}

AudioFileOp::~AudioFileOp(void) {
  if (AudioFD!=-1) close(AudioFD);
}


int AudioFileOp::readChar(long int *mchar,int words) {
  int i;
  if (safeRead(ReadBuffer,1*words)!=1*words) return(0);
  signed temp;
  currentPos+=1*words;
  for (i=0;i<words;++i) {
    temp=ReadBuffer[i];
    if (temp>127) temp=-(256-temp);
    *(mchar+i)=temp;
  }
  return(1);
}

int AudioFileOp::readCharUS(long int *mchar,int words) {
  int i;
  if (safeRead(ReadBuffer,1*words)!=1*words) return(0);
  signed temp;
  currentPos+=1*words;
  for (i=0;i<words;++i) {
    temp=ReadBuffer[i]-128;
    *(mchar+i)=temp;
  }
  return(1);
}

int AudioFileOp::readLEShort(long int *mshort,int words) {
  int i;
  signed long temp;
  if (safeRead(ReadBuffer,2*words)!=2*words) {
    return(0);
  }
  currentPos+=2*words;
  for (i=0;i<words;++i) {
#ifdef WORDS_BIGENDIAN
    temp=ReadBuffer[2*i]+(ReadBuffer[2*i+1]<<8);
#else
    temp=*((short int *) (ReadBuffer+2*i));
#endif
    if (temp>32767) temp=-(65536-temp);
    *(mshort+i)=temp;
  }
  return(1);
}


int AudioFileOp::readBEShort(long int *mshort,int words) {
  int i;
  signed long temp;
  if (safeRead(ReadBuffer,2*words)!=2*words) {
    return(0);
  }
  currentPos+=2*words;
  for (i=0;i<words;++i) {
#ifndef WORDS_BIGENDIAN
    temp=ReadBuffer[2*i+1]+(ReadBuffer[2*i]<<8);
#else
    temp=*((short int *) (ReadBuffer+2*i));
#endif
    if (temp>32767) temp=-(65536-temp);
    *(mshort+i)=temp;
  }
  return(1);
}

int AudioFileOp::readShort(long int *mshort,int words) {
  if (audioInfo->endian()==AudioFileInfo::Endian_Big) {
    return(readBEShort(mshort,words));
  }
  return(readLEShort(mshort,words));
}


int AudioFileOp::readLELong(long int *mlong,int words) {
#ifdef WORDS_BIGENDIAN
  int i;
  signed long long temp;
  if (safeRead(ReadBuffer,4*words)!=4*words) return(0);
  currentPos+=4*words;
  for (i=0;i<words;++i) {
    temp=ReadBuffer[4*i]+(ReadBuffer[4*i+1]<<8)+(ReadBuffer[4*i+2]<<16)+
      (ReadBuffer[4*i+3]<<24);
    if (temp>2147483648LL) temp=temp-4294967296LL;
    *(mlong+i)=temp;
  }
  return(1);
#else
  if (safeRead(mlong,4*words)!=4*words) return(0);
  currentPos+=4*words;
  return(1);
#endif
}

int AudioFileOp::readBELong(long int *mlong,int words) {
#ifndef WORDS_BIGENDIAN
  int i;
  signed long long temp;
  if (safeRead(ReadBuffer,4*words)!=4*words) return(0);
  currentPos+=4*words;
  for (i=0;i<words;++i) {
    temp=ReadBuffer[4*i+3]+(ReadBuffer[4*i+2]<<8)+
      (ReadBuffer[4*i+1]<<16)+(ReadBuffer[4*i]<<24);
    if (temp>2147483648LL) temp=temp-4294967296LL;
    *(mlong+i)=temp;
  }
  return(1);
#else
  if (safeRead(mlong,4*words)!=4*words) return(0);
  currentPos+=4*words;
  return(1);
#endif
}

int AudioFileOp::readLong(long int *mlong,int words) {
  if (audioInfo->endian()==AudioFileInfo::Endian_Big) {
    return(readBELong(mlong,words));
  }
  return(readLELong(mlong,words));
}

int AudioFileOp::readLEShortUS(long int *mshort,int words) {
  int i;
  signed long temp;
  if (safeRead(ReadBuffer,2*words)!=2*words) return(0);
  currentPos+=2*words;
  for (i=0;i<words;++i) {
#ifdef WORDS_BIGENDIAN
    temp=ReadBuffer[2*i]+(ReadBuffer[2*i+1]<<8);
#else
    temp=*((short int *) (ReadBuffer+2*i));
#endif
    temp-=32768;
    *(mshort+i)=temp;
  }
  return(1);
}


int AudioFileOp::readBEShortUS(long int *mshort,int words) {
  int i;
  signed long temp;
  if (safeRead(ReadBuffer,2*words)!=2*words) return(0);
  currentPos+=2*words;
  for (i=0;i<words;++i) {
#ifndef WORDS_BIGENDIAN
    temp=ReadBuffer[2*i+1]+(ReadBuffer[2*i]<<8);
#else
    temp=*((short int *) (ReadBuffer+2*i));
#endif
    temp-=32768;
    *(mshort+i)=temp;
  }
  return(1);
}

int AudioFileOp::readShortUS(long int *mshort,int words) {
  if (audioInfo->endian()==AudioFileInfo::Endian_Big) {
    return(readBEShortUS(mshort,words));
  }
  return(readLEShortUS(mshort,words));
}


int AudioFileOp::readLELongUS(long int *mlong,int words) {
#ifdef WORDS_BIGENDIAN
  int i;
  signed long long temp;
  if (safeRead(ReadBuffer,4*words)!=4*words) return(0);
  currentPos+=4*words;
  for (i=0;i<words;++i) {
    temp=ReadBuffer[4*i]+(ReadBuffer[4*i+1]<<8)+(ReadBuffer[2*i+2]<<16)+
      (ReadBuffer[3]<<24);
    temp-=2147483648LL;
    *(mlong+i)=temp;
  }
  return(1);
#else
  int i; 
  if (safeRead(mlong,4*words)!=4*words) return(0);
  currentPos+=4*words;
  for (i=0;i<words;++i) {
    *(mlong+i)-=2147483648LL;
  }
  return(1);
#endif
}

int AudioFileOp::readBELongUS(long int *mlong,int words) {
#ifndef WORDS_BIGENDIAN
  int i;
  signed long long temp;
  if (safeRead(ReadBuffer,4*words)!=4*words) return(0);
  currentPos+=4*words;
  for (i=0;i<words;++i) {
    temp=ReadBuffer[4*i+3]+(ReadBuffer[4*i+2]<<8)+
      (ReadBuffer[4*i+1]<<16)+(ReadBuffer[4*i]<<24);
    temp-=2147483648LL;
    *(mlong+i)=temp;
  }
  return(1);
#else
  int i;
  if (safeRead(mlong,4*words)!=4*words) return(0);
  currentPos+=4*words;
  for (i=0;i<words;++i) {
    *(mlong+i)-=2147483648LL;
  }
  return(1);
#endif
}

int AudioFileOp::readLongUS(long int *mlong,int words) {
  if (audioInfo->endian()==AudioFileInfo::Endian_Big) {
    return(readBELongUS(mlong,words));
  }
  return(readLELongUS(mlong,words));
}

int AudioFileOp::readLE24(long int *mshort,int words) {
  int i;
  signed long temp;
  if (safeRead(ReadBuffer,3*words)!=3*words) return(0);
  currentPos+=3*words;
  for (i=0;i<words;++i) {
    temp=ReadBuffer[3*i]+(ReadBuffer[3*i+1]<<8)+(ReadBuffer[3*i+2]<<16);
    if (temp>8388607) temp=-(16777216-temp);
    *(mshort+i)=temp;
  }
  return(1);
}


int AudioFileOp::readBE24(long int *mshort,int words) {
  int i;
  signed long temp;
  if (safeRead(ReadBuffer,3*words)!=3*words) return(0);
  currentPos+=3*words;
  for (i=0;i<words;++i) {
    temp=ReadBuffer[3*i+2]+(ReadBuffer[3*i+1]<<8)+(ReadBuffer[3*i]<<16);
    if (temp>8388607) temp=-(16777216-temp);
    *(mshort+i)=temp;
  }
  return(1);
}

int AudioFileOp::read24(long int *mshort,int words) {
  if (audioInfo->endian()==AudioFileInfo::Endian_Big) {
    return(readBE24(mshort,words));
  }
  return(readLE24(mshort,words));
}

int AudioFileOp::readLE24US(long int *mshort,int words) {
  int i;
  signed long temp;
  if (safeRead(ReadBuffer,3*words)!=3*words) return(0);
  currentPos+=3*words;
  for (i=0;i<words;++i) {
    temp=ReadBuffer[3*i]+(ReadBuffer[3*i+1]<<8)+(ReadBuffer[3*i+2]<<16);
    temp-=8388608;
    *(mshort+i)=temp;
  }
  return(1);
}


int AudioFileOp::readBE24US(long int *mshort,int words) {
  int i;
  signed long temp;
  if (safeRead(ReadBuffer,3*words)!=3*words) return(0);
  currentPos+=3*words;
  for (i=0;i<words;++i) {
    temp=ReadBuffer[3*i+2]+(ReadBuffer[3*i+1]<<8)+(ReadBuffer[3*i]<<16);
    temp-=8388608;
    *(mshort+i)=temp;
  }
  return(1);
}

int AudioFileOp::read24US(long int *mshort,int words) {
  if (audioInfo->endian()==AudioFileInfo::Endian_Big) {
    return(readBE24US(mshort,words));
  }
  return(readLE24US(mshort,words));
}


int AudioFileOp::fillSampleBuffer(void) {
  switch (audioInfo->format()) {

  case AudioFileInfo::Samples_ULaw:
    {
      long int restsamples,readburst,i;
      restsamples=audioInfo->channels()*(audioInfo->dataSamples()-AudioPos);
      readburst=(restsamples<=(Read_Cachesize-  
			       (Read_Cachesize%audioInfo->channels())))?
        restsamples:(Read_Cachesize-(Read_Cachesize%audioInfo->channels()));
      if (readburst==0) return(0);
      if (!readCharUS(AudioBuffer,readburst))  return(0);
      AudioPointer=0;
      AudioPos+=readburst/audioInfo->channels();
      AudioCached=readburst;
      for (i=0;i<AudioCached;++i) {
	int t,u_val;
	u_val = ~*(AudioBuffer+i);
	t = ((u_val & 0x0f) << 3) + 0x84;
	t <<= (u_val & 0x70) >> 4;

	*(AudioBuffer+i)=((u_val & 0x80) ? (0x84 - t) : (t - 0x84));
      }
      return(1);
    }

  case AudioFileInfo::Samples_ALaw:
    {
      long int restsamples,readburst,i;
      restsamples=audioInfo->channels()*(audioInfo->dataSamples()-AudioPos);
      readburst=(restsamples<=(Read_Cachesize-  
			       (Read_Cachesize%audioInfo->channels())))?
        restsamples:(Read_Cachesize-(Read_Cachesize%audioInfo->channels()));
      if (readburst==0) return(0);
      if (!readCharUS(AudioBuffer,readburst))  return(0);
      AudioPointer=0;
      AudioPos+=readburst/audioInfo->channels();
      AudioCached=readburst;
      for (i=0;i<AudioCached;++i) {
        int t,seg,a_val;

	a_val = *(AudioBuffer+i)^0x55;
	t = a_val & 0x7f;
	if (t < 16) {
		t = (t << 4) + 8;
	} else {
	  seg = (t >> 4) & 0x07;
	  t = ((t & 0x0f) << 4) + 0x108;
	  t <<= seg -1;
	}
        *(AudioBuffer+i)=((a_val & 0x80) ? t : -t);

      }
      return(1);
    }

  case AudioFileInfo::Samples_PCM:
  case AudioFileInfo::Samples_PCM_Unsigned:
    {
      long int restsamples,readburst;
      bool sig=audioInfo->format()==AudioFileInfo::Samples_PCM;
      restsamples=audioInfo->channels()*(audioInfo->dataSamples()-AudioPos);
      readburst=(restsamples<=(Read_Cachesize-  
			       (Read_Cachesize%audioInfo->channels())))?
        restsamples:(Read_Cachesize-(Read_Cachesize%audioInfo->channels()));
      if (readburst==0) {
        return(0);
      }
      switch (audioInfo->size()) {
      case AudioFileInfo::Samples_8Bit: 
	{
          int i;
          if ( (sig) && (!readChar(AudioBuffer,readburst))) return(0);
          if ( (!sig) && (!readCharUS(AudioBuffer,readburst))) return(0);
          AudioPointer=0;
          AudioPos+=readburst/audioInfo->channels();
          AudioCached=readburst;
          for (i=0;i<readburst;++i) {
	    AudioBuffer[i]=AudioBuffer[i]<<8;
          }
          return(1);
        }
      case AudioFileInfo::Samples_16Bit: 
	{
          if ( (sig) && (!readShort(AudioBuffer,readburst))) {
            return(0);
          }
          if ( (!sig) && (!readShortUS(AudioBuffer,readburst))) return(0);
          AudioPointer=0;
          AudioPos+=readburst/audioInfo->channels();
          AudioCached=readburst;
          return(1);
        }
      case AudioFileInfo::Samples_24Bit: 
	{
          int i;
          if ( (sig) && (!read24(AudioBuffer,readburst))) return(0);
          if ( (!sig) && (!read24US(AudioBuffer,readburst))) return(0);
          AudioPointer=0;
          AudioPos+=readburst/audioInfo->channels();
          AudioCached=readburst;
          for (i=0;i<readburst;++i) {
	    AudioBuffer[i]=AudioBuffer[i]>>8;
          }
          return(1);
        }
      case AudioFileInfo::Samples_32Bit: 
	{
          int i;
          if ( (sig) && (!readLong(AudioBuffer,readburst))) return(0);
          if ( (!sig) && (!readLongUS(AudioBuffer,readburst))) return(0);
          AudioPointer=0;
          AudioPos+=readburst/audioInfo->channels();
          AudioCached=readburst;
          for (i=0;i<readburst;++i) {
	    AudioBuffer[i]=AudioBuffer[i]>>16;
          }
          return(1);
        }
      }
    }
  }
  return(0);
}

int AudioFileOp::flushCDDABuffer(void) {
  if (ImageContent==0) return(1);
  if (write(ImageFD,ImageBuffer,2*ImageContent)!=2*ImageContent) {
    return(0);
  }
  ImageContent=0;
  return(1);
}

int AudioFileOp::openWrite(const char *filename) {
  struct stat mystat;
  closeWrite();
  DisableSwap=0;
  ImageBuffer=(unsigned char *)malloc(Write_Cachesize*2);
  if (ImageBuffer==0) {
    return(0);
  }
  
  isWritePipe=false;
  if (stat(filename,&mystat)==0) {
    if (S_ISFIFO(mystat.st_mode)) {
      isWritePipe=true;
    }
  }

  if (!isWritePipe) {
    ImageFD=open(filename,O_WRONLY|O_CREAT|O_TRUNC,0644);
  } else {
    ImageFD=open(filename,O_WRONLY);
  }
  if (ImageFD==-1) {
    free(ImageBuffer);
    return(0);
  }
  ImageContent=0;
  return(1);
}

void AudioFileOp::closeWrite(void) {
  if (ImageFD!=-1) {
    close(ImageFD);
    ImageFD=-1;
  }
}

int AudioFileOp::openRead(const char *mfilename) {
  struct stat mystat;
  if (AudioFD!=-1) close(AudioFD);

  isReadPipe=false;
  currentPos=0;
  if (stat(mfilename,&mystat)==0) {
    if (S_ISFIFO(mystat.st_mode)) {
      isReadPipe=true;
      totalSize=0;
    } else {
      totalSize=mystat.st_size;
    }
  }

  AudioFD=open(mfilename,O_RDONLY);
  if (AudioFD==-1) return(0);
  AudioCached=0;
  AudioPos=0;
  AudioPointer=0;
  ReadBuffer=(unsigned char *) malloc(Read_Cachesize*4);
  if (ReadBuffer==0) {
    close(AudioFD);
    AudioFD=-1;
  }
  AudioBuffer=(long int *) malloc(Read_Cachesize*sizeof(long int));
  if (AudioBuffer==0) {
    close(AudioFD);
    AudioFD=-1;
    free(ReadBuffer);
  }
  return(1);
}

int AudioFileOp::getFileDescriptor(void) {
  return(AudioFD);
}

int AudioFileOp::getTargetDescriptor(void) {
  return(ImageFD);
}

void AudioFileOp::setSwapping(bool swapping) {
  DisableSwap=swapping?0:1;
}

void AudioFileOp::closeRead(void) {
  if (AudioFD!=-1) {
    close(AudioFD);
    free(ReadBuffer);
    free(AudioBuffer);
  }
  AudioFD=-1;
}

int AudioFileOp::skipRead(int bytes) {
  long soff;
  if (!isReadPipe) {
    if ((soff=lseek(AudioFD,bytes,SEEK_CUR))==-1) return(0);
    if (soff>totalSize) return(0);
  } else {
    if (seekPipe(bytes,SEEK_CUR)==-1) return(0);
  }
  currentPos+=bytes;
  return(1);
}

int AudioFileOp::backRead(int bytes) {
  if (!isReadPipe) {
    if (lseek(AudioFD,-bytes,SEEK_CUR)==-1) return(0);
  } else {
    return(0);  
  }
  currentPos-=bytes;
  return(1);
}


int AudioFileOp::positionSample(long int sample) {
  unsigned long position;
  long soff;
  if  ( (sample>=audioInfo->dataSamples()) || (sample<0) ) return(0);
  position=audioInfo->dataPos()+sample*audioInfo->sampleBytes();

  if (!isReadPipe) {
    if ((soff=lseek(AudioFD,position,SEEK_SET))==-1) return(0);
    if (soff>totalSize) return(0);
  } else {
    if (seekPipe(position,SEEK_SET)==-1) return(0);
   
  }
  AudioPointer=0;
  AudioCached=0;
  AudioPos=sample;
  currentPos=position;
  return(1);
}

int AudioFileOp::seekPipe(int offset,int whence) {
  int seek_bytes,seekread,rbytes;
  char seekBuffer[256];
  switch(whence) {
    case SEEK_SET:
      seek_bytes=offset-currentPos;
      break;
    case SEEK_CUR:
      seek_bytes=offset;
      break;
    default:
      return(-1);
  }
  if (seek_bytes<0) return(-1);
  if (seek_bytes==0) return(currentPos);
  while (seek_bytes>0) {
    seekread=seek_bytes>256?256:seek_bytes;
    rbytes=read(AudioFD,seekBuffer,seekread);
    if (rbytes==-1) {
      return(-1);
    }
    seek_bytes-=rbytes;
  }
  return(currentPos);
}

int AudioFileOp::safeRead(void *buffer,int bytes) {
  char *xbuf=(char *) buffer;
  int leftbytes=bytes;
  int readbytes=0;
  int rbytes;

  if (!isReadPipe) {
    return(read(AudioFD,buffer,bytes));
  }

  while (leftbytes>0) {
    rbytes=read(AudioFD,xbuf,leftbytes);
    if (rbytes==-1) return(-1);
    if (rbytes==0) break;
    xbuf+=rbytes;
    leftbytes-=rbytes;
    readbytes+=rbytes;
  }
  return(readbytes);
}

AudioFileInfo *AudioFileOp::getAudioInfo(void) {
  return(audioInfo);
}
