// KreateCD - CD recording software for the K desktop environment
//
// 2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#include "AudioScanImage.h"
#include "AudioFileInfo.h"
#include "AudioFileOp.h"
#include "AudioScanImage.moc"

#include <stdlib.h>
#include <stdio.h>
#include <strings.h>
#include <unistd.h>
#include <fcntl.h>

#include <sys/stat.h>
#include <sys/ioctl.h>

#include <qtimer.h>
#include <qobject.h>

#include <kapp.h>
#include <klocale.h>

AudioScanImage::AudioScanImage(AudioFileInfo *afil):
  ProcessInterface(scanImageHFork,0) {
  audioInfo=afil;
  fileOp=new AudioFileOp(afil);
}

AudioScanImage::~AudioScanImage(void) {
  if (fileOp!=0) delete fileOp;
}


int AudioScanImage::scanImage(void) {

  setForkArgument(audioInfo->getBurnFile());
  prepareProcess();

  reportAction(i18n("Scanning audio file...\n"));
  ImagizeFlag=0;

  startProcess();
  return(1);
}

int AudioScanImage::processCleanup(int retval) {

  closeProcess();
  return(retval);
}


int AudioScanImage::processExited(void) {
  return(ImagizeFlag);
}

bool AudioScanImage::processStdoutLine(char *linebuffer) {
  if (strncmp("#OK#",linebuffer,4)==0) {
    float mboo;
    char *xptr;
    xptr=linebuffer+4;
    sscanf(xptr,"%f",&mboo);
    audioInfo->setMaxBoost(mboo);
    ImagizeFlag=1;
    return(true);
  }
  if (strncmp("#ERROR#",linebuffer,7)==0) {
    ImagizeFlag=0;
    return(true);
  }

  if (strncmp("#PROGRESS#",linebuffer,10)==0) {
    char *xptr,*yptr;
    unsigned long int cur,max;
    xptr=linebuffer+10;
    while (*xptr==' ') ++xptr;
    if (*xptr==0) return(true);
    cur=strtoul(xptr,&yptr,10);
    xptr=yptr;
    while (*xptr==' ') ++xptr;
    if (*xptr==0) return(true);
    max=strtoul(xptr,&yptr,10);
    reportProgress(cur,max);
    return(true);
  }

  return(true);
}


int AudioScanImage::scanImageHFork(ProcessInterface *afi,const char *fn) {
  float rv;
  rv=((AudioScanImage*)(afi))->maximumBoost(fn);
  if (rv!=-1) {
    fprintf(stdout,"#OK# %f\n",rv);
  } else {
    fprintf(stdout,"#ERROR#\n");
  }
  return(0);
}

float AudioScanImage::maximumBoost(const char *fn) {
  float negboost,posboost,maxval=128;
  long int MaxSamples,Samples,maxneg=0,maxpos=0,sample;
  if (!fileOp->openRead(fn)) return(-1);
  if (fileOp->positionSample(audioInfo->startPos())==-1) {
    fileOp->closeRead();
    return(-1);
  }
  Samples=audioInfo->endPos()-audioInfo->startPos()+1;

  Samples*=audioInfo->channels();
  MaxSamples=Samples;

  while (Samples) {
    if (!fileOp->readSample(&sample)) return(-1);
    if (sample>0) {
      if (sample>maxpos) maxpos=sample;
    } else {
      if (sample<maxneg) maxneg=sample;
    }
    if ((Samples%1000)==0) {
      fprintf(stdout,"#PROGRESS# %ld %ld\n",(MaxSamples-Samples)/1000,
	      (MaxSamples+999)/1000);
    }
    --Samples;
  }
  fileOp->closeRead();
  switch (audioInfo->size()) {
  case AudioFileInfo::Samples_8Bit:
    maxval=128;
    break;
  case AudioFileInfo::Samples_16Bit:
    maxval=32768;
    break;
  case AudioFileInfo::Samples_32Bit:
    maxval=1<<31;
    break;
  }
  negboost=maxval/(-maxneg);
  posboost=(maxval-1)/(maxpos);
  if (negboost<posboost) return(negboost);
  return(posboost);
}

