// KreateCD - CD recording software for the K desktop environment
//
// 1999-2001 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#ifndef CDTRACK_INCLUDED
#define CDTRACK_INCLUDED

class AudioFileInfo;
class ISOImage;
class KConfig;
class ProgressDialog;
class AudioRipper;
class DataRipper;
class QObject;
class PipeCopy;
class AudioBuildImage;
class AudioConversion;

/**
    @short CDTrack - internal handling of a CD track
    @author Alexander Feigl <Alexander.Feigl@gmx.de>

    The CDTrack class is the internal storage class of KreateCD that handles a single track.
    It stores all neccessary information which is needed for a single track of a project.
    It can contain pointers to additional data classes (e.g. ISOImage, AudioFileInfo), it provides
    methods to update the track, create image files etc.

*/


class CDTrack
 {
  public:


  /**
      <pre>
      Track type description enums
      Track_Audio         : Audio track with no pre-emphasis
      Track_Audio_PreEmph : Audio track with pre-emphasis
      Track_DataMode1     : Data Track Mode 1
      Track_DataMode2     : Data Track Mode 2
      Track_DataXA1       : Data Track Mode 1 XA
      Track_DataXA2       : Data Track Mode 2 XA
      Track_CDI           : Data Track CDI Mode
      </pre>
  */
    enum TrackType   {Track_Audio,Track_AudioPreEmph,Track_DataMode1,Track_DataMode2,Track_XA1,Track_XA2,Track_CDI};



  /**
      <pre>
      Track source enums
      Source_None         : No source - initial setting
      Source_CD           : Track has a source from CD
      Source_File         : Track has a source image file
      Source_Filesystem   : Track has a filesystem source (for ISO 9660 mastering)
      </pre>
  */
    enum TrackSource {Source_None,Source_CD,Source_File,Source_Filesystem};

  /**
      <pre>
      Status of the track image
      Image_None          : Track has no image / doesn't need one
      Image_Dirty         : Track image is dirty (outdated or net yet created)
      Image_Clean         : Track image is up to date
      </pre>
  */
    enum TrackImage  {Image_None,Image_Dirty,Image_Clean};

  /**
    This constructor provides an "empty" CDTrack object when a new track is created by
    the user or when KreateCD is about to fill in some information from other sources like
    the CDROM copy tool

  */
    CDTrack(void);

  /**
    This constructor provides an audio track from an AudioFileInfo structure.

    @param AudioFileInfo to use
  */

    CDTrack(AudioFileInfo *afi);


  /**
    This constructor provides a "cloned" CDTrack object from another CDTrack. 
    All settings are cloned but temporary file information is stripped. This is a
    DEEP copy where pointers are not copied but subobjects are duplicated.

    @param object to clone
  */

    CDTrack(const CDTrack &object);
 

  /**
    Constucts a CDTrack object from a KConfig entry. (used to load tracks from a project file).
    This contructor restores the information which is stored in a single KConfig group

    @param config pointer to a KConfig object which is already set to the correct group.
  */
    CDTrack(KConfig *config);           // load track function

  /**
    This method stores a CDTrack track into a specified KConfig object. This is used to
    save tracks in a project file. The information is stored in a single config group for
    each CD track.

    @param config pointer to a KConfig object which is already set to the correct group
  */
    void saveTrack(KConfig *config);    // save track function

  /**
    public destructor for a CDTrack object - it deletes any temporary data and image files
  */
    ~CDTrack(void);

  /**
    This method returns the type of the track.

    @return type of the track
  */
    enum TrackType    getTrackType(void);

  /**
    This method returns the track source.

    @return source of the track
  */
    enum TrackSource  getTrackSource(void);

  /**
    This method returns the current state of the track image - if a image is needed and
    if it is up to date.

    @return status of the track image
  */
    enum TrackImage   getTrackImageMode(void);

  /**
    This method returns a boolean whether this track uses an image file or not.
    This might be forced by the user.

    @return track uses an image file
  */
    bool              getUseImage(void);

  /**
    This method returns a boolean whether this track needs two drives in 
    on the fly mode. 

    @return track needs two drives
  */
    bool              needDualDrive(void);

  /**
    This method copies the description of the track to a supplied character array

    @param string which will be filled with track description
  */
    void              getTrackDescription(char *desc);

  /**
    This method fills a string with the HMSB representation of the track length

    @param string which will be filled with track length (in time)
    @param megabytes should be displayed too
  */
    void              getTrackDurationHMSB(char *duration,bool megabytes=false);

  /**
    This method fills a string with the type of a track ( in words )

    @param string which will be filled with track type
  */
    void              getTrackTypeText(char *type);

  /**
    This method returns a boolean whether this track is an audio track or not.

    @param track is an audio track
  */
    bool               isAudio(void);

  /**
    This method returns a pointer to the associated AudioFileInfo object for this track.
    This is only possible for audio tracks of course.

    @return pointer to AudioFile object or 0 if none available
  */
    class AudioFileInfo  *getAudioFile(void);

  /**
    This method returns a pointer to the associated ISOImage object for this track.
    This is only used when there is a ISO9660 mastering track

    @return pointer to ISOImage object or 0 if none available
  */
    class ISOImage   *getISOImage(void);

  /**
    This method returns the length of a CD track which will be recorded to CD-R. This
    also accounts the selection of an audio track.

    @return length of track in frames/blocks
  */
    long int          getRealDuration(void);

  /**
    This method fills a char array with the filename of the image file which is used to
    burn the CD-R. If the track doesn't need an image file - this method returns a pointer
    to the source file. In any case this file name can be directly feeded into cdrecord

    @return image file is available or net
  */
    bool               getBurnFile(char *burnfile);

  /** 
    This method fills a char array with the filename of the named pipe which is used to
    burn the CD-R in on the fly mode. 
    
    @return image file is available or not
  */
    bool               getOnTheFlyFile(char *burnfile);

  /** 
    This method checks if a track is setup for on the fly burning
    
    @return track set up on the fly burning 
  */
    bool               checkOnTheFly(void);

  /** 
    This method does a clean up of all on the fly stuff
  */
    void               cleanupOnTheFly(void);

  /**
    Static method to obtain a string for a certain track type. A supplied char array is
    filled with the track type name.

    @param type track type for which a string should be created
    @param typestr char array which will be filled with the name of the track type
  */
    static void           getTypeString(enum TrackType type,char *typestr);

  /**
    Static method to get a HMSB representation of a track length or any other length.
    The supplied character array will be filled with the character representation.

    @param durationstr char array which will be filled with the HMSB representation
    @param track length (in frames/blocks) which should be encoded
    @param isaudio - track is an audio track
    @param megabytes - megabytes should be displayed too
  */
    static void           getDurationHMSB(char *durationstr,long int duration,
                                          bool isaudio,bool megabytes=false);

  /**
    This method sets the track description of the CD track

    @param descr new track description
  */
    void setDescription(const char *descr);

  /**
    This method sets the track type of the CD track. The track source might be cleared
    automatically if it is necessary.

    @param ttype new track type
  */
    void setTrackType(enum TrackType ttype);

  /**
    This method will clear the track source and reset it to none. This will delete all
    created image files automatically.
  */
    void clearSource(void);

  /**
    This method sets the source of a track to a CD track. You have to supply the correct
    information to setup the track.

    @param track number of the track in the CD drive (source)
    @param tocsum TOC sum of the source CD
    @param duration duration (length) of the track
    @param lbastart block number where the track starts on the source CD (especially for multisession)
  */
    void setSourceTrack(int track,long int tocsum,unsigned long duration,unsigned long lbastart);

  /**
    This method sets the source of a track to a image file / audio file.

    @param filename filename of the image/audio file
  */
    void setSourceFile(const char *filename);

  /**
    This method sets the source of a track to a ISO 9660 filesystem - which will be mastered
    when the CD-R is created.

    @param image pointer to a ISOImage object for the ISO9660 filesystem
  */
    void setSourceFileSystem(class ISOImage *image=0);

  /**
    This method enables image file creation whenever possible. If not possible it remains in
    the same state.
  */
    void enableImage(void);

  /**
    This method disables image file creation whenever possible. If not possible it remains in
    the same state.
  */
    void disableImage(void);

   /**
    This method clears any created image on harddisk - if any. The image status will be
    restored to none or dirty.
   */
    void clearImage(void);

   /**
    This function saves the current image forever. KreateCD drops the image file
    and won't use or delete it any more 
   */
    void saveImage(void);   

   /**
    This is only a temporary functin, till I get saveImage, Validate, .... working the way I need it for
    the konqueror RMB-Menu extension
   */
    bool  exportISO(class QString filename);

   /**
    This method will ensure that the information in the CD track object is consistent.
    It will update track length from real file sized and selections - it will change
    the track source to none if the source file is no longer available and similar things.
    The track image is cleared when doing this.
   */
    void updateTrack(void);

   /**
    This method will ensure that a valid and up to date image file is available for the
    CD track - ready to burn. It will recreate it if neccessary.

    @param passed ProgressDialog if no own dialog should be used
    @return image file is available
   */
    bool  validateTrack(ProgressDialog *xprog=0,const char *targetfilename=0);

   /**
    This method prepares a track for being written in on the fly mode. It might set up the necessary
    processes to do this.

    @return preparation successful
   */
    bool  prepareOnTheFly(PipeCopy *pipecopy);


   /**
    This method starts everything for a track being written in on the fly mode. It might set up the necessary
    processes to do this.

    @return preparation successful
   */
    bool  startOnTheFly(QObject *trigger=0);


   /**
    This method returs whether a CD track needs an image or not.

    @return needs an image
    */
    bool  needImage(void);

   /**
    This method returs whether a CD track needs a temporary file or not

    @return needs a temporary file
    */
    bool  needFile(void);

   /**
    This method forces the track size when precalculating ISO 9660 size

    @param size size of  track in bytes
   */

   void setTrackSize(long int size);

   /**
     This method creates an unique file name for a track image. It fills a supplied
     character array. You have to provide the duration/track length for this process

     @param fn char array which will be filled with file name
     @param cdrimg image is intended to be a final CD-R image, not a ripping image
     @param duration track length for the image file (used as magic)
   */
    static void createName(char *fn,bool cdrimg,long int duration,bool wav=false);

  protected:

   /**
    This method updates the track length from a image file given. It does not any image
    file clearing or similar - just calculating the length in frames by dividing the file
    size. Needed by validateTrack

    @param fn filename of the image file
   */
    void updateDuration(const char *fn);


  private:

   /**
     This is an internal common constructor method
   */
    void initObject(void);


   /**
    internal track type
   */
    TrackType     TrackType;

   /**
    internal track source
   */
    TrackSource   TrackSource;

   /**
    internal track image status
   */
    TrackImage    TrackImage;

   /**
    cached boolean whether a image is needed
   */
    bool          NeedImage;

   /**
    The CD track has a temporary (ripping) image, whicht might be not the final one
    which will be recorded onto CD-R. This happens when you rip a audio track from
    CD. This track might still change (selecting, boosting)
   */
    bool          IsTemporary;

   /**
    CD Source : source track number
   */
    int           CDTrackNum;                     // CD image : track

   /**
    CD Source : source track CD TOC sum
   */
    long int      CDTocSum;                       // CD image : TOC sum of this CD

   /**
    track description
   */
    char          Description[40];

   /**
    source file name
   */
    char          SourceFile[2048];

   /**
    image file name
   */
    char          ImageFile[512];

   /**
    length of track in bytes
    */
    unsigned long TrackSize;

   /**
    length of track in blocks/frames
   */
    unsigned long TrackDuration;                  // Format : in 1/75 sec blocks

   /**
    CD Track source : LBA of source track
   */
    unsigned long TrackLBA;                       // LBA of Source track

   /**
    internal pointer to AudioFileInfo
   */
    AudioFileInfo      *ThisAudio;

   /**
    internal pointer to ISOImage
   */
    ISOImage       *ThisISO;

   /**
    internal pointer to the AudioRipper for on the fly
   */
    AudioRipper    *onTheFlyRipper;

   /**
    internal pointer to the DataRipper for on the fly
   */
    DataRipper    *onTheFlyDRipper;

   /**
    internal pointer to the AudioBuildImage for on the fly
   */
    AudioBuildImage *onTheFlyABuildImage;

   /**
    internal pointer to the AudioConverter for on the fly
   */
    AudioConversion *onTheFlyAudioConverter;

   /**
    target pipe for on the fly writing
   */
    char            onTheFlyTarget[512];
    
   /**
    pipe where the CD ripper is feeding data
   */
    char            onTheFlyRipPipe[512];

   /**
    pipe where the audio converter feeding data
   */
    char            onTheFlyAudioPipe[512];

   /**
    pipe where the audio decoder feeding data
   */
    char            onTheFlyDecoderPipe[512];
    
 };


#endif
