// KreateCD - CD recording software for the K desktop environment
//
// 2000 by Joseph Wenninger <jowenn@bigfoot.com>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#include "TrackListManager.h"
#include "MainWindow.h"
#include "TrackWindow.h"
#include "NewProjectDialog.h"
#include "ProjectClassicWidget.h"
#include "ProjectSingleDataWidget.h"
#include "ProjectCDCopyWidget.h"
#include "ProjectFromDirWidget.h"
#include "ProjectAdvancedWidget.h"
#include "StartupScreen.h"
#include "AudioFileFormat.h"

#include <qtimer.h>
#include <qfileinfo.h>

#include <kconfig.h>
#include <kapp.h>
#include <klocale.h>
#include <kurl.h>
#include <kurldrag.h>
#include <kmimetype.h>
#include <kmessagebox.h>
#include <kcmdlineargs.h>

#include "TrackListManager.moc"

const char *ProjectTypeNames[]={"Classic mode (full featured)",
				"Audio only CD","Data only CD (MC Style)","CD-Copy","Advanced Mode",0};

const char *ProjectImages[]={"kcd_classic.png","kcd_classic.png","kcd_mcstyle.png","kcd_cdcopy.png","kcd_advanced.png"};

TrackListManager::TrackListManager(CDTrack **PRTR,QWidget* _childParent):QObject(_childParent) {
  shutdown=false;
  _isOpening=false;
  editedTrack=-1;
  currentView=0;
  _TLMType=-1;
  view=0;
  childParent=_childParent;
  ProjectTracks=PRTR;
  startup=true;
  savedTrack=0;
  TrackEdit=0;
  
  initialPath="";

//  newProject();
/*  Nielx: but only if we aren't opening a commandline file!
         : or if we are restoring */
  if ( !( KCmdLineArgs::parsedArgs()->count() >0 || kapp->isRestored() ) )
    {
      if (!(KCmdLineArgs::parsedArgs()->getOption("fromdir").isEmpty()))
      {
	qDebug("from Dir encountered with pathname"+KCmdLineArgs::parsedArgs()->getOption("fromdir"));
        initialPath=KCmdLineArgs::parsedArgs()->getOption("fromdir");
	QTimer::singleShot(0,this,SLOT(newDirProject()));
      }
      else
      {
      	QTimer::singleShot(0,this,SLOT(newProjectSlot()));
      }
    }
}

TrackListManager::~TrackListManager() {
  shutdown=true;
  clearTracks();
}

void TrackListManager::setIsOpening(bool val) {
  _isOpening=val;
  emit isOpeningChanged(_isOpening);
}

bool TrackListManager::isOpening() {
  return _isOpening;
}

void TrackListManager::newDirProject() {
  newProject(TLMFromDir);
}

void TrackListManager::newProjectSlot() {
  newProject();
}

void TrackListManager::newProject(int Type) {
  int res;
  KConfig *config=kapp->config();
  config->setGroup("Project");
  QString tstring=config->readEntry("StartupType");
  if ( (Type==TLMNothing) && ((!startup) || (tstring.isEmpty()))) {
    if (startup) showStartupPage();
    NewProjectDialog *selDialog=new NewProjectDialog(0,"hallo");
    res=selDialog->exec();
    if (res!=-1) {
      setType(res);
      clearTracks();
    } else if (startup) qDebug("Should exit now");
  } else {
    if (Type!=TLMNothing)
      setType(Type);
    else
      setType(tstring.toInt());
    clearTracks();
  }
  initialPath="";
  startup=false;
}


void TrackListManager::clearTracks() {
  for (int i=0;i<MAX_TRACKS_PER_PROJECT;++i) {

    if (ProjectTracks[i]!=0) {
      delete(ProjectTracks[i]);
    }
    ProjectTracks[i]=0;
  }
  emit clearList();
  updateTrackSum();
}

int TrackListManager::getType() {
  return _TLMType;
}

void TrackListManager::setType(int TLMType) {
//  if (TLMType!=_TLMType) {if (currentView!=0) delete currentView; currentView=0;} else return;
  _TLMType=TLMType;
  switch (TLMType)
    {
      case TLMFromDir:
	if (currentView!=0)
		{
                if  ((dynamic_cast<ProjectFromDirWidget*> (currentView))==0)
                        {delete currentView;
                        qDebug("DELETING OLD VIEW");
                        currentView=new ProjectFromDirWidget(childParent,this);}
                } else currentView=new ProjectFromDirWidget(childParent,this);
	break;
      case TLMSingleData:
           if (currentView!=0)
		{
		if  ((dynamic_cast<ProjectSingleDataWidget*> (currentView))==0)
		  	{delete currentView;
			qDebug("DELETING OLD VIEW");
			currentView=new ProjectSingleDataWidget(childParent,this);}
		} else currentView=new ProjectSingleDataWidget(childParent,this);
           break;
      case TLMClassic:
      case TLMAudio:
           if (currentView!=0)
	        {
                  if ((dynamic_cast<ProjectClassicWidget*>(currentView))==0)
			{delete currentView;
                 	currentView=new ProjectClassicWidget(childParent,this);}
		} else currentView=new ProjectClassicWidget(childParent,this);
           break;
      case TLMCDCopy:
	  if (currentView!=0)
		{
		  if ((dynamic_cast<ProjectCDCopyWidget*>(currentView))==0)
			{delete currentView;
			 currentView=new ProjectCDCopyWidget(childParent,this);}
		} else  currentView=new ProjectCDCopyWidget(childParent,this);
        	connect(currentView,SIGNAL(UpdateSizeHint()),this,SIGNAL(updateSizeHint()));
	  break;
      case TLMAdvanced:
          if (currentView!=0)
		{
                  if ((dynamic_cast<ProjectAdvancedWidget*>(currentView))==0)
                        {delete currentView;
                         currentView=new ProjectAdvancedWidget(childParent,this);}
                } else  currentView=new ProjectAdvancedWidget(childParent,this);
	}  

  emit updateSizeHint();
}


void TrackListManager::moveByOne(int itemId,bool up) {
  if (itemId<0) return;
  CDTrack *ttrack;
  int from=itemId;
  int toId=up ? itemId-1: itemId+1;
  if ((toId<0) || (toId>MAX_TRACKS_PER_PROJECT)) return;
  if (ProjectTracks[toId]==0) return;

  ttrack=ProjectTracks[from];
  ProjectTracks[from]=ProjectTracks[toId];
  ProjectTracks[toId]=ttrack;
  emit needUpdate(from);
  emit needUpdate(toId);
  emit needCurrentChanged(toId);

  /*  updateTrackItem();
      CurrentTrack--;
      updateTrackItem();
      MWTracklist->setSelected(TrackItems[CurrentTrack],true);
      MWTracklist->ensureItemVisible(TrackItems[CurrentTrack]);
      checkWidgetEnable();*/

}

void TrackListManager::moveAfter(int /*itemId*/,int /*after*/) {
}

QString TrackListManager::getTrackDescription(int track) {
  char tmp[80];
  if (!trackExists(track)) return QString("");
  ProjectTracks[track]->getTrackDescription(tmp);
  return QString(tmp);
}

QString TrackListManager::getTrackDurationHMSB(int track) {
  char tmp[80];
  if (!trackExists(track)) return QString("");
  ProjectTracks[track]->getTrackDurationHMSB(tmp,true);
  return QString(tmp);
}

QString TrackListManager::getTrackTypeText(int track) {
  char tmp[80];
  if (!trackExists(track)) return QString("");
  ProjectTracks[track]->getTrackTypeText(tmp);
  return QString(i18n(tmp));
}

enum CDTrack::TrackType TrackListManager::getTrackType(int track) {
  if (!trackExists(track)) return CDTrack::Track_DataMode1;
  return ProjectTracks[track]->getTrackType();
}

enum CDTrack::TrackSource  TrackListManager::getTrackSource(int track) {
  if (!trackExists(track)) return CDTrack::Source_None;
  return ProjectTracks[track]->getTrackSource();
}

CDTrack *TrackListManager::getTrack(int track) {
	if (!trackExists(track)) return 0;
	return ProjectTracks[track];
}

bool TrackListManager::trackExists(int track) {
  return (((track >=0 ) && (track<=MAX_TRACKS_PER_PROJECT)) && (ProjectTracks[track]!=0));
}

void TrackListManager::moveItemAfter(int item,int newAfter) {
  CDTrack *track;
  int tmppos;
  if (!trackExists(item)) {qDebug("Item not found"); return;}
  if ((!trackExists(newAfter)) && (newAfter!=-1)) {
    qDebug("New pos not found");
    return;
  }
  qDebug(QString("Item %1 -> %2").arg(item).arg(newAfter));

  if (newAfter<item) tmppos=newAfter+1; else tmppos=newAfter;

  track=ProjectTracks[item];

  if (newAfter<item) {
    for (int i=item;i>tmppos;i--) {
      ProjectTracks[i]=ProjectTracks[i-1];
    }
  } else if (item<newAfter) {
    for (int i=item;i<tmppos;i++) {
      ProjectTracks[i]=ProjectTracks[i+1];}
  }

  ProjectTracks[tmppos]=track;

  emit needCurrentChanged(tmppos);

  if (tmppos>item) {
    newAfter=item; item=tmppos;tmppos=newAfter;
  }
  for (int i=tmppos;i<=item;i++) {
    emit needUpdate(i);
    qDebug(QString("Need update of:%1").arg(i));
  }

}


void TrackListManager::insert(QDropEvent *event,int pos) {
  KURL::List urls;
  QString mtype;
  int tmppos=pos;
  bool isAudio=false;
  if (KURLDrag::decode(event,urls) && urls.count() >0) {
    KURL::List::ConstIterator it(urls.begin());
    for(;it!=urls.end();  ++it) {
      if ((*it).isLocalFile()) {
	qDebug("is Local");
	qDebug((*it).path(0));
/*	KMimeType::Ptr typ=KMimeType::findByURL((*it).path(0),true);
	mtype=typ->name();
	isAudio=((mtype=="audio/x-wav") || (mtype=="audio/x-mp3")||(mtype=="audio/basic") || (mtype=="audio/x-basic"));
	qDebug(QString("Current TLMType is %1").arg(_TLMType));
	qDebug(QString("Mimetype is:")+mtype);
*/

       AudioFileInfo *afi=new AudioFileInfo();
       AudioFileFormat afifo(afi);
       if (!afifo.identifyFile((*it).path(0)))
       {
        delete(afi);
        afi=0;
	isAudio=false;
       } else isAudio=true;

	switch (_TLMType) {
	case TLMAudio:
	  if (!isAudio) {
//	    KMessageBox::sorry(0,i18n("In this mode you can only add files of type: <B>%1</B> or <B>%2</B> or <B>%3</B>").arg(KMimeType::mimeType("audio/x-wav")->comment()).arg(KMimeType::mimeType("audio/x-mp3")->comment()).arg(KMimeType::mimeType("audio/basic")->comment()));
	    KMessageBox::sorry(0,i18n("In this mode you can only add <B>audio</B> files!"));
	    continue;
	  }
	  break;

	case TLMClassic:
	case TLMAdvanced:
	  if (QFileInfo((*it).path(0)).isDir()) {
	    KMessageBox::sorry(0,i18n("In this mode you can only add <B>files</B>, no directories.<BR>The files will be interpreted as <B>ISO-images</B> or audiodata"));
	    continue;
	  }
	  break;
	}
	if (isAudio)
	{
		insertTrack(tmppos,afi);
//		delete afi;
//		afi=0;
	}
	else
	{
		insertTrack(tmppos,(*it).path(0),/*isAudio ? CDTrack::Track_Audio : */ CDTrack::Track_DataMode1);
	}
	tmppos++;
	//addRealObject(new QString((*it).path(0)));
      } else {
	KMessageBox::sorry(0,"You can't add the non local object: <B>"+(*it).prettyURL()+"</B>","Not supported yet");
      }
    }
  }
}



void TrackListManager::insertTrack(int pos,AudioFileInfo *afi) {
  int trackCnt=0;
  for (trackCnt=0; (trackCnt<MAX_TRACKS_PER_PROJECT) &&
	 (ProjectTracks[trackCnt]!=0);trackCnt++);
  if (trackCnt>=MAX_TRACKS_PER_PROJECT) {
    KMessageBox::sorry(0,i18n("You can add no more tracks to this project")); return;
  }
  qDebug("Create Track");
  CDTrack *track=new CDTrack(afi);
  if (track==0) {
    qDebug("Severe Error: couldn't create new CDTRack");
  }
  track->setDescription(QString(afi->getSourceName()).right(38));
  track->setTrackType(CDTrack::Track_Audio);
  qDebug("Free Space");
  for (int i=trackCnt-1;i>=pos;i--) {
    qDebug(QString("Track to Move:%1").arg(i));
    ProjectTracks[i+1]=ProjectTracks[i];
  }
  qDebug("Insert Track");
  ProjectTracks[pos]=track;
  qDebug("Track added");
  emit trackAdded();
  qDebug("after emit trackAdded()");
  for (int i=pos;i<=trackCnt;i++) {
    qDebug(QString("in UpdateLoop: %1").arg(i));
    emit needUpdate(i);
  }
  updateTrackSum();
  qDebug("after all updates");
}





void TrackListManager::insertTrack(int pos,QString filename,enum CDTrack::TrackType trackType) {
  int trackCnt=0;
  for (trackCnt=0; (trackCnt<MAX_TRACKS_PER_PROJECT) &&
	 (ProjectTracks[trackCnt]!=0);trackCnt++);
  if (trackCnt>=MAX_TRACKS_PER_PROJECT) {
    KMessageBox::sorry(0,i18n("You can add no more tracks to this project")); return;
  }
  qDebug("Create Track");
  CDTrack *track=new CDTrack();
  if (track==0) {
    qDebug("Severe Error: couldn't create new CDTRack");
  }
  track->setTrackType(trackType);
  track->setSourceFile(filename);
  // qDebug(QString("Description length:%1").arg(strlen(filename.right(39))));
  track->setDescription(filename.right(38));
  //track->setDescription("DND-File");
  //(trackType==CDTrack::Track_Audio) ?  "AUDIOTRACK - DND" : "DATATRACK - DND");
  qDebug("Free Space");
  for (int i=trackCnt-1;i>=pos;i--) {
    qDebug(QString("Track to Move:%1").arg(i));
    ProjectTracks[i+1]=ProjectTracks[i];
  }
  qDebug("Insert Track");
  ProjectTracks[pos]=track;
  qDebug("Track added");
  emit trackAdded();
  qDebug("after emit trackAdded()");
  for (int i=pos;i<=trackCnt;i++) {
    qDebug(QString("in UpdateLoop: %1").arg(i));
    emit needUpdate(i);
  }
  updateTrackSum();
  qDebug("after all updates");
}


int TrackListManager::addTrack(CDTrack *track) {
  int freetrack=-1,i;
  if (track==0) return -1;

  for (i=0;i<MAX_TRACKS_PER_PROJECT;++i) {
    if (ProjectTracks[i]==0) {
      freetrack=i;
      break;
    }
  }
  if (freetrack==-1) return -1;
  ProjectTracks[freetrack]=track;
  emit trackAdded();
  emit needUpdate(freetrack);
  if ( (freetrack>=1) && (_TLMType==TLMSingleData) ) {
    deleteTrack(0);
    freetrack--;   
    ((ProjectSingleDataWidget*)(currentView))->updateTrack(track);
  }
  updateTrackSum();
  emit needCurrentChanged(freetrack);
  return freetrack;
}

void TrackListManager::addTrack() {
  int trackid;
  CDTrack *track;
  if (_TLMType==TLMSingleData) return;
  if ((trackid=addTrack(track=new CDTrack()))==-1) {
    KMessageBox::sorry(0,i18n("You can add no more tracks to this project")); return;
    delete track;
  } else {
    if (_TLMType==TLMAudio) track->setTrackType(CDTrack::Track_Audio);
    emit needUpdate(trackid);
    editTrackInternal(trackid,true);
  }
}

void TrackListManager::editTrack(int id) {
  editTrackInternal(id,false);
}

void TrackListManager::editTrackInternal(int id,bool newtrack) {

  if (_TLMType==TLMSingleData) return;
  if (!trackExists(id)) return;

  TrackEdit=new TrackWindow(0,"track window",ProjectTracks[id],_TLMType);
  if (TrackEdit==0) {
    editedTrack=-1;
    return;
  }
  if (newtrack) {
    savedTrack=0;
  } else {
    savedTrack=new CDTrack(*ProjectTracks[id]);
  }
  editedTrack=id;
  connect(TrackEdit, SIGNAL(canceled()), this, SLOT(notifyCancelTWin()));
  connect(TrackEdit,SIGNAL(updateTrack()),this,SLOT(updateTrackItem()));
  connect(TrackEdit,SIGNAL(updateTrack()),this,SLOT(updateTrackSum()));
  updateTrackSum();
  emit editWindowStateChanged(); 
  TrackEdit->exec();
  delete TrackEdit;
  TrackEdit=0;
  if (savedTrack!=0) {
    delete savedTrack;
    savedTrack=0;
  }
  emit editWindowStateChanged();
}

void TrackListManager::updateTrackItem() {
  if (editedTrack==-1) return;
  emit needUpdate(editedTrack);
  // if (TrackEdit!=0) TrackEdit->updateWindowContents(); leads to cyclic signal here 
  updateTrackSum();
}

void TrackListManager::updateTrack(int trackId) {
  if (trackId==-1) return;
  //  MWTracklist->ensureItemVisible(TrackItems[CurrentTrack]);
  ProjectTracks[trackId]->updateTrack();
  if (!ProjectTracks[trackId]->validateTrack()) {
    KMessageBox::error(0,i18n("Cannot create Image for track %1: <B>%2</B>").arg(trackId+1).arg(getTrackDescription(trackId)));
  };
  emit needUpdate(trackId);
  updateTrackSum();
}


void TrackListManager::deleteTrack(int id) {
  int i;

  if (_TLMType==TLMSingleData) {
    for (i=0;i<(MAX_TRACKS_PER_PROJECT-1);++i) {
      if (ProjectTracks[i]==0) break;
    }
    if ( (i<=1) || (id!=0) ) return;
  }
  if (!trackExists(id)) return;

  delete(ProjectTracks[id]);
  for (i=id;i<(MAX_TRACKS_PER_PROJECT-1);++i) {
    ProjectTracks[i]=ProjectTracks[i+1];
  }
  ProjectTracks[MAX_TRACKS_PER_PROJECT-1]=0;
  emit trackDeleted();
  updateTrackSum();
  //  for (int i=0;i<MAX_TRACKS_PER_PROJECT;++i)
  //   {
  //  if (ProjectTracks[i]!=0)
  //{
  //  emit needUpdate(i);
  //  qDebug(QString("Need update after delete:%1").arg(i));
  //}
  //}
  while ((id<MAX_TRACKS_PER_PROJECT) && (ProjectTracks[id]!=0)) {
    emit needUpdate(id);
    id++;
  }
}

void TrackListManager::updateTrackSum() {
  bool isaudio=false;
  int cnt=0;
  long dura=0;
  for (int i=0;i<MAX_TRACKS_PER_PROJECT;++i) {
    if (ProjectTracks[i]==0) continue;
    cnt++;
    if (ProjectTracks[i]->isAudio()) {
      isaudio=true;
    }
    dura+=ProjectTracks[i]->getRealDuration();
    dura+=2*75;
  }
  char formated[40];
  CDTrack::getDurationHMSB(formated,dura,isaudio,true);
  emit trackSumUpdated(cnt,QString(formated));
}

void TrackListManager::notifyCancelTWin() {
  // if new, delete
  if (savedTrack==0) {
    deleteTrack(editedTrack);
    savedTrack=0;
  } else {
    delete(ProjectTracks[editedTrack]);
    ProjectTracks[editedTrack]=savedTrack;
    savedTrack=0;
  }
  emit editWindowStateChanged();
  emit needCurrentChanged(editedTrack);
  emit needUpdate(editedTrack);
  TrackEdit->hide();
}

void TrackListManager::showStartupPage()
{
	currentView=new StartupScreen(childParent);
}


bool TrackListManager::openedInTrackWindow(CDTrack *track)
{
	if (!TrackEdit) return false;
	return TrackEdit->isOpenedTrack(track);
}
