// KreateCD - CD recording software for the K desktop environment
//
// 2001 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General      
// Public License.  See the file COPYING in the main directory of the       
// KreateCD distribution for more details.                                     

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "MP3Identify.h"
#include "MP3Converter.h"

#include <stdio.h>

static int const bitrates[][16]={
  {0,32,64,96,128,160,192,224,256,288,320,352,384,416,448,-1},
  {0,32,48,56,64,80,96,112,128,160,192,224,256,320,384,-1},
  {0,32,40,48,56,64,80,96,112,128,160,192,224,256,320,-1},
  {0,32,48,56,64,80,96,112,128,144,160,176,192,224,256,-1},
  {0,8,16,24,32,40,48,56,64,80,96,112,128,144,160,-1}
};

static int const samplerates[][4]={
  {44100,48000,32000,-1},
  {22050,24000,16000,-1},
  {11025,12000,8000,-1}
};

int MP3Identify::identify(void) {
  long totalsize;
  long totalsamples=0;
  long initialsize=0;
  int needheaders=4;
  resetFile();

  while (1) {
    if (seekHeader(4096,true)==false) return(0);
    if (parseHeader()==false) {
      if (skipRead(1)==0) return(0);
    } else {
      break;
    }
  }
  while (1) {
    bool eoffile=false;
    totalsize=4;
    if (mpegCRC) totalsize+=2;
    totalsize+=mpegLength;
    if (needheaders>0) {
      --needheaders;
      initialsize+=totalsize;
    }
    if (skipRead(mpegLength)!=1) {
      if (needheaders>0) return(0);
      break;
    }
    totalsamples+=mpegSamples;
    if (parseHeader()==false) {
      if (needheaders>0) {
	needheaders=4;
	if (backRead(initialsize-1)!=1) return(0);
	while (1) {
	  if (seekHeader(4096)==false) return(0);
	  if (parseHeader()==false) {
	    if (skipRead(1)==0) return(0);
	  } else {
	    break;
	  }
	}
      } else {
	while (1) {
	  if (seekHeader()==false) {
	    eoffile=true;
	    break;
	  }
	  if (parseHeader()==false) {
	    if (skipRead(1)==0) {
	      eoffile=true;
	      break;
	    }
	  } else {
	    break;
	  }  
	}
	if (eoffile==true) break;
      }
    }
  }

  if (totalsamples%588!=0) {
    totalsamples+= 588-(totalsamples%588);
  }

  setSampleFormat(AudioIdentify::Samples_PCM);
  setSampleSize(AudioIdentify::Samples_16Bit);
  setSampleEndian(AudioIdentify::Endian_Big);
  setSampleChannels(mpegMode==3?1:2);
  setSampleRate(mpegSample);
  setDataPos(0);
  setDataSize(totalsamples*sampleChannels()*2);
  setName("MPEG audio file");
  setConverter(new MP3Converter());
  return(1);
} 
  

bool MP3Identify::seekHeader(int bytes,bool initial) {
  long hchar;
  long shifter=0x0000;
  while (bytes>1) {
    if (readChar(&hchar,1)==0) return (false);
    hchar&=0xff;
    bytes--;
    shifter=((shifter<<8)&0xffff00L)|hchar;

    if ( (shifter&0xffe0)==0xffe0) {
	if (backRead(2)==1) return(true);
	return(false);
    }
    if ( (initial) && ((shifter&0xffffff)==0x494433) ) {
      bytes+=96*1024;
    }
  }
  return(false);
}

bool MP3Identify::parseHeader(void) {
  long header;
  int t,idx;
  if (readBELong(&header,1)==0) {
    backRead(4);
    return(false);
  }
  if ( (header&0xffe00000)!=0xffe00000) {
    backRead(4);
    return(false);
  }
  t=header>>19&0x3;
  switch (t) {
  case 0:
    mpegVersion=25;
    break;
  case 2:
    mpegVersion=2;
    break;
  case 3:
    mpegVersion=1;
    break;
  default:
    backRead(4);
    return(false);
  }

  t=header>>17&0x03;
  switch (t) {
  case 1:
    mpegLayer=3;
    break;
  case 2:
    mpegLayer=2;
    break;
  case 3:
    mpegLayer=1;
    break;
  default:
    backRead(4);
    return(false);
  }
  mpegCRC=(header>>16&1)==0;

  if (mpegVersion==1) {
    idx=mpegLayer-1;
  } else if ( (mpegVersion==2) || (mpegVersion==25) ) {
    idx=mpegLayer==1?3:4;
  } else {
    backRead(4);
    return(false);
  }
  t=(header>>12)&15;
  mpegBitrate=bitrates[idx][t];
  if ( (mpegBitrate==-1) || (mpegBitrate==0) ) {
    backRead(4);
    return(false);
  }

  switch (mpegVersion) {
  case 1:
    idx=0;
    break;
  case 2:
    idx=1;
    break;
  case 25:
    idx=2;
    break;
  default:
    backRead(4);
    return(false);
  }

  t=(header>>10)&3;
  mpegSample=samplerates[idx][t];
  if (mpegSample==-1) return(false);

  mpegPadding=(header>>9&1)==1;
  mpegMode=(header>>6)&3;
  mpegExtension=(header>>4)&3;
  mpegCopyright=((header>>3)&1)==1;
  mpegOriginal=((header>>2)&1)==1;
  mpegEmphasis=header&2;

  if (mpegCRC) {
    if (readBEShort(&mpegCRCValue,1)!=1) {
      backRead(4);
      return(false);
    }
  }

  mpegSlotsize=mpegLayer==1?4:1;
  mpegSamples=mpegLayer==1?384:1152;

  if (mpegLayer==1) {
    mpegLength=(12000*mpegBitrate/mpegSample);
  } else {
    mpegLength=144000*mpegBitrate/mpegSample;
  }

  if (mpegPadding) mpegLength++;
  mpegLength*=mpegSlotsize;

  mpegLength-=4;
  if (mpegCRC) mpegLength-=2;

  return(true);
}
