//
//   File : kvi_dcc_manager.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_dcc_manager.cpp)
//   Last major modification : Sat May 15 1999 16:06:19 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#include "kvi_dcc_manager.h"

#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_options.h"
#include "kvi_dcc_chat.h"
#include "kvi_dcc_send.h"
#include "kvi_dcc_voice.h"
#include "kvi_locale.h"
#include "kvi_console.h"
#include "kvi_ircsocket.h"

#include "kvi_netutils.h"
#include "kvi_fileutils.h"
#include "kvi_userlist.h"

#include <qfileinfo.h>

/* FIXME:  "if(g_pOptions->m_bReplyErrormsgOnMalformedDccRequests)..." */

// NOTE:
//        uAddress in this file is in NETWORK BYTE ORDER.
//



KviDccChatMessageBox::KviDccChatMessageBox(KviFrame *parent,const char *nick,const char *username,const char *host,
	unsigned long uAddress,unsigned short uPort,bool bVoice)
:QMessageBox(bVoice ? _CHAR_2_QSTRING(__tr("DCC Voice Request")) : _CHAR_2_QSTRING(__tr("DCC Chat Request")),_CHAR_2_QSTRING(""),QMessageBox::NoIcon,
		QMessageBox::Yes|QMessageBox::Default,QMessageBox::No|QMessageBox::Escape,0,
		parent,0,false)
{
	m_nick = nick;
	m_username = username;
	m_host = host;
	m_uAddress = uAddress;
	m_uPort = uPort;

	struct in_addr addr;
	addr.s_addr = uAddress;
	KviStr humanReadableAddr;

	kvi_binaryIpToString(addr,humanReadableAddr);

	KviStr tmp(KviStr::Format,__tr(
		"%s [%s@%s]\n"\
		"requests a Direct Client Connection in %s mode.\n"\
		"The remote DCC target host is %s on port %u\n"\
		"Do you wish to accept?"
		),nick,username,host,bVoice ? "VOICE" : "CHAT" ,humanReadableAddr.ptr(),uPort);
	setText(_CHAR_2_QSTRING(tmp.ptr()));
}

KviDccChatMessageBox::~KviDccChatMessageBox()
{
}

void KviDccChatMessageBox::done(int result)
{
	QMessageBox::done(result);
	if(result == QMessageBox::Yes){
		emit dccRequestAccepted(m_nick.ptr(),m_username.ptr(),m_host.ptr(),m_uAddress,m_uPort);
	}
//#warning "The delete this in resume is unsafe"
//	delete this;
	g_pApp->destroyLater(this);
}


KviDccSendMessageBox::KviDccSendMessageBox(KviFrame *parent,KviDccSendRequestStruct *dccSend)
:QMessageBox(_CHAR_2_QSTRING(__tr("DCC Send Request")),_CHAR_2_QSTRING(""),QMessageBox::NoIcon,
		QMessageBox::Yes|QMessageBox::Default,QMessageBox::No|QMessageBox::Escape,0,
		parent,0,false)
{
	m_dccSend = dccSend;

	struct in_addr addr;
	addr.s_addr = dccSend->uAddress;
	KviStr humanReadableAddr;

	kvi_binaryIpToString(addr,humanReadableAddr);

	KviStr tmp(KviStr::Format,__tr(
		"%s [%s@%s]\n"\
		"wants to send you a file named\n"\
		"%s, size %u bytes.\n"\
		"The remote DCC target host is %s on port %u\n"\
		"Do you wish to accept?"
		),dccSend->nick.ptr(),dccSend->username.ptr(),dccSend->host.ptr(),
			dccSend->fileName.ptr(),dccSend->fileLength,humanReadableAddr.ptr(),dccSend->uPort);
	setText(_CHAR_2_QSTRING(tmp.ptr()));
}

KviDccSendMessageBox::~KviDccSendMessageBox()
{
	if(m_dccSend != 0)delete m_dccSend;
}

void KviDccSendMessageBox::done(int result)
{
	QMessageBox::done(result);
	if(result == QMessageBox::Yes){
		emit dccRequestAccepted(m_dccSend);
		m_dccSend = 0; //forget it...(do not delete)
	}
//	delete this;
	g_pApp->destroyLater(this);
}

//#warning "Multiple files won't work here...."

KviDccSendFileDialog::KviDccSendFileDialog(KviFrame * parent,KviDccSendRequestStruct * dccSend,bool bOutgoingDcc)
:QFileDialog(QString::null,QString::null,parent,0,false)
{
	m_dccSend = dccSend;
	if(bOutgoingDcc){
		setMode(QFileDialog::ExistingFile);
		KviStr tmp(KviStr::Format,__tr("Select File to Send to %s"),dccSend->nick.ptr());
		setCaption(tmp.ptr());
	} else {
		if(dccSend->filePath.right(1) != "/")dccSend->filePath.append('/');
		dccSend->filePath.append(dccSend->fileName);
		setMode(QFileDialog::AnyFile);
		setCaption(__tr("Save DCC Incoming File"));
		setSelection(_CHAR_2_QSTRING(dccSend->filePath.ptr()));
	}
}

KviDccSendFileDialog::~KviDccSendFileDialog()
{
	if(m_dccSend != 0)delete m_dccSend;
}

void KviDccSendFileDialog::done(int result)
{
	QFileDialog::done(result);
	if(result == Accepted){
		m_dccSend->filePath = selectedFile();
		if(m_dccSend->filePath.hasData()){
			emit dccSaveNameSelected(m_dccSend);
			m_dccSend = 0;
		}
	}
	g_pApp->destroyLater(this);
}

KviDccSendResumeMessageBox::KviDccSendResumeMessageBox(KviFrame *parent,KviDccSendRequestStruct *dccSend)
:QMessageBox(_CHAR_2_QSTRING(__tr("DCC Send Request")),_CHAR_2_QSTRING(""),QMessageBox::NoIcon,
		QMessageBox::Yes|QMessageBox::Escape,QMessageBox::No|QMessageBox::Default,
		QMessageBox::Cancel,
		parent,0,false)
{
	m_dccSend = dccSend;

	setButtonText(QMessageBox::Yes,__tr("&Rename"));
	setButtonText(QMessageBox::No,__tr("Re&sume"));
	setButtonText(QMessageBox::Cancel,__tr("&Overwrite"));

	KviStr tmp(KviStr::Format,__tr(
		"The file %s already exists,\n"\
		"and is smaller than the incoming file.\n"\
		"It could be from a previous failed DCC transfer.\n"\
		"Do you want to resume the previous operation,\n"\
		"overwrite the existing file with the incoming file,\n"\
		"or rename the incoming file?"
		),dccSend->filePath.ptr());
	setText(_CHAR_2_QSTRING(tmp.ptr()));
}

KviDccSendResumeMessageBox::~KviDccSendResumeMessageBox()
{
	if(m_dccSend != 0)delete m_dccSend;
}

void KviDccSendResumeMessageBox::done(int result)
{
	QMessageBox::done(result);
	if(result == QMessageBox::Yes){ //rename
		while(kvi_fileExists(m_dccSend->filePath.ptr()))m_dccSend->filePath.append(".rename");
		m_dccSend->resumeValue = 0;
	}
	if(result == QMessageBox::Cancel){ //overwrite
		m_dccSend->resumeValue = 0;
	}
	// otherwise resume..the file position is already set
	emit dccSendResumeSelectionDone(m_dccSend);
	m_dccSend = 0;
	g_pApp->destroyLater(this);
//
//	delete this;
}

//======================================================================================================
// Dcc send rename message box
// The file is already on disk , but has either a greater size than the offered one
// or it has size 0. Ask wheter the user wants to overwrite it or rename it.
//

KviDccSendRenameMessageBox::KviDccSendRenameMessageBox(KviFrame *parent,KviDccSendRequestStruct *dccSend)
:QMessageBox(_CHAR_2_QSTRING(__tr("DCC Send request")),_CHAR_2_QSTRING(""),QMessageBox::NoIcon,
		QMessageBox::Yes|QMessageBox::Default,QMessageBox::No|QMessageBox::Escape,0,parent,0,false)
{
	m_dccSend = dccSend;

	setButtonText(QMessageBox::Yes,__tr("&Rename"));
	setButtonText(QMessageBox::No,__tr("&Overwrite"));

	KviStr tmp(KviStr::Format,__tr(
		"The file %s already exists.\n"\
		"Do you want to overwrite the existing file\n"\
		"with the incoming one, or rename the incoming file?"
		),dccSend->filePath.ptr());
	setText(_CHAR_2_QSTRING(tmp.ptr()));
}

KviDccSendRenameMessageBox::~KviDccSendRenameMessageBox()
{
	if(m_dccSend != 0)delete m_dccSend;
}

void KviDccSendRenameMessageBox::done(int result)
{
	QMessageBox::done(result);
	if(result == QMessageBox::Yes){ //rename
		while(kvi_fileExists(m_dccSend->filePath.ptr()))m_dccSend->filePath.append(".rename");
	}
	emit dccSendRenameSelectionDone(m_dccSend);
	m_dccSend = 0;
	g_pApp->destroyLater(this);
//
//	delete this;
}

KviDccManager::KviDccManager(KviFrame * frame)
:QObject()
{
	m_pFrm = frame;
}

KviDccManager::~KviDccManager()
{
}


void KviDccManager::handleDccRequest(KviIrcUser &source,KviDccRequest * dcc)
{
	if(g_pOptions->m_bIgnoreDccRequests){
		m_pFrm->m_pConsole->output(KVI_OUT_DCCINFO,
			__tr("Ignoring DCC %s request from %s [%s@%s]: [%s]"),
			dcc->szType.ptr(),source.nick(),source.username(),
			source.host(),dcc->szOriginalRequest.ptr());
		return;
	}

	if(g_pOptions->m_bNotifyAllDccRequestsInConsole){
		m_pFrm->m_pConsole->output(KVI_OUT_DCCINFO,
				__tr("Processing DCC %s request from %s [%s@%s]: [%s]"),
				dcc->szType.ptr(),source.nick(),source.username(),source.host(),
				dcc->szOriginalRequest.ptr());
	}

	if(kvi_strEqualCI(dcc->szType.ptr(),"chat"))
	{
		if(!kvi_strEqualCI(dcc->szParam.ptr(),"chat"))
		{
			m_pFrm->m_pConsole->output(KVI_OUT_DCCWARNING,
				__tr("Invalid parameter '%s' for DCC chat request from %s [%s@%s]: Should be 'chat' : Continuing"),
				dcc->szParam.ptr(),source.nick(),source.username(),source.host());
		}
		handleDccChat(source,dcc->uAddress,dcc->uPort);
	}
	if(kvi_strEqualCI(dcc->szType.ptr(),"send")){
		bool bOk = false;
		unsigned int size = dcc->szLast.toUInt(&bOk);
		if(!bOk){
			m_pFrm->m_pConsole->output(KVI_OUT_DCCERROR,__tr("Invalid file size (%s) for DCC send request from %s [%s@%s] : Ignoring"),
				dcc->szLast.ptr(),source.nick(),source.username(),source.host());
			return;
		}
		handleDccSend(source,dcc->szParam,size,dcc->uAddress,dcc->uPort);
	}
	if(kvi_strEqualCI(dcc->szType.ptr(),"accept")){
		handleDccAccept(source,dcc->szParam,dcc->szAddress.toUShort(),dcc->szPort.toULong());
	}
	if(kvi_strEqualCI(dcc->szType.ptr(),"resume")){
		handleDccResume(source,dcc->szParam,dcc->szAddress.toUShort(),dcc->szPort.toULong());
	}
	if(kvi_strEqualCI(dcc->szType.ptr(),"voice")){
		bool bOk = false;
		unsigned int sampleRate = dcc->szLast.toUInt(&bOk);
		if(!bOk){
			m_pFrm->m_pConsole->output(KVI_OUT_DCCWARNING,
				__tr("No bitrate specified for DCC voice request from %s!%s@%s : Defaulting to 8 kHz"),
				source.nick(),source.username(),source.host());
			sampleRate = 8000;
		}
		if(sampleRate != 8000){
			m_pFrm->m_pConsole->output(KVI_OUT_DCCWARNING,
				__tr("Unsupported sample rate for DCC voice request from %s!%s@%s: %d Hz : Defaulting to 8 kHz"),
				source.nick(),source.username(),source.host(),sampleRate);
			sampleRate = 8000;
		}
		if(!kvi_strEqualCI("ADPCM",dcc->szParam.ptr())){
			m_pFrm->m_pConsole->output(KVI_OUT_DCCWARNING,
				__tr("Unsupported codec for DCC voice request from %s!%s@%s: %s : Defaulting to ADPCM"),
				source.nick(),source.username(),source.host(),dcc->szParam.ptr());
		}
		handleDccVoice(source,dcc->uAddress,dcc->uPort);
	}
//#warning "More DCC Types here..."
}

void KviDccManager::handleDccAccept(KviIrcUser &source,KviStr &filename,unsigned short uPort,unsigned long uResumePos)
{
	KviDccSend *s = m_pFrm->findWaitingDccSend(uPort,source.nick());
	if(!s){
		m_pFrm->m_pConsole->output(KVI_OUT_DCCERROR,__tr("Invalid DCC ACCEPT from %s [%s@%s]. Transfer not initiated for file %s on port %u : Ignoring"),
			source.nick(),source.username(),source.host(),filename.ptr(),uPort);
		return;
	}
	s->output(KVI_OUT_DCCINFO,__tr("RESUME accepted. Transfer will initiate from position %u"),uResumePos);
	s->initiateGet();
}

void KviDccManager::handleDccChat(KviIrcUser &source,unsigned long uAddress,unsigned short uPort)
{
	if(g_pOptions->m_bAutoAcceptDccChat){
		m_pFrm->m_pConsole->output(KVI_OUT_DCCINFO,__tr("Accepting DCC chat request from %s [%s@%s]: [%u,%u]"),
			source.nick(),source.username(),source.host(),uAddress,uPort);
		acceptDccChat(source.nick(),source.username(),source.host(),uAddress,uPort);
	} else {
		KviDccChatMessageBox * box = new KviDccChatMessageBox(m_pFrm,source.nick(),
												source.username(),source.host(),
												uAddress,uPort,false);
		connect(box,
			SIGNAL(dccRequestAccepted(const char *,const char *,const char *,unsigned long,unsigned short)),
			this,
			SLOT(acceptDccChat(const char *,const char *,const char *,unsigned long,unsigned short)));
		if(!box->isVisible())box->show();
	}
}

void KviDccManager::handleDccVoice(KviIrcUser &source,unsigned long uAddress,unsigned short uPort)
{
	if(g_pOptions->m_bAutoAcceptDccVoice){
		m_pFrm->m_pConsole->output(KVI_OUT_DCCINFO,__tr("Accepting DCC voice request from %s [%s@%s]: [%u,%u]"),
			source.nick(),source.username(),source.host(),uAddress,uPort);
		acceptDccVoice(source.nick(),source.username(),source.host(),uAddress,uPort);
	} else {
		KviDccChatMessageBox * box = new KviDccChatMessageBox(m_pFrm,source.nick(),
												source.username(),source.host(),
												uAddress,uPort,true);
		connect(box,
			SIGNAL(dccRequestAccepted(const char *,const char *,const char *,unsigned long,unsigned short)),
			this,
			SLOT(acceptDccVoice(const char *,const char *,const char *,unsigned long,unsigned short)));
		if(!box->isVisible())box->show();

	}
}

// Slot
void KviDccManager::acceptDccVoice(const char *nick,const char *username,const char *host,unsigned long uAddress,unsigned short uPort)
{
	KviStr tmp(KviStr::Format,"DCC-Voice-%s!%s@%s",nick,username,host);
	KviDccVoice * dcc = m_pFrm->createDccVoice(tmp.ptr());
	dcc->acceptDccRequest(nick,username,host,uAddress,uPort);
}

void KviDccManager::requestDccVoice(const char *nick)
{
	KviIrcUser * u = m_pFrm->m_pUserList->findUser(nick);
	KviStr username , host;
	if(!u){
		username = "*";
		host     = "*";
	} else {
		username = u->username();
		host     = u->host();
	}
	KviStr tmp(KviStr::Format,"DCC-Voice-%s!%s@%s",nick,username.ptr(),host.ptr());
	KviDccVoice * dcc = m_pFrm->createDccVoice(tmp.ptr());
	dcc->requestDcc(nick,username.ptr(),host.ptr());
}



void KviDccManager::acceptDccChat(const char *nick,const char *username,const char *host,unsigned long uAddress,unsigned short uPort)
{
	KviStr tmp(KviStr::Format,"DCC-Chat-%s!%s@%s",nick,username,host);
	KviDccChat * dcc = m_pFrm->createDccChat(tmp.ptr());
	dcc->acceptDccRequest(nick,username,host,uAddress,uPort);
}

KviDccChat * KviDccManager::requestDccChat(const char *nick,const char *userandhost)
{
	KviStr username , host;
	if(userandhost){
		KviStr tmp(KviStr::Format,"%s!%s",nick,userandhost);
		KviIrcUser user(tmp.ptr());
		username = user.username();
		host     = user.host();
	} else {
		KviIrcUser * u = m_pFrm->m_pUserList->findUser(nick);
		if(!u){
			username = "*";
			host     = "*";
		} else {
			username = u->username();
			host     = u->host();
		}
	}
	KviStr tmp(KviStr::Format,"DCC-Chat-%s!%s@%s",nick,username.ptr(),host.ptr());
	KviDccChat * dcc = m_pFrm->createDccChat(tmp.ptr());
	dcc->requestDcc(nick,username.ptr(),host.ptr());
	return dcc;
}



void KviDccManager::handleDccSend(KviIrcUser &source,KviStr &filename,unsigned long fileLen,unsigned long uAddress,unsigned short uPort)
{
	KviDccSendRequestStruct * dccSend = new KviDccSendRequestStruct;
	filename.stripWhiteSpace();
	dccSend->nick       = source.nick();
	dccSend->username   = source.username();
	dccSend->host       = source.host();
	dccSend->fileName   = filename.ptr();
	dccSend->originalFileName = filename.ptr();
	if(filename.contains(' ') != 0){
		dccSend->originalFileName.prepend("\"");
		dccSend->originalFileName.append("\"");
	}
	if(g_pOptions->m_bReplaceSpacesInDccSendFileNames)dccSend->fileName.replaceAll(' ',"_"); //replace all spaces with underscores
	dccSend->fileLength = fileLen;
	dccSend->uAddress   = uAddress;
	dccSend->uPort      = uPort;

	if(g_pOptions->m_bAutoAcceptDccSend){
		m_pFrm->m_pConsole->output(KVI_OUT_DCCINFO,__tr("Accepting DCC send request from %s [%s@%s] for file %s [%u bytes]: [%u,%u]"),
		source.nick(),source.username(),source.host(),filename.ptr(),fileLen,uAddress,uPort);

		acceptDccSend(dccSend);

	} else {
		KviDccSendMessageBox * box = new KviDccSendMessageBox(m_pFrm,dccSend);
		connect(box,SIGNAL(dccRequestAccepted(KviDccSendRequestStruct *)),
					this,SLOT(acceptDccSend(KviDccSendRequestStruct *)));
		if(!box->isVisible())box->show();
	}
}

void KviDccManager::acceptDccSend(KviDccSendRequestStruct * dccSend)
{
	int idx = dccSend->fileName.findLastIdx('/');
	if(idx != -1)dccSend->fileName.cutLeft(idx+1); //remove leading directories
	g_pApp->getDefaultDccSaveFilePath(dccSend->filePath,dccSend->fileName.ptr());

	if(g_pOptions->m_bAutoAcceptDccSend){
		// Auto select the file name
		if(dccSend->filePath.right(1) != "/")dccSend->filePath.append('/');
		dccSend->filePath+=dccSend->fileName.ptr();
		dccSendFileNameSelected(dccSend);
	} else {
		// show a dir dialog starting in filePath
		KviDccSendFileDialog * box = new KviDccSendFileDialog(m_pFrm,dccSend,false);
		connect(box,SIGNAL(dccSaveNameSelected(KviDccSendRequestStruct *)),
			this,SLOT(dccSendFileNameSelected(KviDccSendRequestStruct *)));
		if(!box->isVisible())box->show();

	}
}

void KviDccManager::dccSendFileNameSelected(KviDccSendRequestStruct *dccSend)
{
	// check if the file exists on disk
	if(kvi_fileExists(dccSend->filePath.ptr())){
		// the file exists on disk
		// check the file sizes
		QFileInfo fi(dccSend->filePath.ptr());

		if((fi.size() <= dccSend->fileLength) && (fi.size() > 0)){
			// The file on disk is smaller : resume , overwrite or rename ?
			if(g_pOptions->m_bAutoAcceptDccSend){
				// No need to ask , auto accepting.... resume or rename
				if(g_pOptions->m_bEnableResumeOnAutoAccept)dccSend->resumeValue = fi.size();
				else {
					// auto-rename
					dccSend->resumeValue = 0;
					while(kvi_fileExists(dccSend->filePath.ptr()))dccSend->filePath.append(".rnm");
				}
			} else {
				// not auto accepting , ask the user what to do
				dccSend->resumeValue = fi.size();
				KviDccSendResumeMessageBox * box = new KviDccSendResumeMessageBox(m_pFrm,dccSend);
				connect(box,SIGNAL(dccSendResumeSelectionDone(KviDccSendRequestStruct *)),
							this,SLOT(dccSendResumeSelectionDone(KviDccSendRequestStruct *)));
				if(!box->isVisible())box->show();
				return;
			}
		} else {
			// equal file sizes or the file on the disk is bigger , ask wheter to overwrite or rename
			dccSend->resumeValue = 0;
			if(g_pOptions->m_bAutoAcceptDccSend){
				// auto-rename
				while(kvi_fileExists(dccSend->filePath.ptr()))dccSend->filePath.append(".rnm");
			} else {
				// not auto accepting , ask the user what to do
				KviDccSendRenameMessageBox * box = new KviDccSendRenameMessageBox(m_pFrm,dccSend);
				connect(box,SIGNAL(dccSendRenameSelectionDone(KviDccSendRequestStruct *)),
							this,SLOT(dccSendResumeSelectionDone(KviDccSendRequestStruct *)));
				if(!box->isVisible())box->show();
				return;
			}
		}
	} else dccSend->resumeValue = 0;
	dccSendResumeSelectionDone(dccSend);
}

void KviDccManager::dccSendResumeSelectionDone(KviDccSendRequestStruct *dccSend)
{
	//Create the DCC now...:))))))
	KviStr tmp(KviStr::Format,"DCC-Get-%s!%s@%s",dccSend->nick.ptr(),dccSend->username.ptr(),dccSend->host.ptr());
	KviDccSend * s = m_pFrm->createDccSend(tmp.ptr());
	s->acceptDccSendRequest(dccSend);
	delete dccSend;
}

void KviDccManager::requestDccSend(const char *nick,const char *filename)
{
	KviDccSendRequestStruct *dcc = new KviDccSendRequestStruct;
	dcc->nick = nick;
	KviIrcUser * u = m_pFrm->m_pUserList->findUser(nick);
	if(!u){
		dcc->username = "*";
		dcc->host     = "*";
	} else {
		dcc->username = u->username();
		dcc->host     = u->host();
	}

	dcc->filePath = filename ? filename : "";
	if(dcc->filePath.isEmpty()){
		KviDccSendFileDialog * box = new KviDccSendFileDialog(m_pFrm,dcc,true); //existing file only
		connect(box,SIGNAL(dccSaveNameSelected(KviDccSendRequestStruct *)),
			this,SLOT(dccSendFileToSendSelected(KviDccSendRequestStruct *)));
		if(!box->isVisible())box->show();		
	} else dccSendFileToSendSelected(dcc);
}

void KviDccManager::dccSendFileToSendSelected(KviDccSendRequestStruct * dccSend)
{
	KviStr tmp(KviStr::Format,"DCC-Send-%s!%s@%s",dccSend->nick.ptr(),dccSend->username.ptr(),dccSend->host.ptr());
	KviDccSend * s = m_pFrm->createDccSend(tmp.ptr());
	s->requestDccSend(dccSend);
	delete dccSend;
}

void KviDccManager::handleDccResume(KviIrcUser &source,KviStr &filename,unsigned short uPort,unsigned long uResumePos)
{
	KviDccSend *s = m_pFrm->findListeningDccSend(uPort,source.nick());
	if(!s){
		m_pFrm->m_pConsole->output(KVI_OUT_DCCERROR,__tr("Invalid DCC RESUME from %s [%s@%s]. Transfer not initiated for file %s on port %u : Ignoring"),
			source.nick(),source.username(),source.host(),filename.ptr(),uPort);
		return;
	}
	s->output(KVI_OUT_DCCINFO,__tr("Received RESUME request : position %u"),uResumePos);
	if(s->resumeForCurrentSend(uResumePos)){
		m_pFrm->m_pSocket->sendFmtData("PRIVMSG %s :%cDCC ACCEPT %s %u %u%c",
			source.nick(),0x01,filename.ptr(),uPort,uResumePos,0x01);
	} else { // Failed to resume
		m_pFrm->m_pConsole->output(KVI_OUT_DCCERROR,__tr("Invalid DCC RESUME from %s [%s@%s]: Resume position %u out of range for file %s on port %u : Ignoring"),
			source.nick(),source.username(),source.host(),filename.ptr(),uResumePos,uPort);
	}
}

unsigned short int g_uLastDccListenPort = 0;

unsigned short int KviDccManager::getDccSendListenPort()
{
	if(g_pOptions->m_bDccListenOnPortsInRange)
	{
		if((g_pOptions->m_uMaxDccListenPort - g_pOptions->m_uMinDccListenPort) < 1)
		{
			g_pOptions->m_uMinDccListenPort = 1025;
			g_pOptions->m_uMaxDccListenPort = 65000;
		}

		if(g_uLastDccListenPort < g_pOptions->m_uMinDccListenPort)
		{
			g_uLastDccListenPort = g_pOptions->m_uMinDccListenPort;
		} else {
			g_uLastDccListenPort++;
		}

		if(g_uLastDccListenPort > g_pOptions->m_uMaxDccListenPort)
			g_uLastDccListenPort = g_pOptions->m_uMinDccListenPort;

		return g_uLastDccListenPort;

	} else return 0;
}

#include "m_kvi_dcc_manager.moc"
