//
//   File : kvi_dlgscript.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_dlgscript.cpp)
//   Last major modification : Sun Sep 26 1999 20:45 by Till Busch
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//   Copyright (C) 1999-2000 Till Busch (buti@geocities.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_DLG_SCRIPT_CPP_

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_dlgscript.h"
#include "kvi_options.h"
#include "kvi_defines.h"
#include "kvi_app.h"
#include "kvi_locale.h"

#include "kvi_scripteditor.h"
#include "kvi_alias.h"
#include "kvi_event.h"
#include "kvi_rawevent.h"
#include "kvi_popupmenu.h"
#include "kvi_utoolbar.h"

#include "kvi_frame.h"
#include "kvi_uparser.h"
#include "kvi_console.h"

#include "kvi_varcache.h"
#include "kvi_style.h"

#include "kvi_helpwindow.h"

#include "kvi_scriptobject.h"
#include "kvi_listview.h"

#include <qlayout.h>
#include <qevent.h>
#include <qsplitter.h>
#include <qheader.h>
#include <qnamespace.h>
#include <qpixmap.h>
#include <qmessagebox.h>
#include <qasciidict.h>

//#warning "Totally missing quick help"
// declared in kvi_app.cpp
extern KviAliasManager    *g_pAliasManager;
extern KviEventManager    *g_pEventManager;
extern KviRawEventManager *g_pRawEventManager;
extern KviVariableCache   *g_pVarCache;

extern KviPopupMenu *g_pChannelPopup;
extern KviPopupMenu *g_pUserlistPopup;
extern KviPopupMenu *g_pNotifylistPopup;
extern KviPopupMenu *g_pQueryPopup;
extern KviPopupMenu *g_pConsolePopup;
extern KviPopupMenu *g_pDccChatPopup;

extern KviUserToolBarTemplate *g_pUserToolBarTemplate;

extern QPixmap *g_pixViewOut[KVI_OUT_NUM_IMAGES];

KviScriptDataListViewItem::KviScriptDataListViewItem(QListView * lb,const char * buffer,const char * data0,const char *data1,const char * data2,const char * data3)
:QListViewItem(lb,data0,data1,data2,data3)
{
	m_szBuffer = buffer;
}

KviScriptDataListViewItem::~KviScriptDataListViewItem()
{
}

void KviScriptDataListViewItem::makeKey(KviStr &keyBuffer)
{
	keyBuffer.sprintf("%s_%s_%d",text(0).ascii(),text(1).ascii(),m_szBuffer.len());
}


/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////
////
//// RAW EVENTS EDITOR
////
//// This widget allows editing of the raw events
////
/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@quickhelp: KviRawEventEditor
	@widget: Raw event editor
		Here, you can edit the <a href="syntax_events.kvihelp">raw event handlers</a>.<br>
		The left list view displays the list of the currently defined handlers:<br>
		the left column in that list displays the raw server command that the handler
		reacts to and the right column shows the source mask for that handler.<br>
		By clicking on "Add Handler," you will add an empty handler for the UNKNOWN literal reply
		with a full wildcard source mask.<br>
		Double-click on the left column of the newly added item to edit the command name.<br>
		You can insert a numeric value from 0 to 999, or a literal command such as 'WALLOPS' or 'KICK'.<br>
		Double-click on the right column of the item to edit the source mask.<br>
		The source mask is a string that can contain '*' wildcards: something as
		pragma!*@* or *.it (remember that the source of a server message may be either a user mask
		or a server name; some messages may also arrive from IRC services).<br>
		The single "full" wildcard ('*') matches all the sources.<br>
		In the source editor on the right you can edit your handler for the message.<br>
		For example, if you want to filter all privmsgs that arrive from the server:<br>
		- Click on "Add Handler"<br>
		- Double-click on the string, 'UNKNOWN', that just appeared in the list view.<br>
		- Change 'UNKNOWN' to 'PRIVMSG'
		- Leave the source mask as is ('*') so that it will match all sources.<br>
		- Edit your handler.<br>
		If you want to react only to privmsgs sent by users from France:<br>
		- Double-click on the "Source Mask" column of the last added item<br>
		- Change '*' to '*.fr'
		More handlers for the same command are accepted, and will be triggered in an
		arbitrary (unspecified) order.<br>
		See also the <a href="syntax_events.kvihelp">event scripting syntax documentation</a>.<br>		
*/



KviStr g_szLastSelectedRawEvent = "";


KviRawEventEditor::KviRawEventEditor(QWidget * parent,const char * name)
:QWidget(parent,name)
{
	QGridLayout * g = new QGridLayout(this,4,2,4,2);
	m_pListView = new KviListView(this);
	m_pListView->addColumn(__tr("Raw Command"));
	m_pListView->addColumn(__tr("Source Mask (RegExp)"));
	g->addWidget(m_pListView,0,0);

	m_pScriptEditor = new KviScriptEditor(this);
	g->addMultiCellWidget(m_pScriptEditor,0,3,1,1);

	m_pNew = new QPushButton(__tr("&Add Handler"),this);
	g->addWidget(m_pNew,1,0);
	m_pDelete = new QPushButton(__tr("&Remove Handler"),this);
	g->addWidget(m_pDelete,2,0);
	m_pClear = new QPushButton(__tr("&Clear Handlers"),this);
	g->addWidget(m_pClear,3,0);

	g->setRowStretch(0,1);
	g->setColStretch(1,1);

	m_pDelete->setEnabled(false);
	m_pScriptEditor->setEnabled(false);

	m_pCurItem = 0;

	fillEventListView();

	connect(m_pListView,SIGNAL(selectionChanged(QListViewItem *)),this,SLOT(listViewSelectionChanged(QListViewItem *)));
	connect(m_pNew,SIGNAL(clicked()),this,SLOT(makeNewEvent()));
	connect(m_pDelete,SIGNAL(clicked()),this,SLOT(deleteCurEvent()));
	connect(m_pClear,SIGNAL(clicked()),this,SLOT(clearAllEvents()));

	KviScriptDataListViewItem * it = findEventItem(g_szLastSelectedRawEvent.ptr());
	if(it)m_pListView->setSelected(it,true);
}

KviRawEventEditor::~KviRawEventEditor()
{
}

void KviRawEventEditor::makeNewEvent()
{
	KviScriptDataListViewItem * it = new KviScriptDataListViewItem(m_pListView,"",__tr("UNKNOWN"),"*");
	m_pListView->setSelected(it,true);
	m_pScriptEditor->setFocus();
}

void KviRawEventEditor::deleteCurEvent()
{
	if(!m_pCurItem)return;
	KviScriptDataListViewItem * it = m_pCurItem;
	m_pCurItem = 0;
	delete it;
	it = (KviScriptDataListViewItem *)m_pListView->currentItem();
	if(it)m_pListView->setSelected(it,true);
}

void KviRawEventEditor::clearAllEvents()
{
	if(QMessageBox::warning(this,__tr("Clear Handlers"),
		__tr("Are you sure you want to remove all raw event handlers?"),
		QMessageBox::Yes | QMessageBox::Default,
		QMessageBox::No | QMessageBox::Escape) == QMessageBox::Yes)
	{
		m_pCurItem = 0;
		m_pListView->clear();
	}
}

void KviRawEventEditor::saveCurrentItem()
{
	if(!m_pCurItem)return;
	m_pCurItem->m_szBuffer = m_pScriptEditor->text();
}

KviScriptDataListViewItem * KviRawEventEditor::findEventItem(const char * item_key)
{
	for(KviScriptDataListViewItem * it = (KviScriptDataListViewItem * )m_pListView->firstChild();it;it = (KviScriptDataListViewItem *)it->nextSibling())
	{
		KviStr tmp;
		it->makeKey(tmp);
		if(kvi_strEqualCI(tmp.ptr(),item_key))return it;
	}
	return 0;
}

void KviRawEventEditor::commit()
{
	if(m_pCurItem)m_pCurItem->makeKey(g_szLastSelectedRawEvent);
	saveCurrentItem();
	g_pRawEventManager->cleanup();
	for(KviScriptDataListViewItem * it = (KviScriptDataListViewItem * )m_pListView->firstChild();it;it = (KviScriptDataListViewItem *)it->nextSibling())
	{
		g_pRawEventManager->addHandler(it->text(0).ascii(),it->text(1).ascii(),it->m_szBuffer.ptr());
	}
}

void KviRawEventEditor::listViewSelectionChanged(QListViewItem *it)
{
	saveCurrentItem();
	m_pCurItem = 0;
	if(it)
	{
		m_pScriptEditor->setText(((KviScriptDataListViewItem *)it)->m_szBuffer.ptr());
		m_pScriptEditor->setFocus();
	} else {
		m_pScriptEditor->setText("");
	}
	m_pCurItem = (KviScriptDataListViewItem *)it;
	m_pScriptEditor->setEnabled(it);
	m_pDelete->setEnabled(it);
}

void KviRawEventEditor::fillEventListView()
{
	KviScriptDataListViewItem * item = 0;

	for(int i = 0;i< 1000;i++){
		if(g_pRawEventManager->numericEvent(i))
		{
			for(KviRawEvent * e = g_pRawEventManager->numericEvent(i)->first();e;e= g_pRawEventManager->numericEvent(i)->next())
			{
				KviStr tmp(KviStr::Format,"%d",i);
				item = new KviScriptDataListViewItem(m_pListView,e->szBuffer.ptr(),tmp.ptr(),e->szSource.ptr());
			}
		}
	}

	if(g_pRawEventManager->literalEventsDict())
	{
		QAsciiDictIterator<KviRawEventList> it(*(g_pRawEventManager->literalEventsDict()));
		QList<KviRawEvent> * l = 0;
		while((l = it.current()))
		{
			for(KviRawEvent * e = l->first();e;e= l->next())
			{
				item = new KviScriptDataListViewItem(m_pListView,e->szBuffer.ptr(),it.currentKey(),e->szSource.ptr());
			}			
			++it;
		}
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////
////
//// ALIAS EDITOR
////
//// This widget allows editing of the aliases
//// Each alias is kept inside a KviScriptDataListViewItem (QListViewItem)
//// child of the m_pListView.
//// Aliases are saved by calling "commit"
////
/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@quickhelp: KviAliasEditor
	@widget: Aliases tab
		Here, you can edit the aliases (custom commands).<br>
		To create a new alias, click on "Add Alias;" a new "unnamed" alias will be created
		with an empty data buffer.<br>
		Click on "Rename" to set the desired name, and edit your alias body.<br>
		Please note that while you CAN create an alias with a name that
		is already used by a KVIrc command, you will NOT be able to use it.<br>
*/



QValidator::State KviAliasNameValidator::validate(QString & str,int &) const
{
	KviStr s = str;
	const char * p=s.ptr();
	while(*p){
		if((!isalnum(*p))&&(*p != '.')&&(*p != '_'))return QValidator::Invalid;
		p++;
	}
	return QValidator::Acceptable;
}

KviStr g_szLastSelectedAlias = "";

KviAliasEditor::KviAliasEditor(QWidget * parent,const char *name)
:QWidget(parent,name)
{
	QGridLayout * g = new QGridLayout(this,5,3,4,2);
	m_pListView = new QListView(this);
	m_pListView->addColumn(__tr("Alias"));
	m_pListView->setSelectionMode(QListView::Single);
	g->addMultiCellWidget(m_pListView,0,1,0,0);
	m_pLineEdit = new QLineEdit(this);
	g->addWidget(m_pLineEdit,0,1);
	m_pEditName = new QPushButton(__tr("Re&name"),this);
	g->addWidget(m_pEditName,0,2);
	m_pEditor = new KviScriptEditor(this);
	g->addMultiCellWidget(m_pEditor,1,4,1,2);
	m_pNew = new QPushButton(__tr("&Add Alias"),this);
	g->addWidget(m_pNew,2,0);
	m_pDelete = new QPushButton(__tr("&Remove Alias"),this);
	g->addWidget(m_pDelete,3,0);
	m_pClear = new QPushButton(__tr("&Clear Aliases"),this);
	g->addWidget(m_pClear,4,0);
	g->setRowStretch(1,1);
	g->setColStretch(1,1);

	m_pCurItem = 0;
	m_pEditName->setEnabled(false);
	m_pLineEdit->setEnabled(false);
	m_pLineEdit->installEventFilter(this);
	m_pLineEdit->setValidator(new KviAliasNameValidator(m_pLineEdit));
	m_pEditor->setEnabled(false);
	m_pDelete->setEnabled(false);

	fillListView();
	connect(m_pListView,SIGNAL(selectionChanged(QListViewItem *)),this,SLOT(listViewSelectionChanged(QListViewItem *)));
	connect(m_pEditName,SIGNAL(clicked()),this,SLOT(editAliasName()));
	connect(m_pNew,SIGNAL(clicked()),this,SLOT(makeNewAlias()));
	connect(m_pDelete,SIGNAL(clicked()),this,SLOT(deleteCurrentAlias()));
	connect(m_pClear,SIGNAL(clicked()),this,SLOT(clearAllAliases()));

	KviScriptDataListViewItem * it = findAlias(g_szLastSelectedAlias.ptr());
	if(it)m_pListView->setSelected(it,true);
}

KviAliasEditor::~KviAliasEditor()
{
}

void KviAliasEditor::commitAliasName()
{
	if(!m_pLineEdit->isEnabled())return;
	m_pLineEdit->setEnabled(false);
	if(!m_pCurItem)return;
	KviStr txt = m_pLineEdit->text();
	if(txt.isEmpty())
	{
		QMessageBox::warning(this,__tr("Invalid alias name"),__tr("Alias name is empty"),QMessageBox::Ok | QMessageBox::Default,0);
		m_pLineEdit->setText(m_pCurItem->text(0));
		return;
	}
	QListViewItem *it = m_pListView->firstChild();
	while(it)
	{
		if(it != m_pCurItem)
		{
			KviStr tmp = it->text(0);
			if(kvi_strEqualCI(txt.ptr(),tmp.ptr()))
			{
				QMessageBox::warning(this,__tr("Invalid alias name"),__tr("An alias with that name already exists"),QMessageBox::Ok | QMessageBox::Default,0);
				m_pLineEdit->setText(m_pCurItem->text(0));
				return;
			}
		}
		it = it->nextSibling();
	}
	if(m_pCurItem)
	{
		m_pCurItem->setText(0,m_pLineEdit->text());
		return;
	}
	m_pEditor->setFocus();
}

bool KviAliasEditor::eventFilter(QObject *o,QEvent *e)
{
	if((o == this) || (o == m_pLineEdit))
	{
		if((e->type() == QEvent::Accel)||(e->type() == QEvent::KeyPress)){
			if(m_pLineEdit->isEnabled()){
				if(((QKeyEvent *)e)->key() == Qt::Key_Return){
					((QKeyEvent *)e)->accept();
					commitAliasName();
					return true;
				}
			}
		}
	}
	if(o == m_pLineEdit)
	{
		if(m_pLineEdit->isEnabled())
		{
			if(e->type() == QEvent::FocusOut)
			{
				commitAliasName();
				return true;
			}
		}
	}
	return QWidget::eventFilter(o,e);
}

void KviAliasEditor::saveCurrentItem()
{
	if(m_pCurItem)
	{
		m_pCurItem->m_szBuffer = m_pEditor->text();
	}
}

void KviAliasEditor::editAliasName()
{
	m_pLineEdit->setEnabled(true);
	m_pLineEdit->setFocus();
}

void KviAliasEditor::deleteCurrentAlias()
{
	if(m_pCurItem)
	{
		KviScriptDataListViewItem * it = m_pCurItem;
		m_pCurItem = 0;
		delete it;
		it = (KviScriptDataListViewItem * )m_pListView->currentItem();
		if(it)m_pListView->setSelected(it,true);
//			listViewSelectionChanged(it);
	}
}

void KviAliasEditor::clearAllAliases()
{
	if(QMessageBox::warning(this,__tr("Clear Aliases"),
		__tr("Are you sure you want to remove all aliases?"),
		QMessageBox::Yes | QMessageBox::Default,
		QMessageBox::No | QMessageBox::Escape) == QMessageBox::Yes)
	{
		m_pCurItem = 0;
		m_pListView->clear();
	}
}

KviScriptDataListViewItem * KviAliasEditor::findAlias(const char *name)
{
	for(QListViewItem * it = m_pListView->firstChild();it;it = it->nextSibling())
	{
		KviStr tmp = it->text(0);
		if(kvi_strEqualCI(tmp.ptr(),name))return (KviScriptDataListViewItem *)it;
	}
	return 0;
}

void KviAliasEditor::makeNewAlias()
{
	KviStr tmp = __tr("unnamed");
	int i = 1;
	while(findAlias(tmp.ptr()))
	{
		tmp.sprintf(__tr("unnamed%d"),i);
		i++;
	}
	KviScriptDataListViewItem *it = new KviScriptDataListViewItem(m_pListView,"",tmp.ptr());
	m_pListView->setSelected(it,true);
	m_pEditor->setFocus();
}

void KviAliasEditor::listViewSelectionChanged(QListViewItem *it)
{
	saveCurrentItem();
	m_pCurItem = 0;
	if(it)
	{
		m_pEditor->setText(((KviScriptDataListViewItem *)it)->m_szBuffer.ptr());
		m_pLineEdit->setText(((QListViewItem *)it)->text(0));
		m_pEditor->setFocus();
	} else {
		m_pEditor->setText("");
		m_pLineEdit->setText("");
	}
	m_pCurItem = (KviScriptDataListViewItem *)it;
	m_pEditor->setEnabled(it);
	m_pEditName->setEnabled(it);
	m_pDelete->setEnabled(it);
}

void KviAliasEditor::fillListView()
{
	KviScriptDataListViewItem * it = 0;
	for(KviAlias * a = g_pAliasManager->m_pAliasList->first();a;a = g_pAliasManager->m_pAliasList->next())
	{
		it = new KviScriptDataListViewItem(m_pListView,a->szBuffer.ptr(),a->szName.ptr());
	}
}

void KviAliasEditor::commit()
{
	if(m_pCurItem)g_szLastSelectedAlias = m_pCurItem->text(0);
	saveCurrentItem();
	g_pAliasManager->clearAll();
	for(KviScriptDataListViewItem * it = (KviScriptDataListViewItem * )m_pListView->firstChild();it;it = (KviScriptDataListViewItem *)it->nextSibling())
	{
		KviAlias * a = new KviAlias;
		a->szName = it->text(0);
		a->szBuffer = it->m_szBuffer;
		g_pAliasManager->addAlias(a);
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////
////
//// OBJECT VIEWER
////
/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@quickhelp: KviObjectViewer
	@widget: Object viewer
		This tab displays the tree of currently existing <a href="syntax_objects.kvihelp">object classes</a>,
		and the tree of existing object instances.<br>
*/



KviObjectViewer::KviObjectViewer(QWidget * parent,const char *name)
:QWidget(parent,name)
{
	QGridLayout *g = new QGridLayout(this,1,1,4,2);
	QSplitter * s = new QSplitter(QSplitter::Horizontal,this);
	g->addWidget(s,0,0);
	QSplitter * v = new QSplitter(QSplitter::Vertical,s);

	m_pClassView = new QListView(v);
	m_pClassView->addColumn(__tr("Class"));
	m_pClassView->addColumn(__tr("Definition"));
	m_pClassView->setRootIsDecorated(true);
	connect(m_pClassView,SIGNAL(currentChanged(QListViewItem *)),this,SLOT(classViewCurrentChanged(QListViewItem *)));
	m_pFncView = new QListView(v);
	m_pFncView->addColumn(__tr("Function"));
	m_pFncView->addColumn(__tr("Comments"));
	m_pObjectView = new QListView(s);
	m_pObjectView->addColumn(__tr("Object"));
	m_pObjectView->addColumn(__tr("Name"));
	m_pObjectView->addColumn(__tr("Class"));
	m_pObjectView->addColumn(__tr("Comments"));
	m_pObjectView->setRootIsDecorated(true);
	fillViews();
}

KviObjectViewer::~KviObjectViewer()
{
}

// Global class definitions (kvi_scriptobject.cpp)
extern KviScriptObjectClassDefinition * g_pClassDefinitions;

void KviObjectViewer::classViewCurrentChanged(QListViewItem *it)
{
	m_pFncView->clear();
	KviStr cl = it->text(0);
	KviScriptObjectClassDefinition * def = 0;
	if(kvi_strEqualCI(cl.ptr(),g_pClassDefinitions->getClass()))def = g_pClassDefinitions;
	else def = g_pClassDefinitions->lookupChildClassDefinition(cl.ptr());
	if(!def)
	{
		debug("Oops... %s: No such class definition",cl.ptr());
		return;
	}
	QAsciiDictIterator<KviScriptObjectFunctionStruct> iter(*(def->functionDict()));
	while(iter.current())
	{
		KviStr tmp;
		if(iter.current()->fncHandler)tmp = __tr("Core code handler");
		else tmp = __tr("Script handler");
		if(iter.current()->flags & KVI_SCRIPTOBJECTFUNCTION_FLAG_INHERITED)tmp.append(__tr(", Inherited"));
		if(iter.current()->flags & KVI_SCRIPTOBJECTFUNCTION_FLAG_OVERRIDE)tmp.append(__tr(", Override"));
		if(iter.current()->flags & KVI_SCRIPTOBJECTFUNCTION_FLAG_BUILTIN)tmp.append(__tr(", Built-in"));
		else tmp.append(__tr(", User-defined"));
		it = new QListViewItem(m_pFncView,iter.currentKey(),tmp.ptr());
		++iter;
	}
}

void KviObjectViewer::fillViews()
{
	m_pClassView->clear();
	m_pObjectView->clear();
	m_pFncView->clear();
	fillClassDefinition(0,g_pClassDefinitions);
	for(KviFrame * f = g_pApp->m_pFrameList->first();f;f=g_pApp->m_pFrameList->next())
	{
		fillObjectItem(0,f->m_pUserParser->m_pScriptObjectController->topLevelObject());
	}
}

void KviObjectViewer::fillObjectItem(QListViewItem * p,KviScriptObject * o)
{
	if(!o)return;
	QListViewItem * it = 0;
	if(p)it = new QListViewItem(p,o->id(),o->getName(),o->getClass(),__tr("-"));
	else it = new QListViewItem(m_pObjectView,o->id(),o->getName(),o->getClass(),__tr("Built-in root object"));
	it->setOpen(true);
	QList<KviScriptObject> * l = o->childrenList();
	for(KviScriptObject * obj=l->first();obj;obj=l->next())
	{
		fillObjectItem(it,obj);
	}
}

void KviObjectViewer::fillClassDefinition(QListViewItem * p,KviScriptObjectClassDefinition * d)
{
	if(!d)return;
	QListViewItem * it = p ? new QListViewItem(p,d->getClass(),d->isBuiltin() ? __tr("Built-in") : __tr("User-defined")) : new QListViewItem(m_pClassView,d->getClass(),d->isBuiltin() ? __tr("Built-in") : __tr("User-defined"));
	it->setOpen(true);
	QList<KviScriptObjectClassDefinition> * l = d->childClasses();
	if(l)
	{
		for(KviScriptObjectClassDefinition * def = l->first();def;def = l->next())
		{
			fillClassDefinition(it,def);
		}
	}
}


/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////
////
//// EVENT EDITOR
////
//// This widget allows editing of the events
////
/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@quickhelp: KviEventEditor
	@widget: Event editor
		Here you edit the <a href="syntax_events.kvihelp">event handlers</a>.<br>
		To edit an event handler, select its name in the left list box.<br>
		When you are done editing the event, enable it by checking the "Enable" checkbox.<br>
*/

KviStr g_szLastSelectedEvent = "";


KviEventEditor::KviEventEditor(QWidget * parent,const char * name)
:QWidget(parent,name)
{
	QGridLayout * g = new QGridLayout(this,4,2,4,2);
	m_pListView = new QListView(this);
	m_pListView->addColumn(__tr("Event"));
	m_pListView->addColumn(__tr("-"));
	m_pListView->addColumn(__tr("!"));
	m_pListView->setSelectionMode(QListView::Single);
	g->addMultiCellWidget(m_pListView,0,2,0,0);

	m_pEnabled = new QCheckBox(__tr("Enable event"),this);
	g->addWidget(m_pEnabled,0,1);

	m_pParamsLabel = new QLabel(this);
	m_pParamsLabel->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	g->addWidget(m_pParamsLabel,1,1);

	m_pScriptEditor = new KviScriptEditor(this);
	g->addMultiCellWidget(m_pScriptEditor,2,3,1,1);

	m_pClear = new QPushButton(__tr("&Clear Handlers"),this);
	g->addWidget(m_pClear,3,0);

	g->setRowStretch(2,1);
	g->setColStretch(1,1);

	m_pScriptEditor->setEnabled(false);
	m_pEnabled->setEnabled(false);
	m_pParamsLabel->setEnabled(false);

	m_pCurItem = 0;

	fillEventListView();

	connect(m_pListView,SIGNAL(selectionChanged(QListViewItem *)),this,SLOT(listViewSelectionChanged(QListViewItem *)));
	connect(m_pEnabled,SIGNAL(toggled(bool)),this,SLOT(enableCheckToggled(bool)));
	connect(m_pClear,SIGNAL(clicked()),this,SLOT(clearAllHandlers()));

	KviScriptDataListViewItem * it = findEventItem(g_szLastSelectedEvent.ptr());
	if(it)m_pListView->setSelected(it,true);
}

KviEventEditor::~KviEventEditor()
{
}

KviScriptDataListViewItem * KviEventEditor::findEventItem(const char * item_key)
{
	for(KviScriptDataListViewItem * it = (KviScriptDataListViewItem * )m_pListView->firstChild();it;it = (KviScriptDataListViewItem *)it->nextSibling())
	{
		KviStr tmp;
		it->makeKey(tmp);
		if(kvi_strEqualCI(tmp.ptr(),item_key))return it;
	}
	return 0;
}

void KviEventEditor::fillEventListView()
{
	KviScriptDataListViewItem * it=0;

	for(int i=0;i<KVI_NUM_EVENTS;i++)
	{
		it = new KviScriptDataListViewItem(m_pListView,g_pEventManager->m_szEventBuffer[i].ptr(),
					g_pEventManager->m_eventDescriptor[i].eventName,
					g_pEventManager->m_szEventBuffer[i].hasData() ? "X" : "",
					g_pEventManager->m_bEventEnabled[i] ? "X" : "",
					g_pEventManager->m_eventDescriptor[i].eventParams);
	}
}

void KviEventEditor::clearAllHandlers()
{
	if(QMessageBox::warning(this,__tr("Clear Handlers"),
		__tr("Are you sure you want to clear all event handlers?"),
		QMessageBox::Yes | QMessageBox::Default,
		QMessageBox::No | QMessageBox::Escape) == QMessageBox::Yes)
	{
		for(KviScriptDataListViewItem * it = (KviScriptDataListViewItem * )m_pListView->firstChild();it;it = (KviScriptDataListViewItem *)it->nextSibling())
		{
			it->m_szBuffer = "";
			it->setText(1,"");
			it->setText(2,"");
		}
		m_pScriptEditor->setText("");
		m_pEnabled->setChecked(false);
	}
}

void KviEventEditor::saveCurrentItem()
{
	if(!m_pCurItem)return;
	m_pCurItem->m_szBuffer = m_pScriptEditor->text();
	if(m_pCurItem->m_szBuffer.hasNonWhiteSpaceData())
	{
		m_pCurItem->setText(1,"X");
		m_pCurItem->setText(2,m_pEnabled->isChecked() ? "X" : "");
	} else {
		// Empty implementation
		m_pCurItem->m_szBuffer = "";
		m_pCurItem->setText(1,"");
		m_pCurItem->setText(2,"");
	}
}

void KviEventEditor::enableCheckToggled(bool bEnabled)
{
	if(!m_pCurItem)return;
	KviStr tmp = m_pScriptEditor->text();
	m_pCurItem->setText(1,tmp.hasNonWhiteSpaceData() ? "X" : "");
	m_pCurItem->setText(2,bEnabled ? "X" : "");
}

void KviEventEditor::listViewSelectionChanged(QListViewItem *it)
{
	saveCurrentItem();
	m_pCurItem = 0;
	if(it)
	{
		m_pScriptEditor->setText(((KviScriptDataListViewItem *)it)->m_szBuffer.ptr());
		KviStr tmp = ((QListViewItem *)it)->text(2);
		m_pEnabled->setChecked(tmp.hasData());
		m_pScriptEditor->setFocus();
		m_pParamsLabel->setText(((QListViewItem *)it)->text(3));
	} else {
		m_pScriptEditor->setText("");
		m_pParamsLabel->setText("");
	}
	m_pCurItem = (KviScriptDataListViewItem *)it;
	m_pScriptEditor->setEnabled(it);
	m_pParamsLabel->setEnabled(it);
	m_pEnabled->setEnabled(it);
}

void KviEventEditor::commit()
{
	if(m_pCurItem)m_pCurItem->makeKey(g_szLastSelectedEvent);

	saveCurrentItem();
	g_pEventManager->cleanup();
	for(KviScriptDataListViewItem * it = (KviScriptDataListViewItem * )m_pListView->firstChild();it;it = (KviScriptDataListViewItem *)it->nextSibling())
	{
		KviStr tmp = it->text(1);
		if(tmp.hasData())
		{
			tmp = it->text(2);
			bool bEnabled = tmp.hasData();
			g_pEventManager->setHandler(it->text(0).ascii(),it->m_szBuffer.ptr(),bEnabled);
		}
	}
	g_pEventManager->disableEmptyHandlers();

	for(KviFrame * f = g_pApp->m_pFrameList->first(); f; f = g_pApp->m_pFrameList->next())f->restartIdleTimer();
}

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////
//// POPUP EDITOR
////
//// This widget allows editing of the popups
////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

/*
	@quickhelp: KviPopupEditor
	@widget: Popups tab
		Here, you can edit your popup menus.<br>
		In the upper list box, you can select the popup to edit:<br>
		&nbsp;Channel Window - Shown when right clicking in the channel window.<br>
		&nbsp;Query Window - Shown when right clicking in a query window.<br>
		&nbsp;Console Window - Shown when right clicking in the console window.<br>
		&nbsp;DCC Chat Window - Shown when right clicking in a DCC chat window.<br>
		&nbsp;Notify List - Shown when right clicking on a nickname in the right pane of the Console window (the notify list).<br>
		&nbsp;User List - Shown when right clicking on a nickname in the right pane of a channel window (the user list).<br>
		The lower tree view displays the current popup.<br>
		A right click in the tree view shows a menu that allows adding/removing
		items from the popup.<br>
		In the lower editor, you edit the popup code associated with the selected popup item.<br>
		In the 'Icon' line edit, you can enter the number of an icon to use with the popup (see the <a href="echo.kvihelp">echo</a> -i option).<br>
		The 'Icon' can be also an absolute or relative (to the KVIrc pics directory) file name.<br>
		Right clicking in the editor displays a helper menu.<br>
		The 'Name' line edit contains the current item name.<br>
		The "Test" button allows you to test your popup in its current state, and clicking an item
		in the popup selects it in the tree view.<br>
*/


KviPopupListViewItem::KviPopupListViewItem(PopupItemType t,KviPopupListViewItem * parent,const char * name,const char *type,const char *buffer,const char *icon)
:QListViewItem(parent,_CHAR_2_QSTRING(name),_CHAR_2_QSTRING(type))
{
	if(buffer)m_buffer = buffer;
	if(icon)m_icon = icon;
	m_type = t;
	m_id = 0;
	tryLoadImage();
}

KviPopupListViewItem::KviPopupListViewItem(PopupItemType t,KviPopupListViewItem * parent,KviPopupListViewItem * after,const char * name,const char *type,const char *buffer,const char *icon)
:QListViewItem(parent,after,_CHAR_2_QSTRING(name),_CHAR_2_QSTRING(type))
{
	if(buffer)m_buffer = buffer;
	if(icon)m_icon = icon;
	m_type = t;
	m_id = 0;
	tryLoadImage();
}
KviPopupListViewItem::KviPopupListViewItem(PopupItemType t,QListView * parent,const char *name,const char *type,const char *buffer,const char *icon)
:QListViewItem(parent,_CHAR_2_QSTRING(name),_CHAR_2_QSTRING(type))
{
	if(buffer)m_buffer = buffer;
	if(icon)m_icon = icon;
	m_type = t;
	m_id = 0;
	tryLoadImage();
}
KviPopupListViewItem::KviPopupListViewItem(PopupItemType t,QListView * parent,KviPopupListViewItem * after,const char *name,const char *type,const char *buffer,const char *icon)
:QListViewItem(parent,after,_CHAR_2_QSTRING(name),_CHAR_2_QSTRING(type))
{
	if(buffer)m_buffer = buffer;
	if(icon)m_icon = icon;
	m_type = t;
	m_id = 0;
	tryLoadImage();
}
KviPopupListViewItem::~KviPopupListViewItem()
{
}

KviPopupClipboardItem * KviPopupListViewItem::getClipboardItem()
{
	KviPopupClipboardItem * it = new KviPopupClipboardItem;
//	it->pItemList = 0;
	it->szName = text(0);
	it->szBuffer = m_buffer;
	it->szIcon = m_icon;
	it->type = m_type;
	KviPopupListViewItem * ch = (KviPopupListViewItem *)firstChild();
	if(ch)
	{
		it->pItemList = new QList<KviPopupClipboardItem>;
		it->pItemList->setAutoDelete(false);
		while(ch)
		{
			it->pItemList->append(ch->getClipboardItem());
			ch = (KviPopupListViewItem *)ch->nextSibling();
		}
	} else it->pItemList = 0;
	return it;
}

void KviPopupListViewItem::tryLoadImage()
{
	m_icon.stripWhiteSpace();
	if(m_icon.hasData()){
		if(m_icon.isNum()){
			bool bOk=false;
			int imId = m_icon.toInt(&bOk);
			if(imId >= 0){
				imId = imId % KVI_OUT_NUM_IMAGES;
				setPixmap(0,*(g_pixViewOut[imId]));
			}
		} else {
			KviStr imagePath;
			if(g_pApp->findImage(imagePath,m_icon.ptr())){
				QPixmap pix(imagePath.ptr());
				if(!pix.isNull()){
					setPixmap(0,pix);
				} else debug("Unable to find image file %s",m_icon.ptr());
			}
		}
	}
}

int g_iLastSelectedPopup = -1;


KviPopupEditor::KviPopupEditor(QWidget * parent,const char *name)
:QWidget(parent,name)
{
	m_pChannelPopup = new KviPopupMenu(0,"dummy");
	m_pChannelPopup->copyFromPopup(g_pChannelPopup);
	m_pQueryPopup = new KviPopupMenu(0,"dummy");
	m_pQueryPopup->copyFromPopup(g_pQueryPopup);
	m_pNotifylistPopup = new KviPopupMenu(0,"dummy");
	m_pNotifylistPopup->copyFromPopup(g_pNotifylistPopup);
	m_pUserlistPopup = new KviPopupMenu(0,"dummy");
	m_pUserlistPopup->copyFromPopup(g_pUserlistPopup);
	m_pConsolePopup = new KviPopupMenu(0,"dummy");
	m_pConsolePopup->copyFromPopup(g_pConsolePopup);
	m_pDccChatPopup = new KviPopupMenu(0,"dummy");
	m_pDccChatPopup->copyFromPopup(g_pDccChatPopup);

	QGridLayout * g = new QGridLayout(this,4,4,4,2);

	QPushButton * b = new QPushButton(_CHAR_2_QSTRING(__tr("&Test Popup")),this);
	g->addWidget(b,0,2);
	connect(b,SIGNAL(clicked()),this,SLOT(testCurrentPopup()));

	b = new QPushButton(_CHAR_2_QSTRING(__tr("&Clear Popup")),this);
	g->addWidget(b,0,3);
	connect(b,SIGNAL(clicked()),this,SLOT(clearCurrentPopup()));

	m_pPopupsBox = new QComboBox(false,this);
	m_pPopupsBox->insertItem(_CHAR_2_QSTRING("Channel Window (channel)"));
	m_pPopupsBox->insertItem(_CHAR_2_QSTRING("Console Window (console)"));
	m_pPopupsBox->insertItem(_CHAR_2_QSTRING("DCC Chat Window (dccchat)"));
	m_pPopupsBox->insertItem(_CHAR_2_QSTRING("Console Notify List (notifylist)"));
	m_pPopupsBox->insertItem(_CHAR_2_QSTRING("Query Window (query)"));
	m_pPopupsBox->insertItem(_CHAR_2_QSTRING("Channel Userlist (userlist)"));
	g->addMultiCellWidget(m_pPopupsBox,0,0,0,1);

	connect(m_pPopupsBox,SIGNAL(activated(int)),this,SLOT(popupComboSelectionChanged(int)));

	QSplitter * splitter = new QSplitter(Qt::Vertical,this);

	m_pPopupView = new QListView(splitter);
	m_pPopupView->setFrameStyle(QFrame::WinPanel|QFrame::Sunken);
	m_pPopupView->addColumn(_CHAR_2_QSTRING("Item"));
	m_pPopupView->addColumn(_CHAR_2_QSTRING("Type"));
	m_pPopupView->setRootIsDecorated(true);
	m_pPopupView->setMultiSelection(false);
	m_pPopupView->header()->setClickEnabled(false);
	m_pPopupView->setAllColumnsShowFocus(true);
	m_pPopupView->viewport()->installEventFilter(this); // Need to adjust the behaviour of mousePressEvent

	m_pPopupPopup = new QPopupMenu(this);

	connect(m_pPopupView,SIGNAL(selectionChanged(QListViewItem *)),this,SLOT(popupItemSelected(QListViewItem *)));
	connect(m_pPopupView,SIGNAL(rightButtonPressed(QListViewItem *,const QPoint &,int)),
			this,SLOT(popupViewRightButtonPressed(QListViewItem *,const QPoint &,int)));

	m_pPopupEditor = new KviScriptEditor(splitter);
	g->addMultiCellWidget(splitter,1,1,0,3);

	m_pIconPopup = new QPopupMenu(this);
	QPopupMenu *ppp=m_pIconPopup;
	connect(ppp,SIGNAL(activated(int)),this,SLOT(iconPopupActivated(int)));
	for(int i=0;i<KVI_OUT_NUM_IMAGES;i++){
		ppp->insertItem(*(g_pixViewOut[i]),i);
		if(((i % 15) == 0)&&(i != 0)){
			QPopupMenu * sub = new QPopupMenu(ppp);
			ppp->insertItem(__tr("More..."),sub);
			ppp = sub;
			connect(ppp,SIGNAL(activated(int)),this,SLOT(iconPopupActivated(int)));
		}
	}

	QLabel *l=new QLabel(_CHAR_2_QSTRING("Icon:"),this);
	g->addWidget(l,2,0);

	m_pPopupIconEditor = new QLineEdit(this);
	m_pPopupIconEditor->installEventFilter(this);

	g->addMultiCellWidget(m_pPopupIconEditor,2,2,1,3);

	l = new QLabel(_CHAR_2_QSTRING("Name:"),this);
	g->addWidget(l,3,0);

	m_pPopupNameEditor = new QLineEdit(this);
	g->addMultiCellWidget(m_pPopupNameEditor,3,3,1,3);

	g->setColStretch(1,10);

	m_pEditedMenu = 0;
	m_pEditedPopupItem = 0;

	m_pClipboardItem = 0;

	if((g_iLastSelectedPopup < 0)||(g_iLastSelectedPopup > 5))g_iLastSelectedPopup = 0;
	m_pPopupsBox->setCurrentItem(g_iLastSelectedPopup);
	popupComboSelectionChanged(g_iLastSelectedPopup);
}

KviPopupEditor::~KviPopupEditor()
{
	destroyClipboard();
	delete m_pConsolePopup;
	delete m_pChannelPopup;
	delete m_pQueryPopup;
	delete m_pUserlistPopup;
	delete m_pNotifylistPopup;
	delete m_pDccChatPopup;
}

void KviPopupEditor::destroyClipboardItem(KviPopupClipboardItem * i)
{
	if(!i)return;
	if(i->pItemList)
	{
		for(KviPopupClipboardItem * it = i->pItemList->first();it;it = i->pItemList->next())
		{
			destroyClipboardItem(it);
		}
		delete i->pItemList;
	}
	delete i;
}

void KviPopupEditor::destroyClipboard()
{
	if(m_pClipboardItem)
	{
		destroyClipboardItem(m_pClipboardItem);
		m_pClipboardItem = 0;
	}
}

void KviPopupEditor::commit()
{
	if(m_pEditedMenu)saveMenu(m_pEditedMenu);
	g_iLastSelectedPopup = m_pPopupsBox->currentItem();

	g_pChannelPopup->copyFromPopup(m_pChannelPopup);
	g_pQueryPopup->copyFromPopup(m_pQueryPopup);
	g_pNotifylistPopup->copyFromPopup(m_pNotifylistPopup);
	g_pUserlistPopup->copyFromPopup(m_pUserlistPopup);
	g_pConsolePopup->copyFromPopup(m_pConsolePopup);
	g_pDccChatPopup->copyFromPopup(m_pDccChatPopup);
}

bool KviPopupEditor::eventFilter(QObject * o,QEvent *e)
{
	// Poor hack
	if(o == m_pPopupIconEditor){
		if(e->type() == QEvent::MouseButtonPress){
			if(((QMouseEvent *)e)->button() & RightButton){
				m_pIconPopup->popup(QCursor::pos());
				return true;
			}
		}
		return false;
	}
	if(e->type() == QEvent::MouseButtonPress){
		if(o == m_pPopupView->viewport()){
			if(m_pPopupView->firstChild() == 0){
				popupViewRightButtonPressed(0,m_pPopupView->mapToGlobal(((QMouseEvent *)e)->pos()),0);
				return true;
			}
		}
	}
	return QWidget::eventFilter(o,e);
}


void KviPopupEditor::popupComboSelectionChanged(int index)
{
	switch(index){
		case 0: startEditPopup(m_pChannelPopup); break;
		case 1: startEditPopup(m_pConsolePopup); break;
		case 2: startEditPopup(m_pDccChatPopup); break;
		case 3: startEditPopup(m_pNotifylistPopup); break;
		case 4: startEditPopup(m_pQueryPopup); break;
		case 5: startEditPopup(m_pUserlistPopup); break;
		default: /* newer here */ break;
	}
}

void KviPopupEditor::startEditPopup(KviPopupMenu *p)
{
	if(m_pEditedMenu)saveMenu(m_pEditedMenu);
	m_pEditedMenu = p;
	m_pPopupView->clear();
	m_pPopupView->setSorting(-1);
	appendChildPopup(0,p);
	m_pEditedPopupItem = 0;
	if(m_pPopupView->firstChild() == 0)disablePopupEditors();
	else m_pPopupView->setSelected(m_pPopupView->firstChild(),true);
}

void KviPopupEditor::testCurrentPopup()
{
	if(m_pEditedMenu){
		saveMenu(m_pEditedMenu);
		m_pEditedMenu->doTestPopup(this,SLOT(testPopupSlot(int)));
	}
}

void KviPopupEditor::clearCurrentPopup()
{
	if(QMessageBox::warning(this,__tr("Clear Popup"),
		__tr("Are you sure you want to clear this popup?"),
		QMessageBox::Yes | QMessageBox::Default,
		QMessageBox::No | QMessageBox::Escape) == QMessageBox::Yes)
	{
		m_pPopupView->clear();
		m_pPopupView->setSorting(-1);
	}
}

void KviPopupEditor::testPopupSlot(int id)
{
	if(!selectPopupItemById(0,id))debug("oops...popup item not found in the list view ?");
}

bool KviPopupEditor::selectPopupItemById(KviPopupListViewItem * par,int id)
{
	KviPopupListViewItem *it = (KviPopupListViewItem *)(par ? par->firstChild() : m_pPopupView->firstChild());
	while(it){
		if(it->m_type == KviPopupListViewItem::Item){
			if(it->m_id == id){
				// open all parents
				par = (KviPopupListViewItem *)it->parent();
				while(par){
					par->setOpen(true);
					par = (KviPopupListViewItem *)par->parent();
				}
				// select it
				m_pPopupView->setSelected(it,true);
				// and ensure it is visible
				m_pPopupView->ensureItemVisible(it);
				return true;
			}
		} else if(it->m_type == KviPopupListViewItem::Submenu){
			if(selectPopupItemById(it,id))return true;
		}
		it = (KviPopupListViewItem *)it->nextSibling();
	}
	return false;
}

void KviPopupEditor::saveMenu(KviPopupMenu *p)
{
	if(m_pEditedPopupItem)savePopupItem(m_pEditedPopupItem); // do not forget to save the last one
	p->clearAll();
	saveSubmenu(p,0);
}

void KviPopupEditor::saveSubmenu(KviPopupMenu *p,KviPopupListViewItem *par)
{
	KviPopupListViewItem *it = (KviPopupListViewItem *)(par ? par->firstChild() : m_pPopupView->firstChild());
	while(it){
		if(it->m_type == KviPopupListViewItem::Separator){
			p->insertSeparatorItem();
			it->m_id = 0;
		} else {
			if(it->m_type == KviPopupListViewItem::Item){
				KviStr tmp = it->text(0);
				tmp.stripWhiteSpace();
				if(tmp.isEmpty())tmp = __tr("Unnamed");
				it->m_id = p->insertNormalItem(tmp.ptr(),it->m_icon.ptr(),it->m_buffer.ptr());
			} else {
				KviStr tmp = it->text(0);
				tmp.stripWhiteSpace();
				if(tmp.isEmpty())tmp = __tr("Unnamed");
				KviStr yourName(KviStr::Format,"utoolbar_script_popup_%s",tmp.ptr());
				KviPopupMenu *sub = new KviPopupMenu(0,yourName.ptr());
				saveSubmenu(sub,it);
				p->insertPopupItem(tmp.ptr(),it->m_icon.ptr(),sub);
				it->m_id = 0;
			}
		}
		it = (KviPopupListViewItem *)it->nextSibling();
	}
}

void KviPopupEditor::disablePopupEditors()
{
	m_pPopupEditor->setText(_CHAR_2_QSTRING(""));
	m_pPopupIconEditor->setText(_CHAR_2_QSTRING(""));
	m_pPopupNameEditor->setText(_CHAR_2_QSTRING(""));
	m_pPopupEditor->setEnabled(false);
	m_pPopupIconEditor->setEnabled(false);
	m_pPopupNameEditor->setEnabled(false);
}

void KviPopupEditor::iconPopupActivated(int id)
{
//	if(currentPage())
//		if(kvi_strEqualCSN(((currentPage())->name()),"popupstab", 9)) {
	if((id >= 0)&&(id <= KVI_OUT_NUM_IMAGES) && m_pPopupIconEditor->isEnabled()){
		QString tmp;
		tmp.setNum(id);
		m_pPopupIconEditor->setText(tmp);
		if(m_pEditedPopupItem)savePopupItem(m_pEditedPopupItem);
		return;
	}
//		} else {
//			if((id >= 0)&&(id <= KVI_OUT_NUM_IMAGES) && m_pToolBarIconEditor->isEnabled()) {
//				QString tmp;
//				tmp.setNum(id);
//				m_pToolBarIconEditor->setText(tmp);
//				if(m_pEditedToolButton)saveToolButton(m_pEditedToolButton);
//			}
//		}
}

void KviPopupEditor::appendChildPopup(KviPopupListViewItem * parent,KviPopupMenu *p)
{
	KviPopupListViewItem *it = 0;
	for(KviPopupMenuData * d = p->m_pDataList->first();d;d=p->m_pDataList->next()){
		if(d->menu != 0){
			KviStr tmp(p->text(d->id));
			if(parent){
				if(it){ // Insert after (Qt strangely sometimes inverts the insertion order)
					it = new KviPopupListViewItem(KviPopupListViewItem::Submenu,parent,it,
						tmp.ptr(),"submenu",0,d->image.ptr());
				} else {
					it = new KviPopupListViewItem(KviPopupListViewItem::Submenu,parent,
						tmp.ptr(),"submenu",0,d->image.ptr());
				}
			} else {
				if(it){
					it = new KviPopupListViewItem(KviPopupListViewItem::Submenu,m_pPopupView,it,
						tmp.ptr(),"submenu",0,d->image.ptr());
				} else {
					it = new KviPopupListViewItem(KviPopupListViewItem::Submenu,m_pPopupView,
						tmp.ptr(),"submenu",0,d->image.ptr());
				}
			}
			appendChildPopup(it,d->menu);
		} else {
			if((d->id == 0) && kvi_strEqualCS(d->buffer.ptr()," ")){
				if(parent){
					if(it){
						it = new KviPopupListViewItem(KviPopupListViewItem::Separator,parent,it,
							"----","separator",0,0);
					} else {
						it = new KviPopupListViewItem(KviPopupListViewItem::Separator,parent,
							"----","separator",0,0);
					}
				} else {
					if(it){
						it = new KviPopupListViewItem(KviPopupListViewItem::Separator,m_pPopupView,it,
							"----","separator",0,0);
					} else {
						it = new KviPopupListViewItem(KviPopupListViewItem::Separator,m_pPopupView,
							"----","separator",0,0);
					}
				}
			} else {
				KviStr tmp(p->text(d->id));
				if(parent){
					if(it){
						it = new KviPopupListViewItem(KviPopupListViewItem::Item,parent,it,
							tmp.ptr(),"item",d->buffer.ptr(),d->image.ptr());
					} else {
						it = new KviPopupListViewItem(KviPopupListViewItem::Item,parent,
							tmp.ptr(),"item",d->buffer.ptr(),d->image.ptr());
					}
				} else {
					if(it){
						it = new KviPopupListViewItem(KviPopupListViewItem::Item,m_pPopupView,it,
							tmp.ptr(),"item",d->buffer.ptr(),d->image.ptr());
					} else {
						it = new KviPopupListViewItem(KviPopupListViewItem::Item,m_pPopupView,
							tmp.ptr(),"item",d->buffer.ptr(),d->image.ptr());
					}
				}
			}
		}
	}
}

void KviPopupEditor::popupItemSelected(QListViewItem *i)
{
	if(m_pEditedPopupItem)savePopupItem(m_pEditedPopupItem);

	KviPopupListViewItem *it = (KviPopupListViewItem *)i;
	m_pEditedPopupItem = it;
	if(it == 0){
//		m_pPopupView->clearSelection();		// test -- Qt 2.0
		disablePopupEditors();
		return;
	}

	if(it->m_type == KviPopupListViewItem::Submenu){
		m_pPopupEditor->setText(_CHAR_2_QSTRING(""));
		m_pPopupEditor->setEnabled(false);
		m_pPopupIconEditor->setText(_CHAR_2_QSTRING(it->m_icon.ptr()));
		m_pPopupIconEditor->setEnabled(true);
		m_pPopupNameEditor->setText(it->text(0));
		m_pPopupNameEditor->setEnabled(true);
	} else if(it->m_type == KviPopupListViewItem::Separator){
		m_pPopupEditor->setText(_CHAR_2_QSTRING(""));
		m_pPopupEditor->setEnabled(false);
		m_pPopupIconEditor->setText(_CHAR_2_QSTRING(""));
		m_pPopupIconEditor->setEnabled(false);
		m_pPopupNameEditor->setText(_CHAR_2_QSTRING(""));
		m_pPopupNameEditor->setEnabled(false);
	} else {
		m_pPopupEditor->setText(_CHAR_2_QSTRING(it->m_buffer.ptr()));
		m_pPopupEditor->setEnabled(true);
		m_pPopupIconEditor->setText(_CHAR_2_QSTRING(it->m_icon.ptr()));
		m_pPopupIconEditor->setEnabled(true);
		m_pPopupNameEditor->setText(it->text(0));
		m_pPopupNameEditor->setEnabled(true);
	}
}

void KviPopupEditor::savePopupItem(KviPopupListViewItem * p)
{
	if(m_pEditedPopupItem->m_type == KviPopupListViewItem::Item){
		m_pEditedPopupItem->m_buffer = m_pPopupEditor->text();
		m_pEditedPopupItem->m_icon   = m_pPopupIconEditor->text();
		QString s = m_pPopupNameEditor->text();
		s.stripWhiteSpace();
		if(s.isEmpty())s = _CHAR_2_QSTRING(__tr("Unnamed"));
		m_pEditedPopupItem->setText(0,s);
		m_pEditedPopupItem->tryLoadImage();
	} else if(m_pEditedPopupItem->m_type == KviPopupListViewItem::Submenu){
		m_pEditedPopupItem->m_icon   = m_pPopupIconEditor->text();
		QString s = m_pPopupNameEditor->text();
		s.stripWhiteSpace();
		if(s.isEmpty())s = _CHAR_2_QSTRING(__tr("Unnamed"));
		m_pEditedPopupItem->setText(0,s);
		m_pEditedPopupItem->tryLoadImage();
	}
}

void KviPopupEditor::popupViewRightButtonPressed(QListViewItem *i,const QPoint &p,int)
{
	if(i)m_pPopupView->setSelected(i,true);
	else {
		m_pPopupView->clearSelection();		// test -- Qt 2.0
		if(m_pEditedPopupItem)savePopupItem(m_pEditedPopupItem);
	}
	m_pEditedPopupItem = (KviPopupListViewItem *)i;

	if(m_pEditedPopupItem){
		m_pPopupPopup->clear();
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("&Remove Item")),this,SLOT(removeEditedPopupItem()));
		m_pPopupPopup->insertSeparator();
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("&Add Item Below")),this,SLOT(addEmptyPopupItem()));
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add &Submenu Below")),this,SLOT(addSubmenuItem()));
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add S&eparator Below")),this,SLOT(addSeparatorItem()));
		if(m_pEditedPopupItem->parent() == m_pEditedPopupItem->itemAbove()){
			m_pPopupPopup->insertSeparator();
			m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("A&dd Item Above")),this,SLOT(addEmptyPopupItemAbove()));
			m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add S&ubmenu Above")),this,SLOT(addSubmenuItemAbove()));
			m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add Separat&or Above")),this,SLOT(addSeparatorItemAbove()));
		}
		if(m_pEditedPopupItem->m_type == KviPopupListViewItem::Submenu){
			m_pPopupPopup->insertSeparator();
			m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add I&tem Inside")),this,SLOT(addEmptyPopupItemInside()));
			m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add Su&bmenu Inside")),this,SLOT(addSubmenuItemInside()));
		}// else if(m_pEditedPopupItem->m_type == KviPopupListViewItem::Item){
		m_pPopupPopup->insertSeparator();
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("&Copy Item")),this,SLOT(copyItem()));
		//}
		if(m_pClipboardItem){
			m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("&Paste Item Below")),this,SLOT(pasteItemBelow()));
			if(m_pEditedPopupItem->parent() == m_pEditedPopupItem->itemAbove())
				m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Paste &Item Above")),this,SLOT(pasteItemAbove()));
			if(m_pEditedPopupItem->m_type == KviPopupListViewItem::Submenu)
				m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Paste Item I&nside")),this,SLOT(pasteItemInside()));
		}
	} else {
		// At the end ?
		disablePopupEditors();
		m_pPopupPopup->clear();
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("&Add Item")),this,SLOT(addEmptyPopupItem()));
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add &Submenu")),this,SLOT(addSubmenuItem()));
		m_pPopupPopup->insertItem(_CHAR_2_QSTRING(__tr("Add S&eparator")),this,SLOT(addSeparatorItem()));
	}
	m_pPopupPopup->popup(p);
}

void KviPopupEditor::removeEditedPopupItem()
{
	if(!m_pEditedPopupItem)return;
	QListViewItem * p = m_pEditedPopupItem->parent();
	KviPopupListViewItem * tmp = m_pEditedPopupItem;
	// Needs to be 0 before the delete (the QListViewItem destructor
	// triggers an update in the QListView that calls selectionChanged that
	// attempts to save the popup while it is being deleted!
	m_pEditedPopupItem = 0; 
	delete tmp;
	if(p)m_pPopupView->setSelected(p,true);
    else {
		if(m_pPopupView->firstChild())m_pPopupView->setSelected(m_pPopupView->firstChild(),true);
		else disablePopupEditors();
	}
//	updateToolBar(m_pEditedToolBar, 0);
}

void KviPopupEditor::copyItem()
{
	if(!m_pEditedPopupItem)return;
	destroyClipboard();
	m_pClipboardItem = m_pEditedPopupItem->getClipboardItem();
//	if(m_pEditedPopupItem->m_type != KviPopupListViewItem::Item)return;
//	m_szClipboardItemBuffer = m_pEditedPopupItem->m_buffer;
//	m_szClipboardItemIcon   = m_pEditedPopupItem->m_icon;
//	m_szClipboardItemName   = m_pEditedPopupItem->text(0);
}

void KviPopupEditor::addEmptyPopupItem()
{
	KviPopupListViewItem *it;
	if(m_pEditedPopupItem){
		if(m_pEditedPopupItem->parent()){
			it = new KviPopupListViewItem(KviPopupListViewItem::Item,((KviPopupListViewItem *)m_pEditedPopupItem->parent()),
				m_pEditedPopupItem,"unnamed","item",__tr("# Enter the script for the new item here"),"");
		} else {
			it = new KviPopupListViewItem(KviPopupListViewItem::Item,m_pPopupView,
				m_pEditedPopupItem,"unnamed","item",__tr("# Enter the script for the new item here"),"");
		}
	} else {
		it = new KviPopupListViewItem(KviPopupListViewItem::Item,m_pPopupView,
			"unnamed","item",__tr("# Enter the script for the new item here"),"");
	}
	m_pPopupView->setSelected(it,true);
}

KviPopupListViewItem * KviPopupEditor::pasteClipboard(KviPopupListViewItem * par,KviPopupListViewItem * aft,KviPopupClipboardItem * it,bool bNullAfter)
{
	if(!it)return 0;
	KviPopupListViewItem *i;
	const char * type = 0;
	static const char * types[4] = 
	{
		"item","submenu","separator","???"
	};

	switch(it->type)
	{
		case KviPopupListViewItem::Item: type = types[0]; break;
		case KviPopupListViewItem::Submenu: type = types[1]; break;
		case KviPopupListViewItem::Separator: type = types[2]; break;
		default: type = types[3]; break;
	}
	if(par)
	{
		if(aft || bNullAfter)
		{
			i = new KviPopupListViewItem(it->type,par,aft,it->szName.ptr(),type,it->szBuffer.ptr(),it->szIcon.ptr());
		} else {
			i = new KviPopupListViewItem(it->type,par,it->szName.ptr(),type,it->szBuffer.ptr(),it->szIcon.ptr());
		}
	} else {
		if(aft || bNullAfter)
		{
			i = new KviPopupListViewItem(it->type,m_pPopupView,aft,it->szName.ptr(),type,it->szBuffer.ptr(),it->szIcon.ptr());
		} else {
			i = new KviPopupListViewItem(it->type,m_pPopupView,it->szName.ptr(),type,it->szBuffer.ptr(),it->szIcon.ptr());
		}
	}
	KviPopupListViewItem * last = 0;
	if(it->pItemList)
	{
		for(KviPopupClipboardItem * c = it->pItemList->first();c;c = it->pItemList->next())
		{
			last = pasteClipboard(i,last,c);
		}
	}
//	m_pPopupView->setSelected(i,true);
	return i;
}

void KviPopupEditor::pasteItemBelow()
{
	KviPopupListViewItem * it;
	if(m_pEditedPopupItem)
	{
		if(m_pEditedPopupItem->parent())it = pasteClipboard((KviPopupListViewItem *)m_pEditedPopupItem->parent(),m_pEditedPopupItem,m_pClipboardItem);
		else it = pasteClipboard(0,m_pEditedPopupItem,m_pClipboardItem);

	} else it = pasteClipboard(0,0,m_pClipboardItem);

	if(it)m_pPopupView->setSelected(it,true);
}

void KviPopupEditor::pasteItemInside()
{
	if(!m_pEditedPopupItem)return;
	if(m_pEditedPopupItem->m_type != KviPopupListViewItem::Submenu)return;

	KviPopupListViewItem * it = pasteClipboard(m_pEditedPopupItem,0,m_pClipboardItem);

	if(it)m_pPopupView->setSelected(it,true);

}

void KviPopupEditor::pasteItemAbove()
{
	KviPopupListViewItem *it;
	if(!m_pEditedPopupItem)return;
	if(m_pEditedPopupItem->parent() != m_pEditedPopupItem->itemAbove())return;

//	if(m_szClipboardItemName.isEmpty())return;
	if(m_pEditedPopupItem->parent())it = pasteClipboard((KviPopupListViewItem *)m_pEditedPopupItem->parent(),0,m_pClipboardItem,true);
	else it = pasteClipboard(0,0,m_pClipboardItem,true);

	if(it)m_pPopupView->setSelected(it,true);
}

void KviPopupEditor::addEmptyPopupItemAbove()
{
	KviPopupListViewItem *it;
	if(!m_pEditedPopupItem)return;
	if(m_pEditedPopupItem->parent() != m_pEditedPopupItem->itemAbove())return;
	if(m_pEditedPopupItem->parent()){
		it = new KviPopupListViewItem(KviPopupListViewItem::Item,((KviPopupListViewItem *)m_pEditedPopupItem->parent()),
			0,"unnamed","item",__tr("# Enter the script for the new item here"),"");

	} else {
		it = new KviPopupListViewItem(KviPopupListViewItem::Item,m_pPopupView,
			0,"unnamed","item",__tr("# Enter the script for the new item here"),"");
	}
	m_pPopupView->setSelected(it,true);
}

void KviPopupEditor::addSubmenuItemAbove()
{
	KviPopupListViewItem *sb;
	if(!m_pEditedPopupItem)return;
	if(m_pEditedPopupItem->parent() != m_pEditedPopupItem->itemAbove())return;
	if(m_pEditedPopupItem->parent()){
		sb = new KviPopupListViewItem(KviPopupListViewItem::Submenu,((KviPopupListViewItem *)m_pEditedPopupItem->parent()),
			0,"unnamed","submenu","","");
	} else {
		sb = new KviPopupListViewItem(KviPopupListViewItem::Submenu,m_pPopupView,
			0,"unnamed","submenu","","");
	}

	KviPopupListViewItem * it = new KviPopupListViewItem(KviPopupListViewItem::Item,sb,
		"unnamed","item",__tr("# Enter the script for the new item here"),"");
	sb->setOpen(true);

	m_pPopupView->setSelected(it,true);
}

void KviPopupEditor::addSeparatorItemAbove()
{
	KviPopupListViewItem * it;
	if(!m_pEditedPopupItem)return;
	if(m_pEditedPopupItem->parent() != m_pEditedPopupItem->itemAbove())return;
	if(m_pEditedPopupItem->parent()){
		it = new KviPopupListViewItem(KviPopupListViewItem::Separator,((KviPopupListViewItem *)m_pEditedPopupItem->parent()),
			0,"----","separator","","");
	} else {
		it = new KviPopupListViewItem(KviPopupListViewItem::Separator,m_pPopupView,
			0,"----","separator","","");
	}

	m_pPopupView->setSelected(it,true);
}

void KviPopupEditor::addEmptyPopupItemInside()
{
	if(!m_pEditedPopupItem)return;
	if(m_pEditedPopupItem->m_type != KviPopupListViewItem::Submenu)return;
	KviPopupListViewItem *it = new KviPopupListViewItem(KviPopupListViewItem::Item,m_pEditedPopupItem,
		"unnamed","item",__tr("# Enter the script for the new item here"),"");
	m_pPopupView->setSelected(it,true);
}

void KviPopupEditor::addSubmenuItemInside()
{
	if(!m_pEditedPopupItem)return;
	KviPopupListViewItem *sb = new KviPopupListViewItem(KviPopupListViewItem::Submenu,m_pEditedPopupItem,
		"unnamed","submenu","","");
	KviPopupListViewItem * it = new KviPopupListViewItem(KviPopupListViewItem::Item,sb,
		"unnamed","item",__tr("# Enter the script for the new item here"),"");
	sb->setOpen(true);
	m_pPopupView->setSelected(it,true);
}

void KviPopupEditor::addSubmenuItem()
{
	KviPopupListViewItem *sb;
	if(m_pEditedPopupItem){
		if(m_pEditedPopupItem->parent()){
			sb = new KviPopupListViewItem(KviPopupListViewItem::Submenu,((KviPopupListViewItem *)m_pEditedPopupItem->parent()),
				m_pEditedPopupItem,"unnamed","submenu","","");
		} else {
			sb = new KviPopupListViewItem(KviPopupListViewItem::Submenu,m_pPopupView,
				m_pEditedPopupItem,"unnamed","submenu","","");
		}
	} else {
		sb = new KviPopupListViewItem(KviPopupListViewItem::Submenu,m_pPopupView,
			"unnamed","submenu","","");
	}

	KviPopupListViewItem * it = new KviPopupListViewItem(KviPopupListViewItem::Item,sb,
		"unnamed","item",__tr("# Enter the script for the new item here"),"");
	sb->setOpen(true);

	m_pPopupView->setSelected(it,true);
}

void KviPopupEditor::addSeparatorItem()
{
	KviPopupListViewItem * it;
	if(m_pEditedPopupItem){
		if(m_pEditedPopupItem->parent()){
			it = new KviPopupListViewItem(KviPopupListViewItem::Separator,((KviPopupListViewItem *)m_pEditedPopupItem->parent()),
				m_pEditedPopupItem,"----","separator","","");
		} else {
			it = new KviPopupListViewItem(KviPopupListViewItem::Separator,m_pPopupView,
				m_pEditedPopupItem,"----","separator","","");
		}
	} else {
		it = new KviPopupListViewItem(KviPopupListViewItem::Separator,m_pPopupView,
			"separator","separator","","");
	}

	m_pPopupView->setSelected(it,true);
}


/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////
////
//// TOOLBAR EDITOR
////
//// This widget allows editing of the user toolbar
////
/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@quickhelp: QWidget_ToolBarTab
	@widget: Toolbar editor
		Here you can edit your toolbar.<br>
*/


//==================== Constructor: Add ======================

KviToolBarListViewItem::KviToolBarListViewItem(ToolButtonType t,QListView * parent,const char *name,const char *type,const char *buffer,const char *icon, const char *sbar)
:QListViewItem(parent,_CHAR_2_QSTRING(name),_CHAR_2_QSTRING(type))
{
	if(buffer)m_buffer = buffer;
	if(icon)m_icon = icon;
	m_sbar = sbar;
	m_id = 0;
	m_type = t;
	tryLoadImage();
}

//==================== Constructor: Add below =================
KviToolBarListViewItem::KviToolBarListViewItem(ToolButtonType t,QListView * parent,KviToolBarListViewItem * after,const char *name,const char *type,const char *buffer,const char *icon, const char *sbar)
:QListViewItem(parent,after,_CHAR_2_QSTRING(name),_CHAR_2_QSTRING(type))
{
	if(buffer)m_buffer = buffer;
	if(icon)m_icon = icon;
	m_sbar = sbar;
	m_type = t;
	m_id = 0;
	tryLoadImage();
}

//==================== Constructor: Add menu =================
KviToolBarListViewItem::KviToolBarListViewItem(ToolButtonType t,KviToolBarListViewItem * parent,const char * name,const char *type,const char *buffer,const char *icon, const char *sbar)
:QListViewItem(parent,_CHAR_2_QSTRING(name),_CHAR_2_QSTRING(type))
{
	if(buffer)m_buffer = buffer;
	if(icon)m_icon = icon;
	m_sbar = sbar;
	m_type = t;
	m_id = 0;
	tryLoadImage();
}

//==================== Constructor: Add menu below ============
KviToolBarListViewItem::KviToolBarListViewItem(ToolButtonType t,KviToolBarListViewItem * parent,KviToolBarListViewItem * after,const char * name,const char *type,const char *buffer,const char *icon, const char *sbar)
:QListViewItem(parent,after,_CHAR_2_QSTRING(name),_CHAR_2_QSTRING(type))
{
	if(buffer)m_buffer = buffer;
	if(icon)m_icon = icon;
	m_sbar = sbar;
	m_type = t;
	m_id = 0;
	tryLoadImage();
}

//==================== Destructor =================

KviToolBarListViewItem::~KviToolBarListViewItem()
{
}

//==================== tryLoadImage =================
void KviToolBarListViewItem::tryLoadImage()
{
	m_icon.stripWhiteSpace();
	if(m_icon.hasData()){
		if(m_icon.isNum()){
			bool bOk=false;
			int imId = m_icon.toInt(&bOk);
			if(imId >= 0){
				imId = imId % KVI_OUT_NUM_IMAGES;
				setPixmap(0,*(g_pixViewOut[imId]));
			}
		} else {
			KviStr imagePath;
			if(g_pApp->findImage(imagePath,m_icon.ptr())){
				QPixmap pix(imagePath.ptr());
				if(!pix.isNull()){
					setPixmap(0,pix);
				} else debug("Unable to find image file %s",m_icon.ptr());
			}
		}
	}
}


KviUToolBarEditor::KviUToolBarEditor(QWidget * parent,const char *name)
: QWidget(parent,name)
{
	m_pEditedToolButton = 0;

	QGridLayout * g = new QGridLayout(this,5,3,4,2);

	m_pToolBar = new KviUserToolBar((KviFrame *)(g_pApp->m_pFrameList->first()), this);
	m_pToolBar->copyFromTemplate(g_pUserToolBarTemplate,false);

	m_pToolBar->setStretchableWidget(new KviTransparentWidget(m_pToolBar));
	g->addMultiCellWidget(m_pToolBar,0,0,0,1);
	
	QPushButton * b = new QPushButton(_CHAR_2_QSTRING(__tr("&Clear Toolbar")),this);
	g->addWidget(b,0,2);
	connect(b,SIGNAL(clicked()),this,SLOT(clearToolBar()));

	QSplitter * splitter = new QSplitter(Qt::Vertical,this);

	m_pToolBarView = new QListView(splitter);
	m_pToolBarView->setFrameStyle(QFrame::WinPanel|QFrame::Sunken);
	m_pToolBarView->addColumn(_CHAR_2_QSTRING(__tr("Item")));
	m_pToolBarView->addColumn(_CHAR_2_QSTRING(__tr("Type")));
	m_pToolBarView->setRootIsDecorated(true);
	m_pToolBarView->setMultiSelection(false);
	m_pToolBarView->header()->setClickEnabled(false);
	m_pToolBarView->setAllColumnsShowFocus(true);
	m_pToolBarView->viewport()->installEventFilter(this); // Need to adjust the behaviour of mousePressEvent

	m_pToolBarPopup = new QPopupMenu(this);

	connect(m_pToolBarView,SIGNAL(selectionChanged(QListViewItem *)),this,SLOT(toolButtonSelected(QListViewItem *)));
	connect(m_pToolBarView,SIGNAL(rightButtonPressed(QListViewItem *,const QPoint &,int)),
			this,SLOT(toolBarViewRightButtonPressed(QListViewItem *,const QPoint &,int)));

	m_pToolBarEditor = new KviScriptEditor(splitter);
	g->addMultiCellWidget(splitter,1,1,0,2);

	m_pIconPopup = new QPopupMenu(this);
	QPopupMenu *ppp=m_pIconPopup;
	connect(ppp,SIGNAL(activated(int)),this,SLOT(iconPopupActivated(int)));
	for(int i=0;i<KVI_OUT_NUM_IMAGES;i++){
		ppp->insertItem(*(g_pixViewOut[i]),i);
		if(((i % 15) == 0)&&(i != 0)){
			QPopupMenu * sub = new QPopupMenu(ppp);
			ppp->insertItem(__tr("More..."),sub);
			ppp = sub;
			connect(ppp,SIGNAL(activated(int)),this,SLOT(iconPopupActivated(int)));
		}
	}


	QLabel * l=new QLabel(_CHAR_2_QSTRING("Icon:"),this);
	g->addWidget(l,2,0);

	m_pToolBarIconEditor = new QLineEdit(this);
	m_pToolBarIconEditor->installEventFilter(this);

	g->addMultiCellWidget(m_pToolBarIconEditor,2,2,1,2);

	l = new QLabel(_CHAR_2_QSTRING("Name:"),this);
	g->addWidget(l,3,0);

	m_pToolBarNameEditor = new QLineEdit(this);
	g->addMultiCellWidget(m_pToolBarNameEditor,3,3,1,2);

	l=new QLabel(_CHAR_2_QSTRING("Statusbar text:"),this);
	g->addWidget(l,4,0);

	m_pToolBarStatusBarEditor = new QLineEdit(this);
	g->addMultiCellWidget(m_pToolBarStatusBarEditor,4,4,1,2);


	g->setColStretch(1,10);

	startEditToolBar(m_pToolBar);
}

KviUToolBarEditor::~KviUToolBarEditor()
{
	delete m_pToolBar;
}

void KviUToolBarEditor::commit()
{
	if(m_pEditedToolBar)saveToolBar(m_pEditedToolBar);
	g_pUserToolBarTemplate->copyFromToolBar(m_pEditedToolBar);
}

bool KviUToolBarEditor::eventFilter(QObject *o,QEvent *e)
{
	if(o == m_pToolBarIconEditor){
		if(e->type() == QEvent::MouseButtonPress){
			if(((QMouseEvent *)e)->button() & RightButton){
				m_pIconPopup->popup(QCursor::pos());
				return true;
			}
		}
		return false;
	}

	if(e->type() == QEvent::MouseButtonPress){
		if(o == m_pToolBarView->viewport()){
			if(m_pToolBarView->firstChild() == 0){
				toolBarViewRightButtonPressed(0,m_pToolBarView->mapToGlobal(((QMouseEvent *)e)->pos()),0);
				return true;
			}
		}
	}
	return QWidget::eventFilter(o,e);
}

void KviUToolBarEditor::iconPopupActivated(int id)
{
	if((id >= 0)&&(id <= KVI_OUT_NUM_IMAGES) && m_pToolBarIconEditor->isEnabled()) {
		QString tmp;
		tmp.setNum(id);
		m_pToolBarIconEditor->setText(tmp);
		if(m_pEditedToolButton)saveToolButton(m_pEditedToolButton);
	}
}


//==================== startEditToolBar =================
void KviUToolBarEditor::startEditToolBar(KviUserToolBar *p)
{
	if(m_pEditedToolBar)saveToolBar(m_pEditedToolBar);
	m_pEditedToolBar = p;
	m_pToolBarView->clear();
	m_pToolBarView->setSorting(-1);
	appendToolButton(p);
//	m_pToolBarView->setSorting(-1);
	m_pEditedToolButton = 0;
	if(m_pToolBarView->firstChild() == 0)disableToolBarEditors();
	else m_pToolBarView->setSelected(m_pToolBarView->firstChild(),true);
}

//==================== clearToolBar =================
void KviUToolBarEditor::clearToolBar()
{
	if(QMessageBox::warning(this,__tr("Clear User Toolbar"),
		__tr("Are you sure you want to clear the user toolbar?"),
		QMessageBox::Yes | QMessageBox::Default,
		QMessageBox::No | QMessageBox::Escape) == QMessageBox::Yes)
	{
		m_pToolBarView->clear();
		m_pToolBarView->setSorting(-1);
	}
	m_pEditedToolButton = 0;
	disableToolBarEditors();
	updateToolBar(m_pEditedToolBar, 0);
}

//==================== selectToolButtonById =================
bool KviUToolBarEditor::selectToolButtonById(KviToolBarListViewItem * par,int id)
{
	KviToolBarListViewItem *it = (KviToolBarListViewItem *)(par ? par->firstChild() : m_pToolBarView->firstChild());
	while(it){
		if(it->m_id == id){
			// open all parents
			par = (KviToolBarListViewItem *)it->parent();
			while(par){
				par->setOpen(true);
				par = (KviToolBarListViewItem *)par->parent();
			}
			// select it
			m_pToolBarView->setSelected(it,true);
			// and ensure it is visible
			m_pToolBarView->ensureItemVisible(it);
			return true;
		}
		it = (KviToolBarListViewItem *)it->nextSibling();
	}
	return false;
}

//==================== saveToolBar =================
void KviUToolBarEditor::saveToolBar(KviUserToolBar *p)
{
	if(m_pEditedToolButton)saveToolButton(m_pEditedToolButton); // do not forget to save the last one
}

//==================== saveToolButton =================
void KviUToolBarEditor::saveToolButton(KviToolBarListViewItem * p)
{
//	debug("saveToolButton: ");
	if(m_pEditedToolButton->m_type == KviToolBarListViewItem::Button || 
			m_pEditedToolButton->m_type == KviToolBarListViewItem::Menu ||
			m_pEditedToolButton->m_type == KviToolBarListViewItem::Item) {
		m_pEditedToolButton->m_buffer = m_pToolBarEditor->text();
		m_pEditedToolButton->m_icon   = m_pToolBarIconEditor->text();
		m_pEditedToolButton->m_sbar   = m_pToolBarStatusBarEditor->text();
		QString s = m_pToolBarNameEditor->text();
		s.stripWhiteSpace();
		if(s.isEmpty())s = _CHAR_2_QSTRING(__tr("Unnamed"));
//		debug("    %s",s.ascii());
		m_pEditedToolButton->setText(0,s);
		m_pEditedToolButton->tryLoadImage();
	} else if(m_pEditedToolButton->m_type == KviToolBarListViewItem::Submenu) {
		m_pEditedToolButton->m_icon   = m_pToolBarIconEditor->text();
		QString s = m_pToolBarNameEditor->text();
		s.stripWhiteSpace();
		if(s.isEmpty())s = _CHAR_2_QSTRING(__tr("Unnamed"));
		m_pEditedToolButton->setText(0,s);
		m_pEditedToolButton->tryLoadImage();
	}
		updateToolBar(m_pEditedToolBar, 0);
}

void KviUToolBarEditor::saveToolBarMenu(KviPopupMenu *p,KviToolBarListViewItem *par)
{
	KviToolBarListViewItem *it = (KviToolBarListViewItem *)(par ? par->firstChild() : m_pToolBarView->firstChild());
	while(it){
		if(it->m_type == KviToolBarListViewItem::Separator){
			p->insertSeparatorItem();
			it->m_id = 0;
		} else {
			if(it->m_type == KviToolBarListViewItem::Item){
//				debug("saving item: %s",it->text(0).ascii());
				KviStr tmp = it->text(0);
				tmp.stripWhiteSpace();
				if(tmp.isEmpty())tmp = __tr("Unnamed");
				it->m_id = p->insertNormalItem(tmp.ptr(),it->m_icon.ptr(),it->m_buffer.ptr());
			} else if(it->m_type == KviToolBarListViewItem::Menu) {
//				debug("saving menu: %s",it->text(0).ascii());
				KviStr tmp = it->text(0);
				tmp.stripWhiteSpace();
				if(tmp.isEmpty())tmp = __tr("Unnamed");
				KviStr yourName(KviStr::Format,"utoolbar_script_popup_%s",tmp.ptr());

				KviPopupMenu *sub = new KviPopupMenu(0,yourName.ptr());
				saveToolBarMenu(sub,it);

				p->insertPopupItem(tmp.ptr(),it->m_icon.ptr(),sub);
				it->m_id = 0;
			}
		}
		it = (KviToolBarListViewItem *)it->nextSibling();
	}
}

//==================== updateToolBar =================
void KviUToolBarEditor::updateToolBar(KviUserToolBar *p,KviToolBarListViewItem *par)
{
	p->setUpdatesEnabled(false);
	p->clearAll();
	KviToolBarListViewItem *it = (KviToolBarListViewItem *)(par ? par->firstChild() : m_pToolBarView->firstChild());
	while(it){
		if(it->m_type == KviToolBarListViewItem::Separator){
			p->insertSeparatorItem();
			it->m_id = 0;
		} else
		if(it->m_type == KviToolBarListViewItem::Button){
			KviStr tmp = it->text(0);
			tmp.stripWhiteSpace();
			if(tmp.isEmpty())tmp = __tr("Unnamed");
//			it->m_id = p->insertNormalItem(tmp.ptr(),it->m_sbar.ptr(),it->m_icon.ptr(),it->m_buffer.ptr());
			p->insertNormalItem(tmp.ptr(),it->m_sbar.ptr(),it->m_icon.ptr(),it->m_buffer.ptr());
		} else
		if(it->m_type == KviToolBarListViewItem::Menu){

			KviStr tmp = it->text(0);
			tmp.stripWhiteSpace();
			if(tmp.isEmpty())tmp = __tr("Unnamed");
			KviStr yourName(KviStr::Format,"utoolbar_script_popup_%s",tmp.ptr());
			KviPopupMenu *m = new KviPopupMenu(0,yourName.ptr());
			saveToolBarMenu(m,it);

			p->insertNormalItem(tmp.ptr(),it->m_sbar.ptr(),it->m_icon.ptr(),it->m_buffer.ptr(),m);
		}
		it = (KviToolBarListViewItem *)it->nextSibling();
	}
	QWidget *w = new QWidget(p,"bar");
	p->setStretchableWidget(w);
	p->setUpdatesEnabled(true);
//	p->hide();
//	p->show();
	p->update();
}


//==================== disableToolBarEditors =================
void KviUToolBarEditor::disableToolBarEditors()
{
	m_pToolBarEditor->setText(_CHAR_2_QSTRING(""));
	m_pToolBarIconEditor->setText(_CHAR_2_QSTRING(""));
	m_pToolBarNameEditor->setText(_CHAR_2_QSTRING(""));
	m_pToolBarStatusBarEditor->setText(_CHAR_2_QSTRING(""));
	m_pToolBarEditor->setEnabled(false);
	m_pToolBarIconEditor->setEnabled(false);
	m_pToolBarNameEditor->setEnabled(false);
	m_pToolBarStatusBarEditor->setEnabled(false);
}

//==================== appendToolButton ========================
void KviUToolBarEditor::appendToolButton(KviUserToolBar *p)
{
	KviToolBarListViewItem *it = 0;
	for(KviUserToolBarData * d = p->m_pDataList->first();d;d=p->m_pDataList->next()){
		if(it){
			switch (d->type) {
				case KviUserToolBar::Separator:
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,m_pToolBarView,it,
						"----","separator",0,0);
				break;
				case KviUserToolBar::Button:
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,m_pToolBarView,it,
						d->text.ptr(),"button",d->buffer.ptr(),d->image.ptr(),d->sbar.ptr());
				break;
				case KviUserToolBar::Menu:
					{
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,m_pToolBarView,it,
						d->text.ptr(),"menu",d->buffer.ptr(),d->image.ptr(),d->sbar.ptr());
					appendToolBarMenu(it, d->menu);
					}
				break;
			}
		} else {
			switch (d->type) {
				case KviUserToolBar::Separator:
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,m_pToolBarView,
						"----","separator",0,0);
				break;
				case KviUserToolBar::Button:
					{
					KviStr tmp(d->text);
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,m_pToolBarView,
						tmp.ptr(),"button",d->buffer.ptr(),d->image.ptr(),d->sbar.ptr());
					}
				break;
				case KviUserToolBar::Menu:
					{
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,m_pToolBarView,
						d->text.ptr(),"menu",d->buffer.ptr(),d->image.ptr(),d->sbar.ptr());
					appendToolBarMenu(it, d->menu);
					}
				break;
			}
		}
	}
}

void KviUToolBarEditor::appendToolBarMenu(KviToolBarListViewItem *parent, KviPopupMenu *p)
{
	KviToolBarListViewItem *it = 0;
	for(KviPopupMenuData * d = p->m_pDataList->first();d;d=p->m_pDataList->next()){
		if(d->menu != 0){
			KviStr tmp(p->text(d->id));
				if(it){ // Insert after (Qt strangely sometimes inverts the insertion order)
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,parent,it,
						tmp.ptr(),"submenu",0,d->image.ptr());
				} else {
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,parent,
						tmp.ptr(),"submenu",0,d->image.ptr());
				}
			appendToolBarMenu(it,d->menu);
		} else {
			if((d->id == 0) && kvi_strEqualCS(d->buffer.ptr()," ")){
				if(it){
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,parent,it,
						"----","separator",0,0);
				} else {
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,parent,
						"----","separator",0,0);
				}
			} else {
				KviStr tmp(p->text(d->id));
				if(it){
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,parent,it,
						tmp.ptr(),"item",d->buffer.ptr(),d->image.ptr());
				} else {
					it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,parent,
						tmp.ptr(),"item",d->buffer.ptr(),d->image.ptr());
				}
			}
		}
	}
}


//==================== toolButtonSelected ========================
void KviUToolBarEditor::toolButtonSelected(QListViewItem *i)
{
	if(m_pEditedToolButton)saveToolButton(m_pEditedToolButton);

	KviToolBarListViewItem *it = (KviToolBarListViewItem *)i;
	m_pEditedToolButton = it;
	if(it == 0){
//		m_pToolBarView->clearSelection();		// test -- Qt 2.0
		disableToolBarEditors();
		return;
	}

	switch (it->m_type) {
		case KviToolBarListViewItem::Separator:
			m_pToolBarEditor->setText(_CHAR_2_QSTRING(""));
			m_pToolBarEditor->setEnabled(false);
			m_pToolBarIconEditor->setText(_CHAR_2_QSTRING(""));
			m_pToolBarIconEditor->setEnabled(false);
			m_pToolBarNameEditor->setText(_CHAR_2_QSTRING(""));
			m_pToolBarNameEditor->setEnabled(false);
			m_pToolBarStatusBarEditor->setText(_CHAR_2_QSTRING(""));
			m_pToolBarStatusBarEditor->setEnabled(false);
		break;					
		case KviToolBarListViewItem::Menu:
			m_pToolBarEditor->setText(_CHAR_2_QSTRING(""));
			m_pToolBarEditor->setEnabled(false);
			m_pToolBarIconEditor->setText(_CHAR_2_QSTRING(it->m_icon.ptr()));
			m_pToolBarIconEditor->setEnabled(true);
			m_pToolBarNameEditor->setText(it->text(0));
			m_pToolBarNameEditor->setEnabled(true);
			m_pToolBarStatusBarEditor->setText(_CHAR_2_QSTRING(it->m_sbar.ptr()));
			m_pToolBarStatusBarEditor->setEnabled(true);
		break;					
		case KviToolBarListViewItem::Item:
			m_pToolBarEditor->setText(_CHAR_2_QSTRING(it->m_buffer.ptr()));
			m_pToolBarEditor->setEnabled(true);
			m_pToolBarIconEditor->setText(_CHAR_2_QSTRING(it->m_icon.ptr()));
			m_pToolBarIconEditor->setEnabled(true);
			m_pToolBarNameEditor->setText(it->text(0));
			m_pToolBarNameEditor->setEnabled(true);
			m_pToolBarStatusBarEditor->setText(_CHAR_2_QSTRING(""));
			m_pToolBarStatusBarEditor->setEnabled(false);
		break;
		default:
			m_pToolBarEditor->setText(_CHAR_2_QSTRING(it->m_buffer.ptr()));
			m_pToolBarEditor->setEnabled(true);
			m_pToolBarIconEditor->setText(_CHAR_2_QSTRING(it->m_icon.ptr()));
			m_pToolBarIconEditor->setEnabled(true);
			m_pToolBarNameEditor->setText(it->text(0));
			m_pToolBarNameEditor->setEnabled(true);
			m_pToolBarStatusBarEditor->setText(_CHAR_2_QSTRING(it->m_sbar.ptr()));
			m_pToolBarStatusBarEditor->setEnabled(true);
		break;
	}
}

//==================== toolBarViewRightButtonPressed ========================
void KviUToolBarEditor::toolBarViewRightButtonPressed(QListViewItem *i,const QPoint &p,int)
{
	if(i)m_pToolBarView->setSelected(i,true);
	else {
		m_pToolBarView->clearSelection();		// test -- Qt 2.0
		if(m_pEditedToolButton)saveToolButton(m_pEditedToolButton);
	}
	m_pEditedToolButton = (KviToolBarListViewItem *)i;

	if(m_pEditedToolButton){
		m_pToolBarPopup->clear();
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(__tr("&Remove Item")),this,SLOT(removeEditedToolButton()));
		m_pToolBarPopup->insertSeparator();
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(__tr("&Add Item Below")),this,SLOT(addEmptyToolButton()));
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(__tr("Add &Submenu Below")),this,SLOT(addToolBarMenu()));
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(__tr("Add S&eparator Below")),this,SLOT(addToolBarSeparatorItem()));
		if(m_pEditedToolButton->parent() == m_pEditedToolButton->itemAbove()){
			m_pToolBarPopup->insertSeparator();
			m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(__tr("A&dd Item Above")),this,SLOT(addEmptyToolButtonAbove()));
			m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(__tr("Add S&ubmenu Above")),this,SLOT(addToolBarMenuAbove()));
			m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(__tr("Add Separat&or Above")),this,SLOT(addToolBarSeparatorItemAbove()));
		}
		if(m_pEditedToolButton->m_type == KviToolBarListViewItem::Button || m_pEditedToolButton->m_type == KviToolBarListViewItem::Item){
			m_pToolBarPopup->insertSeparator();
			m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(__tr("&Copy Item")),this,SLOT(copyToolButton()));
		}
		if(m_szClipboardItemName.hasData()){
			m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(__tr("&Paste Item Below")),this,SLOT(pasteToolButtonBelow()));
			if(m_pEditedToolButton->parent() == m_pEditedToolButton->itemAbove())
				m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(__tr("Paste &Item Above")),this,SLOT(pasteToolButtonAbove()));
		}
	} else {
		// At the end ?
		disableToolBarEditors();
		m_pToolBarPopup->clear();
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(__tr("&Add Item")),this,SLOT(addEmptyToolButton()));
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(__tr("Add &Submenu")),this,SLOT(addToolBarMenu()));
		m_pToolBarPopup->insertItem(_CHAR_2_QSTRING(__tr("Add S&eparator")),this,SLOT(addToolBarSeparatorItem()));
	}
	m_pToolBarPopup->popup(p);
}

//==================== removeEditedToolButton ========================
void KviUToolBarEditor::removeEditedToolButton()
{
	if(!m_pEditedToolButton)return;
	QListViewItem * p = m_pEditedToolButton->parent();
	KviToolBarListViewItem * tmp = m_pEditedToolButton;
	// Needs to be 0 before the delete (the QListViewItem destructor
	// triggers an update in the QListView that calls selectionChanged that
	// attempts to save the popup while it is being deleted!
	m_pEditedToolButton = 0;
	delete tmp;
	if(p)m_pToolBarView->setSelected(p,true);
    else {
		if(m_pToolBarView->firstChild())m_pToolBarView->setSelected(m_pToolBarView->firstChild(),true);
		else disableToolBarEditors();
	}
	updateToolBar(m_pEditedToolBar, 0);
}

//==================== copyToolButton ========================
void KviUToolBarEditor::copyToolButton()
{
	if(!m_pEditedToolButton)return;
	if(m_pEditedToolButton->m_type != KviToolBarListViewItem::Button && m_pEditedToolButton->m_type != KviToolBarListViewItem::Item) return;
	m_szClipboardItemBuffer = m_pEditedToolButton->m_buffer;
	m_szClipboardItemIcon   = m_pEditedToolButton->m_icon;
	m_szClipboardItemName   = m_pEditedToolButton->text(0);
	m_szClipboardItemSbar	= m_pEditedToolButton->m_sbar;
}

//==================== addEmptyToolButton ========================
void KviUToolBarEditor::addEmptyToolButton()
{
	KviToolBarListViewItem *it;
	if(m_pEditedToolButton){
		if(m_pEditedToolButton->parent())
			it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,(KviToolBarListViewItem *)m_pEditedToolButton->parent(),
				m_pEditedToolButton,"unnamed","item",__tr("# Enter the script for the new item here"),"");
		else
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,m_pToolBarView,
			m_pEditedToolButton,"unnamed","button",__tr("# Enter the script for the new item here"),"");
	} else {
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,m_pToolBarView,
			"unnamed","button",__tr("# Enter the script for the new item here"),"");
	}
	m_pToolBarView->setSelected(it,true);
	updateToolBar(m_pEditedToolBar, 0);
}

void KviUToolBarEditor::addToolBarMenu()
{
	KviToolBarListViewItem *sb;
	if(m_pEditedToolButton){
		if(m_pEditedToolButton->parent()){
			sb = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,(KviToolBarListViewItem *)m_pEditedToolButton->parent(),
				m_pEditedToolButton,"unnamed","submenu","","");
		} else {
			sb = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,m_pToolBarView,
				m_pEditedToolButton,"unnamed","menu","","");
		}
	} else {
		sb = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,m_pToolBarView,
			"unnamed","menu","","");
	}

	KviToolBarListViewItem * it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,sb,
		"unnamed","item",__tr("# Enter the script for the new item here"),"");
	sb->setOpen(true);

	m_pToolBarView->setSelected(it,true);
	updateToolBar(m_pEditedToolBar, 0);
}

void KviUToolBarEditor::addToolBarMenuAbove()
{
	KviToolBarListViewItem *sb;
	if(!m_pEditedToolButton)return;
	if(m_pEditedToolButton->parent() != m_pEditedToolButton->itemAbove())return; // ensure first item
	if(m_pEditedToolButton->parent())
		sb = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,(KviToolBarListViewItem *)m_pEditedToolButton->parent(),
			"unnamed","submenu","","");
	else
		sb = new KviToolBarListViewItem(KviToolBarListViewItem::Menu,m_pToolBarView,
			"unnamed","menu","","");

	KviToolBarListViewItem * it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,sb,
		"unnamed","item",__tr("# Enter the script for the new item here"),"");
	sb->setOpen(true);

	m_pToolBarView->setSelected(it,true);
	updateToolBar(m_pEditedToolBar, 0);
}

//==================== pasteToolButtonBelow ========================
void KviUToolBarEditor::pasteToolButtonBelow()
{
	KviToolBarListViewItem *it;
	if(m_szClipboardItemName.isEmpty())return;
	if(m_pEditedToolButton){
		if(m_pEditedToolButton->parent())
			it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,(KviToolBarListViewItem *)m_pEditedToolButton->parent(),
				m_pEditedToolButton,m_szClipboardItemName.ptr(),"item",m_szClipboardItemBuffer.ptr(),m_szClipboardItemIcon.ptr());
		else
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,m_pToolBarView,
			m_pEditedToolButton,m_szClipboardItemName.ptr(),"button",m_szClipboardItemBuffer.ptr(),m_szClipboardItemIcon.ptr(),m_szClipboardItemSbar.ptr());
	} else {
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,m_pToolBarView,
			m_szClipboardItemName.ptr(),"button",m_szClipboardItemBuffer.ptr(),m_szClipboardItemIcon.ptr(),m_szClipboardItemSbar.ptr());
	}
	m_pToolBarView->setSelected(it,true);
}

//==================== pasteToolButtonAbove ========================
void KviUToolBarEditor::pasteToolButtonAbove()
{
	KviToolBarListViewItem *it;
	if(!m_pEditedToolButton)return;
	if(m_pEditedToolButton->parent() != m_pEditedToolButton->itemAbove())return; // ensure first item
	if(m_szClipboardItemName.isEmpty())return;
	if(m_pEditedToolButton->parent())
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,(KviToolBarListViewItem *)m_pEditedToolButton->parent(),
			m_szClipboardItemName.ptr(),"item",m_szClipboardItemBuffer.ptr(),m_szClipboardItemIcon.ptr(),"");
	else
	it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,m_pToolBarView,
		m_szClipboardItemName.ptr(),"button",m_szClipboardItemBuffer.ptr(),m_szClipboardItemIcon.ptr(),m_szClipboardItemSbar.ptr());

	m_pToolBarView->setSelected(it,true);
}
//
//==================== addEmptyToolButtonAbove ========================
void KviUToolBarEditor::addEmptyToolButtonAbove()
{
	KviToolBarListViewItem *it;
	if(!m_pEditedToolButton)return;
	if(m_pEditedToolButton->parent() != m_pEditedToolButton->itemAbove())return; // ensure first item
	if(m_pEditedToolButton->parent())
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Item,(KviToolBarListViewItem *)m_pEditedToolButton->parent(),
			"unnamed","item",__tr("# Enter the script for the new item here"),"","");
	else
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Button,m_pToolBarView,
			"unnamed","button",__tr("# Enter the script for the new item here"),"","");
	m_pToolBarView->setSelected(it,true);
}

//==================== addToolBarSeparatorItemAbove ========================
void KviUToolBarEditor::addToolBarSeparatorItemAbove()
{
	KviToolBarListViewItem * it;
	if(!m_pEditedToolButton)return;
	if(m_pEditedToolButton->parent() != m_pEditedToolButton->itemAbove())return; // ensure first item
	if(m_pEditedToolButton->parent())
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,(KviToolBarListViewItem *)m_pEditedToolButton->parent(),
			"----","separator","","","");
	else
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,m_pToolBarView,
			"----","separator","","","");
	m_pToolBarView->setSelected(it,true);
}

//==================== addToolBarSeparatorItem ========================
void KviUToolBarEditor::addToolBarSeparatorItem()
{
	KviToolBarListViewItem * it;
	if(m_pEditedToolButton){
		if(m_pEditedToolButton->parent())
			it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,(KviToolBarListViewItem *)m_pEditedToolButton->parent(),
				m_pEditedToolButton,"----","separator",__tr("# Enter the script for the new item here"),"");
		else
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,m_pToolBarView,
			m_pEditedToolButton,"----","separator","","");
	} else {
		it = new KviToolBarListViewItem(KviToolBarListViewItem::Separator,m_pToolBarView,
			"----","separator","","");
	}
	m_pToolBarView->setSelected(it,true);
}


/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////
////
//// VARIABLES EDITOR
////
//// This widget allows editing of the global variables
////
/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@quickhelp: KviVariablesEditor
	@widget: Variables tab
		This tab displays an automatically generated script buffer.<br>
		It contains all the currently set global variables.<br>
		You can edit it to set the variable contents to match your test needs.<br>
		This script buffer will only be executed if you click on "OK" while in the dialog.<br>
		Please note that if you want to unset a variable, you have to
		remove the part after the '=' sign, but NOT remove the variable name.<br>
*/


KviVariablesEditor::KviVariablesEditor(QWidget * parent,const char * name)
: QWidget(parent,name)
{
	QGridLayout * g = new QGridLayout(this,1,1,4,2);

	m_pVariablesEditor = new KviScriptEditor(this);
	g->addWidget(m_pVariablesEditor,0,0);
	fillVariablesEditor();
	m_bVariablesChanged = false;
	connect(m_pVariablesEditor->editor(),SIGNAL(textChanged()),this,SLOT(variablesTextChanged()));
}

KviVariablesEditor::~KviVariablesEditor()
{
}

void KviVariablesEditor::fillVariablesEditor()
{
	KviStr tmp;
	for(KviVariable * v=g_pVarCache->m_pVarList->first();v;v=g_pVarCache->m_pVarList->next()){
		tmp.append('%');
		tmp.append(v->szName);
		tmp.append(" = ");
		tmp.append(v->szValue);
		tmp.append('\n');
	}
	for(KviDictionary * d=g_pVarCache->m_pDictList->first();d;d=g_pVarCache->m_pDictList->next()){
		for(KviVariable * var=d->m_pVarCache->m_pVarList->first();var;var=d->m_pVarCache->m_pVarList->next()){
			tmp.append('%');
			tmp.append(d->szName);
			tmp.append('[');
			tmp.append(var->szName);
			tmp.append(']');
			tmp.append(" = ");
			tmp.append(var->szValue);
			tmp.append('\n');
		}
	}
	m_pVariablesEditor->setText(tmp.ptr());
}

void KviVariablesEditor::variablesTextChanged()
{
	m_bVariablesChanged = true;
}

void KviVariablesEditor::commit()
{
	if(m_bVariablesChanged){
		KviStr tmp = m_pVariablesEditor->text();
		// any frame can parse this
		KviFrame * aFrame = g_pApp->m_pFrameList->first();
		if(aFrame)aFrame->m_pUserParser->parseCommand(tmp.ptr(),aFrame->m_pConsole);
	}
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////
////
//// CODE TESTER
////
//// This widget allows testing of the script code
////
/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@quickhelp: KviCodeTester
	@widget: Script tester
		This tab allows you to test multiline code.<br>
		Since the commandline widget does not allow multiline code to be executed,
		you can enter it here and click "Execute".<br>
		You can specify a window for the command to be executed in.<br>
		If you don't specify a window, or if the target window does not exist,
		the command buffer will be executed in the first session's Console.<br>
		If you still don't understand what this tab does, type in the editor:<br>
		<example>
			<a href="echo.kvihelp">echo</a> Hello in <a href="s_window.kvihelp">$window</a>!"
		</example>
		click "Execute," and watch the Console window.<br>
*/

KviStr g_szLastCodeTesterBuffer = "";

KviCodeTester::KviCodeTester(QWidget * parent,const char * name)
: QWidget(parent,name)
{
	QGridLayout * g = new QGridLayout(this,2,3,4,2);

	m_pCodeEditor = new KviScriptEditor(this);
	g->addMultiCellWidget(m_pCodeEditor,0,0,0,2);
	m_pCodeEditor->setText(g_szLastCodeTesterBuffer.ptr());
	QPushButton *pb =new QPushButton(__tr("E&xecute"),this);
	g->addWidget(pb,1,2);
	QLabel * lb = new QLabel(__tr("Window:"),this);
	g->addWidget(lb,1,0);
	m_pWindowNameEdit = new QLineEdit(this);
	g->addWidget(m_pWindowNameEdit,1,1);
	g->setColStretch(1,1);
	g->setRowStretch(0,1);
	connect(pb,SIGNAL(clicked()),this,SLOT(test()));
}

KviCodeTester::~KviCodeTester()
{
}

void KviCodeTester::test()
{
	KviStr tmp = m_pCodeEditor->text();

	KviStr szWnd = m_pWindowNameEdit->text();
	KviWindow *wnd = 0;
	KviFrame * theFrame = 0;
	if(szWnd.hasData())
	{
		for(KviFrame *f = g_pApp->m_pFrameList->first();f;f=g_pApp->m_pFrameList->next())
		{
			wnd = f->findWindow(szWnd.ptr());
			if(wnd){
				theFrame = f;
				break;
			}
		}
	}
	if(!wnd)
	{
		theFrame = g_pApp->m_pFrameList->first();
		wnd = theFrame->m_pConsole;
	}
	if(theFrame)theFrame->m_pUserParser->parseCommand(tmp.ptr(),wnd);
}

/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////
////
//// SCRIPT CENTER
////
/////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@quickhelp: KviScriptCenter
	@widget: Script Center
		This dialog allows you to edit KVI++ script elements.<br>
		Don't be scared by the number of the tabs in this dialog;
		once you understand the meaning of each tab, you will find this dialog very useful.<br>
		Have fun exploring the Script Center!<br>
		<a href="qh_kvialiaseditor.kvihelp">Alias editor</a><br>
		<a href="qh_kvieventeditor.kvihelp">Event editor</a><br>
		<a href="qh_kviraweventeditor.kvihelp">Raw event editor</a><br>
		<a href="qh_kvipopupeditor.kvihelp">Popup editor</a><br>
		<a href="qh_kviutoolbareditor.kvihelp">Toolbar editor</a><br>
		<a href="qh_kvivariableseditor.kvihelp">Variables editor</a><br>
		<a href="qh_kviobjectviewer.kvihelp">Object viewer</a><br>
		<a href="qh_kvicodetester.kvihelp">Code tester</a><br>
*/

KviStr g_szLastScriptDialogTab = "";

//KviDlgScript::KviDlgScript()
//:QTabDialog()
//{
//	setCaption(__tr("Script Center"));
//
////	KviScriptCenter * c = new KviScriptCenter(0);
////	c->show();
////	c = new KviScriptCenter(g_pApp->m_pFrameList->first()->m_pSplitter);
////	c->show();
//
//	m_pAliasEditor = new KviAliasEditor(this,"script_tab_aliases");
//	addTab(m_pAliasEditor,__tr("Aliases"));
//
//	m_pHiLevelEventEditor = new KviEventEditor(this,"script_tab_events");
//	addTab(m_pHiLevelEventEditor,__tr("Events"));
//
//	m_pRawEventEditor = new KviRawEventEditor(this,"script_tab_rawevents");
//	addTab(m_pRawEventEditor,__tr("Raw Events"));
//
//	m_pThePopupEditor = new KviPopupEditor(this,"script_tab_popups");
//	addTab(m_pThePopupEditor,__tr("Popups"));
//
//	m_pUToolBarEditor = new KviUToolBarEditor(this,"script_tab_utoolbar");
//	addTab(m_pUToolBarEditor,__tr("Toolbar"));
//
//	m_pVariablesEditor = new KviVariablesEditor(this,"script_tab_variables");
//	addTab(m_pVariablesEditor,__tr("Variables"));
//
//	KviObjectViewer * vw = new KviObjectViewer(this,"script_tab_objectviewer");
//	addTab(vw,__tr("Objects"));
//
//	KviCodeTester * ct = new KviCodeTester(this,"script_tab_codetester");
//	addTab(ct,__tr("Code Tester"));
//
//	KviHelpWindow * hw = new KviHelpWindow(g_pApp->m_pFrameList->first(),this,"script_tab_helpbrowser");
//	addTab(hw,__tr("Help Browser"));
//
//	if(g_szLastScriptDialogTab.hasData())
//	{
//		QObject * o = child(g_szLastScriptDialogTab.ptr());
//		if(o)
//		{
//			if(o->inherits("QWidget"))
//			{
//				showPage((QWidget *)o);
//			}
//		}
//	}
//
//	setCancelButton(__tr("Cancel"));
//	setDefaultButton(_CHAR_2_QSTRING(__tr("What is this ?")));
//	connect(this,SIGNAL(defaultButtonPressed()),g_pApp,SLOT(slot_whatIsThisRequest()));
//
//	setGeometry(g_pOptions->m_rectScriptDialog);
//}
//
////============ ~KviDlgScript() ===================
//
//KviDlgScript::~KviDlgScript()
//{
//	if(g_pOptions->m_bUseHackedFrameGeometry){
//		g_pOptions->m_rectScriptDialog.moveTopLeft(frameGeometry().topLeft());
//		g_pOptions->m_rectScriptDialog.setSize(size());
//	} else g_pOptions->m_rectScriptDialog = geometry();
//}
//
////============ KviDlgScript::done ===================
//
//void KviDlgScript::done(int r)
//{
//	QTabDialog::done(r);
//	// Copy the modified data
//	if(r == Accepted){
//
//		QWidget * w = currentPage();
//		if(w)g_szLastScriptDialogTab = w->name();
//
//		m_pAliasEditor->commit();
//		m_pRawEventEditor->commit();
//		m_pHiLevelEventEditor->commit();
//		m_pThePopupEditor->commit();
//		m_pUToolBarEditor->commit();
//		m_pVariablesEditor->commit();
//	}
//	emit finished((r == Accepted));
//}
//
//============ KviDlgScript::closeEvent ===================
//
//void KviDlgScript::closeEvent(QCloseEvent *)
//{
//	emit finished(false);
//}
//
//bool KviDlgScript::focusNextPrevChild(bool next)
//{
//	QWidget * w = focusWidget();
//	if(w){
//		if(next && w->inherits("KviScriptEditorWidget"))return false; // Do not change the focus widget!
//	}
//	return QTabDialog::focusNextPrevChild(next);
//}
//



KviScriptCenter::KviScriptCenter(QWidget * parent)
: QWidget(parent,"script_center")
{
	setCaption(__tr("KVIrc: Script Center"));

	QGridLayout * g = new QGridLayout(this,2,5,parent ? 2 : 10,4); //8,2

	m_pTabWidget = new QTabWidget(this);
	g->addMultiCellWidget(m_pTabWidget,0,0,0,4);

	m_pAliasEditor = new KviAliasEditor(m_pTabWidget,"script_tab_aliases");
	m_pTabWidget->addTab(m_pAliasEditor,__tr("&Aliases"));

	m_pHiLevelEventEditor = new KviEventEditor(m_pTabWidget,"script_tab_events");
	m_pTabWidget->addTab(m_pHiLevelEventEditor,__tr("&Events"));

	m_pRawEventEditor = new KviRawEventEditor(m_pTabWidget,"script_tab_rawevents");
	m_pTabWidget->addTab(m_pRawEventEditor,__tr("&Raw Events"));

	m_pThePopupEditor = new KviPopupEditor(m_pTabWidget,"script_tab_popups");
	m_pTabWidget->addTab(m_pThePopupEditor,__tr("&Popups"));

	m_pUToolBarEditor = new KviUToolBarEditor(m_pTabWidget,"script_tab_utoolbar");
	m_pTabWidget->addTab(m_pUToolBarEditor,__tr("&User Toolbar"));

	m_pVariablesEditor = new KviVariablesEditor(m_pTabWidget,"script_tab_variables");
	m_pTabWidget->addTab(m_pVariablesEditor,__tr("&Variables"));

	m_pObjectViewer = new KviObjectViewer(m_pTabWidget,"script_tab_objectviewer");
	m_pTabWidget->addTab(m_pObjectViewer,__tr("&Objects"));

	m_pCodeTester = new KviCodeTester(m_pTabWidget,"script_tab_codetester");
	m_pTabWidget->addTab(m_pCodeTester,__tr("&Tester"));

	if(g_pOptions->m_bScriptCenterAddHelpBrowser)
	{
		m_pHelpWindow = new KviHelpWindow(g_pApp->m_pFrameList->first(),m_pTabWidget,"script_tab_helpbrowser");
		m_pTabWidget->addTab(m_pHelpWindow,__tr("&Help Browser"));
	} else m_pHelpWindow = 0;

	if(g_szLastScriptDialogTab.hasData())
	{
		QObject * o = m_pTabWidget->child(g_szLastScriptDialogTab.ptr());
		if(o)
		{
			if(o->inherits("QWidget"))
			{
				m_pTabWidget->showPage((QWidget *)o);
			}
		}
	}

	QPushButton * b = new QPushButton(__tr("What's &This?"),this);
	g->addWidget(b,1,0);
	connect(b,SIGNAL(clicked()),g_pApp,SLOT(slot_whatIsThisRequest()));

	b = new QPushButton(__tr("&OK"),this);
	b->setDefault(true);
	g->addWidget(b,1,2);
	connect(b,SIGNAL(clicked()),this,SLOT(acceptClicked()));

	b = new QPushButton(__tr("&Apply"),this);
	g->addWidget(b,1,3);
	connect(b,SIGNAL(clicked()),this,SLOT(applyClicked()));

	b = new QPushButton(__tr("&Cancel"),this);
	g->addWidget(b,1,4);
	connect(b,SIGNAL(clicked()),this,SLOT(rejectClicked()));

	g->setRowStretch(0,1);
	g->setColStretch(1,1);

	setGeometry(g_pOptions->m_rectScriptDialog);
}

KviScriptCenter::~KviScriptCenter()
{
	emit closed();
//	debug("Script center destroyed");
	if(g_pOptions->m_bUseHackedFrameGeometry){
		g_pOptions->m_rectScriptDialog.moveTopLeft(frameGeometry().topLeft());
		g_pOptions->m_rectScriptDialog.setSize(size());
	} else g_pOptions->m_rectScriptDialog = geometry();
}

void KviScriptCenter::applyClicked()
{
	m_pAliasEditor->commit();
	m_pRawEventEditor->commit();
	m_pHiLevelEventEditor->commit();
	m_pThePopupEditor->commit();
	m_pUToolBarEditor->commit();
	m_pVariablesEditor->commit();
	m_pVariablesEditor->fillVariablesEditor();
	m_pObjectViewer->fillViews();
	for(KviFrame *f = g_pApp->m_pFrameList->first();f;f=g_pApp->m_pFrameList->next()){
		f->updateUserToolBar();
	}
}

void KviScriptCenter::acceptClicked()
{
	QWidget * w = m_pTabWidget->currentPage();
	if(w)g_szLastScriptDialogTab = w->name();

	m_pAliasEditor->commit();
	m_pRawEventEditor->commit();
	m_pHiLevelEventEditor->commit();
	m_pThePopupEditor->commit();
	m_pUToolBarEditor->commit();
	m_pVariablesEditor->commit();

	for(KviFrame *f = g_pApp->m_pFrameList->first();f;f=g_pApp->m_pFrameList->next()){
		f->updateUserToolBar();
	}

	if(g_pOptions->m_bScriptCenterAutoSaveWhenAccepted)
	{
		g_pOptions->saveEvents();
		g_pOptions->saveRawEvents();
		g_pOptions->savePopups();
		g_pOptions->saveAliases();
		g_pOptions->saveToolBar();
	}

//	emit closed(true);
	delete this; //die!
}

void KviScriptCenter::rejectClicked()
{
//	emit closed(false);
	delete this; //die!
}

void KviScriptCenter::closeEvent(QCloseEvent *e)
{
//	emit closed(false);
	e->accept();
	delete this; //die!
}

bool KviScriptCenter::focusNextPrevChild(bool next)
{
	QWidget * w = focusWidget();
	if(w){
		if(next && w->inherits("KviScriptEditorWidget"))return false; // Do not change the focus widget!
	}
	return QWidget::focusNextPrevChild(next);
}




#include "m_kvi_dlgscript.moc"
