//
//   File : kvi_dlgserver.cpp (/usr/build/NEW_kvirc/kvirc/kvirc/kvi_dlgserver.cpp)
//   Last major modification : Sun Jan 17 1999 15:37:49 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//#define _KVI_DEBUG_CLASS_NAME_ "KviDlgOptServer"
//#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#define _KVI_DLGSERVER_CPP_

#include "kvi_dlgserver.h"

#include "kvi_selectors.h"
#include "kvi_string.h"
#include "kvi_netutils.h"
#include "kvi_fileutils.h"

#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_options.h"
#include "kvi_defines.h"
#include "kvi_settings.h"

#include "kvi_locale.h"
//#include "kvi_quickhelp.h"

#include <qlayout.h>
#include <qcombobox.h>
#include <qpushbutton.h>
#include <qcursor.h>
#include <qkeycode.h>
#include <qframe.h>
#include <qheader.h>
#include <qtooltip.h>

/* FIXME:  "TODO : Move to the KviListView implementation..." */

//=============================================================================
//
// ### Field editor for QListView ###
//
//=============================================================================

KviIrcServerFieldEditor::KviIrcServerFieldEditor(QWidget *parent)
:QLineEdit(parent)
{
	hide();
	connect(this,SIGNAL(returnPressed()),this,SLOT(pressedReturnKey()));
}

KviIrcServerFieldEditor::~KviIrcServerFieldEditor(){}

void KviIrcServerFieldEditor::edit(const char *text)
{
	setText(_CHAR_2_QSTRING(text));
	show();
	grabMouse();
	setFocus();
}

void KviIrcServerFieldEditor::mousePressEvent(QMouseEvent *e)
{
	//if the mouse is outside of this widget release...
	if(!rect().contains(e->pos()))terminateEdit(false);
	else QLineEdit::mousePressEvent(e);
}

void KviIrcServerFieldEditor::terminateEdit(bool bCommitChanges)
{
	releaseMouse();
	if(isVisible()){
		hide();
		QCString tmp = text().local8Bit();
		if(bCommitChanges)emit editFinished(tmp.data());
		setText("");
	}
}

void KviIrcServerFieldEditor::handleKeyEvent(QKeyEvent *e){ keyPressEvent(e); }
void KviIrcServerFieldEditor::focusOutEvent(QFocusEvent *){ terminateEdit(true); }
void KviIrcServerFieldEditor::pressedReturnKey(){ terminateEdit(true); }

//=============================================================================
//
// ### A server entry list item for QListView ###
//
//=============================================================================

// /me hates ComboBoxes not in sync with internal data...

KviIrcServerListItem::KviIrcServerListItem(QListView *w,KviIrcServer * ptr)
:QListViewItem(w,ptr->szHost.ptr(),ptr->szPort.ptr(),
	_CHAR_2_QSTRING(ptr->szDescription.ptr()),ptr->szPassword.ptr(),
	ptr->szIp.ptr())
{
	__range_valid(ptr);
	m_ptr = ptr;
#ifdef COMPILE_NEED_IPV6
	setText(5,ptr->bIpV6 ? _CHAR_2_QSTRING("yes") : _CHAR_2_QSTRING("no"));
#endif
}
KviIrcServerListItem::~KviIrcServerListItem(){}

//=============================================================================
//
// ### A proxy entry list item for QListView ###
//
//=============================================================================

KviIrcProxyListItem::KviIrcProxyListItem(QListView *w,KviIrcProxy * ptr)
:QListViewItem(w,ptr->szHost.ptr(),ptr->szPort.ptr(),ptr->szUsername.ptr(),
	ptr->szPassword.ptr(),ptr->szIp.ptr())
{
	__range_valid(ptr);
	m_ptr = ptr;
}
KviIrcProxyListItem::~KviIrcProxyListItem(){}

//=============================================================================
//
// ### The server options dialog ###
//
//=============================================================================

KviDlgOptServer::KviDlgOptServer(QWidget *parent)
:QTabDialog(parent,"dlgserver")
{
	setCaption(_i18n_("KVIrc: Server Options"));
	// Local copy of data
	m_pManager = new KviIrcServerManager();
	m_pManager->copyFrom(g_pOptions->m_pServerManager);
	m_pProxyManager = new KviIrcProxyManager();
	m_pProxyManager->copyFrom(g_pOptions->m_pProxyManager);

	m_curEditedServerItem = 0;
	m_curEditedProxyItem = 0;
	//
	// Server tab...
	//

	//  0           1                        2
	//0 +++++++++++ ++++++++++++++++++++++++ ++++++++++++++++
	//  +  Label  + +      m_pNetCombo     + +  new Network +
	//  +++++++++++ ++++++++++++++++++++++++ ++++++++++++++++
	//1 ++++++++++++++++++++++++++++++++++++ ++++++++++++++++
	//  +                                  + +  m_pDelNet   +
	//  +                  m_pSrvList      + ++++++++++++++++
	//2 +                                  + ++++++++++++++++
	//  +                                  + +  m_pNewSrv   +
	//  +                                  + ++++++++++++++++
	//3 +                                  + ++++++++++++++++
	//  +                                  + +  m_pEditSrv  +
	//  +                                  + ++++++++++++++++
	//4 +                                  + ++++++++++++++++
	//  +                                  + +  m_pDelSrv   +
	//  +                                  + ++++++++++++++++
	//5 +                                  + ++++++++++++++++
	//  +                                  + +  m_pSort     +
	//  +                                  + ++++++++++++++++
	//6 +                                  + ++++++++++++++++
	//  +                                  + + import *.ini +
	//  ++++++++++++++++++++++++++++++++++++ ++++++++++++++++
	//
	QWidget * tab = new QWidget(this);
	//tab,rows,cols,
	QGridLayout *g = new QGridLayout(tab,7,3,10,4);

	QLabel *l= new QLabel(__tr("Network:"),tab);
	g->addWidget(l,0,0);
	
	m_pNetCombo = new QComboBox(true,tab);
	m_pNetCombo->setInsertionPolicy(QComboBox::NoInsertion);
	//row,column
	g->addWidget(m_pNetCombo,0,1);
	connect(m_pNetCombo,SIGNAL(highlighted(const QString &)),
		this,SLOT(currentNetworkChanged(const QString &)));

	m_pSrvList = new QListView(tab);

	m_pSrvList->setFrameStyle(QFrame::Panel|QFrame::Sunken);
	m_pSrvList->addColumn(_i18n_("Server"));
	m_pSrvList->addColumn(_i18n_("Port"));
	m_pSrvList->addColumn(_i18n_("Description"));
	m_pSrvList->addColumn(_i18n_("Password"));
	m_pSrvList->addColumn(_i18n_("IP Address"));
#ifdef COMPILE_NEED_IPV6
	m_pSrvList->addColumn(_i18n_("IPv6"));
#endif
	m_pSrvList->setMultiSelection(false);
	m_pSrvList->setAllColumnsShowFocus(true);
	m_pSrvList->setSorting(-1);
	m_pSrvList->header()->setClickEnabled(false,-1); //disable click
	m_pSrvList->header()->setMovingEnabled(false);
	m_pSrvList->setFocusPolicy(QWidget::StrongFocus);
	QToolTip::add(m_pSrvList,__tr("Double-click to edit"));

	m_pEditor = new KviIrcServerFieldEditor(m_pSrvList->viewport());
	connect(m_pEditor,SIGNAL(editFinished(const char *)),
		this,SLOT(serverEditFinished(const char *)));

	//fromRow,toRow,fromCol,toCol
	g->addMultiCellWidget(m_pSrvList,1,6,0,1);
	connect(m_pSrvList,SIGNAL(currentChanged(QListViewItem *)),
		this,SLOT(currentServerChanged(QListViewItem *)));
	connect(m_pSrvList,SIGNAL(doubleClicked(QListViewItem *)),
		this,SLOT(serverDoubleClicked(QListViewItem *)));

	QPushButton * newNet = new QPushButton(_i18n_("Add &Network"),tab);
	newNet->setMinimumSize(newNet->sizeHint());
	g->addWidget(newNet,0,2);
	connect(newNet,SIGNAL(clicked()),this,SLOT(newNetwork()));

	m_pDelNet = new QPushButton(_i18n_("Re&move Network"),tab);
	g->addWidget(m_pDelNet,1,2);
	connect(m_pDelNet,SIGNAL(clicked()),this,SLOT(deleteNetwork()));

	m_pNewSrv = new QPushButton(_i18n_("&Add Server"),tab);
	g->addWidget(m_pNewSrv,2,2);
	connect(m_pNewSrv,SIGNAL(clicked()),this,SLOT(newServer()));

	m_pEditSrv = new QPushButton(_i18n_("&Edit Server"),tab);
	g->addWidget(m_pEditSrv,3,2);
	connect(m_pEditSrv,SIGNAL(clicked()),this,SLOT(editServer()));

	m_pDelSrv = new QPushButton(_i18n_("&Remove Server"),tab);
	g->addWidget(m_pDelSrv,4,2);
	connect(m_pDelSrv,SIGNAL(clicked()),this,SLOT(deleteServer()));

	m_pSort = new QPushButton(_i18n_("&Sort by Domain"),tab);
	g->addWidget(m_pSort,5,2);
	connect(m_pSort,SIGNAL(clicked()),this,SLOT(sortServers()));

	QPushButton *import=new QPushButton(_i18n_("&Import from *.ini"),tab);
	g->addWidget(import,6,2);
	connect(import,SIGNAL(clicked()),this,SLOT(importFromIni()));

	g->setColStretch(1,1);
	g->activate();

	addTab(tab,_i18n_("&Servers"));

	//
	// Other misc options tab
	//

	tab = new QWidget(this);
	//tab,rows,cols,
	g = new QGridLayout(tab,10,1,10,4);

	KviNewStringSelector *ss = new KviNewStringSelector(tab,__tr("Local host IP address:"),&(g_pOptions->m_szLocalHostIp),true);
	g->addWidget(ss,0,0);


	KviNewBoolSelector *bs = new KviNewBoolSelector(tab,__tr("Bind IRC sockets to the local host IP address"),&(g_pOptions->m_bBindIrcSocketToSpecificInterface));
	g->addWidget(bs,1,0);

	KviNewIntegerSelector *is = new KviNewIntegerSelector(tab,__tr("Connection timeout:"),
		(void *)&(g_pOptions->m_iSocketTimeout),KviNewIntegerSelector::Int,
		5,600,60,true,100,0,0," second(s)");
	g->addWidget(is,2,0);
//	g->setRowStretch( 2,2 );

	bs = new KviNewBoolSelector(tab,__tr("Automatically reconnect on disconnect"),&(g_pOptions->m_bAutoReconnectOnDisconnect));
	g->addWidget(bs,3,0);

	bs = new KviNewBoolSelector(tab,__tr("Close channel and query windows on disconnect"),&(g_pOptions->m_bCloseWindowsOnDisconnect));
	g->addWidget(bs,4,0);

	bs = new KviNewBoolSelector(tab,__tr("Rejoin channels on reconnect"),&(g_pOptions->m_bRejoinChannelsOnReconnect));
	g->addWidget(bs,5,0);

	bs = new KviNewBoolSelector(tab,__tr("Try next server in the network if connect fails"),&(g_pOptions->m_bTryNextServerOnConnectFailed));
	g->addWidget(bs,6,0);
//	g->setRowStretch( 6,2 );

	QFrame *r = new QFrame(tab);
	r->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addWidget(r,7,0);
//	g->setRowStretch( 7,2 );

	bs = new KviNewBoolSelector(tab,__tr("Force synchronous DNS calls"),&(g_pOptions->m_bForceSyncDns));
	g->addWidget(bs,8,0);
//	g->setRowStretch( 8,1 );

	r = new QFrame(tab);
	g->addWidget(r,9,0);
	g->setRowStretch(9,1);

	g->activate();

	addTab(tab,_i18n_("&Connection"));

	//
	// Other misc options tab
	//

	tab = new QWidget(this);
	//tab,rows,cols,
	g = new QGridLayout(tab,7,2,10,4);

	bs = new KviNewBoolSelector(tab,__tr("Suppress auto-login with nickname and username"),&(g_pOptions->m_bUseTelnetProxy));
	g->addMultiCellWidget(bs,0,0,0,1);

	KviNewBoolSelector * bs1 = new KviNewBoolSelector(tab,__tr("Use HTTP/1.0 proxy server"),&(g_pOptions->m_bProxyProtoHttp));
	g->addMultiCellWidget(bs1,1,1,0,1);

	KviNewBoolSelector * bs2 = new KviNewBoolSelector(tab,__tr("Use SOCKS proxy server"),&(g_pOptions->m_bUseProxy));
	g->addMultiCellWidget(bs2,2,2,0,1);
	connect(bs2,SIGNAL(toggled(bool)),bs1,SLOT(setDisabled(bool)));
	connect(bs1,SIGNAL(toggled(bool)),bs2,SLOT(setDisabled(bool)));

	bs = new KviNewBoolSelector(tab,__tr("Use SOCKS protocol version 5 instead of version 4"),
		&(g_pOptions->m_bProxyProtoV5),g_pOptions->m_bUseProxy);
	connect(bs2,SIGNAL(toggled(bool)),bs,SLOT(setEnabled(bool)));
	g->addMultiCellWidget(bs,3,3,0,1);
//	g->setRowStretch( 1,2 );

	m_pProxyList = new QListView(tab);

	m_pProxyList->setFrameStyle(QFrame::Panel|QFrame::Sunken);
	m_pProxyList->addColumn(_i18n_("Proxy"));
	m_pProxyList->addColumn(_i18n_("Port"));
	m_pProxyList->addColumn(_i18n_("Username"));
	m_pProxyList->addColumn(_i18n_("Password"));
	m_pProxyList->addColumn(_i18n_("IP Address"));
	m_pProxyList->setMultiSelection(false);
	m_pProxyList->setAllColumnsShowFocus(true);
	m_pProxyList->setSorting(-1);
	m_pProxyList->header()->setClickEnabled(false,-1); //disable click
	m_pProxyList->header()->setMovingEnabled(false);
	m_pProxyList->setFocusPolicy(QWidget::StrongFocus);
	QToolTip::add(m_pProxyList,__tr("Double-click to edit"));


	m_pProxyEditor = new KviIrcServerFieldEditor(m_pProxyList->viewport());
	connect(m_pProxyEditor,SIGNAL(editFinished(const char *)),
		this,SLOT(proxyEditFinished(const char *)));

	//fromRow,toRow,fromCol,toCol
	g->addMultiCellWidget(m_pProxyList,4,6,0,0);
	connect(m_pProxyList,SIGNAL(currentChanged(QListViewItem *)),
		this,SLOT(currentProxyChanged(QListViewItem *)));
	connect(m_pProxyList,SIGNAL(doubleClicked(QListViewItem *)),
		this,SLOT(proxyDoubleClicked(QListViewItem *)));

	QPushButton *nProxy = new QPushButton(_i18n_("&Add Proxy"),tab);
	g->addWidget(nProxy,4,1);
	connect(nProxy,SIGNAL(clicked()),this,SLOT(newProxy()));

	m_pEditProxy = new QPushButton(_i18n_("&Edit Proxy"),tab);
	g->addWidget(m_pEditProxy,5,1);
	connect(m_pEditProxy,SIGNAL(clicked()),this,SLOT(editProxy()));

	m_pDelProxy = new QPushButton(_i18n_("&Remove Proxy"),tab);
	g->addWidget(m_pDelProxy,6,1);
	connect(m_pDelProxy,SIGNAL(clicked()),this,SLOT(deleteProxy()));

	g->setRowStretch(4,1);

	g->activate();

	addTab(tab,_i18n_("&Proxy Servers"));

	//
	// Init stuff
	//

	setHelpButton(__tr("What's &This?"));
	connect(this,SIGNAL(helpButtonPressed()),g_pApp,SLOT(slot_whatIsThisRequest()));
	setOkButton(__tr("&OK"));
	setCancelButton(__tr("&Cancel"));

	installEventFilter(this);

	fillNetCombo();
	fillProxyList();
}

//=============================================================================
//
// ### Destruction ###
//
//=============================================================================

KviDlgOptServer::~KviDlgOptServer()
{
	delete m_pManager;
	delete m_pProxyManager;
}

void KviDlgOptServer::done(int r)
{
	QTabDialog::done(r);
	//Commit the changes
	if(r == Accepted){
		KviNewIntegerSelector::commitAll(this);
		KviNewBoolSelector::commitAll(this);
		KviNewStringSelector::commitAll(this);
		g_pOptions->m_pServerManager->copyFrom(m_pManager);
		g_pOptions->m_pProxyManager->copyFrom(m_pProxyManager);
//		KviIrcNetwork * net = g_pOptions->m_pServerManager->getCurrentNetwork();
//		KviIrcServer * srv = net->currentServer();
//		m_pFrm->m_global.szConnectNetwork    = net->name();
//		m_pFrm->m_global.szConnectServerHost = srv->szHost;
//		m_pFrm->m_global.szConnectServerIp   = srv->szIp;
//		m_pFrm->m_global.szConnectServerPort = srv->szPort;
//		m_pFrm->m_global.szConnectPassword   = srv->szPassword;
//#ifdef COMPILE_NEED_IPV6
//		m_pFrm->m_global.bCIpV6Mode          = srv->bIpV6;
//#endif
//		KviIrcProxy * prx = g_pOptions->m_pProxyManager->currentProxy();
//		m_pFrm->m_global.szConnectProxyHost     = prx->szHost;
//		m_pFrm->m_global.szConnectProxyIp       = prx->szIp;
//		m_pFrm->m_global.szConnectProxyPort     = prx->szPort;
//		m_pFrm->m_global.szConnectProxyPassword = prx->szPassword;
//		m_pFrm->m_global.szConnectProxyUsername = prx->szUsername;
	}
	emit finished((r == Accepted));
}

void KviDlgOptServer::closeEvent(QCloseEvent *)
{ emit finished(false); }


//=============================================================================
//
// ### Servers tab stuff ###
//
//=============================================================================


void KviDlgOptServer::fillNetCombo()
{
	//This will clear the combo , 
	//Fill it with the network list
	//Make sure that exists a current network
	//and select it.
	//If the network list is empty , it will disable the comboBox , and the deleteNet button
	m_pNetCombo->clear();
	__range_valid(m_pManager->networkList());
	m_pNetCombo->setEnabled(!m_pManager->networkList()->isEmpty());
	m_pDelNet->setEnabled(!m_pManager->networkList()->isEmpty());
	m_pNewSrv->setEnabled(!m_pManager->networkList()->isEmpty());
	if(m_pManager->networkList()->isEmpty()){
		m_pSrvList->clear();
		enableServerButtons(false);
		return;
	}
	int idx = 0;
	int cur = -1;
	KviIrcNetwork * net = m_pManager->getCurrentNetwork();
	__range_valid(net);
	//fill the list
	for(KviIrcNetwork * n=m_pManager->networkList()->first();n;
		n=m_pManager->networkList()->next()){

		m_pNetCombo->insertItem(n->name()); //append it
		if(net == n)cur = idx; //this is the current...
		idx++;
	}
	__range_valid(cur >= 0); //now must be set!
	//select it
	m_pNetCombo->setCurrentItem(cur);
	fillServerList(net);
}

void KviDlgOptServer::deleteNetwork()
{
	KviIrcNetwork *net = m_pManager->getCurrentNetwork();
	__range_valid(net);
#ifdef _KVI_DEBUG_CHECK_RANGE_
	__range_valid(m_pManager->removeNetwork(net));
#else
	m_pManager->removeNetwork(net);
#endif
	fillNetCombo();
}

void KviDlgOptServer::deleteServer()
{
	KviIrcServerListItem *it=(KviIrcServerListItem *)m_pSrvList->currentItem();
	if(!it)return;
	KviIrcNetwork *net = m_pManager->getCurrentNetwork();
	__range_valid(net);
#ifdef _KVI_DEBUG_CHECK_RANGE_
	__range_valid(net->removeServer(it->m_ptr));
#else
	net->removeServer(it->m_ptr);
#endif
	fillServerList(net);
}

void KviDlgOptServer::editServer()
{
	KviIrcServerListItem *it=(KviIrcServerListItem *)m_pSrvList->currentItem();
	if(it)editServerField(it,Host);
}

void KviDlgOptServer::newServer()
{
	KviIrcNetwork *net = m_pManager->getCurrentNetwork();
	if(!net)return;
	KviIrcServer *srv = new KviIrcServer;
	srv->szHost = "irc.unknown.net";
	srv->szIp = "";
	srv->szDescription = "None";
	srv->szPort = "6667";
	srv->szPassword = "";
#ifdef COMPILE_NEED_IPV6
	srv->bIpV6 = false;
#endif
	net->appendServer(srv);
	fillServerList(net);
}

void KviDlgOptServer::serverEditFinished(const char *text)
{
	__enter("serverEditFinished");
	__range_valid(m_curEditedServerItem);
	KviStr szT(text);

	switch(m_curEditedServerField){
		case Host:
			szT.stripWhiteSpace();
			if((szT.findFirstIdx('.') != -1) && (szT.findFirstIdx(' ') == -1)){
				//must contain at least one dot and no spaces
				m_curEditedServerItem->m_ptr->szHost = szT.ptr();
				m_curEditedServerItem->setText(0,szT.ptr());
			}
			break;
		case Port:
			szT.stripWhiteSpace();
			if(!szT.isUnsignedNum())szT = "6667"; //reasonable default
			m_curEditedServerItem->m_ptr->szPort = szT.ptr();
			m_curEditedServerItem->setText(1,szT.ptr());
			break;
		case Description:
			m_curEditedServerItem->m_ptr->szDescription = szT.ptr();
			m_curEditedServerItem->setText(2,szT.ptr());
			break;
		case Password:
			m_curEditedServerItem->m_ptr->szPassword = szT.ptr();
			m_curEditedServerItem->setText(3,szT.ptr());
			break;
		case Ip:
			szT.stripWhiteSpace();
//			if(kvi_isValidStringIp(szT.ptr())){
				m_curEditedServerItem->m_ptr->szIp = szT.ptr();
				m_curEditedServerItem->setText(4,szT.ptr());
//			} else m_curEditedServerItem->m_ptr->szIp = "";
			break;
		case IpV6:
#ifdef COMPILE_NEED_IPV6
			szT.stripWhiteSpace();
			m_curEditedServerItem->m_ptr->bIpV6 = (kvi_strEqualCI(szT.ptr(),"yes") || kvi_strEqualCI(szT.ptr(),"1") || kvi_strEqualCI(szT.ptr(),"true"));
			m_curEditedServerItem->setText(5,szT.ptr());
			break;
#endif
		default:
			return;
			break;
	}

	m_pSrvList->setFocus();
}

void KviDlgOptServer::importFromIni()
{
	KviStr szFName(kvi_askForOpenFileName(0,"*.ini"));
	if(!szFName.isEmpty()){
		if(!m_pManager->importFromIni(szFName.ptr()))
			g_pApp->warningBox(_i18n_("Can not open file %s\n"),szFName.ptr());
		fillNetCombo();
	}
}

void KviDlgOptServer::editServerField(KviIrcServerListItem *it,
	KviOptDlgServerField field)
{
	__range_valid(it);
	__range_invalid(m_pEditor->isVisible());
	QRect rct = m_pSrvList->itemRect(it);
	int fieldWidth = m_pSrvList->columnWidth(field);
	int fieldX = - m_pSrvList->contentsX();
	for(int i = 0;i< field;i++)fieldX += m_pSrvList->columnWidth(i);
	m_pEditor->move(fieldX,rct.top());
	m_pEditor->resize(fieldWidth,rct.height());
	char * field_ptr = 0;
	switch(field){
		case Host: field_ptr = it->m_ptr->szHost.ptr(); break;
		case Port: field_ptr = it->m_ptr->szPort.ptr(); break;
		case Description: field_ptr = it->m_ptr->szDescription.ptr(); break;
		case Password: field_ptr = it->m_ptr->szPassword.ptr(); break;
		case Ip: field_ptr = it->m_ptr->szIp.ptr(); break;
#ifdef COMPILE_NEED_IPV6
		case IpV6: field_ptr = (it->m_ptr->bIpV6 ? (char *)"yes" : (char *)"no"); break;
#endif
		default: return; break;
	}
	m_curEditedServerField = field;
	m_curEditedServerItem = it;
	if(field_ptr)m_pEditor->edit(field_ptr);
}

void KviDlgOptServer::serverDoubleClicked(QListViewItem *i)
{
	KviIrcServerListItem *it=(KviIrcServerListItem *)i;
	QPoint pnt = QCursor::pos();
	pnt = m_pSrvList->viewport()->mapFromGlobal(pnt);
	int xLeft =m_pSrvList->columnWidth(0) - m_pSrvList->contentsX();
	if(pnt.x() < xLeft)editServerField(it,Host);
	else {
		xLeft += m_pSrvList->columnWidth(1); //port
		if(pnt.x() < xLeft)editServerField(it,Port);
		else {
			xLeft += m_pSrvList->columnWidth(2); //description
			if(pnt.x() < xLeft)editServerField(it,Description);
			else {
				xLeft += m_pSrvList->columnWidth(3); //password
				if(pnt.x() < xLeft)editServerField(it,Password);
				else {
					xLeft += m_pSrvList->columnWidth(4); //ip
					if(pnt.x() < xLeft)editServerField(it,Ip);
#ifdef COMPILE_NEED_IPV6
					else {
						xLeft += m_pSrvList->columnWidth(5); //Ipv6
						if(pnt.x() < xLeft)editServerField(it,IpV6);
					}
#endif
				}
			}
		}
	}
}

void KviDlgOptServer::sortServers()
{
	KviIrcNetwork *net = m_pManager->getCurrentNetwork();
	__range_valid(net);
	net->sortServers();
	fillNetCombo();	
}

void KviDlgOptServer::newNetwork()
{
	KviStr newNet = "New Network";
	//unique name please!
	while(m_pManager->getNetworkByName(newNet.ptr()))newNet+="!";
	KviIrcNetwork * net = new KviIrcNetwork(newNet.ptr());
	m_pManager->insertNetwork(net);
	fillNetCombo();
}

void KviDlgOptServer::currentNetworkChanged(const QString &damn)
{
	__enter("currentNetworkChanged");
	KviStr text=damn; //mmmmh, I don't like it, but it is here...
#ifdef _KVI_DEBUG_CHECK_RANGE_
	__range_valid(m_pManager->setCurrentNetwork(text.ptr()));
#else
	m_pManager->setCurrentNetwork(text.ptr());
#endif
	KviIrcNetwork *net = m_pManager->getCurrentNetwork();
	__range_valid(stricmp(text.ptr(),net->name())==0);
	fillServerList(net);
}

void KviDlgOptServer::currentServerChanged(QListViewItem *it)
{
	__enter("currentServerChanged");
#ifdef _KVI_DEBUG_CHECK_RANGE_
	__range_valid(m_pManager->setCurrentServer(((KviIrcServerListItem *)it)->m_ptr));
#else
	m_pManager->setCurrentServer(((KviIrcServerListItem *)it)->m_ptr);
#endif
}

void KviDlgOptServer::enableServerButtons(bool bEnable)
{
	m_pSort->setEnabled(bEnable);
	m_pEditSrv->setEnabled(bEnable);
	m_pDelSrv->setEnabled(bEnable);
}

void KviDlgOptServer::fillServerList(KviIrcNetwork *net)
{
	__range_valid(net);
	m_pSrvList->clear();
	__range_valid(net->serverList());
	if(net->serverList()->isEmpty()){
		//Disable everything
		enableServerButtons(false);
		return;
	}
	//fill the server list
	KviIrcServer * srv = net->currentServer();
	KviIrcServerListItem *cur = 0;
	__range_valid(srv); //MUST BE SET!!!! the list is not empty!
	m_pSrvList->setUpdatesEnabled(false);
	int idx = 0;
	int curIdx = -1;
	for(KviIrcServer *s = net->serverList()->first();s;s=net->serverList()->next()){
		KviIrcServerListItem *it = new KviIrcServerListItem(m_pSrvList,s);
		if(srv == s){
			m_pSrvList->setCurrentItem(it);
			m_pSrvList->setSelected(it,true);
			cur = it;
			curIdx = idx;
		}
		idx++;
	}
	m_pSrvList->setUpdatesEnabled(true);
	__range_valid(cur); //Must be set now!!!
	m_pSrvList->ensureItemVisible(cur);
	__range_valid(curIdx >= 0);
	enableServerButtons(true);
}

bool KviDlgOptServer::eventFilter(QObject *,QEvent *e)
{
	if(m_pEditor->isVisible()){
		//poor hack....but there is no other way...
		if((e->type() == QEvent::Accel)||(e->type() == QEvent::KeyPress)){
			if(((QKeyEvent *)e)->key() != Qt::Key_Tab)m_pEditor->handleKeyEvent((QKeyEvent *)e);
			else {
				//Loop thru fields
				KviOptDlgServerField field;
				switch(m_curEditedServerField){
					case Host: field = Port; break;
					case Port: field = Description; break;
					case Description: field = Password; break;
					case Password: field = Ip; break;
#ifdef COMPILE_NEED_IPV6
					case Ip: field = IpV6; break;
					case IpV6: field = Host; break;
#else
					case Ip: field = Host; break;
#endif
					default: field = Host; break;
				}
				//Accept the previous changes...
				m_pEditor->terminateEdit(true);
				//And edit the next field
				editServerField(m_curEditedServerItem,field);
			}
			((QKeyEvent *)e)->accept();
			return true;
		}
	} else if(m_pProxyEditor->isVisible()){
		//poor hack....but there is no other way...
		if((e->type() == QEvent::Accel)||(e->type() == QEvent::KeyPress)){
			if(((QKeyEvent *)e)->key() != Qt::Key_Tab)m_pProxyEditor->handleKeyEvent((QKeyEvent *)e);
			else {
				//Loop thru fields
				KviOptDlgProxyField field;
				switch(m_curEditedProxyField){
					case ProxyHost: field = ProxyPort; break;
					case ProxyPort: field = ProxyUsername; break;
					case ProxyUsername: field = ProxyPassword; break;
					case ProxyPassword: field = ProxyIp; break;
					case ProxyIp: field = ProxyHost; break;
					default: field = ProxyHost; break;
				}
				//Accept the previous changes...
				m_pProxyEditor->terminateEdit(true);
				//And edit the next field
				editProxyField(m_curEditedProxyItem,field);
			}
			((QKeyEvent *)e)->accept();
			return true;
		}
	} else {
		if((e->type() == QEvent::Accel)||(e->type() == QEvent::KeyPress)){
			if(m_pSrvList->hasFocus()){
				if(((QKeyEvent *)e)->key() == Qt::Key_Space){
					editServer();
					((QKeyEvent *)e)->accept();
				}
			} else if(m_pProxyList->hasFocus()){
				if(((QKeyEvent *)e)->key() == Qt::Key_Space){
					editProxy();
					((QKeyEvent *)e)->accept();
				}
			} else if(m_pNetCombo->hasFocus()){
				//Name of the current network changed ?
				if(((QKeyEvent *)e)->key() == Qt::Key_Return){
					((QKeyEvent *)e)->accept(); //always block
					KviIrcNetwork *net = m_pManager->getCurrentNetwork();
					__range_valid(net);
					if(!net)return true; //should not happen
					KviStr szNet=m_pNetCombo->currentText();
					if(!szNet.isEmpty()){
						KviIrcNetwork *newNet = m_pManager->getNetworkByName(szNet.ptr());
						//net != 0 now
						//newNet now can be == 0   --> new name for the current network
						//                  == net --> new name for the current network (changed just case)
						//          != 0 && != net --> another existing network name selected
						if(newNet && (newNet != net)){
							//another network name
							m_pManager->setCurrentNetwork(newNet);
						} else {
							//renamed the current network
							newNet = new KviIrcNetwork(net);
							newNet->setName(szNet.ptr());
							m_pManager->removeNetwork(net);
							m_pManager->insertNetwork(newNet);
						}
					}
					fillNetCombo();
				}
			}
			return true;
		}
	}
	return false;
}

//=============================================================================
//
// ### Proxy tab stuff ###
//
//=============================================================================

void KviDlgOptServer::fillProxyList()
{
	m_pProxyList->clear();
	if(m_pProxyManager->proxyList()->isEmpty()){
		//Disable everything
		enableProxyButtons(false);
		return;
	}
	//fill the server list
	KviIrcProxy * prx = m_pProxyManager->currentProxy();
	KviIrcProxyListItem *cur = 0;
	__range_valid(prx); //MUST BE SET!!!! the list is not empty!
	m_pProxyList->setUpdatesEnabled(false);
	int idx = 0;
	int curIdx = -1;
	for(KviIrcProxy *s = m_pProxyManager->proxyList()->first();s;s=m_pProxyManager->proxyList()->next()){
		KviIrcProxyListItem *it = new KviIrcProxyListItem(m_pProxyList,s);
		if(prx == s){
			m_pProxyList->setCurrentItem(it);
			m_pProxyList->setSelected(it,true);
			cur = it;
			curIdx = idx;
		}
		idx++;
	}
	m_pProxyList->setUpdatesEnabled(true);
	__range_valid(cur); //Must be set now!!!
	m_pProxyList->ensureItemVisible(cur);
	__range_valid(curIdx >= 0);
	enableProxyButtons(true);
}

void KviDlgOptServer::enableProxyButtons(bool bEnable)
{
	m_pEditProxy->setEnabled(bEnable);
	m_pDelProxy->setEnabled(bEnable);
}

void KviDlgOptServer::deleteProxy()
{
	KviIrcProxyListItem *it=(KviIrcProxyListItem *)m_pProxyList->currentItem();
	if(!it)return;
#ifdef _KVI_DEBUG_CHECK_RANGE_
	__range_valid(m_pProxyManager->removeProxy(it->m_ptr));
#else
	m_pProxyManager->removeProxy(it->m_ptr);
#endif
	fillProxyList();	
}

void KviDlgOptServer::editProxy()
{
	KviIrcProxyListItem *it=(KviIrcProxyListItem *)m_pProxyList->currentItem();
	if(it)editProxyField(it,ProxyHost);
}

void KviDlgOptServer::newProxy()
{
	KviIrcProxy *prx = new KviIrcProxy;
	prx->szHost = "proxy.localdomain";
	prx->szIp = "";
	prx->szUsername = "";
	prx->szPort = "1080";
	prx->szPassword = "";
	m_pProxyManager->appendProxy(prx);
	fillProxyList();
}

void KviDlgOptServer::editProxyField(KviIrcProxyListItem *it,
	KviOptDlgProxyField field)
{
	__range_valid(it);
	__range_invalid(m_pProxyEditor->isVisible());
	QRect rct = m_pProxyList->itemRect(it);
	int fieldWidth = m_pProxyList->columnWidth(field);
	int fieldX = - m_pProxyList->contentsX();
	for(int i = 0;i< field;i++)fieldX += m_pProxyList->columnWidth(i);
	m_pProxyEditor->move(fieldX,rct.top());
	m_pProxyEditor->resize(fieldWidth,rct.height());
	char * field_ptr;
	switch(field){
		case ProxyHost: field_ptr = it->m_ptr->szHost.ptr(); break;
		case ProxyPort: field_ptr = it->m_ptr->szPort.ptr(); break;
		case ProxyUsername: field_ptr = it->m_ptr->szUsername.ptr(); break;
		case ProxyPassword: field_ptr = it->m_ptr->szPassword.ptr(); break;
		case ProxyIp: field_ptr = it->m_ptr->szIp.ptr(); break;
		default: return; break;
	}
	m_curEditedProxyField = field;
	m_curEditedProxyItem = it;
	m_pProxyEditor->edit(field_ptr);
}

void KviDlgOptServer::proxyEditFinished(const char *text)
{
	__enter("proxyEditFinished");
	__range_valid(m_curEditedProxyItem);
	KviStr szT(text);

	switch(m_curEditedProxyField){
		case ProxyHost:
			szT.stripWhiteSpace();
			if((szT.findFirstIdx('.') != -1) && (szT.findFirstIdx(' ') == -1)){
				//must contain at least one dot and no spaces
				m_curEditedProxyItem->m_ptr->szHost = szT.ptr();
				m_curEditedProxyItem->setText(0,szT.ptr());
			}
			break;
		case ProxyPort:
			szT.stripWhiteSpace();
			if(!szT.isUnsignedNum())szT = "1080"; //reasonable default
			m_curEditedProxyItem->m_ptr->szPort = szT.ptr();
			m_curEditedProxyItem->setText(1,szT.ptr());
			break;
		case ProxyUsername:
			m_curEditedProxyItem->m_ptr->szUsername = szT.ptr();
			m_curEditedProxyItem->setText(2,szT.ptr());
			break;
		case ProxyPassword:
			m_curEditedProxyItem->m_ptr->szPassword = szT.ptr();
			m_curEditedProxyItem->setText(3,szT.ptr());
			break;
		case ProxyIp:
			szT.stripWhiteSpace();
//			if(kvi_isValidStringIp(szT.ptr())){
				m_curEditedProxyItem->m_ptr->szIp = szT.ptr();
				m_curEditedProxyItem->setText(4,szT.ptr());
//			}
			break;
		default:
			return;
			break;
	}
	m_pProxyList->setFocus();
}

void KviDlgOptServer::currentProxyChanged(QListViewItem *it)
{
	__enter("currentProxyChanged");
#ifdef _KVI_DEBUG_CHECK_RANGE_
	__range_valid(m_pProxyManager->setCurrentProxy(((KviIrcProxyListItem *)it)->m_ptr));
#else
	m_pProxyManager->setCurrentProxy(((KviIrcProxyListItem *)it)->m_ptr);
#endif
}

void KviDlgOptServer::proxyDoubleClicked(QListViewItem *i)
{
	KviIrcProxyListItem *it=(KviIrcProxyListItem *)i;
	QPoint pnt = QCursor::pos();
	pnt = m_pProxyList->viewport()->mapFromGlobal(pnt);
	int xLeft =m_pProxyList->columnWidth(0) - m_pSrvList->contentsX();
	if(pnt.x() < xLeft)editProxyField(it,ProxyHost);
	else {
		xLeft += m_pProxyList->columnWidth(1); //port
		if(pnt.x() < xLeft)editProxyField(it,ProxyPort);
		else {
			xLeft += m_pProxyList->columnWidth(2); //username
			if(pnt.x() < xLeft)editProxyField(it,ProxyUsername);
			else {
				xLeft += m_pProxyList->columnWidth(3); //password
				if(pnt.x() < xLeft)editProxyField(it,ProxyPassword);
				else {
					xLeft += m_pProxyList->columnWidth(4); //ip
					if(pnt.x() < xLeft)editProxyField(it,ProxyIp);
				}
			}
		}
	}
}

#include "m_kvi_dlgserver.moc"
