//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define _KVI_DEBUG_CHECK_RANGE_

#include "kvi_debug.h"
#include "kvi_scriptobject.h"
#include "kvi_scriptsocket.h"
#include "kvi_scriptwidget.h"
#include "kvi_scriptlayout.h"
#include "kvi_scriptmisc.h"
#include "kvi_uparser.h"
#include "kvi_varcache.h"
#include "kvi_window.h"
#include "kvi_defines.h"
#include "kvi_locale.h"
#include "kvi_error.h"
#include "kvi_console.h"

//
// Want to IRC ?  kvirc is the answer
// Want to talk to people around the world ? kvirc is the answer
// Want to browse your hd ? kvirc is the answer
// Want to play minesweeper ? kvirc is the answer
// Want to setup an interactive httpd daemon ? kvirc is the answer
//
// So, where don't you want to go today ;) ?
//

//////////////////////////////////////////////////////////////////////////////////////
//
//  Script object controller
//
//  Unique for each KviFrame object
//
//////////////////////////////////////////////////////////////////////////////////////

// This will overflow at 2^32 (4.294.967.296) , and when it will happen
// the lowest numbered objects will be SURELY dead
// (otherwise all the objects will occupy some terabytes of memory)
static unsigned int g_uObjectUniqueId = 1;

// Global class definitions
KviScriptObjectClassDefinition * g_pClassDefinitions = 0;

// Global list of object controllers
QList<KviScriptObjectController> * g_pObjectControllers = 0;

KviScriptObjectController::KviScriptObjectController(KviUserParser * pParser)
{
	m_pUserParser     = pParser;
	m_pTopLevelObject = 0;

	// First one creates the list
	if(!g_pObjectControllers)
	{
		// global init
		g_pObjectControllers = new QList<KviScriptObjectController>;
		g_pObjectControllers->setAutoDelete(false);
		allocateBuiltInClassDefinitions();
	}

	// register ourself
	g_pObjectControllers->append(this);

	// and create the toplevel object
	createTopLevelObject();
}

KviScriptObjectController::~KviScriptObjectController()
{
	// Kill the children one by one
	if(m_pTopLevelObject){
		delete m_pTopLevelObject;
		m_pTopLevelObject = 0;
	}

	// Last one kills the list
	g_pObjectControllers->removeRef(this);
	if(g_pObjectControllers)
	{
		if(g_pObjectControllers->count() == 0){
			// The last one closes the door
			delete g_pObjectControllers;
			g_pObjectControllers = 0;
			delete g_pClassDefinitions;
			g_pClassDefinitions = 0;
		}
	}
}

KviFrame * KviScriptObjectController::mainFrame()
{
	return m_pUserParser->m_pFrm;
}

void KviScriptObjectController::createTopLevelObject()
{
	m_pTopLevelObject = new KviScriptObject(this,0,"frame",g_pClassDefinitions);
}

void KviScriptObjectController::dumpClasses(KviWindow *wnd)
{
	g_pClassDefinitions->dump(wnd);
}

void KviScriptObjectController::dumpObjects(KviWindow *wnd)
{
	m_pTopLevelObject->dump(wnd);
}

void KviScriptObjectController::allocateBuiltInClassDefinitions()
{
// I changed this one a little bit - now it is easier to add new objects. If
// your object derrives from widget _and_ object its name must be added to
// widget_classes array.
// --
// Kristoff

	// Object
	g_pClassDefinitions = new KviScriptObjectClassDefinition("object", 0,
		true);
	KviScriptObject::initializeClassDefinition(g_pClassDefinitions);
// classes that derrive just from object
	void * misc_classes[] = {
	(void *)"socket",		(void *)KviScriptSocket::initializeClassDefinition,
	(void *)"dns",			(void *)KviScriptDns::initializeClassDefinition,
	(void *)"timer",		(void *)KviScriptTimer::initializeClassDefinition,
	(void *)"file",			(void *)KviScriptFile::initializeClassDefinition,
	(void *)"layout",		(void *)KviScriptLayout::initializeClassDefinition,
	(void *)"treeviewitem",		(void *)KviScriptTreeViewItem::initializeClassDefinition
				   };
// classes that derrive from widget
	void * widget_classes[] =	{
	// name, 		// init function
	(void *)"button",		(void *)KviScriptButton::initializeClassDefinition,
	(void *)"label",		(void *)KviScriptLabel::initializeClassDefinition,
	(void *)"toolbutton",		(void *)KviScriptToolButton::initializeClassDefinition,
	(void *)"lineedit",		(void *)KviScriptLineEdit::initializeClassDefinition,
	(void *)"multilineedit",	(void *)KviScriptMultiLineEdit::initializeClassDefinition,
	(void *)"toolbar",		(void *)KviScriptToolBar::initializeClassDefinition,
	(void *)"qtwrapper",		(void *)KviScriptQtWrapper::initializeClassDefinition,
	(void *)"checkbox",		(void *)KviScriptCheckBox::initializeClassDefinition,
	(void *)"splitter",		(void *)KviScriptSplitter::initializeClassDefinition,
	(void *)"popupmenu",		(void *)KviScriptPopupMenu::initializeClassDefinition,
	(void *)"spinbox",		(void *)KviScriptSpinBox::initializeClassDefinition,
	(void *)"combobox",		(void *)KviScriptComboBox::initializeClassDefinition,
	(void *)"tabwidget",		(void *)KviScriptTabWidget::initializeClassDefinition,
	(void *)"treeview",		(void *)KviScriptTreeView::initializeClassDefinition
					};

	KviScriptObjectClassDefinition * d1 = 0;
	void (*pFunc)(KviScriptObjectClassDefinition *) = 0;
	for(int i = 0; i < (int)(sizeof(misc_classes) / sizeof(misc_classes[0])); i++)
	{
		d1 = new KviScriptObjectClassDefinition((char *)misc_classes[i],
			g_pClassDefinitions, true);
		g_pClassDefinitions->addChildClassDefinition(d1);
		pFunc = (void(*)(KviScriptObjectClassDefinition*))misc_classes[++i];
		pFunc(d1);
	}
	
	// Widget
	d1 = new KviScriptObjectClassDefinition("widget", g_pClassDefinitions,
		true);
	KviScriptWidget::initializeClassDefinition(d1);
	g_pClassDefinitions->addChildClassDefinition(d1);

// Ok, the widget is ready, now we can add classes that derrive from it
	KviScriptObjectClassDefinition * d2 = 0;	
	for(int i=0; i < (int)(sizeof(widget_classes) / sizeof(widget_classes[0]));i++)
	{
		d2 = new KviScriptObjectClassDefinition((char *)widget_classes[i], d1, true);
		d1->addChildClassDefinition(d2);
		pFunc = (void(*)(KviScriptObjectClassDefinition*))widget_classes[++i];
		pFunc(d2);
	}
}

void KviScriptObjectController::globalReset()
{
	for(KviScriptObjectController * c = g_pObjectControllers->first();c;c=g_pObjectControllers->next())
	{
		delete c->m_pTopLevelObject;
		c->m_pTopLevelObject = 0;
	}

	__range_valid(g_pClassDefinitions);

	if(g_pClassDefinitions){
		delete g_pClassDefinitions;
		g_pClassDefinitions = 0;
		allocateBuiltInClassDefinitions();
	}

	for(KviScriptObjectController * c = g_pObjectControllers->first();c;c=g_pObjectControllers->next())
	{
		c->createTopLevelObject();
	}
}

bool KviScriptObjectController::killClass(const char * szClassName)
{
	KviScriptObjectClassDefinition * d = lookupClassDefinition(szClassName);
	if(!d)return false;

	killClass(d);
	return true;
}

void KviScriptObjectController::killClass(KviScriptObjectClassDefinition *d)
{
	// first of all delete the derived classes
	if(d->childClasses())
	{
		QList<KviScriptObjectClassDefinition> l(*(d->childClasses()));

		for(KviScriptObjectClassDefinition *chdef=l.first();chdef;chdef=l.next())
		{
			killClass(chdef);
		}
	}

	// now kill all the objects with THIS class
	for(KviScriptObjectController * c = g_pObjectControllers->first();c;c=g_pObjectControllers->next())
	{
		c->killAllLocalObjectsWithClass(d->getClass());
		if(c->m_pTopLevelObject == 0){
			// We were deleting the "object" class (so the toplevel object too!)
			__range_valid(d == g_pClassDefinitions); // object class destroyed!
			c->createTopLevelObject();
		}
	}

	// finally delete the class definition , if not builtin
	if(!(d->isBuiltin())){
		// Has at least one parent class!
		d->inheritedClass()->childClasses()->removeRef(d); //ouh! :)....a "deleted" path!
	}
}

void KviScriptObjectController::killAllLocalObjectsWithClass(const char *szClassName)
{
	while(KviScriptObject * o = findLocalObjectByClass(szClassName))delete o;
}

// Application wide unique ID
void KviScriptObjectController::getUniqueId(KviStr &idBuf)
{
	idBuf.setNum(g_uObjectUniqueId);
	idBuf.prepend("@");
	g_uObjectUniqueId++;
}

KviScriptObjectClassDefinition * KviScriptObjectController::lookupClassDefinition(const char *szClass)
{
	if(!g_pClassDefinitions){
		debug("No existing class definitions");
		return 0;
	}
	if(kvi_strEqualCI(g_pClassDefinitions->getClass(),szClass))return g_pClassDefinitions;
	return g_pClassDefinitions->lookupChildClassDefinition(szClass);
}

KviScriptObject * KviScriptObjectController::allocateObject(const char *szClass,const char *szName,KviScriptObject * par,QList<KviStr> * params,bool * pbDeletedParams)
{
	// builtin classes

	__range_valid(par);

	KviScriptObject *o = 0;
	KviScriptObjectClassDefinition *d = lookupClassDefinition(szClass);
	if(d)
	{
		KviScriptObjectClassDefinition *builtin = d;
		while(!builtin->isBuiltin())
		{
			builtin = builtin->inheritedClass();
			if(!builtin)debug("oops..something went wrong with class definitions ?");
		}
		if(kvi_strEqualCI(builtin->getClass(),"object"))
		{
			o = new KviScriptObject(this,par,szName,d);
		} else if(kvi_strEqualCI(builtin->getClass(),"socket"))
		{
			o = new KviScriptSocket(this,par,szName,d);
		} else if(kvi_strEqualCI(builtin->getClass(),"timer"))
		{
			o = new KviScriptTimer(this,par,szName,d);
		} else if(kvi_strEqualCI(builtin->getClass(),"dns"))
		{
			o = new KviScriptDns(this,par,szName,d);
		} else if(kvi_strEqualCI(builtin->getClass(),"widget"))
		{
			o = new KviScriptWidget(this,par,szName,d);
			if( !(((KviScriptWidget *)o)->init(0)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"file"))
		{
			o = new KviScriptFile(this,par,szName,d);
			if( !(((KviScriptFile *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"button"))
		{
			o = new KviScriptButton(this,par,szName,d);
			if( !(((KviScriptButton *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"splitter"))
		{
			o = new KviScriptSplitter(this,par,szName,d);
			if( !(((KviScriptSplitter *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"popupmenu"))
		{
			o = new KviScriptPopupMenu(this,par,szName,d);
			if( !(((KviScriptPopupMenu *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"toolbutton"))
		{
			o = new KviScriptToolButton(this,par,szName,d);
			if( !(((KviScriptToolButton *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"label"))
		{
			o = new KviScriptLabel(this,par,szName,d);
			if( !(((KviScriptLabel *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"layout"))
		{
			o = new KviScriptLayout(this,par,szName,d);
			if( !(((KviScriptLayout *)o)->init(0)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"lineedit"))
		{
			o = new KviScriptLineEdit(this,par,szName,d);
			if( !(((KviScriptLineEdit *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"multilineedit"))
		{
			o = new KviScriptMultiLineEdit(this,par,szName,d);
			if( !(((KviScriptMultiLineEdit *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"toolbar"))
		{
			o = new KviScriptToolBar(this,par,szName,d);
			if( !(((KviScriptToolBar *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"qtwrapper"))
		{
			o = new KviScriptQtWrapper(this,par,szName,d);
			if( !(((KviScriptQtWrapper *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"checkbox"))
		{
			o = new KviScriptCheckBox(this,par,szName,d);
			if( !(((KviScriptCheckBox *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"spinbox"))
		{
			o = new KviScriptSpinBox(this, par, szName, d);
			if( !(((KviScriptSpinBox *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"combobox"))
		{
			o = new KviScriptComboBox(this, par, szName, d);
			if( !(((KviScriptComboBox *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(),"tabwidget"))
		{
			o = new KviScriptTabWidget(this, par, szName, d);
			if( !(((KviScriptTabWidget *)o)->init(params)) )
			{
				// Failed to create it
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(), "treeviewitem"))
		{
			o = new KviScriptTreeViewItem(this, par, szName, d);
			if( !(((KviScriptTreeViewItem *)o)->init(params)) )
			{
				delete o;
				return 0;
			}
		} else if(kvi_strEqualCI(builtin->getClass(), "treeview"))
		{
			o = new KviScriptTreeView(this, par, szName, d);
			if( !(((KviScriptTreeView *)o)->init(params)) )
			{
				delete o;
				return 0;
			}
		}
		// need to insert the default event handlers from all the inherited classes
		KviScriptObjectClassDefinition *def = d;
		while(def)
		{
			for(KviScriptEventStruct * s = def->eventList()->first();s;s=def->eventList()->next())
			{
				// Insert unless we have already got it from a lower class in inheritance tree
				if(!(o->hasEventHandler(s->szName.ptr())))o->setEventHandler(s->szName.ptr(),s->szBuffer.ptr());
			}
			def = def->inheritedClass();
		}

	} else return 0;

	if(o)
	{
		KviStr buffer;

		*pbDeletedParams = true; // the params will be deleted by callFunction!

		if(o->callFunction("constructor",params,buffer) != KVI_ERROR_Success)
		{
			debug("Can't execute object constructor : implementation failure");
			delete o;
			return 0;
		}

		if(kvi_strEqualCI(buffer.ptr(),"0"))
		{
			debug("Object constructor failed (return value 0)");
			delete o;
			return 0;
		}
	}

	return o;
}
/*
KviScriptObject * KviScriptObjectController::findGlobalObjectById(const char *szId)
{
	KviScriptObject *o;
	for(KviScriptObjectController * c = g_pObjectControllers->first();c;c=g_pObjectControllers->next())
	{
		if(kvi_strEqualCI(c->m_pTopLevelObject->id(),szId))return c->m_pTopLevelObject;
		o = c->m_pTopLevelObject->findObjectById(szId);
		if(o)return o;
	}
	return 0;
}

KviScriptObject * KviScriptObjectController::findGlobalObjectByName(const char *szName)
{
	KviScriptObject *o;
	for(KviScriptObjectController * c = g_pObjectControllers->first();c;c=g_pObjectControllers->next())
	{
		if(kvi_strEqualCI(c->m_pTopLevelObject->getName(),szName))return c->m_pTopLevelObject;
		o = c->m_pTopLevelObject->findObjectByName(szName);
		if(o)return o;
	}
	return 0;
}

KviScriptObject * KviScriptObjectController::findGlobalObjectByClass(const char *szClass)
{
	KviScriptObject *o;
	for(KviScriptObjectController * c = g_pObjectControllers->first();c;c=g_pObjectControllers->next())
	{
		if(kvi_strEqualCI(c->m_pTopLevelObject->getClass(),szClass))return c->m_pTopLevelObject;
		o = c->m_pTopLevelObject->findObjectByClass(szClass);
		if(o)return o;
	}
	return 0;
}
*/
KviScriptObject * KviScriptObjectController::findLocalObjectByClass(const char *szClass)
{
	if(kvi_strEqualCI(m_pTopLevelObject->getClass(),szClass))return m_pTopLevelObject;
	return m_pTopLevelObject->findObjectByClass(szClass);
}
/*
KviScriptObject * KviScriptObjectController::findObjectByName(const char *szName)
{
	if(kvi_strEqualCI(m_pTopLevelObject->getName(),szName))return m_pTopLevelObject;
	return m_pTopLevelObject->findObjectByName(szName);
}
*/
KviScriptObject * KviScriptObjectController::findObjectById(const char *szId)
{
	KviScriptObject *o;
	for(KviScriptObjectController * c = g_pObjectControllers->first();c;c=g_pObjectControllers->next())
	{
		if(kvi_strEqualCI(c->m_pTopLevelObject->id(),szId))return c->m_pTopLevelObject;
		o = c->m_pTopLevelObject->findObjectById(szId);
		if(o)return o;
	}
	return 0;
//	if(kvi_strEqualCI(m_pTopLevelObject->id(),szId))return m_pTopLevelObject;
//	return m_pTopLevelObject->findObjectById(szId);
}
/*
	@class: object
	@short:
		The base class of all the objects.
	@inherits:
		itself only
	@functions:
		!fn: $name()
		Returns the name of this object.

		!fn: $class()
		Returns the class name of this object.

		!fn: $parent()		
		Returns the ID of the parent object or 0 if this is a toplevel object.

		!fn: $inheritedClasses()
		Returns a comma separated list of inherited classes.

		!fn: $inherits(&lt;class_name&gt;)
		Returns '1' if this object inherits &lt;class_name&gt;.
		An instance of a class always inherits the class itself (eg. object inherits object).

		!fn: $children()
		Returns a comma separated list of children ID , or nothing
		if this object has no children.
	
		!fn: $findChild(&lt;child_name&gt;)
		Searches for a direct child with a specified name and returns its ID
		if found, 0 otherwise.<br>
		If more than one child has the same name , only the first one is returned.
	
		!fn: $findDirectChild(&lt;child_name&gt;)
		Searches for a direct child with a specified name and returns its ID
		if found, 0 otherwise.<br>
		This function descends only one level in the children tree,
		and is really faster than $findChild().<br>
		If more than one child has the same name , only the first one is returned.

		!fn: $emit(&lt;signal_name&gt;[,parameters])
		Emits a specified signal, passing the (optional) parameters.<br>
		See the <a href="syntax_objects.kvihelp#signals">signals and slots documentation</a>.<br>

	@events:
		!ev: OnDestroy()
		Triggered just before the object destruction.

	@description:
		The base type of all the objects.<br>
*/

KviScriptObject::KviScriptObject(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: QObject(p,name)
{
	m_pClassDefinition = pDef;
	m_pController      = cntrl;
	m_bDying           = false;
	m_bInDelayedDestroy = false;
	m_pChildList       = new QList<KviScriptObject>;
	m_pChildList->setAutoDelete(true);
	m_pInheritanceList = new QList<KviStr>;
	m_pInheritanceList->setAutoDelete(true);
	m_pVarCache        = new KviVariableCache();
	m_pEventDict       = new QAsciiDict<KviStr>(5,false);
	m_pEventDict->setAutoDelete(true);
	m_pController->getUniqueId(m_szId);
//	m_bInEventOrFunction = false;
	m_bNeedRecreate      = false;
	m_pSlotDict          = new QAsciiDict<KviScriptObjectSlotList>(5,false);
	m_pSlotDict->setAutoDelete(true);
	m_pSignalList        = new QList<KviScriptObjectSignal>;
	m_pSignalList->setAutoDelete(true);
	KviScriptObjectClassDefinition *def = pDef;
	while(def)
	{
		m_pInheritanceList->append(new KviStr(def->getClass()));
		def = def->inheritedClass();
	}

	if(p)p->addChild(this);
}

KviScriptObject::~KviScriptObject()
{
	KviStr parms;
	triggerEvent("OnDestroy",parms);
	KviStr buffer;
	callFunction("destructor",0,buffer);

	m_bDying = true;


	// Kill the children one by one
	m_pChildList->setAutoDelete(false);
	while(m_pChildList->first())
	{
		KviScriptObject * tmp = m_pChildList->first();
		delete tmp; // <-- trigger ondestroy inside (*this* is still valid there! Tricky :)
		m_pChildList->removeRef(tmp);
	}
	delete m_pChildList;

	if(parentObject())parentObject()->removeChild(this,false);
	else {
		__range_valid(m_pController->m_pTopLevelObject == this);
		m_pController->m_pTopLevelObject = 0;
	}
	delete m_pInheritanceList;
	delete m_pEventDict;
	delete m_pVarCache;

	// disconnect all signals
	for(KviScriptObjectSignal * s = m_pSignalList->first();s;s = m_pSignalList->next())
	{
		s->pObject->unregisterSlot(s->szSignalName.ptr(),s->szSlotName.ptr(),this);
	}
	// notify all slots that we are dying (disconnect all the slots)
	QAsciiDictIterator<KviScriptObjectSlotList> it(*m_pSlotDict);
	while(KviScriptObjectSlotList * l = it.current())
	{
		for(KviScriptObjectSlot * sl = l->first();sl;sl = l->next())
		{
			sl->pObject->unregisterSignal(it.currentKey(),sl->szFunction.ptr(),this);
		}
		++it;
	}
	delete m_pSlotDict;
	delete m_pSignalList;

	if(m_bNeedRecreate)
	{
		__range_invalid(parentObject());
		m_pController->createTopLevelObject();
	}
}

void KviScriptObject::registerSlot(const char *signalName,const char *slotName,KviScriptObject *object)
{
	KviScriptObjectSlot * sl = new KviScriptObjectSlot;
	sl->pObject = object;
	sl->szFunction = slotName;
	KviScriptObjectSlotList * l = m_pSlotDict->find(signalName);
	if(!l){
		l = new KviScriptObjectSlotList;
		l->setAutoDelete(true);
		m_pSlotDict->insert(signalName,l);
	}
	l->append(sl);
}

void KviScriptObject::unregisterSlot(const char *signalName,const char *slotName,KviScriptObject *object)
{
	KviScriptObjectSlotList * l = m_pSlotDict->find(signalName);
	if(l){
		for(KviScriptObjectSlot * sl = l->first();sl;sl = l->next())
		{
			if(sl->pObject == object)
			{
				if(kvi_strEqualCI(sl->szFunction.ptr(),slotName))
				{
					l->removeRef(sl); //autodeleting!
					if(l->isEmpty())m_pSlotDict->remove(signalName);
					return;
				}
			}
		}
	}
	debug("No such slot (%s) for signal (%s)",slotName,signalName);
}

void KviScriptObject::registerSignal(const char *signalName,const char *slotName,KviScriptObject *object)
{
	KviScriptObjectSignal * s = new KviScriptObjectSignal;
	s->szSlotName = slotName;
	s->szSignalName = signalName;
	s->pObject = object;
	m_pSignalList->append(s);
}

void KviScriptObject::unregisterSignal(const char *signalName,const char *slotName,KviScriptObject *object)
{
	for(KviScriptObjectSignal * s = m_pSignalList->first();s;s= m_pSignalList->next())
	{
		if(s->pObject == object)
		{
			if(kvi_strEqualCI(s->szSlotName.ptr(),slotName))
			{
				if(kvi_strEqualCI(s->szSignalName.ptr(),signalName))
				{
					m_pSignalList->removeRef(s);
					return;
				}
			}
		}
	}
	debug("No such signal (%s) for slot (%s)",signalName,slotName);
}

void KviScriptObject::connectSignalSlot(KviScriptObject *src,const char *signalName,KviScriptObject *dst,const char *slotName)
{
	src->registerSlot(signalName,slotName,dst);
	dst->registerSignal(signalName,slotName,src);
}

void KviScriptObject::disconnectSignalSlot(KviScriptObject *src,const char *signalName,KviScriptObject *dst,const char *slotName)
{
	src->unregisterSlot(signalName,slotName,dst);
	dst->unregisterSignal(signalName,slotName,src);
}

void KviScriptObject::die(bool bDelayed)
{
	if(m_bInDelayedDestroy){
		debug("OOPS... OBJECT ALREADY IN ZOMBIE STATE");
		return;
	}
	if(bDelayed)dieOutOfThisEventStep();
	else delete this; // die now
}

void KviScriptObject::dieAndRecreate(bool bDelayed)
{
	m_bNeedRecreate = true;
	die(bDelayed);
}

void KviScriptObject::dieOutOfThisEventStep()
{
	if(m_bInDelayedDestroy){
		debug("OOPS... object already in zombie state");
		return;
	}
	m_bInDelayedDestroy = true;
	QTimer::singleShot(0,this,SLOT(delayedDie()));
}

void KviScriptObject::delayedDie()
{
//	if(inEventOrFunction())
//	{
//		debug("OOOPS : DELAYED DIE IN EVENT OR FUNCTION! (OBJECT ID %s) (OBJECT NAME %s) (OBJECT CLASS %s)",id(),getName(),getClass());
//	}
	delete this;
}

void KviScriptObject::triggerEvent(const char *evName,const KviStr &parms)
{
//	if(m_bInDelayedDestroy)return; // dangerous
//	bool bWasInEventOrFunction = m_bInEventOrFunction; // Nested function & event calls are allowed!
//	m_bInEventOrFunction = true;
	KviStr * pEvBuf = m_pEventDict->find(evName);
	if(pEvBuf)m_pController->m_pUserParser->triggerObjectEvent(this,evName,pEvBuf->ptr(),parms);
//	m_bInEventOrFunction = bWasInEventOrFunction;
}

void KviScriptObject::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("name",&KviScriptObject::builtinFunction_NAME);
	d->addBuiltinFunction("class",&KviScriptObject::builtinFunction_CLASS);
	d->addBuiltinFunction("parent",&KviScriptObject::builtinFunction_PARENT);
	d->addBuiltinFunction("inherits",&KviScriptObject::builtinFunction_INHERITS);
	d->addBuiltinFunction("inheritedClasses",&KviScriptObject::builtinFunction_INHERITEDCLASSES);
	d->addBuiltinFunction("children",&KviScriptObject::builtinFunction_CHILDREN);
	d->addBuiltinFunction("findDirectChild",&KviScriptObject::builtinFunction_FINDDIRECTCHILD);
	d->addBuiltinFunction("findChild",&KviScriptObject::builtinFunction_FINDCHILD);
	d->addBuiltinFunction("emit",&KviScriptObject::builtinFunction_EMIT);
}

bool KviScriptObject::emitSignal(const char *sigName,QList<KviStr> * params,KviStr &buffer)
{
	if(*sigName == '\0')return false;
	KviScriptObjectSlotList * slotsL = m_pSlotDict->find(sigName);
	if(slotsL)
	{
//		if(m_bInDelayedDestroy)return false; // dangerous
//		bool bWasInEventOrFunction = m_bInEventOrFunction; // Nested function & event calls are allowed!
//		m_bInEventOrFunction = true;
		// trick...copy the list of slots and use that one
		// since the slots may be unregistered inside its handlers
		KviScriptObjectSlotList l(*slotsL); //shallow copy
		l.setAutoDelete(false);
		int idx = 0;
		// loop thru all the slots
		for(KviScriptObjectSlot * sl = l.first();sl;sl = l.next())
		{
			// ensure that it has not been unregistered
			if(slotsL->findRef(sl) != -1)
			{
				// ok , slot not deleted
				if(idx > 0)buffer.append(',');
				idx++;
				// copy the param list (it will be deleted inside callFunction())
				QList<KviStr> * tmp = new QList<KviStr>;
				tmp->setAutoDelete(true);
				if(params){
					for(KviStr * s = params->first();s;s= params->next())tmp->append(new KviStr(*s));
				}
				// Call it
				m_pController->m_pUserParser->setLastSignalSender(id());
				int iRet = sl->pObject->callFunction(sl->szFunction.ptr(),tmp,buffer);
				// and eventually print the error
				if(iRet != KVI_ERROR_Success)
				{
					m_pController->m_pUserParser->m_pConsole->output(KVI_OUT_INTERNAL,"[parser] : Error in SLOT call: %s",kvi_getErrorString(iRet));
					m_pController->m_pUserParser->m_pConsole->output(KVI_OUT_INTERNAL,"[parser] : Caller object: %s class: %s name: %s",id(),getClass(),getName());
					m_pController->m_pUserParser->m_pConsole->output(KVI_OUT_INTERNAL,"[parser] : Caller signal: %s",sigName);
					m_pController->m_pUserParser->m_pConsole->output(KVI_OUT_INTERNAL,"[parser] : Receiver object: %s class: %s name: %s",sl->pObject->id(),sl->pObject->getClass(),sl->pObject->getName());
					m_pController->m_pUserParser->m_pConsole->output(KVI_OUT_INTERNAL,"[parser] : Receiver slot: %s",sl->szFunction.ptr());
				}
				// tmp should be deleted now!
			}
		}
	}
	return true;
}


int KviScriptObject::builtinFunction_EMIT(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			// trick.....remove the first param
			bool bAutoDelete = params->autoDelete(); //<--necessary ? or it is always true ?
			params->setAutoDelete(false);
			params->removeFirst();
			params->setAutoDelete(bAutoDelete);
			// emit the signal
			emitSignal(pS->ptr(),params,buffer);
			// reinsert it for deletion
			params->insert(0,pS);
			return KVI_ERROR_Success;
		}
	}

	return KVI_ERROR_MissingSignalName;
}

int KviScriptObject::builtinFunction_NAME(QList<KviStr> * params,KviStr &buffer)
{
	buffer.append(name());
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_CLASS(QList<KviStr> * params,KviStr &buffer)
{
	buffer.append(getClass());
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_PARENT(QList<KviStr> * params,KviStr &buffer)
{
	buffer.append(parentObject() ? parentObject()->id() : "0");
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_CHILDREN(QList<KviStr> * params,KviStr &buffer)
{
	bool bFirst = true;
	for(KviScriptObject * o = m_pChildList->first();o;o = m_pChildList->next())
	{
		if(!bFirst)
		{
			bFirst = false;
			buffer.append(',');
		} else bFirst = false;
		buffer.append(o->id());
	}
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_INHERITEDCLASSES(QList<KviStr> * params,KviStr &buffer)
{
	bool bFirst = true;
	for(KviStr * s = m_pInheritanceList->first();s;s = m_pInheritanceList->next())
	{
		if(!bFirst)
		{
			bFirst = false;
			buffer.append(',');
		} else bFirst = false;
		buffer.append(s->ptr());
	}
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_FINDDIRECTCHILD(QList<KviStr> * params,KviStr &buffer)
{
	KviStr * pStr = params->first();
	if(!pStr) return KVI_ERROR_MissingParameter;
	for(KviScriptObject * o = m_pChildList->first();o;o = m_pChildList->next())
	{
		if(kvi_strEqualCI(o->getName(),pStr->ptr()))
		{
			buffer.append(o->id());
			return KVI_ERROR_Success;
		}
	}
	buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_FINDCHILD(QList<KviStr> * params,KviStr &buffer)
{
	KviStr * pStr = params->first();
	if(!pStr)return KVI_ERROR_MissingParameter;
	KviScriptObject *o = findObjectByName(pStr->ptr());
	if(o)buffer.append(o->id());
	else buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_INHERITS(QList<KviStr> * params,KviStr &buffer)
{
	if(!params)return KVI_ERROR_MissingParameter;
	KviStr * pStr = params->first();
	if(!pStr)return KVI_ERROR_MissingParameter;
	for(KviStr * s = m_pInheritanceList->first();s;s = m_pInheritanceList->next())
	{
		if(kvi_strEqualCI(s->ptr(),pStr->ptr()))
		{
			buffer.append('1');
			return KVI_ERROR_Success;
		}
	}
	buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptObject::callFunction(const char *fncName,QList<KviStr> * params,KviStr &buffer,const char *classOverride)
{
	// This will delete the params!
	KviScriptObjectClassDefinition * d = classOverride ? m_pClassDefinition->lookupParentClassDefinition(classOverride) : m_pClassDefinition;
	if(!d){
		if(params)delete params;
		return KVI_ERROR_InvalidClassOverride;
	}
	KviScriptObjectFunctionStruct * f = d->lookupFunction(fncName);
	if(!f){
		if(params)delete params;
		return KVI_ERROR_NoSuchObjectFunction;
	}
	if(f->fncHandler){
		int retVal = (this->*(f->fncHandler))(params,buffer);
		if(params)delete params;
		return retVal;		
	} else {
		// The params will be deleted inside callObjectFunction
		bool bRet = m_pController->m_pUserParser->callObjectFunction(this,fncName,f->szBuffer.ptr(),params,buffer);
		return bRet ? KVI_ERROR_Success : KVI_ERROR_ObjectFunctionFailure;
	}
}

void KviScriptObject::addChild(KviScriptObject * o)
{
	childrenList()->append(o);
}

void KviScriptObject::dump(KviWindow *wnd,const char *margin)
{
	KviStr tmp(KviStr::Format,__tr("%s%cObject %s, class %s , name %s"),margin ? margin : "",KVI_TEXT_BOLD,id(),getClass(),getName());
	wnd->output(KVI_OUT_KVIRC,tmp.ptr());
	for(KviScriptObject * o = m_pChildList->first();o;o = m_pChildList->next())
	{
		KviStr marg = margin;
		marg.append("    ");
		o->dump(wnd,marg.ptr());
	}
}

void KviScriptObject::removeChild(KviScriptObject *o,bool bDelete)
{
	if(m_bDying)return; // All children will be removed by delete
	if(!bDelete)m_pChildList->setAutoDelete(false);
	if(!(m_pChildList->removeRef(o)))debug("Oops...no such child : %s",o->id());
	if(!bDelete)m_pChildList->setAutoDelete(true);
}

KviScriptObject * KviScriptObject::childByName(const char * szName)
{
	for(KviScriptObject * o = m_pChildList->first();o;o=m_pChildList->next())
	{
		if(kvi_strEqualCI(o->getName(),szName))return o;
	}
	return 0;
}

KviScriptObject * KviScriptObject::findObjectByName(const char *szName)
{
	for(KviScriptObject *o = m_pChildList->first();o;o = m_pChildList->next())
	{
		if(kvi_strEqualCI(o->getName(),szName))return o;
		o = o->findObjectByName(szName);
		if(o)return o;
	}

	return 0;
}

KviScriptObject * KviScriptObject::findObjectByClass(const char *szClass)
{
	for(KviScriptObject *o = m_pChildList->first();o;o = m_pChildList->next())
	{
		if(kvi_strEqualCI(o->getClass(),szClass))return o;
		o = o->findObjectByClass(szClass);
		if(o)return o;
	}

	return 0;
}

KviScriptObject * KviScriptObject::findObjectById(const char *szId)
{
//	debug("%s : Child findObjectById : %s",id(),szId);
	for(KviScriptObject *o = m_pChildList->first();o;o = m_pChildList->next())
	{
		if(kvi_strEqualCI(o->id(),szId))return o;
		o = o->findObjectById(szId);
		if(o)return o;
	}

	return 0;
}

void KviScriptObject::setEventHandler(const char *evName,const char *evBuffer)
{
	m_pEventDict->replace(evName,new KviStr(evBuffer));
}


//////////////////////////////////////////////////////////////////////////////////////
//
//  Script object class definition
//  All kept in an application-widget global list
//
//
//////////////////////////////////////////////////////////////////////////////////////


KviScriptObjectClassDefinition::KviScriptObjectClassDefinition(const char *className,KviScriptObjectClassDefinition * inheritedClass,bool bBuiltin)
{
	m_pInheritedClass = inheritedClass;
	m_szClassName     = className;
	m_pChildClasses   = new QList<KviScriptObjectClassDefinition>;
	m_pChildClasses->setAutoDelete(true);
	m_bBuiltin        = bBuiltin;
	m_pEventList      = new QList<KviScriptEventStruct>;
	m_pEventList->setAutoDelete(true);
	m_pFunctionDict   = new QAsciiDict<KviScriptObjectFunctionStruct>(23,false);
	m_pFunctionDict->setAutoDelete(true);
	if(inheritedClass)
	{
		QAsciiDictIterator<KviScriptObjectFunctionStruct> it(*(inheritedClass->m_pFunctionDict));
		while(it.current())
		{
			KviScriptObjectFunctionStruct * f = new KviScriptObjectFunctionStruct;
			f->fncHandler = it.current()->fncHandler;
			f->flags = it.current()->flags | KVI_SCRIPTOBJECTFUNCTION_FLAG_INHERITED;
			f->szBuffer   = it.current()->szBuffer;
			m_pFunctionDict->replace(it.currentKey(),f);
			++it;
		}
		if(inheritedClass->isBuiltin())
		{
			addFunction("constructor","setreturn 1",true);
			addFunction("destructor","setreturn 1",true);
		} else {
			KviStr tmp(KviStr::Format,"setreturn $this->$%s:constructor()",inheritedClass->getClass());
			addFunction("constructor",tmp.ptr(),true);
			tmp.sprintf("setreturn $this->$%s:destructor()",inheritedClass->getClass());
			addFunction("destructor",tmp.ptr(),true);
		}
	} else {
		addFunction("constructor","setreturn 1",true);
		addFunction("destructor","setreturn 1",true);
	}
}

KviScriptObjectClassDefinition::~KviScriptObjectClassDefinition()
{
	delete m_pChildClasses;
	delete m_pFunctionDict;
}

KviScriptObjectClassDefinition * KviScriptObjectClassDefinition::lookupParentClassDefinition(const char *szClass)
{
	if(m_pInheritedClass)
	{
		if(kvi_strEqualCI(m_pInheritedClass->getClass(),szClass))return m_pInheritedClass;
		return m_pInheritedClass->lookupParentClassDefinition(szClass);
	} else return 0;
}

KviScriptObjectClassDefinition * KviScriptObjectClassDefinition::lookupChildClassDefinition(const char *szClass)
{
	for(KviScriptObjectClassDefinition * d = m_pChildClasses->first();d;d=m_pChildClasses->next())
	{
		if(kvi_strEqualCI(d->getClass(),szClass))return d;
		KviScriptObjectClassDefinition  * child = d->lookupChildClassDefinition(szClass);
		if(child)return child;
	}
	return 0;
}

void KviScriptObjectClassDefinition::addDefaultEvent(KviScriptEventStruct *s)
{
	for(KviScriptEventStruct *str = m_pEventList->first();str;str=m_pEventList->next())
	{
		if(kvi_strEqualCI(str->szName.ptr(),s->szName.ptr()))
		{
			str->szBuffer = s->szBuffer;
			delete s;
			return;
		}
	}
	m_pEventList->append(s);
}

void KviScriptObjectClassDefinition::addFunction(const char *fncName,const char * buffer,bool bBuiltin)
{
	KviScriptObjectFunctionStruct * f = new KviScriptObjectFunctionStruct;
	f->szBuffer = buffer;
	f->fncHandler = 0;
	f->flags = bBuiltin ? KVI_SCRIPTOBJECTFUNCTION_FLAG_BUILTIN : 0;
	if(m_pInheritedClass)
	{
		KviScriptObjectFunctionStruct * old = m_pInheritedClass->m_pFunctionDict->find(fncName);
		if(old)f->flags |= KVI_SCRIPTOBJECTFUNCTION_FLAG_OVERRIDE;
	}
	m_pFunctionDict->replace(fncName,f);
}

void KviScriptObjectClassDefinition::addBuiltinFunction(const char *fncName,scriptObjectFunction fncHandler)
{
	KviScriptObjectFunctionStruct * f = new KviScriptObjectFunctionStruct;
	f->fncHandler = fncHandler;
	f->flags = KVI_SCRIPTOBJECTFUNCTION_FLAG_BUILTIN;
	if(m_pInheritedClass)
	{
		KviScriptObjectFunctionStruct * old = m_pInheritedClass->m_pFunctionDict->find(fncName);
		if(old)f->flags |= KVI_SCRIPTOBJECTFUNCTION_FLAG_OVERRIDE;
	}
	m_pFunctionDict->replace(fncName,f);
}

void KviScriptObjectClassDefinition::dump(KviWindow *wnd,const char *margin)
{
	KviStr tmp(KviStr::Format,__tr("%s%cClass definition : %s"),margin ? margin : "",KVI_TEXT_BOLD,m_szClassName.ptr());	
	wnd->output(KVI_OUT_KVIRC,tmp.ptr());
	tmp.sprintf(__tr("%sInherited class : %s"),margin ? margin : "",m_pInheritedClass ? m_pInheritedClass->m_szClassName.ptr() : __tr("none"));
	wnd->output(KVI_OUT_KVIRC,tmp.ptr());
	tmp.sprintf(__tr("%sIs builtin ? : %s"),margin ? margin : "",isBuiltin() ? __tr("yes") : __tr("no"));
	wnd->output(KVI_OUT_KVIRC,tmp.ptr());
	for(KviScriptObjectClassDefinition * d = m_pChildClasses->first(); d ; d = m_pChildClasses->next())
	{
		KviStr marg = margin;
		marg.append("    ");
		d->dump(wnd,marg.ptr());
	}
}


#include "m_kvi_scriptobject.moc"
