#ifndef _KVI_STAT_CONTROLLER_H_INCLUDED_
#define _KVI_STAT_CONTROLLER_H_INCLUDED_

// =============================================================================
//
//        --- kvi_stat_controller.h ---
//
//   This is a plugin for KVIrc 2. It allows you to see
//   statistics of your IRC sessions - total words said etc.
//   More features are still to come.
//
//   (c) Krzysztof Godlewski <kristoff@poczta.wprost.pl>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#include <qobject.h>
#include <qptrlist.h>

#include "kvi_string.h"

class KviStatChan;
class KviStatSysTray;

/**
 * Main object: acts as a filter, data keeper and so on...
 */
typedef struct
{
	// Some options:
	bool          sysTrayOnStartup;

	// Type of stat showing on channel join (No/Full/Custom)
	int           joinStatType;
	// On channel join show:
	bool          showJoins;
	bool          showWords;
	bool          showKicks;
	bool          showBans;
	bool          showTopics;
	// In systray widget show:
	bool          STWshowJoins;
	bool          STWshowWords;
	bool          STWshowKicks;
	bool          STWshowBans;
	bool          STWshowTopics;
	// Systray widget scrolling options:
	int           ScrollingWhen;
	int           ScrollingDelay;
	int           ScrollingTextJump;
	bool          ScrollingDirLeft;
} KviStatConfig;

typedef struct
{
	unsigned int  queriesWords;
	unsigned int  queriesLetters;
	unsigned int  dccsWords;
	unsigned int  dccsLetters;
	unsigned int  chansWords;
	unsigned int  chansLetters;
	unsigned int  totalWords;
	unsigned int  totalLetters;
	unsigned int  kicks;
	unsigned int  bans;
	unsigned int  onIrc;
	unsigned int  joins;
	unsigned int  topics;
	unsigned int  sessionWords;
	unsigned int  wordsRecord;
	KviStr        startDate;
} KviStatData;

class KviStatController : public QObject
{
	Q_OBJECT
public:
	KviStatController();
	~KviStatController();

protected:
	QPtrList<KviStatSysTray> *m_pSysTrayList;
	QPtrList<KviStatChan>    *m_pStatChanList;
	KviStatData               m_stats;
	KviStatConfig             m_config;
	bool                      m_bShowConsoleStats;

private:
	void loadStats();
	void loadConfig();
	void saveStats()  const;
	void saveConfig() const;

public:
	void registerStatTray  (KviStatSysTray *ptr);
	void unregisterStatTray(KviStatSysTray *ptr);

	void setCurrentChan(KviStatChan *ptr);
	KviStatChan *const findStatChan(const char *channame) const;
	const KviStatChan *const currentChan() const;

	void addChan   (KviStatChan *ptr);
	void removeChan(KviStatChan *ptr);

	const KviStatData &statData() const;
	QPtrList<KviStatChan> *const chanList() const;

	bool showConsoleStats() const;
	bool sysTrayOnStartup() const;

	// Add to number of total actions:
	void addTotalKick();
	void addTotalBan();
	void addOnIrc();
	void addTotalJoin();
	void addTotalTopic();

	// Add letters to total stats:
	void addChansLetters(unsigned int num);
	void addQueriesLetters(unsigned int num);
	void addDccsLetters(unsigned int num);
	void addTotalLetters(unsigned int num);

	// Add words to total stats:
	void addChansWords(unsigned int num);
	void addQueriesWords(unsigned int num);
	void addDccsWords(unsigned int num);
	void addTotalWords(unsigned int num);

	// Return number of total words...
	unsigned int totalWords()   const;
	unsigned int chansWords()   const;
	unsigned int queriesWords() const;
	unsigned int dccsWords()    const;

	// ... and letters :
	unsigned int totalLetters()   const;
	unsigned int chansLetters()   const;
	unsigned int queriesLetters() const;
	unsigned int dccsLetters()    const;

	// Return some total stats:
	unsigned int numOnIrc()  const;
	unsigned int numKicks()  const;
	unsigned int numBans()   const;
	unsigned int numTopics() const;
	unsigned int numJoins()  const;

	// Return last reset date (stats start date)
	const char *startDate() const;
	// Add session words
	void addSessionWords(int num);
	// This session words
	unsigned int sessionWords() const;
	// Reset session words
	void resetSessionWords();
	// Session record
	unsigned int wordsRecord() const;
	// Set words record
	void setWordsRecord(unsigned int n);

	unsigned int joinStatType() const;
	// Checks for plugin options
	bool showJoins()  const;
	bool showWords()  const;
	bool showKicks()  const;
	bool showBans()   const;
	bool showTopics() const;

	bool doReset();

protected:
	virtual bool eventFilter(QObject *, QEvent *);

public slots:
	void slotKillOptionsWindow();
	void slotKillStatWindow();
	void slotApplyOptions();
	void setSysTrayOptions();
	void slotShowConfig();
	void slotShowStats();
	void slotReset();
	void slotSaveStats();
	void slotRemoveAllChans();
};

#endif // _KVI_STAT_CONTROLLER_H_INCLUDED_
