// =============================================================================
//
//      --- kvi_functions.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviFunctions"

#include "kvi_settings.h"

#if defined(HAVE_SYS_UTSNAME_H) && defined(HAVE_UNAME)
	#include <sys/utsname.h>
#endif

#include <qfile.h>
#include <qregexp.h>

#include "kvi_app.h"
#include "kvi_banmask.h"
#include "kvi_channel.h"
#include "kvi_command.h"
#include "kvi_config.h"
#include "kvi_error.h"
#include "kvi_fileutils.h"
#include "kvi_frame.h"
#include "kvi_input.h"
#include "kvi_irc_user.h"
#include "kvi_irc_userlist.h"
#include "kvi_locale.h"
#include "kvi_malloc.h"
#include "kvi_netutils.h"
#include "kvi_options.h"
#include "kvi_plugin.h"
#include "kvi_regusersdb.h"
#include "kvi_script_object.h"
#include "kvi_script_objectcontroller.h"
#include "kvi_userlistbox.h"
#include "kvi_userparser.h"
#include "kvi_userwindow.h"
#include "kvi_variablecache.h"

// TODO: $frameCaption

// Created in KviApp constructor
extern KviVariableCache *g_pVarCache;
extern KviConfig        *g_pUserConfig;

#ifdef COMPILE_PLUGIN_SUPPORT
	extern KviPluginManager *g_pPluginManager;
#endif

KviFunctionEntry KviUserParser::fncTable_A_G[] =
{
	{ "ALLONCHAN"       , &KviUserParser::parseFnc_ALLONCHAN         }, // AllOnChan(<channel>)
	{ "ASCII"           , &KviUserParser::parseFnc_ASCII             },
	{ "CALC"            , &KviUserParser::parseFnc_CALC              },
	{ "GETENV"          , &KviUserParser::parseFnc_GETENV            },
	{ "DEFLOGFILE"      , &KviUserParser::parseFnc_DEFLOGFILE        }, // DefLogFile(<window>)
	{ "CHAR"            , &KviUserParser::parseFnc_CHAR              },
	{ "COMMANDLINE"     , &KviUserParser::parseFnc_COMMANDLINE       }, // Commandline(<window>)
	{ "FILEEXISTS"      , &KviUserParser::parseFnc_FILEEXISTS        }, // FileExists(<filename>)
	{ "DIREXISTS"       , &KviUserParser::parseFnc_DIREXISTS         }, // DirExists(<dirname>)
	{ "FILESIZE"        , &KviUserParser::parseFnc_FILESIZE          }, // FileSize(<filename>)
	{ "DICTLIST"        , &KviUserParser::parseFnc_DICTLIST          }, // DictList(<dictname>)
	{ "DICTCOUNT"       , &KviUserParser::parseFnc_DICTCOUNT         }, // DictCount(<dictname>)
	{ "GETFLAGS"        , &KviUserParser::parseFnc_GETFLAGS          }, // GetFlags(<mask>)
	{ "GETFLAGSEXACT"   , &KviUserParser::parseFnc_GETFLAGSEXACT     }, // GetFlags(<mask>)
	{ "GETPASS"         , &KviUserParser::parseFnc_GETPASS           }, // GetPass(<mask>)
	{ "GETCOMMENT"      , &KviUserParser::parseFnc_GETCOMMENT        }, // GetComment(<mask>)
	{ "CONFIG"          , &KviUserParser::parseFnc_CONFIG            }, // Config(<section>, <key>)
	{ "DURATIONTOSTRING", &KviUserParser::parseFnc_DURATIONTOSTRING  }, // DurationToString(<durationinsecs>)
	{ "CLASSDEFINED"    , &KviUserParser::parseFnc_CLASSDEFINED      },
	{ "CHANMODE"        , &KviUserParser::parseFnc_CHANMODE          },
	{ "CHANLIMIT"       , &KviUserParser::parseFnc_CHANLIMIT         },
	{ "CHANKEY"         , &KviUserParser::parseFnc_CHANKEY           },
	{ "BANLIST"         , &KviUserParser::parseFnc_BANLIST           },
	{ "BANEXCEPTIONLIST", &KviUserParser::parseFnc_BANEXCEPTIONLIST  },
	{ 0                 , 0                                          }
};

// TODO: $StripWhiteSpace()

KviFunctionEntry KviUserParser::fncTable_H_N[] =
{
	{ "ISME"            , &KviUserParser::parseFnc_ISME              }, // IsMe(nickname)
	{ "ISNUMBER"        , &KviUserParser::parseFnc_ISNUMBER          },
	{ "ISON"            , &KviUserParser::parseFnc_ISON              }, // IsOn(#kaos, pragma)
	{ "ISOWNER"         , &KviUserParser::parseFnc_ISOWNER           },
	{ "ISOP"            , &KviUserParser::parseFnc_ISOP              }, // IsOp(#kaos, pragma)
	{ "ISHALFOP"        , &KviUserParser::parseFnc_ISHALFOP          },
	{ "ISVOICE"         , &KviUserParser::parseFnc_ISVOICE           }, // IsVoice(#kaos, pragma)
	{ "ISUSEROP"        , &KviUserParser::parseFnc_ISUSEROP          },
	{ "ICON"            , &KviUserParser::parseFnc_ICON              }, // Icon(<iconname>)
	{ "ISREG"           , &KviUserParser::parseFnc_ISREG             }, // IsReg(<mask>)
	{ "ISWINDOW"        , &KviUserParser::parseFnc_ISWINDOW          }, // IsWindow(<window name>)
	{ "ISDOCKED"        , &KviUserParser::parseFnc_ISDOCKED          }, // IsDocked(<window name>)
	{ "ISKNOWN"         , &KviUserParser::parseFnc_ISKNOWN           }, // IsKnown(<nick>)
	{ "ISWELLKNOWN"     , &KviUserParser::parseFnc_ISWELLKNOWN       }, // IsWellKnown(<nick>)
	{ "ISVALIDIP"       , &KviUserParser::parseFnc_ISVALIDIP         }, // IsValidIp(<ipaddress>)
	{ "INETATON"        , &KviUserParser::parseFnc_INETATON          }, // IpToStr(<number>)
	{ "INETNTOA"        , &KviUserParser::parseFnc_INETNTOA          }, // StrToIp(<ip_address>)
	{ "LOWCASE"         , &KviUserParser::parseFnc_LOWCASE           }, // Lowcase(<string>);
	{ "MASK"            , &KviUserParser::parseFnc_MASK              }, // Mask(<type>, <nick>)
	{ "NOPONCHAN"       , &KviUserParser::parseFnc_NOPONCHAN         }, // NOpOnChan(<channel>)
	{ "ITEMCOUNT"       , &KviUserParser::parseFnc_ITEMCOUNT         }, // ItemCount(<window>)
	{ "ISTIMER"         , &KviUserParser::parseFnc_ISTIMER           }, // IsTimer(<timer_mask>);
	{ "HOSTNAME"        , &KviUserParser::parseFnc_HOSTNAME          }, // Hostname(<nick>)
	{ "NUMTIMETOSTRING" , &KviUserParser::parseFnc_NUMTIMETOSTRING   }, // NumTimeToString(<timeinsecs>)
	{ "HASEVENTHANDLER" , &KviUserParser::parseFnc_HASEVENTHANDLER   },
	{ "NEW"             , &KviUserParser::parseFnc_NEW               }, // Obj_Create(...)
	{ "HEXTOSTR"        , &KviUserParser::parseFnc_HEXTOSTR          }, // HexToStr()
	{ "INVITEXCEPTIONLIST", &KviUserParser::parseFnc_INVITEEXCEPTIONLIST },
	{ 0                 , 0                                          }
};

KviFunctionEntry KviUserParser::fncTable_O_S[] =
{
	{ "SELECTED"          , &KviUserParser::parseFnc_SELECTED            }, // Selected(#kaos)
	{ "SELECTEDMASKS"     , &KviUserParser::parseFnc_SELECTEDMASKS       }, // SelectedMasks(#kaos)
	{ "SELECTEDHOSTS"     , &KviUserParser::parseFnc_SELECTEDHOSTS       }, // SelectedHosts(#kaos)
	{ "SELECTEDUSERNAMES" , &KviUserParser::parseFnc_SELECTEDUSERNAMES   }, // SelectedUsernames(#kaos)
	{ "STRLEN"            , &KviUserParser::parseFnc_STRLEN              },
	{ "STRCAT"            , &KviUserParser::parseFnc_STRCAT              },
	{ "STRLEFT"           , &KviUserParser::parseFnc_STRLEFT             }, // StrLeft(<len>, <string>)
	{ "STRRIGHT"          , &KviUserParser::parseFnc_STRRIGHT            }, // StrRight(<len>, <string>)
	{ "STRCUTLEFT"        , &KviUserParser::parseFnc_STRCUTLEFT          }, // StrCutLeft(<len>, <string>)
	{ "STRCUTRIGHT"       , &KviUserParser::parseFnc_STRCUTRIGHT         }, // StrCutRight(<len>, <string>)
	{ "STRMID"            , &KviUserParser::parseFnc_STRMID              }, // StrMid(<idx>, <len>, <string>)
	{ "STRFIND"           , &KviUserParser::parseFnc_STRFIND             }, // StrFind(<substr>, <string>)
	{ "STRFINDCS"         , &KviUserParser::parseFnc_STRFINDCS           }, // StrFindCS(<substr>, <string>)
	{ "STRREVFIND"        , &KviUserParser::parseFnc_STRREVFIND          }, // StrRevFind(<substr>, <string>)
	{ "STRREVFINDCS"      , &KviUserParser::parseFnc_STRREVFINDCS        }, // StrRevFindCS(<substr>, <string>)
	{ "STRREPLACE"        , &KviUserParser::parseFnc_STRREPLACE          }, // StrReplace(<string>, <toFind>, <replacement>)
	{ "STRREPLACECS"      , &KviUserParser::parseFnc_STRREPLACECS        }, // StrReplaceCS(<string>, <toFind>, <replacement>)
	{ "STRLEFTTOFIRST"    , &KviUserParser::parseFnc_STRLEFTTOFIRST      }, // StrLeftToFirst(<string>, <string>)
	{ "STRLEFTTOLAST"     , &KviUserParser::parseFnc_STRLEFTTOLAST       }, // StrLeftToLast(<string>, <string>)
	{ "STRRIGHTFROMFIRST" , &KviUserParser::parseFnc_STRRIGHTFROMFIRST   }, // StrRightFromFirst(<string>, <string>)
	{ "STRRIGHTFROMLAST"  , &KviUserParser::parseFnc_STRRIGHTFROMLAST    }, // StrRightFromLast(<string>, <string>)
	{ "STRLEFTTOFIRSTCS"  , &KviUserParser::parseFnc_STRLEFTTOFIRST      }, // StrLeftToFirstCS(<string>, <string>)
	{ "STRLEFTTOLASTCS"   , &KviUserParser::parseFnc_STRLEFTTOLAST       }, // StrLeftToLastCS(<string>, <string>)
	{ "STRRIGHTFROMFIRSTCS", &KviUserParser::parseFnc_STRRIGHTFROMFIRSTCS },// StrRightFromFirstCS(<string>, <string>)
	{ "STRRIGHTFROMLASTCS", &KviUserParser::parseFnc_STRRIGHTFROMLASTCS  }, // StrRightFromLastCS(<string>, <string>)
	{ "STRMATCH"          , &KviUserParser::parseFnc_STRMATCH            }, // StrMatch(<string1>, <string2>)
	{ "STRMATCHCS"        , &KviUserParser::parseFnc_STRMATCHCS          }, // StrMatchCS(<string1>, <string2>)
	{ "STRTOHEX"          , &KviUserParser::parseFnc_STRTOHEX            },
	{ "RAND"              , &KviUserParser::parseFnc_RAND                }, // Rand(<max>) (max is never returned, is sup)
	{ "OPTION"            , &KviUserParser::parseFnc_OPTION              }, // Option(<option name>)
	{ "RANGE"             , &KviUserParser::parseFnc_RANGE               }, // Range(<min>, <max>, <step>)
	{ "OPONCHAN"          , &KviUserParser::parseFnc_OPONCHAN            }, // OpOnChan(<channel>)
	{ "READFILE"          , &KviUserParser::parseFnc_READFILE            }, // Readfile(<filename>)
	{ "PLUGINLOADED"      , &KviUserParser::parseFnc_PLUGINLOADED        }, // PluginLoaded(<pluginname>)
	{ 0                   , 0                                            }
};

KviFunctionEntry KviUserParser::fncTable_T_Z[] =
{
	{ "TOPIC"           , &KviUserParser::parseFnc_TOPIC             }, // Topic(#channel)
	{ "UPCASE"          , &KviUserParser::parseFnc_UPCASE            }, // Upcase(<string>)
	{ "VOICEONCHAN"     , &KviUserParser::parseFnc_VOICEONCHAN       }, // VoiceOnChan(<channel>)
	{ "UNAME"           , &KviUserParser::parseFnc_UNAME             }, // Uname(sysname|nodename|release|version|machine|domainname)
	{ "USERNAME"        , &KviUserParser::parseFnc_USERNAME          }, // Username(<nick>)
	{ 0                 , 0                                          }
};

const char *iconNamesTable[] = {
	"none",            "internal",          "error",
	"kvirc",           "echo",              "socket",
	"info",            "unhandled",         "motd",
	"unhandled_error", "nickname_error",    "ping",
	"join",            "topic",             "time",
	"desync",          "names",             "part",
	"kick",            "quit",              "quitsplit",
	"split",           "umode",             "op",
	"deop",            "voice",             "devoice",
	"ban",             "unban",             "except",
	"unexcept",        "key",               "limit",
	"chanmode",        "who",               "privmsg",
	"ctcp_request",    "ctcp_reply",        "ctcp_error",
	"raw",             "own",               "antispam",
	"nick",            "flood",             "action",
	"notice",          "dns",               "watch",
	"highlight",       "invite",            "dccinfo",
	"dccwarning",      "dccerror",          "help",
	"stdin",           "stdout",            "stderr",
	"directory",       "file",              "plugin",
	"happy",           "unhappy",           "angry",
	"kill",            "killed",            "multimedia",
	"wallops",         "log",               "idea",
	"input",           "packet",            "talk",
	"script",          "colors",            "onotice",
	"mirc",            "widget",            "server_error",
	"console",         "channel",           "query",
	"chat",            "send",              "helpwindow",
	"dirbrowser",      "voice",             "links",
	"list",            "uwindow"
};

/*
	@function: Icon
	@short:
		Finds icon numbers by descriptive names
	@syntax:
		$Icon(<icon_name>)
	@description:
		Returns the number associated with a specified icon descriptive name.<br>
		The available icon names are:<br>
		none, internal, error, kvirc, echo, socket, <br>
		info, unhandled, motd, unhandled_error, nickname_error, ping, <br>
		join, topic, time, desync, names, part, <br>
		kick, quit, quitsplit, split, umode, op, <br>
		deop, voice, devoice, ban, unban, except, <br>
		unexcept, key, limit, chanmode, who, privmsg, <br>
		ctcp_request, ctcp_reply, ctcp_error, raw, own, antispam, <br>
		nick, flood, action, notice, dns, watch, <br>
		highlight, invite, dccinfo, dccwarning, dccerror, help, <br>
		stdin, stdout, stderr, directory, file, plugin, <br>
		happy, angry, unhappy, kill, killed, multimedia, wallops, log, <br>
		idea, input, packet, talk, script, colors, onotice, mirc, <br>
		widget, server_error, console, channel, query, chat, send, <br>
		helpwindow, dirbrowser, voice, links, list, uwindow.<br>
		This function returns 0 if the &lt;icon_name&gt; does not match
		any name in this list.<br>
		In time critical functions is REALLY better to use directly
		the number of the requested icon instead of this identifier.<br>
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> -i $icon(ping) Ping Pong!
		</example>
	@seealso:
		<a href="echo.kvihelp">echo</a><br>
		<a href="popup.kvihelp">popup</a><br>
*/

bool KviUserParser::parseFnc_ICON(KviCommand *cmd, KviStr &buffer)
{
	KviStr icon_name;
	if( !processFncFinalPart(cmd, icon_name) )
		return false;

	// There are NO spaces in icon names.
	// This is to help the user, otherwise he gets an error when using an "iconname "
	icon_name.stripWhiteSpace();

	icon_name.toLower();
	for( int i = 0; iconNamesTable[i]; i++ ) {
		if( kvi_strEqualNoLocaleCI(iconNamesTable[i], icon_name.ptr()) ) {
			icon_name.setNum(i);
			buffer.append(icon_name);
			return true;
		}
	}
	buffer.append('0');
	return true;
}

/*
	@function: Option
	@short:
		Read option values
	@syntax:
		$Option(&lt;option_name&gt;)
	@description:
		Returns the current value for the specified option.<br>
		The options are of three types: integer, boolean and string.<br>
		For integer options this function returns... yes... a (signed) integer value.<br>
		For boolean options returns '1' for 'true' or 'enabled' and
		'0' for 'false' or 'disabled'.<br>
		For string options returns a string value that may also be empty.<br>
		The option names usually correspond to the entries of the
		kvi.*.conf files (look in the config subdir of the "KVIrc home directory").<br>
		This function aborts with an error if the &lt;option_name&gt; does not match
		any known option name.<br>
		/<a href="option.kvihelp">OPTION</a> -l shows a list of available option names
		and their relative types.
	@examples:
		<example>
		<a href="echo.kvihelp">echo</a> $option(PartMessage)
		</example>
	@seealso:
		<a href="option.kvihelp">option</a>
*/

bool KviUserParser::parseFnc_OPTION(KviCommand *cmd, KviStr &buffer)
{
	KviStr option_name;
	if( !processFncFinalPart(cmd, option_name) )
		return false;

	option_name.stripWhiteSpace(); // There are NO spaces in option names

	KviOptionEntry *optTable;

	switch( tolower(*(option_name.ptr())) ) {
		case 'c': optTable = g_pOptions->optColorsTable;  break;
		case 'b': optTable = g_pOptions->optBoolTable;    break;
		case 'f': optTable = g_pOptions->optFontsTable;   break;
		case 's': optTable = g_pOptions->optStringsTable; break;
		case 't': optTable = g_pOptions->optTextTable;    break;
		case 'i': optTable = g_pOptions->optIntTable;     break;
		default:
			cmd->setError(KVI_ERROR_NoSuchOption, "$Option()", option_name.ptr());
			return false;
		break;
	}

	for( int i = 0; optTable[i].optName; i++ ) {
		if( kvi_strEqualNoLocaleCI(optTable[i].optName, option_name.ptr()) ) {
			switch( optTable[i].optType & KVI_OPT_TYPE_MASK ) {
				case KVI_OPT_BOOL:
					buffer.append(((*((bool *) (optTable[i].optValue))) ? '1' : '0'));
					break;
				case KVI_OPT_STR:
					buffer.append(*((KviStr *) (optTable[i].optValue)));
					break;
				case KVI_OPT_CLR: {
					KviStr tmp(KviStr::Format, "%d,%d,%d",
						((QColor *) (optTable[i].optValue))->red(),
						((QColor *) (optTable[i].optValue))->green(),
						((QColor *) (optTable[i].optValue))->blue()
					);
					buffer.append(tmp);
					break;
				}
				case KVI_OPT_TXTF: // Fall-through
				case KVI_OPT_TXTB: {
					KviStr tmp(KviStr::Format, "%u", *((unsigned char *) (optTable[i].optValue)));
					buffer.append(tmp);
					break;
				}
				case KVI_OPT_INT: {
					KviStr tmp(KviStr::Format, "%d", *((int *) (optTable[i].optValue)));
					buffer.append(tmp);
					break;
				}
				case KVI_OPT_PCLR: {
					if( ((QColor **) (optTable[i].optValue)) ) {
						KviStr tmp(KviStr::Format, "%d,%d,%d",
							(*((QColor **) (optTable[i].optValue)))->red(),
							(*((QColor **) (optTable[i].optValue)))->green(),
							(*((QColor **) (optTable[i].optValue)))->blue()
						);
						buffer.append(tmp);
					} else buffer.append("Oops... null pointer?");
					break;
				}
				case KVI_OPT_FNT: {
					KviStr tmp;
					KviConfig::getFontProperties(tmp, ((QFont *) (optTable[i].optValue)));
					buffer.append(tmp);
					break;
				}
				default: // Never here
					break;
			}
			return true;
		}
	}
	cmd->setError(KVI_ERROR_NoSuchOption, "$Option()", option_name.ptr());
	return false;
}

/*
	@function: ClassDefined
	@short:
		Checks if a class is defined
	@syntax:
		$classDefined(&lt;class_name&gt;)
	@description:
		Returns 1 if a the class &lt;class_name&gt; is defined
		0 otherwise.<br>
	@seealso:
		<a href="class.kvihelp">CLASS</a><br>
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/
bool KviUserParser::parseFnc_CLASSDEFINED(KviCommand *cmd, KviStr &buffer)
{
	KviStr className;
	if( !processFncFinalPart(cmd, className) )
		return false;

	buffer.append(m_pScriptObjectController->lookupClassDefinition(className.ptr()) ? '1' : '0');
	return true;
}

/*
	@function: NEW
	@short:
		Allocates a new object
	@syntax:
		$new(&lt;object_class&gt;, &lt;object_parent&gt;, &lt;object_name&gt;[, constructor_parameters])
	@description:
		Allocates a new object of class &lt;object_class&gt;,
		name &lt;object_name&gt; and the specified parent, and returns its unique identifier.<br>
		The &lt;object_parent&gt; may be <a href="s_root.kvihelp">$root</a> for top level objects.<br>
		If the &lt;object_type&gt; is invalid, this function stops the
		command parsing when the parser is in pedantic mode and prints
		a warning otherwise.<br>
		The "constructor_parameters" are passed to the constructor of the object.<br>
		The &lt;object_name&gt; can be also an empty string.<br>
		If the function fails for some reason and the execution
		continues, the <b>STRING</b> "0" is returned.<br>
	@seealso:
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/
bool KviUserParser::parseFnc_NEW(KviCommand *cmd, KviStr &buffer)
{
	KviStr type, name, parent;
	if( !processFncSingleParam(cmd, type)   ) return false;
	if( !processFncSingleParam(cmd, parent) ) return false;
	if( !processFncSingleParam(cmd, name)   ) return false;
	QPtrList<KviStr> *params = new QPtrList<KviStr>;
	params->setAutoDelete(true);
	KviStr tmp;
	do {
		tmp = "";
		if( !processFncSingleParam(cmd, tmp) )
			return false;
		if( tmp.hasData() )
			params->append(new KviStr(tmp));
	} while( tmp.hasData() );

	if( !processFncFinalPart(cmd, tmp) ) {
		delete params;
		return false;
	}

	if( type.isEmpty() ) {
		buffer.append('0');
		delete params;
		return recoverableError(cmd, KVI_ERROR_MissingObjectClass, "NEW");
	}
	if( parent.isEmpty() ) {
		buffer.append('0');
		delete params;
		return recoverableError(cmd, KVI_ERROR_MissingObjectParent, "NEW");
	}
	if( !m_pScriptObjectController->isKnownObjectClass(type.ptr()) ) {
		buffer.append('0');
		delete params;
		return recoverableError(cmd, KVI_ERROR_UnknownObjectClass, "NEW", type.ptr());
	}
	KviScriptObject *p = m_pScriptObjectController->findObjectById(parent.ptr());
	if( !p ) {
		buffer.append('0');
		delete params;
		return recoverableError(cmd, KVI_ERROR_NonexistantParentObject, "NEW", parent.ptr());
	}
	bool bDeletedParams = false;

	KviScriptObject *o = m_pScriptObjectController->allocateObject(type.ptr(), name.ptr(), p, params, &bDeletedParams);
	if( o )
		buffer.append(o->id());
	else
		buffer.append('0');

	if( !bDeletedParams )
		delete params;
	return true;
}

/*
	@function: HasEventHandler
	@short:
		Checks if an object's event handler has been set
	@syntax:
		$obj_haseventhandler(&lt;object_id&gt;, &lt;object_event_name&gt;)
	@description:
		Returns 1 if the specified event handler is set for the specified object,
		0 otherwise.<br>
	@seealso:
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/
bool KviUserParser::parseFnc_HASEVENTHANDLER(KviCommand *cmd, KviStr &buffer)
{
	KviStr id, evName;
	if( !processFncSingleParam(cmd, id)   ) return false;
	if( !processFncFinalPart(cmd, evName) ) return false;
	if( id.isEmpty() )
		return recoverableError(cmd, KVI_ERROR_MissingObjectId, "HASEVENTHANDLER");

	if( evName.isEmpty() )
		return recoverableError(cmd, KVI_ERROR_MissingObjectEventName, "HASEVENTHANDLER");
	KviScriptObject *o = m_pScriptObjectController->findObjectById(id.ptr());
	if( !o )
		return recoverableError(cmd, KVI_ERROR_ObjectNotFound, "HASEVENTHANDLER", id.ptr());

	buffer.append(o->hasEventHandler(evName.ptr()) ? '1' : '0');
	return true;
}

/*
	@function: IsMe
	@short:
		Checks if a nickname is the one used by *this* KVIrc user.
	@syntax:
		$isMe(&lt;nickname&gt;)
	@description:
		Returns 1 if the &lt;nickname&gt; is yours, 0 otherwise.<br>
	@examples:
		<example>
		if($isMe(%thatNick))echo %thatNick is me!
		</example>
*/
bool KviUserParser::parseFnc_ISME(KviCommand *cmd, KviStr &buffer)
{
	KviStr nick;
	if( !processFncFinalPart(cmd, nick) ) return false;
	nick.stripWhiteSpace(); // There are NO spaces in nicknames...
	buffer.append((kvi_strEqualCI(nick.ptr(), m_pFrm->m_global.szCurrentNick.ptr()) ? '1' : '0'));
	return true;
}

/*
	@function: FileExists
	@short:
		Checks the existance of a file
	@syntax:
		$FileExists(&lt;filename&lt;);
	@description:
		Returns 1 if the file &lt;filename&gt; exists and is readable.<br>
		This function is NOT valid for directories<br>
		(see the <a href="s_direxists.kvihelp">$direxists()</a> function).
	@seealso:
		<a href="s_direxists.kvihelp">$direxists()</a>
*/

bool KviUserParser::parseFnc_FILEEXISTS(KviCommand *cmd, KviStr &buffer)
{
	KviStr filename;
	if( !processFncFinalPart(cmd, filename) ) return false;
	if( filename.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingFileName, "$FileExists()");
		return false;
	}
	buffer.append((kvi_fileIsReadable(filename.ptr()) ? '1' : '0'));
	return true;
}

/*
	@function: FileSize
	@short:
		Returns the size of a file
	@syntax:
		$FileSize(&lt;filename&lt;);
	@description:
		Returns the size in bytes of the file &lt;filename&gt;.<br>
		If the file cannot be found this function returns -1.<br>
	@seealso:
		<a href="s_fileexists.kvihelp">$fileexists()</a>
*/
bool KviUserParser::parseFnc_FILESIZE(KviCommand *cmd, KviStr &buffer)
{
	KviStr filename;
	if( !processFncFinalPart(cmd, filename) ) return false;
	if( filename.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingFileName, "$FileSize()");
		return false;
	}
	QFile f(filename.ptr());
	if( !f.exists() )
		buffer.append("-1");
	else {
		KviStr size(KviStr::Format, "%u", f.size());
		buffer.append(size);
	}
	return true;
}

/*
	@function: DirExists
	@short:
		Checks the existance of a directory
	@syntax:
		$DirExists(&lt;dirname&lt;);
	@description:
		Returns 1 if the directory &lt;dirname&gt; exists.<br>
	@seealso:
		<a href="s_fileexists.kvihelp">$fileexists()</a>
*/
bool KviUserParser::parseFnc_DIREXISTS(KviCommand *cmd, KviStr &buffer)
{
	KviStr filename;
	if( !processFncFinalPart(cmd, filename) ) return false;
	if( filename.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingDirName, "$DirExists()");
		return false;
	}
	buffer.append((kvi_directoryExists(filename.ptr()) ? '1' : '0'));
	return true;
}

/*
	@function: Readfile
	@short:
		Returns the contents of a file
	@syntax:
		$Readfile(&lt;filename&lt;);
	@description:
		Reads the file &lt;filename&gt; and returns its contents.<br>
		The &lt;filename&gt; must be the COMPLETE path to the file.<br>
		If the file cannot be found or opened this function aborts with an error.<br>
		WARNING : this function will read correctly files that
		do not contain the NULL (ASCII 0) character (ASCII files).<br>
	@examples:
		<example>
		%var = $readfile(/tmp/fun.txt);<br>
		<a href="echo.kvihelp">echo</a> %var
		</example>
	@seealso:
		<a href="writefile.kvihelp">WRITEFILE</a>
*/
bool KviUserParser::parseFnc_READFILE(KviCommand *cmd, KviStr &buffer)
{
	KviStr filename;
	if( !processFncFinalPart(cmd, filename) ) return false;
	if( filename.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingFileName, "$readfile()");
		return false;
	}
	KviStr fileBuf;
	if( !kvi_loadFile(filename.ptr(), fileBuf) ) {
		cmd->setError(KVI_ERROR_CannotOpenFileForReading, "$readfile()", filename.ptr());
		return false;
	}
	buffer.append(fileBuf);
	return true;
}

/*
	@function: IsOwner
	@short:
		Checks if a user has channel owner status
	@syntax:
		$isOwner(&lt;channel&gt;, &lt;nickname&gt;)
	@description:
		Returns 1 if the user &lt;nickname&gt; is an owner on &lt;channel&gt;.
		If you have not joined &lt;channel&gt;, this function returns 0.
	@examples:
		<example>
		if($isOwner($chan, $me))echo I am an owner!
		</example>
*/
bool KviUserParser::parseFnc_ISOWNER(KviCommand *cmd, KviStr &buffer)
{
	KviStr channel, nickname;
	if( !processFncSingleParam(cmd, channel)  ) return false;
	if( !processFncSingleParam(cmd, nickname) ) return false;
	channel.stripWhiteSpace(); // No spaces in nicknames or channel names
	nickname.stripWhiteSpace();
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan ) {
		buffer.append('0');
		return processFncFinalPart(cmd, nickname);
	}
	if( chan->isOwner(nickname.ptr()) )
		buffer.append('1');
	else
		buffer.append('0');
	return processFncFinalPart(cmd, nickname);
}

/*
	@function: IsOp
	@short:
		Checks if a user has operator status
	@syntax:
		$isOp(&lt;channel&gt;, &lt;nickname&gt;)
	@description:
		Returns 1 if the user &lt;nickname&gt; is an operator on &lt;channel&gt;.
		If you have not joined &lt;channel&gt;, this function returns 0.
	@examples:
		<example>
		if($isOp($chan, $me))echo I am an operator!
		</example>
*/
bool KviUserParser::parseFnc_ISOP(KviCommand *cmd, KviStr &buffer)
{
	KviStr channel, nickname;
	if( !processFncSingleParam(cmd, channel)  ) return false;
	if( !processFncSingleParam(cmd, nickname) ) return false;
	channel.stripWhiteSpace(); // No spaces in nicknames or channel names
	nickname.stripWhiteSpace();
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan ) {
		buffer.append('0');
		return processFncFinalPart(cmd, nickname);
	}
	if( chan->isOp(nickname.ptr()) )
		buffer.append('1');
	else
		buffer.append('0');
	return processFncFinalPart(cmd, nickname);
}

/*
	@function: IsHalfOp
	@short:
		Checks if a user has half-operator status
	@syntax:
		$isHalfOp(&lt;channel&gt;, &lt;nickname&gt;)
	@description:
		Returns 1 if the user &lt;nickname&gt; is a half-operator on &lt;channel&gt;.
		If you have not joined &lt;channel&gt;, this function returns 0.
	@examples:
		<example>
		if($isHalfOp($chan, $me))echo I am a halfop!
		</example>
*/
bool KviUserParser::parseFnc_ISHALFOP(KviCommand *cmd, KviStr &buffer)
{
	KviStr channel, nickname;
	if( !processFncSingleParam(cmd, channel)  ) return false;
	if( !processFncSingleParam(cmd, nickname) ) return false;
	channel.stripWhiteSpace(); // No spaces in nicknames or channel names
	nickname.stripWhiteSpace();
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan ) {
		buffer.append('0');
		return processFncFinalPart(cmd, nickname);
	}
	if( chan->isHalfOp(nickname.ptr()) )
		buffer.append('1');
	else
		buffer.append('0');
	return processFncFinalPart(cmd, nickname);
}

/*
	@function: IsUserOp
	@short:
		Checks if a user has user operator status
	@syntax:
		$isOp(&lt;channel&gt;, &lt;nickname&gt;)
	@description:
		Returns 1 if the user &lt;nickname&gt; is a user operator on &lt;channel&gt;.
		If you have not joined &lt;channel&gt;, this function returns 0.
	@examples:
		<example>
		if($isUserOp($chan, $me))echo I am a userop!
		</example>
*/
bool KviUserParser::parseFnc_ISUSEROP(KviCommand *cmd, KviStr &buffer)
{
	KviStr channel, nickname;
	if( !processFncSingleParam(cmd, channel)  ) return false;
	if( !processFncSingleParam(cmd, nickname) ) return false;
	channel.stripWhiteSpace(); // No spaces in nicknames or channel names
	nickname.stripWhiteSpace();
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan ) {
		buffer.append('0');
		return processFncFinalPart(cmd, nickname);
	}
	if( chan->isUserOp(nickname.ptr()) )
		buffer.append('1');
	else
		buffer.append('0');
	return processFncFinalPart(cmd, nickname);
}

/*
	@function: Topic
	@short:
		Returns the topic of the specified channel
	@syntax:
		$Topic(&lt;channel&gt;)
	@description:
		Returns the topic of the specified channel.<br>
		This function aborts with an error if the specified
		channel is not a valid one.
	@seealso:
		[fnc]$chanmode[/fnc](), <br>
		[fnc]$chanlimit[/fnc](), <br>
		[fnc]$chankey[/fnc]()
*/
bool KviUserParser::parseFnc_TOPIC(KviCommand *cmd, KviStr &buffer)
{
	KviStr channel;
	if( !processFncFinalPart(cmd, channel) ) return false;
	channel.stripWhiteSpace(); // No spaces in channel names
	if( channel.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingChannelName, "$topic()");
		return false;
	}
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan ) {
		cmd->setError(KVI_ERROR_ChannelNotFound, "$topic()", channel.ptr());
		return true;
	}
	chan->appendTopic(buffer);
	return true;
}

/*
	@function: Chanmode
	@short:
		Returns the mode string of the specified channel
	@syntax:
		$chanMode(&lt;channel&gt;)
	@description:
		Returns the mode string of the specified channel.<br>
		This function aborts with an error if the specified
		channel is not a valid one.
	@seealso:
		[fnc]$topic[/fnc](), <br>
		[fnc]$chanlimit[/fnc](), <br>
		[fnc]$chankey[/fnc]()
*/
bool KviUserParser::parseFnc_CHANMODE(KviCommand *cmd, KviStr &buffer)
{
	KviStr channel;
	if( !processFncFinalPart(cmd, channel) ) return false;
	channel.stripWhiteSpace(); // No spaces in channel names
	if( channel.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingChannelName, "$chanMode()");
		return false;
	}
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan ) {
		cmd->setError(KVI_ERROR_ChannelNotFound, "$chanMode()", channel.ptr());
		return true;
	}
	buffer.append(chan->m_szChanMode);
	return true;
}

/*
	@function: ChanLimit
	@short:
		Returns the user limit of the specified channel
	@syntax:
		$chanLimit(&lt;channel&gt;)
	@description:
		Returns the user limit of the specified channel (or 0 if there is no limit).<br>
		This function aborts with an error if the specified
		channel is not a valid one.
	@seealso:
		[fnc]$topic[/fnc](), <br>
		[fnc]$chanmode[/fnc](), <br>
		[fnc]$chankey[/fnc]()
*/
bool KviUserParser::parseFnc_CHANLIMIT(KviCommand *cmd, KviStr &buffer)
{
	KviStr channel;
	if( !processFncFinalPart(cmd, channel) ) return false;
	channel.stripWhiteSpace(); // No spaces in channel names
	if( channel.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingChannelName, "$chanLimit()");
		return false;
	}
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan ) {
		cmd->setError(KVI_ERROR_ChannelNotFound, "$chanLimit()", channel.ptr());
		return true;
	}
	buffer.append(chan->m_szUsersLimit.hasData() ? chan->m_szUsersLimit.ptr() : "0");
	return true;
}

/*
	@function: ChanKey
	@short:
		Returns the key of the specified channel
	@syntax:
		$chanKey(&lt;channel&gt;)
	@description:
		Returns the key of the specified channel (or an empty string if no key has been set).<br>
		This function aborts with an error if the specified
		channel is not a valid one.
	@seealso:
		[fnc]$topic[/fnc](), <br>
		[fnc]$chanlimit[/fnc](), <br>
		[fnc]$chankey[/fnc]()
*/
bool KviUserParser::parseFnc_CHANKEY(KviCommand *cmd, KviStr &buffer)
{
	KviStr channel;
	if( !processFncFinalPart(cmd, channel) ) return false;
	channel.stripWhiteSpace(); // No spaces in channel names
	if( channel.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingChannelName, "$chanKey()");
		return false;
	}
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan ) {
		cmd->setError(KVI_ERROR_ChannelNotFound, "$chanKey()", channel.ptr());
		return true;
	}
	buffer.append(chan->m_szChanKey);
	return true;
}

/*
	@function: IsOn
	@short:
		Checks if a user is on a specified channel
	@syntax:
		$isOn(&lt;channel&gt;, &lt;nickname&gt;)
	@description:
		Returns 1 if the user &lt;nickname&gt; is on the channel &lt;channel&gt;.<br>
		If you have not joined the &lt;channel&gt; this function returns 0.
	@examples:
		<example>
		if($isOn($chan, $me))echo I am on this channel!
		</example>
*/
bool KviUserParser::parseFnc_ISON(KviCommand *cmd, KviStr &buffer)
{
	KviStr channel, nickname;
	if( !processFncSingleParam(cmd, channel)  ) return false;
	if( !processFncSingleParam(cmd, nickname) ) return false;
	channel.stripWhiteSpace(); // No spaces in channels or nicknames
	nickname.stripWhiteSpace();
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan ) {
		buffer.append('0');
		return processFncFinalPart(cmd, nickname);
	}
	if( chan->isOnChannel(nickname.ptr()) )
		buffer.append('1');
	else
		buffer.append('0');
	return processFncFinalPart(cmd, nickname);
}

/*
	@function: IsWindow
	@short:
		Checks if a window exists
	@syntax:
		$isWindow(&lt;window name&gt;)
	@description:
		Returns 1 if the window &lt;window name&gt; exists, 0 otherwise.
	@examples:
		<example>
		if($isWindow(<a href="s_window.kvihelp">$window</a>))<a href="echo.kvihelp">echo</a> Yeah... it is working!
		</example>
*/
bool KviUserParser::parseFnc_ISWINDOW(KviCommand *cmd, KviStr &buffer)
{
	KviStr window;
	if( !processFncFinalPart(cmd, window) ) return false;
	window.stripWhiteSpace(); // Window names have NO spaces
	buffer.append((m_pFrm->findWindow(window.ptr()) ? '1' : '0'));
	return true;
}

/*
	@function: IsDocked
	@short:
		Checks if a window is docked
	@syntax:
		$isDocked(&lt;window name&gt;)
	@description:
		Returns 1 if the window &lt;window name&gt; exists and is docked to the MDI manager,
		0 otherwise.
	@seealso:
		<a href="s_isdocked.kvihelp">$IsDocked</a>
*/
bool KviUserParser::parseFnc_ISDOCKED(KviCommand *cmd, KviStr &buffer)
{
	KviStr window;
	if( !processFncFinalPart(cmd, window) ) return false;
	window.stripWhiteSpace(); // Window names have no spaces
	KviWindow *win = m_pFrm->findWindow(window.ptr());
	if( !win )
		buffer.append('0');
	else
		buffer.append(win->isAttached() ? '1' : '0');
	return true;
}

/*
	@function: Commandline
	@short:
		Returns the commanline input text of a window
	@syntax:
		$Commandline(&lt;window name&gt;)
	@description:
		Returns the text currently displayed in the commandline input of the
		specified window.<br>
		If the window has no commandline input box, this function returns an
		empty string.<br>
	@seealso:
		<a href="window.kvihelp">WINDOW</a> <a href="s_window.kvihelp">$window</a>
*/
bool KviUserParser::parseFnc_COMMANDLINE(KviCommand *cmd, KviStr &buffer)
{
	KviStr window;
	if( !processFncFinalPart(cmd, window) ) return false;
	window.stripWhiteSpace(); // Window names have no leading or trailing spaces
	KviWindow *win = m_pFrm->findWindow(window.ptr());
	if( win ) {
		if( win->m_pInput )
			buffer.append(win->m_pInput->m_szTextBuffer);
	}
	return true;
}

/*
	@function: Calc
	@short:
		Evaluate arithmetic and logic expressions
	@syntax:
		$Calc(&lt;expression&gt;)
	@description:
		Evaluates arithmetic and/or logic expression.<br>
		<docsubtitle>Binary algebraic operators supported</docsubtitle>
		+ : addition<br>
		- : subtraction<br>
		* : multiplication<br>
		/ : division<br>
		% : modulus<br>
		^ : power<br>
		<docsubtitle>Other binary operators supported</docsubtitle>
		& : bitwise and<br>
		| : bitwise or<br>
		&lt;&lt; : shift left<br>
		&gt;&gt; : shift right<br>
		&gt; : greater than<br>
		&lt; : lower than<br>
		&gt;= : greater or equal<br>
		&lt;= : lower or equal<br>
		== : equal to<br>
		!= : not equal to<br>
		&& : logical and<br>
		|| : logical or<br>
		<docsubtitle>Special operators for string operands</docsubtitle>
		== : case insensitive 'equal to'<br>
		@= : case sensitive 'equal to'<br>
		&lt; : case insensitive 'lower than'  (alphabetic order)<br>
		&gt; : case insensitive 'greater than (alphabetic order)<br>
		&lt;= : case insensitive 'lower or equal to'        (...)<br>
		&gt;= : case insensitive 'greater or equal to'      (...)<br>
		!= : not equal to<br>
		<docsubtitle>Unary operators supported</docsubtitle>
		! : logical not<br>
		~ : bitwise not<br>
		<docsubtitle>Other specs</docsubtitle>
		Logical (boolean) operators return 0 for false and 1 for true:<br>
		<example>
			$calc(1 &gt;= 2) will return 0
			$calc(1 != (3 &lt;= 10-8)) will return 1
		</example>
		You can use nested parenthesis to override the precedence of operators:<br>
		<example>
			$calc((2+3)*2) will return 10
		</example>
		You can use numbers with different bases.<br>
		The result of calculation will be always returned in base 10:<br>
		<example>
			$calc(0xFF + 1) will return 256
		</example>
		The base of a number is encoded in the following way:<br>
		0&lt;base-code-character&gt;&lt;number&gt;<br>
		<docsubtitle>Base code characters</docsubtitle>
		2 : i, 3 : j, 4 : k, 5 : l<br>
		6 : m, 7 : n, 8 : o, 9 : p<br>
		10: q, 11: r, 12: s, 13: t<br>
		14: u, 15: v, 16: x, 16: h<br>
		So 0xFF (or 0hFF) is 255 in 'hexadecimal', 0i00000010 is 2 in 'binary', <br>
		0o32 is 26 in 'octal', 0t32 is 41 in base 13...<br>
		<docsubtitle>Numbers</docsubtitle>
		This function performs only integer calculations:<br>
		floating point values are not allowed, and will break the computation
		with an error.<br>
		The valid range for numbers is [-2147483647, 2147483647]=(-(2^31), (2^31)):
		overflows do not generate any error, you will just get "absurd" numeric results.<br>
		<docsubtitle>String operations</docsubtitle>
		You can compare strings in the expression using the special operators.<br>
		Anythig that should be treated as a string must be included in double-quotes:<br>
		<example>
			$calc("String" == "string") will return 1.
			$calc("1" &lt;= "4") will return 1.
			$calc("1" == 1) will NOT work.
			$calc(("1" == "1") == 1) will work (and will return 1) :)
		</example>
		Quoted variables and identifiers are still evaluated:
		<example>
			$calc("%var" == "text")
		</example>
		This will return 1 if the variable %var contains the string 'text'.
		In string comparisons it is valid for a variable to be unset.
		In that case the string is considered to be empty:
		<example>
			$calc("%var" != "")
		</example>
		This will return 0 if the variable is not set.<br>
		Note that for numeric calculations all variables must be set,
		and must contain valid numbers.<br>
		<example>
			$calc(%var &lt; 3)
		</example>
		The example above will terminate with an error if the variable %var is not set
		or contains a something that cannot be interpreted as a number.<br>
	@examples:
		<example>
			echo $calc(2^2 + 1 * 3 &lt;&lt; ~7)
			%var = 10; echo $calc(1/(1+4 && 3)* (!(%var &gt; 1) && 2&1))
			echo $calc(~(0xFFFF & 0xFFFF0000))
			echo $calc(0i11011011 | 0i00100100)
			echo $calc((0h100 + -1)==0i11111111)
		</example>
*/
bool KviUserParser::parseFnc_CALC(KviCommand *cmd, KviStr &buffer)
{
	long res;
	if( !evaluateExpr(cmd, &res) ) return false;
	KviStr tmp(KviStr::Format, "%d", res);
	buffer.append(tmp);
	return true;
}

/*
	@function: Range
	@short:
		Returns a string with numbers in a specified range
	@syntax:
		$Range(&lt;from&gt;, &lt;to&gt;[, step])
	@description:
		Returns a string with a comma separated sequence of numbers starting
		from &lt;from&gt; and ending at &lt;to&gt; with the specified step.<br>
		Step MUST be a positive number: if it is negative (or 0) it defaults to 1.<br>
		If step is not specified, it defaults to 1.<br>
		This function will abort with an error if &lt;from&gt; or &lt;to&gt; are
		not numbers.<br>
		WARNING: this function will likely generate a really long string:
		remember that you cannot send strings longer than 512 bytes to the server.<br>
		For really long ranges, you should use a <a href="while.kvihelp">while</a> loop.<br>
	@examples:
		<example>
			<a href="echo.kvihelp">echo</a> $Range(0, 10);
			<a href="echo.kvihelp">echo</a> $Range(50, -50, 10);
			<a href="foreach.kvihelp">foreach</a>(%tmp, $Range(0, 25, 2))<a href="echo.kvihelp">echo</a> <a href="s_calc.kvihelp">$Calc</a>(%tmp &lt;&lt; 8)
		</example>
*/
bool KviUserParser::parseFnc_RANGE(KviCommand *cmd, KviStr &buffer)
{
	KviStr min, max, step;
	if( !processFncSingleParam(cmd, min) ) return false;
	if( !processFncSingleParam(cmd, max) ) return false;
	if( !processFncFinalPart(cmd, step)  ) return false;
	bool bOk = false;
	int iMin = min.toInt(&bOk);
	if( !bOk ) {
		cmd->setError(KVI_ERROR_InvalidRangeValue, "$Range()", min.ptr());
		return false;
	}
	int iMax = max.toInt(&bOk);
	if( !bOk ) {
		cmd->setError(KVI_ERROR_InvalidRangeValue, "$Range()", max.ptr());
		return false;
	}
	int iStep = step.toInt(&bOk);
	if( (!bOk) || (iStep <= 0) )
		iStep = 1;
	KviStr tmp;
	if( iMin < iMax ) {
		for( int i = iMin; i <= iMax; i += iStep ) {
			if( i > iMin )
				buffer.append(',');
			tmp.setNum(i);
			buffer.append(tmp);
		}
	} else {
		for( int i = iMin; i >= iMax; i -= iStep ) {
			if( i < iMin )
				buffer.append(',');
			tmp.setNum(i);
			buffer.append(tmp);
		}
	}
	return true;
}

/*
	@function: UpCase
	@short:
		Returns a string converted to upper case
	@syntax:
		$upCase(&lt;string&gt;)
	@description:
		Returns the &lt;string&gt; converted to upper case.
*/
bool KviUserParser::parseFnc_UPCASE(KviCommand *cmd, KviStr &buffer)
{
	KviStr string;
	if( !processFncFinalPart(cmd, string) ) return false;
	string.toUpper();
	buffer.append(string);
	return true;
}

/*
	@function: LowCase
	@short:
		Returns a string converted to lower case
	@syntax:
		$lowCase(&lt;string&gt;)
	@description:
		Returns the &lt;string&gt; converted to lower case.
*/
bool KviUserParser::parseFnc_LOWCASE(KviCommand *cmd, KviStr &buffer)
{
	KviStr string;
	if( !processFncFinalPart(cmd, string) ) return false;
	string.toLower();
	buffer.append(string);
	return true;
}

/*
	@function: IsVoice
	@short:
		Checks if a user has voice status
	@syntax:
		$isVoice(&lt;channel&gt;, &lt;nickname&gt;)
	@description:
		Returns 1 if the user &lt;nickname&gt; is has the +v flag on channel &lt;channel&gt;.
		If you have not joined the &lt;channel&gt; this function returns 0.
	@examples:
		<example>
		if($isVoice(<a href="s_chan.kvihelp">$chan</a>, <a href="s_me.kvihelp">$me</a>))<a href="echo.kvihelp">echo</a> I have the +v flag
		</example>
*/
bool KviUserParser::parseFnc_ISVOICE(KviCommand *cmd, KviStr &buffer)
{
	KviStr channel, nickname;
	if( !processFncSingleParam(cmd, channel)  ) return false;
	if( !processFncSingleParam(cmd, nickname) ) return false;
	channel.stripWhiteSpace(); // No leading or trailing spaces
	nickname.stripWhiteSpace();
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan ) {
		buffer.append('0');
		return true;
	}
	if( chan->isVoice(nickname.ptr()) )
		buffer.append('1');
	else
		buffer.append('0');
	return processFncFinalPart(cmd, nickname);
}

/*
	@function: StrCat
	@short:
		Concatenates strings
	@syntax:
		$strCat(&lt;string&gt;[, string[, ...]])
	@description:
		Concatenates all the strings.
*/
bool KviUserParser::parseFnc_STRCAT(KviCommand *cmd, KviStr &buffer)
{
	KviStr params;
	while( *(cmd->m_ptr) != ')' ) {
		params = "";
		if( !processFncSingleParam(cmd, params) ) return false;
		buffer.append(params);
	}
	++(cmd->m_ptr);
	return true;
}

/*
	@function: StrLen
	@short:
		Returns the length of a string
	@syntax:
		$strLen(&lt;string&gt;)
	@description:
		Returns the length in characters of the &lt;string&gt;.<br>
		The leading and trailing spaces are ignored.
*/
bool KviUserParser::parseFnc_STRLEN(KviCommand *cmd, KviStr &buffer)
{
	KviStr params;
	if( !processFncSingleParam(cmd, params) ) return false;
	params.setNum(params.len());
	buffer.append(params);
	return processFncFinalPart(cmd, params);
}

/*
	@function: StrCutLeft
	@short:
		Extracts the right part of a string
	@syntax:
		$strCutLeft(&lt;length&gt;, &lt;string&gt;)
	@description:
		Extracts the right part of the &lt;string&gt; returning (maximum) ($strlen(string) - &lt;length&gt) characters;<br>
		Alternative definition : skips the first &lt;length&gt; characters of the &lt;string&gt; and returns the remaining part.<br>
*/
bool KviUserParser::parseFnc_STRCUTLEFT(KviCommand *cmd, KviStr &buffer)
{
	KviStr length;
	if( !processFncSingleParam(cmd, length) ) return false;
	bool bOk = false;
	int len = length.toInt(&bOk);
	if( (!bOk) || (len < 0) ) {
		cmd->setError(KVI_ERROR_StringLengthExpected, "$StrCutLeft()", length.ptr());
		return false;
	}
	length = "";
	if( !processFncFinalPart(cmd, length) ) return false;
	buffer.append(length.right(length.len() - len));
	return true;
}

/*
	@function: StrCutRight
	@short:
		Extracts the left part of a string
	@syntax:
		$strCutRight(&lt;length&gt;, &lt;string&gt;)
	@description:
		Extracts the left part of the &lt;string&gt; returning (maximum) ($strlen(string) - &lt;length&gt) characters;<br>
		Alternative definition : cuts the last &lt;length&gt; characters of the &lt;string&gt; and returns the remaining part.<br>
*/
bool KviUserParser::parseFnc_STRCUTRIGHT(KviCommand *cmd, KviStr &buffer)
{
	KviStr length;
	if( !processFncSingleParam(cmd, length) ) return false;
	bool bOk = false;
	int len = length.toInt(&bOk);
	if( (!bOk) || (len < 0) ) {
		cmd->setError(KVI_ERROR_StringLengthExpected, "$StrCutRight()", length.ptr());
		return false;
	}
	length = "";
	if( !processFncFinalPart(cmd, length) ) return false;
	buffer.append(length.left(length.len() - len));
	return true;
}

/*
	@function: StrLeft
	@short:
		Extracts the left part of a string
	@syntax:
		$strLeft(&lt;length&gt;, &lt;string&gt;)
	@description:
		Extracts the left part of the &lt;string&gt; returning (maximum) &lt;length&gt; characters.
*/
bool KviUserParser::parseFnc_STRLEFT(KviCommand *cmd, KviStr &buffer)
{
	KviStr length;
	if( !processFncSingleParam(cmd, length) ) return false;
	bool bOk = false;
	int len = length.toInt(&bOk);
	if( (!bOk) || (len < 0) ) {
		cmd->setError(KVI_ERROR_StringLengthExpected, "$StrLeft()", length.ptr());
		return false;
	}
	length = "";
	if( !processFncFinalPart(cmd, length) ) return false;
	buffer.append(length.left(len));
	return true;
}

/*
	@function: StrRight
	@short:
		Extracts the right part of a string
	@syntax:
		$strRight(&lt;length&gt;, &lt;string&gt;)
	@description:
		Extracts the right part of the &lt;string&gt; returning (maximum) &lt;length&gt; characters.
*/
bool KviUserParser::parseFnc_STRRIGHT(KviCommand *cmd, KviStr &buffer)
{
	KviStr length;
	if( !processFncSingleParam(cmd, length) ) return false;
	bool bOk = false;
	int len = length.toInt(&bOk);
	if( (!bOk) || (len < 0) ) {
		cmd->setError(KVI_ERROR_StringLengthExpected, "$StrRight()", length.ptr());
		return false;
	}
	length = "";
	if( !processFncFinalPart(cmd, length) ) return false;
	buffer.append(length.right(len));
	return true;
}

/*
	@function: StrMid
	@short:
		Extracts a substring
	@syntax:
		$strMid(&lt;index&gt;, &lt;length&gt;, &lt;string&gt;)
	@description:
		Extracts a substring from &lt;string&gt; starting at zero based index &lt;index&gt;.<br>
		The returned string is at most &lt;length&gt; characters long.
*/
bool KviUserParser::parseFnc_STRMID(KviCommand *cmd, KviStr &buffer)
{
	KviStr number;
	if( !processFncSingleParam(cmd, number) ) return false;
	bool bOk = false;
	int idx = number.toInt(&bOk);
	if( (!bOk) || (idx < 0) ) {
		cmd->setError(KVI_ERROR_StringIndexExpected, "$StrMid()", number.ptr());
		return false;
	}
	number = "";
	if( !processFncSingleParam(cmd, number) ) return false;
	bOk = false;
	int len = number.toInt(&bOk);
	if( (!bOk) || (len < 0) ) {
		cmd->setError(KVI_ERROR_StringLengthExpected, "$StrMid()", number.ptr());
		return false;
	}
	number = "";
	if( !processFncFinalPart(cmd, number) ) return false;
	buffer.append(number.middle(idx, len));
	return true;
}

/*
	@function: StrLeftToFirst
	@short:
		Extracts the left part of a string up to a specified substring (case insensitive)
	@syntax:
		$strLeftToFirst(&lt;substring&gt;, &lt;string&gt;)
	@description:
		Extracts the left part of the &lt;string&gt; up to (and excluding)
		the first occurrence of &lt;substring&gt;.<br>
		If an occurrence of &lt;substring&gt; is not found,
		the entire &lt;string&gt; is returned.
		This function is <b>case-insensitive</b>.
	@examples:
		The following example will echo "/usr/local/share/" in the current window.<br>
		<example>
			<a href="echo.kvihelp">echo</a> $strlefttofirst(kvirc, /usr/local/share/kvirc/config/kvirc.config)
		</example>
	@seealso:
		<a href="s_strlefttofirstcs.kvihelp">$StrLeftToFirstCS()</a>
		<a href="s_strlefttolast.kvihelp">$StrLeftToLast()</a>
		<a href="s_strlefttolastcs.kvihelp">$StrLeftToLastCS()</a>
		<a href="s_strrightfromfirst.kvihelp">$StrRightFromFirst()</a>
		<a href="s_strrightfromfirstcs.kvihelp">$StrRightFromFirstCS()</a>
		<a href="s_strrightfromlast.kvihelp">$StrRightFromLast()</a>
		<a href="s_strrightfromlastcs.kvihelp">$StrRightFromLastCS()</a>
*/
bool KviUserParser::parseFnc_STRLEFTTOFIRST(KviCommand *cmd, KviStr &buffer)
{
	KviStr substring;
	if( !processFncSingleParam(cmd, substring) ) return false;
	KviStr string;
	if( !processFncFinalPart(cmd, string) ) return false;
	int idx = string.findFirstIdx(substring.ptr(), false);
	if( idx == -1 )
		buffer.append(string);
	else
		buffer.append(string.left(idx));
	return true;
}

/*
	@function: StrLeftToFirstCS
	@short:
		Extracts the left part of a string up to a specified substring (case-sensitive)
	@syntax:
		$strLeftToFirstCS(&lt;substring&gt;, &lt;string&gt;)
	@description:
		Extracts the left part of the &lt;string&gt; up to (and excluding)
		the first occurrence of &lt;substring&gt;.<br>
		If an occurrence of &lt;substring&gt; is not found,
		the entire &lt;string&gt; is returned.
		This function is <b>case-insensitive</b>.
	@examples:
		The following example will echo "/usr/local/share/" in the current window.<br>
		<example>
			<a href="echo.kvihelp">echo</a> $strlefttofirstcs(kvirc, /usr/local/share/kvirc/config/kvirc.config)
		</example>
	@seealso:
		<a href="s_strlefttofirst.kvihelp">$StrLeftToFirst()</a>
		<a href="s_strlefttolast.kvihelp">$StrLeftToLast()</a>
		<a href="s_strlefttolastcs.kvihelp">$StrLeftToLastCS()</a>
		<a href="s_strrightfromfirst.kvihelp">$StrRightFromFirst()</a>
		<a href="s_strrightfromfirstcs.kvihelp">$StrRightFromFirstCS()</a>
		<a href="s_strrightfromlast.kvihelp">$StrRightFromLast()</a>
		<a href="s_strrightfromlastcs.kvihelp">$StrRightFromLastCS()</a>
*/
bool KviUserParser::parseFnc_STRLEFTTOFIRSTCS(KviCommand *cmd, KviStr &buffer)
{
	KviStr substring;
	if( !processFncSingleParam(cmd, substring) ) return false;
	KviStr string;
	if( !processFncFinalPart(cmd, string) ) return false;
	int idx = string.findFirstIdx(substring.ptr(), true);
	if( idx == -1 )
		buffer.append(string);
	else
		buffer.append(string.left(idx));
	return true;
}

/*
	@function: StrLeftToLast
	@short:
		Extracts the left part of a string up to a specified substring (case insensitive)
	@syntax:
		$strLeftToLast(&lt;substring&gt;, &lt;string&gt;)
	@description:
		Extracts the left part of the &lt;string&gt; up to (and excluding)
		the last occurrence of &lt;substring&gt;.<br>
		If an occurrence of &lt;substring&gt; is not found,
		the entire &lt;string&gt; is returned.
		This function is <b>case-insensitive</b>.
	@examples:
		The following example will echo "/usr/local/bin" in the current window.<br>
		<example>
			<a href="echo.kvihelp">echo</a> $strlefttolast(/, /usr/local/bin/kvirc)
		</example>
	@seealso:
		<a href="s_strlefttofirst.kvihelp">$StrLeftToFirst()</a>
		<a href="s_strlefttofirstcs.kvihelp">$StrLeftToFirstCS()</a>
		<a href="s_strlefttolastcs.kvihelp">$StrLeftToLastCS()</a>
		<a href="s_strrightfromfirst.kvihelp">$StrRightFromFirst()</a>
		<a href="s_strrightfromfirstcs.kvihelp">$StrRightFromFirstCS()</a>
		<a href="s_strrightfromlast.kvihelp">$StrRightFromLast()</a>
		<a href="s_strrightfromlastcs.kvihelp">$StrRightFromLastCS()</a>
*/
bool KviUserParser::parseFnc_STRLEFTTOLAST(KviCommand *cmd, KviStr &buffer)
{
	KviStr substring;
	if( !processFncSingleParam(cmd, substring) ) return false;
	KviStr string;
	if( !processFncFinalPart(cmd, string) ) return false;
	int idx = string.findLastIdx(substring.ptr(), false);
	if( idx == -1 )
		buffer.append(string);
	else
		buffer.append(string.left(idx));
	return true;
}

/*
	@function: StrLeftToLastCS
	@short:
		Extracts the left part of a string up to a specified substring (case-sensitive)
	@syntax:
		$strLeftToLastCS(&lt;substring&gt;, &lt;string&gt;)
	@description:
		Extracts the left part of the &lt;string&gt; up to (and excluding)
		the last occurrence of &lt;substring&gt;.<br>
		If an occurrence of &lt;substring&gt; is not found,
		the entire &lt;string&gt; is returned.
		This function is <b>case-sensitive</b>.
	@examples:
		The following example will echo "/usr/local/bin" in the current window.<br>
		<example>
			<a href="echo.kvihelp">echo</a> $strlefttolastcs(/, /usr/local/bin/kvirc2)
		</example>
	@seealso:
		<a href="s_strlefttofirst.kvihelp">$StrLeftToFirst()</a>
		<a href="s_strlefttofirstcs.kvihelp">$StrLeftToFirstCS()</a>
		<a href="s_strlefttolast.kvihelp">$StrLeftToLast()</a>
		<a href="s_strrightfromfirst.kvihelp">$StrRightFromFirst()</a>
		<a href="s_strrightfromfirstcs.kvihelp">$StrRightFromFirstCS()</a>
		<a href="s_strrightfromlast.kvihelp">$StrRightFromLast()</a>
		<a href="s_strrightfromlastcs.kvihelp">$StrRightFromLastCS()</a>
*/
bool KviUserParser::parseFnc_STRLEFTTOLASTCS(KviCommand *cmd, KviStr &buffer)
{
	KviStr substring;
	if( !processFncSingleParam(cmd, substring) ) return false;
	KviStr string;
	if( !processFncFinalPart(cmd, string) ) return false;
	int idx = string.findLastIdx(substring.ptr(), true);
	if( idx == -1 )
		buffer.append(string);
	else
		buffer.append(string.left(idx));
	return true;
}

/*
	@function: StrRightFromFirst
	@short:
		Extracts the right part of a string starting after a specified substring (case insensitive)
	@syntax:
		$strRightFromFirst(&lt;substring&gt;, &lt;string&gt;)
	@description:
		Extracts the right part of the &lt;string&gt; starting after
		the first occurrence of &lt;substring&gt;.<br>
		If an occurrence of &lt;substring&gt; is not found,
		an empty string is returned.<br>
		This function is <b>case-insensitive</b>.
	@examples:
		The following example will echo "1-2-3-4-5" in the current window.<br>
		<example>
			<a href="echo.kvihelp">echo</a> $strrightfromfirst(-, 0-1-2-3-4-5)
		</example>
	@seealso:
		<a href="s_strlefttofirst.kvihelp">$StrLeftToFirst()</a>
		<a href="s_strlefttofirstcs.kvihelp">$StrLeftToFirstCS()</a>
		<a href="s_strlefttolast.kvihelp">$StrLeftToLast()</a>
		<a href="s_strlefttolastcs.kvihelp">$StrLeftToLastCS()</a>
		<a href="s_strrightfromfirstcs.kvihelp">$StrRightFromFirstCS()</a>
		<a href="s_strrightfromlast.kvihelp">$StrRightFromLast()</a>
		<a href="s_strrightfromlastcs.kvihelp">$StrRightFromLastCS()</a>
*/
bool KviUserParser::parseFnc_STRRIGHTFROMFIRST(KviCommand *cmd, KviStr &buffer)
{
	KviStr substring;
	if( !processFncSingleParam(cmd, substring) ) return false;
	KviStr string;
	if( !processFncFinalPart(cmd, string) ) return false;
	int idx = string.findFirstIdx(substring.ptr(), false);
	if( idx != -1 ) {
		idx += substring.len();
		buffer.append(string.right(string.len() - idx));
	}
	return true;
}

/*
	@function: StrRightFromFirstCS
	@short:
		Extracts the right part of a string starting after a specified substring (case-sensitive)
	@syntax:
		$strRightFromFirstCS(&lt;substring&gt;, &lt;string&gt;)
	@description:
		Extracts the right part of the &lt;string&gt; starting after
		the first occurrence of &lt;substring&gt;.<br>
		If an occurrence of &lt;substring&gt; is not found,
		an empty string is returned.<br>
		This function is <b>case-sensitive</b>.
	@examples:
		The following example will echo "1-2-3-4-5" in the current window.<br>
		<example>
			<a href="echo.kvihelp">echo</a> $strrightfromfirstcs(-, 0-1-2-3-4-5)
		</example>
	@seealso:
		<a href="s_strlefttofirst.kvihelp">$StrLeftToFirst()</a>
		<a href="s_strlefttofirstcs.kvihelp">$StrLeftToFirstCS()</a>
		<a href="s_strlefttolast.kvihelp">$StrLeftToLast()</a>
		<a href="s_strlefttolastcs.kvihelp">$StrLeftToLastCS()</a>
		<a href="s_strrightfromfirst.kvihelp">$StrRightFromFirst()</a>
		<a href="s_strrightfromlast.kvihelp">$StrRightFromLast()</a>
		<a href="s_strrightfromlastcs.kvihelp">$StrRightFromLastCS()</a>
*/
bool KviUserParser::parseFnc_STRRIGHTFROMFIRSTCS(KviCommand *cmd, KviStr &buffer)
{
	KviStr substring;
	if( !processFncSingleParam(cmd, substring) ) return false;
	KviStr string;
	if( !processFncFinalPart(cmd, string) ) return false;
	int idx = string.findFirstIdx(substring.ptr(), true);
	if( idx != -1 ) {
		idx += substring.len();
		buffer.append(string.right(string.len() - idx));
	}
	return true;
}

/*
	@function: StrRightFromLast
	@short:
		Extracts the right part of a string starting after a specified substring (case insensitive)
	@syntax:
		$strRightFromLast(&lt;substring&gt;, &lt;string&gt;)
	@description:
		Extracts the right part of the &lt;string&gt; starting after
		the last occurrence of &lt;substring&gt;.<br>
		If an occurrence of &lt;substring&gt; is not found,
		an empty string is returned.<br>
		This function is <b>case-insensitive</b>.
	@examples:
		The following example will echo "5" in the current window.<br>
		<example>
			<a href="echo.kvihelp">echo</a> $strrightfromlast(-, 0-1-2-3-4-5)
		</example>
	@seealso:
		<a href="s_strlefttofirst.kvihelp">$StrLeftToFirst()</a>
		<a href="s_strlefttofirstcs.kvihelp">$StrLeftToFirstCS()</a>
		<a href="s_strlefttolast.kvihelp">$StrLeftToLast()</a>
		<a href="s_strlefttolastcs.kvihelp">$StrLeftToLastCS()</a>
		<a href="s_strrightfromfirst.kvihelp">$StrRightFromFirst()</a>
		<a href="s_strrightfromfirstcs.kvihelp">$StrRightFromFirstCS()</a>
		<a href="s_strrightfromlastcs.kvihelp">$StrRightFromLastCS()</a>
*/
bool KviUserParser::parseFnc_STRRIGHTFROMLAST(KviCommand *cmd, KviStr &buffer)
{
	KviStr substring;
	if( !processFncSingleParam(cmd, substring) ) return false;
	KviStr string;
	if( !processFncFinalPart(cmd, string) ) return false;
	int idx = string.findLastIdx(substring.ptr(), false);
	if( idx != -1 ) {
		idx += substring.len();
		buffer.append(string.right(string.len() - idx));
	}
	return true;
}

/*
	@function: StrRightFromLastCS
	@short:
		Extracts the right part of a string starting after a specified substring (case-sensitive)
	@syntax:
		$strRightFromLastCS(&lt;substring&gt;, &lt;string&gt;)
	@description:
		Extracts the right part of the &lt;string&gt; starting after
		the last occurrence of &lt;substring&gt;.<br>
		If an occurrence of &lt;substring&gt; is not found,
		an empty string is returned.<br>
		This function is <b>case-sensitive</b>.
	@examples:
		The following example will echo "5" in the current window.<br>
		<example>
			<a href="echo.kvihelp">echo</a> $strrightfromlastcs(-, 0-1-2-3-4-5)
		</example>
	@seealso:
		<a href="s_strlefttofirst.kvihelp">$StrLeftToFirst()</a>
		<a href="s_strlefttofirstcs.kvihelp">$StrLeftToFirstCS()</a>
		<a href="s_strlefttolast.kvihelp">$StrLeftToLast()</a>
		<a href="s_strlefttolastcs.kvihelp">$StrLeftToLastCS()</a>
		<a href="s_strrightfromfirst.kvihelp">$StrRightFromFirst()</a>
		<a href="s_strrightfromfirstcs.kvihelp">$StrRightFromFirstCS()</a>
		<a href="s_strrightfromlast.kvihelp">$StrRightFromLast()</a>
*/
bool KviUserParser::parseFnc_STRRIGHTFROMLASTCS(KviCommand *cmd, KviStr &buffer)
{
	KviStr substring;
	if( !processFncSingleParam(cmd, substring) ) return false;
	KviStr string;
	if( !processFncFinalPart(cmd, string) ) return false;
	int idx = string.findLastIdx(substring.ptr(), true);
	if( idx != -1 ) {
		idx += substring.len();
		buffer.append(string.right(string.len() - idx));
	}
	return true;
}

/*
	@function: StrMatch
	@short:
		Matches two strings containing wildcards
	@syntax:
		$strMatch(&lt;string1&gt;, &lt;string2&gt;)
	@description:
		Compares &lt;string1&gt; and &lt;string2&gt;.<br>
		Both strings can contain the '*' wildcard, that matches
		any sequence of characters.<br>
		This function is <b>case-insensitive</b>.<br>
	@examples:
		<example>
			<a href="echo.kvihelp">echo</a> $strmatch(pragma!nobody@*.it, *!nobody@ahost.adomain.it)
		</example>
	@seealso:
		<a href="s_strlefttofirst.kvihelp">$StrMatchCS()</a>
*/
bool KviUserParser::parseFnc_STRMATCH(KviCommand *cmd, KviStr &buffer)
{
	KviStr str1, str2;
	if( !processFncSingleParam(cmd, str1) ) return false;
	if( !processFncFinalPart(cmd, str2)   ) return false;
	buffer.append(kvi_matchWildExpr(str1.ptr(), str2.ptr()) ? '1' : '0');
	return true;
}

/*
	@function: StrMatchCS
	@short:
		Matches two strings containing wildcards (case-sensitive)
	@syntax:
		$strMatchCS(&lt;string1&gt;, &lt;string2&gt;)
	@description:
		Compares &lt;string1&gt; and &lt;string2&gt;.<br>
		Both strings can contain the '*' wildcard, that matches
		any sequence of characters.<br>
		This function is <b>case-sensitive</b>.<br>
	@examples:
		<example>
			<a href="echo.kvihelp">echo</a> $strmatchCS(pragma!Nobody@*.it, *!nobody@ahost.adomain.it)
		</example>
	@seealso:
		<a href="s_strlefttofirst.kvihelp">$StrMatch()</a>
*/
bool KviUserParser::parseFnc_STRMATCHCS(KviCommand *cmd, KviStr &buffer)
{
	KviStr str1, str2;
	if( !processFncSingleParam(cmd, str1) ) return false;
	if( !processFncFinalPart(cmd, str2)   ) return false;
	buffer.append(kvi_matchWildExprCS(str1.ptr(), str2.ptr()) ? '1' : '0');
	return true;
}

/*
	@function: IsNumber
	@short:
		Checks if a string is a number
	@syntax:
		$IsNumber(&lt;string&gt;)
	@description:
		Returns 1 if &lt;string&gt; represents a number is decimal notation (base &lt= 10),
		0 otherwise<br>
*/
bool KviUserParser::parseFnc_ISNUMBER(KviCommand *cmd, KviStr &buffer)
{
	KviStr params;
	if( !processFncSingleParam(cmd, params) ) return false;
	buffer.append(params.isNum() ? '1' : '0');
	return processFncFinalPart(cmd, params);
}

/*
	@function: GetEnv
	@short:
		Read enivoinement variables
	@syntax:
		$GetEnv(&lt;variable&gt;)
	@description:
		Reads environment variables.<br>
	@examples:
		<example>
			<a href="echo.kvihelp">echo</a> My home directory is $getEnv(HOME)
		</example>
*/
bool KviUserParser::parseFnc_GETENV(KviCommand *cmd, KviStr &buffer)
{
	KviStr params;
	if( !processFncSingleParam(cmd, params) ) return false;
	char *tmp = getenv(params.ptr());
	if( tmp )
		buffer.append(tmp);
	return processFncFinalPart(cmd, params);
}

/*
	@function: DefLogFile
	@short:
		Returns the default log filename
	@syntax:
		$DefLogFile(&lt;window&gt;)
	@description:
		Returns the default log file name for the specified &lt;window&gt;.<br>
	@examples:
		<example>
			<a href="echo.kvihelp">echo</a> $deflogname(<a href="s_window.kvihelp">$window</a>)
		</example>
*/
bool KviUserParser::parseFnc_DEFLOGFILE(KviCommand *cmd, KviStr &buffer)
{
	KviStr window;
	if( !processFncFinalPart(cmd, window) ) return false;
	window.stripWhiteSpace(); // No spaces in window names
	KviWindow *wnd = m_pFrm->findWindow(window.ptr());
	if( !wnd ) {
		g_pApp->getLocalKVIrcDirectory(window, KviApp::Log, "UNKNOWN_WINDOW.log");
		buffer.append(window.ptr());
	} else {
		wnd->getDefaultLogName(window);
		buffer.append(window);
	}
	return true;
}

/*
	@function: IsReg
	@short:
		Checks if a user is registered
	@syntax:
		$IsReg(&lt;mask&gt;)
	@description:
		Returns 1 if &lt;mask&gt; matches at least one of the users registered in KVIrc.<br>
		(See the registered users database dialog)
	@seealso:
		<a href="s_getflags.kvihelp">$GetFlags</a><br>
		<a href="s_getflagsexact.kvihelp">$GetFlagaExact</a><br>
		<a href="s_getpass.kvihelp">$GetPass</a><br>
		<a href="s_getcomment.kvihelp">$GetComment</a><br>
		<a href="setflags.kvihelp">SETFLAGS</a><br>
		<a href="register.kvihelp">REGISTER</a><br>
		<a href="unregister.kvihelp">UNREGISTER</a><br>
		<a href="notify.kvihelp">NOTIFY</a><br>
		<a href="ignore.kvihelp">IGNORE</a>
*/
bool KviUserParser::parseFnc_ISREG(KviCommand *cmd, KviStr &buffer)
{
	KviStr mask;
	if( !processFncFinalPart(cmd, mask) ) return false;
	mask.stripWhiteSpace(); // No leading or trailing spaces in masks
	buffer.append((g_pOptions->m_pRegUsersDb->findUserByMask(mask.ptr()) ? '1' : '0'));
	return true;
}

/*
	@function: GetFlags
	@short:
		Returns the flags of registered users
	@syntax:
		$GetFlags(&lt;mask&gt;)
	@description:
		This function returns a string containing the flags from all users that match the &lt;mask&gt;.<br>
	@seealso:
		<a href="s_isreg.kvihelp">$IsReg</a><br>
		<a href="s_getflagsexact.kvihelp">$GetFlagsExact</a><br>
		<a href="s_getpass.kvihelp">$GetPass</a><br>
		<a href="s_getcomment.kvihelp">$GetComment</a><br>
		<a href="setflags.kvihelp">SETFLAGS</a><br>
		<a href="register.kvihelp">REGISTER</a><br>
		<a href="unregister.kvihelp">UNREGISTER</a><br>
		<a href="notify.kvihelp">NOTIFY</a><br>
		<a href="ignore.kvihelp">IGNORE</a>
*/
bool KviUserParser::parseFnc_GETFLAGS(KviCommand *cmd, KviStr &buffer)
{
	KviStr mask;
	if( !processFncFinalPart(cmd, mask) ) return false;
	mask.stripWhiteSpace(); // No leading or trailing spaces in masks
	KviStr tmp;
	g_pOptions->m_pRegUsersDb->getFlags(mask.ptr(), tmp);
	buffer.append(tmp);
	return true;
}

/*
	@function: GetFlagsExact
	@short:
		Returns the flags of registered users
	@syntax:
		$GetFlagsExact(&lt;mask&gt;)
	@description:
		This function returns a string containing the flags from the users that exactly matches the &lt;mask&gt;.<br>
	@seealso:
		<a href="s_isreg.kvihelp">$IsReg</a><br>
		<a href="s_getflags.kvihelp">$GetFlags</a><br>
		<a href="s_getpass.kvihelp">$GetPass</a><br>
		<a href="s_getcomment.kvihelp">$GetComment</a><br>
		<a href="setflags.kvihelp">SETFLAGS</a><br>
		<a href="register.kvihelp">REGISTER</a><br>
		<a href="unregister.kvihelp">UNREGISTER</a><br>
		<a href="notify.kvihelp">NOTIFY</a><br>
		<a href="ignore.kvihelp">IGNORE</a>
*/
bool KviUserParser::parseFnc_GETFLAGSEXACT(KviCommand *cmd, KviStr &buffer)
{
	KviStr mask;
	if( !processFncFinalPart(cmd, mask) ) return false;
	mask.stripWhiteSpace(); // No leading or trailing spaces in masks
	KviStr tmp;
	g_pOptions->m_pRegUsersDb->getFlagsExact(mask.ptr(), tmp);
	buffer.append(tmp);
	return true;
}

/*
	@function: GetPass
	@short:
		Returns the password of a registered user
	@syntax:
		$GetPass(&lt;mask&gt;)
	@description:
		This function returns a string containing the password from the users that exactly matches the &lt;mask&gt;.<br>
		NOTE: this function will return 0 if the user was not found. So setting a password to "0" is not a good idea.<br>
	@seealso:
		<a href="s_isreg.kvihelp">$IsReg</a><br>
		<a href="s_getflags.kvihelp">$GetFlags</a><br>
		<a href="s_getflagsexact.kvihelp">$GetFlagsExact</a><br>
		<a href="s_getpass.kvihelp">$GetPass</a><br>
		<a href="s_getcomment.kvihelp">$GetComment</a><br>
		<a href="setflags.kvihelp">SETFLAGS</a><br>
		<a href="register.kvihelp">REGISTER</a><br>
		<a href="unregister.kvihelp">UNREGISTER</a><br>
		<a href="notify.kvihelp">NOTIFY</a><br>
		<a href="ignore.kvihelp">IGNORE</a>
*/
bool KviUserParser::parseFnc_GETPASS(KviCommand *cmd, KviStr &buffer)
{
	KviStr mask;
	if( !processFncFinalPart(cmd, mask) ) return false;
	mask.stripWhiteSpace(); // No leading or trailing spaces in masks

	KviRegisteredUser *u = g_pOptions->m_pRegUsersDb->findExactMatch(mask.ptr());
	if( !u ) {
		buffer.append("0");
		return true;
	}
	KviStr tmp;
	g_pOptions->m_pRegUsersDb->getPasswd(u, tmp);
	buffer.append(tmp);
	return true;
}

/*
	@function: GetComment
	@short:
		Returns the comment of a registered user
	@syntax:
		$GetPass(&lt;mask&gt;)
	@description:
		This function returns a string containing the comment from the users that exactly matches the &lt;mask&gt;.<br>
		NOTE: this function will return 0 if the user was not found. So setting a comment to "0" can cause confusion.<br>
	@seealso:
		<a href="s_isreg.kvihelp">$IsReg</a><br>
		<a href="s_getflags.kvihelp">$GetFlags</a><br>
		<a href="s_getflagsexact.kvihelp">$GetFlagsExact</a><br>
		<a href="s_getpass.kvihelp">$GetPass</a><br>
		<a href="setflags.kvihelp">SETFLAGS</a><br>
		<a href="register.kvihelp">REGISTER</a><br>
		<a href="unregister.kvihelp">UNREGISTER</a><br>
		<a href="notify.kvihelp">NOTIFY</a><br>
		<a href="ignore.kvihelp">IGNORE</a>
*/
bool KviUserParser::parseFnc_GETCOMMENT(KviCommand *cmd, KviStr &buffer)
{
	KviStr mask;
	if( !processFncFinalPart(cmd, mask) ) return false;
	mask.stripWhiteSpace(); // No leading or trailing spaces in masks

	KviRegisteredUser *u = g_pOptions->m_pRegUsersDb->findExactMatch(mask.ptr());
	if( !u ) {
		buffer.append("0");
		return true;
	}
	KviStr tmp;
	g_pOptions->m_pRegUsersDb->getComment(u, tmp);
	buffer.append(tmp);
	return true;
}

/*
	@function: Rand
	@short:
		Returns random integers
	@syntax:
		$Rand(&lt;max&gt;)
	@description:
		Returns a random positive integer between 0 and &lt;max&gt; (excluding &lt;max&gt;!)
*/

/**
 * The  versions of rand() and srand() in the GNU C Library
 * use the same random number generator as random() and
 * srandom(),  so the lower-order bits should be as random as the
 * higher-order bits.
 */
bool KviUserParser::parseFnc_RAND(KviCommand *cmd, KviStr &buffer)
{
	KviStr max;
	if( !processFncFinalPart(cmd, max) ) return false;
	bool bOk = false;
	int iMax = max.toInt(&bOk);
	if( (iMax <= 0) || (!bOk) )
		iMax = RAND_MAX;
	max.setNum((rand() % iMax));
	buffer.append(max);
	return true;
}

/*
	@function: Uname
	@short:
		Returns system information
	@syntax:
		$Uname(&lt;param&gt;)
	@description:
		Returns the uname() information corresponding to &lt;param&gt;, one of:<br>
		sysname, nodename, release, version, machine.
*/
bool KviUserParser::parseFnc_UNAME(KviCommand *cmd, KviStr &buffer)
{
	KviStr param;
	KviStr info = "unknown";
	if( !processFncFinalPart(cmd, param) ) return false;
#if defined(HAVE_SYS_UTSNAME_H) && defined(HAVE_UNAME)
	struct utsname u;
	if( uname(&u) == 0 ) {
		switch( *param.ptr() ) {
			case 's':
				info = u.sysname;
				break;
			case 'n':
				info = u.nodename;
				break;
			case 'r':
				info = u.release;
				break;
			case 'v':
				info = u.version;
				break;
			case 'm':
				info = u.machine;
				break;
		}
	}
#endif
	buffer.append(info);
	return true;
}

/*
	@function: StrFind
	@short:
		Finds a substring in a string
	@syntax:
		$StrFind(&lt;substring&gt;, &lt;string&gt;)
	@description:
		Returns the zero-based index of the first occurrence of &lt;substring&gt;
		in &lt;string&gt;, or -1 if there are no matches.<br>
		This function is case INSENSITIVE.<br>
	@seealso:
		<a href="s_strrevfind.kvihelp">$StrRevFind</a>, <br>
		<a href="s_strfindcs.kvihelp">$StrFindCS</a>, <br>
		<a href="s_strrevfindcs.kvihelp">$StrRevFindCS</a><br>
	@examples:
		<example>
			echo $StrFind(linux, LINUX RULES... and Pragma uses Linux!)
		</example>
*/
bool KviUserParser::parseFnc_STRFIND(KviCommand *cmd, KviStr &buffer)
{
	KviStr sub, str;
	if( !processFncSingleParam(cmd, sub) ) return false;
	if( !processFncFinalPart(cmd, str)   ) return false;
	sub.setNum(str.findFirstIdx(sub.ptr(), false)); // Case insensitive
	buffer.append(sub);
	return true;
}

/*
	@function: StrRevFind
	@short:
		Finds a substring in a string starting from the end
	@syntax:
		$StrRevFind(&lt;substring&gt;, &lt;string&gt;)
	@description:
		Returns the zero-based index of the last occurrence of &lt;substring&gt;
		in &lt;string&gt;, or -1 if there are no matches.<br>
		This function is <b>case-insensitive</b>.<br>
	@seealso:
		<a href="s_strfind.kvihelp">$StrFind</a>, <br>
		<a href="s_strfindcs.kvihelp">$StrFindCS</a>, <br>
		<a href="s_strrevfindcs.kvihelp">$StrRevFindCS</a><br>
	@examples:
		<example>
			echo $StrRevFind(linux, LINUX Rules... and Pragma uses Linux!)
		</example>
*/
bool KviUserParser::parseFnc_STRREVFIND(KviCommand *cmd, KviStr &buffer)
{
	KviStr sub, str;
	if( !processFncSingleParam(cmd, sub) ) return false;
	if( !processFncFinalPart(cmd, str)   ) return false;
	sub.setNum(str.findLastIdx(sub.ptr(), false)); // Case insensitive
	buffer.append(sub);
	return true;
}

/*
	@function: StrFindCS
	@short:
		Finds a substring in a string (case-sensitive)
	@syntax:
		$StrFindCS(&lt;substring&gt;, &lt;string&gt;)
	@description:
		Returns the zero-based index of the first occurrence of &lt;substring&gt;
		in &lt;string&gt;, or -1 if there are no matches.<br>
		This function is <b>case-sensitive</b>.<br>
	@seealso:
		<a href="s_strrevfind.kvihelp">$StrRevFind</a>, <br>
		<a href="s_strfind.kvihelp">$StrFind</a>, <br>
		<a href="s_strrevfindcs.kvihelp">$StrRevFindCS</a><br>
	@examples:
		<example>
			echo $StrFindCS(linux, LINUX RULES... and Pragma uses Linux!)
		</example>
*/
bool KviUserParser::parseFnc_STRFINDCS(KviCommand *cmd, KviStr &buffer)
{
	KviStr sub, str;
	if( !processFncSingleParam(cmd, sub) ) return false;
	if( !processFncFinalPart(cmd, str)   ) return false;
	sub.setNum(str.findFirstIdx(sub.ptr(), true)); // Case sensitive
	buffer.append(sub);
	return true;
}

/*
	@function: StrRevFindCS
	@short:
		Finds a substring in a string starting from end (case-sensitive)
	@syntax:
		$StrRevFindCS(&lt;substring&gt;, &lt;string&gt;)
	@description:
		Returns the zero-based index of the last occurrence of &lt;substring&gt;
		in &lt;string&gt;, or -1 if there are no matches.<br>
		This function is <b>case-sensitive</b>.<br>
	@seealso:
		<a href="s_strfind.kvihelp">$StrFind</a>, <br>
		<a href="s_strfindcs.kvihelp">$StrFindCS</a>, <br>
		<a href="s_strrevfind.kvihelp">$StrRevFind</a><br>
	@examples:
		<example>
			echo $StrRevFindCS(linux, LINUX Rules... and Pragma uses Linux!)
		</example>
*/
bool KviUserParser::parseFnc_STRREVFINDCS(KviCommand *cmd, KviStr &buffer)
{
	KviStr sub, str;
	if( !processFncSingleParam(cmd, sub) ) return false;
	if( !processFncFinalPart(cmd, str)   ) return false;
	sub.setNum(str.findLastIdx(sub.ptr(), true)); // Case sensitive
	buffer.append(sub);
	return true;
}

/*
	@function: StrReplace
	@short:
		Replace all occurrences of a substring with a string
	@syntax:
		$StrReplace(&lt;string&gt;, &lt;substing&gt;, &lt;replacement&gt;)
	@description:
		Returns the &lt;string&gt; with all the occurrences of
		&lt;substring&gt; replaced by &lt;replacement&gt;.<br>
		Note that the &lt;replacement&gt; may be also an empty sting.<br>
		This function is case INSENSITIVE.<br>
	@seealso:
		<a href="s_strreplacecs.kvihelp">$StrReplaceCS</a>, <br>
*/
bool KviUserParser::parseFnc_STRREPLACE(KviCommand *cmd, KviStr &buffer)
{
	KviStr string, toFind, replacement;
	if( !processFncSingleParam(cmd, string)    ) return false;
	if( !processFncSingleParam(cmd, toFind)    ) return false;
	if( !processFncFinalPart(cmd, replacement) ) return false;
	if( toFind.isEmpty() ) return true;
	string.replaceAll(toFind.ptr(), replacement.ptr(), false);
	buffer.append(string);
	return true;
}

/*
	@function: StrReplaceCS
	@short:
		Replace all occurrences of a substring with a string (case-sensitive)
	@syntax:
		$StrReplaceCS(&lt;string&gt;, &lt;substing&gt;, &lt;replacement&gt;)
	@description:
		Returns the &lt;string&gt; with all the occurrences of
		&lt;substring&gt; replaced by &lt;replacement&gt;.<br>
		Note that the &lt;replacement&gt; may be also an empty sting.<br>
		This function is <b>case-sensitive</b>.<br>
	@seealso:
		<a href="s_strreplace.kvihelp">$StrReplace</a>, <br>
*/
bool KviUserParser::parseFnc_STRREPLACECS(KviCommand *cmd, KviStr &buffer)
{
	KviStr string, toFind, replacement;
	if( !processFncSingleParam(cmd, string)    ) return false;
	if( !processFncSingleParam(cmd, toFind)    ) return false;
	if( !processFncFinalPart(cmd, replacement) ) return false;
	if( toFind.isEmpty() ) return true;
	string.replaceAll(toFind.ptr(), replacement.ptr(), true);
	buffer.append(string);
	return true;
}

/*
	@function: Mask
	@short:
		Returns user mask
	@syntax:
		$Mask(&lt;mask_type&gt;, &lt;nickname&gt;)
	@description:
		Returns the mask of the specified &lt;mask_type&gt; of the user &lt;nickname&gt;<br>
		If the user host and username is not known this function returns &lt;nickname&gt;!*@*.<br>
		The mask type is a digit and can be one of the following:<br>
		0 : nick!user@machine.host.top (nick!user@XXX.XXX.XXX.XXX)<br>
		1 : nick!user@*.host.top (nick!user@XXX.XXX.XXX.*)<br>
		2 : nick!user@*<br>
		3 : nick!*@machine.host.top (nick!user@XXX.XXX.XXX.XXX)<br>
		4 : nick!*@*.host.top (nick!user@XXX.XXX.XXX.*)<br>
		5 : nick!*@*<br>
		6 : *!user@machine.host.top (*!user@XXX.XXX.XXX.XX)<br>
		7 : *!user@*.host.top (*!user@XXX.XXX.XXX.*)<br>
		8 : *!user@*<br>
		9 : *!*@machine.host.top (*!*@XXX.XXX.XXX.XXX)<br>
		10: *!*@*.host.top (*!*@XXX.XXX.XXX.*)<br>
		11: nick!*user@machine.host.top (nick!*user@XXX.XXX.XXX.XXX)<br>
		12: nick!*user@*.host.top (nick!*user@XXX.XXX.XXX.*)<br>
		13: nick!*user@*<br>
		14: *!*user@machine.host.top (*!*user@XXX.XXX.XXX.XXX)<br>
		15: *!*user@*.host.top (*!*user@XXX.XXX.XXX.*)<br>
		16: *!*user@*<br>
		If the host or username is not known, the mask may contain less information
		than requested.<br>
		For example, if hostname is missing, the mask type 3 or 4 may be reduced to type 5.<br>
	@seealso:
		<a href="s_isknown.kvihelp">$IsKnown</a>,
		<a href="s_hostname.kvihelp">$Hostname</a>,
		<a href="s_username.kvihelp">$Username</a>,
		<a href="s_iswellknown.kvihelp">$IsWellKnown</a>
*/
bool KviUserParser::parseFnc_MASK(KviCommand *cmd, KviStr &buffer)
{
	KviStr type, nick;
	if( !processFncSingleParam(cmd, type) ) return false;
	if( !processFncFinalPart(cmd, nick)   ) return false;
	bool bOk = false;
	int iType = type.toInt(&bOk);
	if( (!bOk) || (iType < 0) || (iType > 16) )
		iType = 0;
	nick.stripWhiteSpace(); // No trailing or leading spaces
	if( nick.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingNickname, "$Mask()");
		return false;
	}
	KviIrcUser *u = m_pFrm->m_pUserList->findUser(nick.ptr());
	if( !u ) {
		// Append default
		buffer.append(nick);
		buffer.append("!*@*");
	} else {
		u->mask(type, iType);
		buffer.append(type);
	}
	return true;
}

/*
	@function: Hostname
	@short:
		Returns a cached user's hostname
	@syntax:
		$Hostname(&lt;nickname&gt;)
	@description:
		If the user &lt;nickname&gt; is in the internal userlist and his hostname
		has been already received, this function returns his hostname.<br>
		Otherwise an empty string is returned.<br>
	@seealso:
		<a href="s_mask.kvihelp">$Mask</a>,
		<a href="s_isknown.kvihelp">$IsKnown</a>,
		<a href="s_iswellknown.kvihelp">$IsWellKnown</a>,
		<a href="s_username.kvihelp">$Username</a>
*/
bool KviUserParser::parseFnc_HOSTNAME(KviCommand *cmd, KviStr &buffer)
{
	KviStr nick;
	if( !processFncFinalPart(cmd, nick) ) return false;
	nick.stripWhiteSpace(); // No trailing or leading spaces in nicknames
	if( nick.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingNickname, "$Hostname()");
		return false;
	}
	KviIrcUser *u = m_pFrm->m_pUserList->findUser(nick.ptr());
	if( u && u->hasHost() )
		buffer.append(u->host());
	return true;
}

/*
	@function: Username
	@short:
		Returns a cached user's username
	@syntax:
		$Username(&lt;nickname&gt;)
	@description:
		If the user &lt;nickname&gt; is in the internal userlist and his username
		has been already received, this function returns his username.<br>
		Otherwise an empty string is returned.<br>
	@seealso:
		<a href="s_mask.kvihelp">$Mask</a>,
		<a href="s_isknown.kvihelp">$IsKnown</a>,
		<a href="s_iswellknown.kvihelp">$IsWellKnown</a>,
		<a href="s_hostname.kvihelp">$Hostname</a>
*/
bool KviUserParser::parseFnc_USERNAME(KviCommand *cmd, KviStr &buffer)
{
	KviStr nick;
	if( !processFncFinalPart(cmd, nick) ) return false;
	nick.stripWhiteSpace(); // No trailing or leading spaces in nicknames
	if( nick.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingNickname, "$Username()");
		return false;
	}
	KviIrcUser *u = m_pFrm->m_pUserList->findUser(nick.ptr());
	if( u && u->hasUsername() )
		buffer.append(u->username());
	return true;
}

/*
	@function: IsKnown
	@short:
		Checks whether the user is in the internal userlist
	@syntax:
		$IsKnown(&lt;nickname&gt;)
	@description:
		Returns 1 if the user &lt;nickname&gt; is in the internal userlist,
		0 otherwise.<br> A user is in the userlist if he is on a channel with you
		or in the notify list (actually on irc).<br>
	@seealso:
		<a href="s_mask.kvihelp">$Mask</a>,
		<a href="s_iswellknown.kvihelp">$IsWellKnown</a>
*/
bool KviUserParser::parseFnc_ISKNOWN(KviCommand *cmd, KviStr &buffer)
{
	KviStr nick;
	if( !processFncFinalPart(cmd, nick) ) return false;
	nick.stripWhiteSpace(); // No trailing or leading spaces in nicknames
	if( nick.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingNickname, "$IsKnown()");
		return false;
	}
	buffer.append((m_pFrm->m_pUserList->findUser(nick.ptr()) ? '1' : '0'));
	return true;
}

/*
	@function: IsWellKnown
	@short:
		Checks whether the user is in the internal userlist and his hostname is known
	@syntax:
		$IsWellKnown(&lt;nickname&gt;)
	@description:
		Returns 1 if the user &lt;nickname&gt; is in the internal userlist and
		his hostname/username have been already received,
		0 otherwise.<br> A user is in the userlist if he is on a channel with you
		or in the notify list (actually on irc).<br>
		If this function returns 1 you may be sure that <a href="s_hostname.kvihelp">$hostname(&lt;nickname&gt;)</a>
		will return a valid string.<br>
		This function is more or less equivalent to:<br>
		<b>("<a href="s_hostname.kvihelp">$Hostname</a>(&lt;nickname&gt;)" != "")</b><br>
		or to:<br>
		<b>("<a href="s_username.kvihelp">$Hostname</a>(&lt;nickname&gt;)" != "")</b><br>
		But is really faster than either.<br>
	@seealso:
		<a href="s_mask.kvihelp">$Mask()</a>,
		<a href="s_isknown.kvihelp">$IsKnown()</a>,
		<a href="s_hostname.kvihelp">$Hostname()</a>,
		<a href="s_username.kvihelp">$Username()</a>
*/
bool KviUserParser::parseFnc_ISWELLKNOWN(KviCommand *cmd, KviStr &buffer)
{
	KviStr nick;
	if( !processFncFinalPart(cmd, nick) ) return false;
	nick.stripWhiteSpace(); // No trailing or leading spaces in nicknames
	if( nick.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingNickname, "$IsWellKnown()");
		return false;
	}
	KviIrcUser *u = m_pFrm->m_pUserList->findUser(nick.ptr());
	if( !u )
		buffer.append('0');
	else
		buffer.append(u->hasMask() ? '1' : '0');
	return true;
}

/*
	@function: StrToHex
	@short:
		Returns the hex translation of the string
	@syntax:
		$StrToHex(&lt;string&gt;)
	@description:
		Returns a sequence of hexadecimal digits that encodes
		the arument string.<br>
		For example, if you call $StrToHex("hello") you will obtain "68656C6C6F, "
		the hexadecimal form of each character in "hello."<br>
	@seealso:
		<a href="s_hextostr.kvihelp">$HexToStr()</a>
*/
bool KviUserParser::parseFnc_STRTOHEX(KviCommand *cmd, KviStr &buffer)
{
	KviStr str;
	if( !processFncFinalPart(cmd, str) ) return false;
	KviStr hex;
	hex.bufferToHex(str.ptr(), str.len());
	buffer.append(hex.ptr());
	return true;
}

/*
	@function: HexToStr
	@short:
		Returns the string translation of the hexadecimal string
	@syntax:
		$HexToStr(&lt;hexstring&gt;)
	@description:
		Returns the human-readable sequence of characters
		that are encoded in the hexadecimal string.<br>
		For example, if you call $HexToStr("68656C6C6F") you will obtain "hello".<br>
		This function will translate all "00" hex sequences that correspond to the null characters
		to newlines.<br>
	@seealso:
		<a href="s_strtohex.kvihelp">$StrToHex()</a>
*/
bool KviUserParser::parseFnc_HEXTOSTR(KviCommand *cmd, KviStr &buffer)
{
	KviStr hex;
	if( !processFncFinalPart(cmd, hex) ) return false;
	char *buf;
	int len = hex.hexToBuffer(&buf, true);
	buffer.append(buf, len);
	kvi_free(buf);
	return true;
}

/*
	@function: IsValidIp
	@short:
		Checks if a string represents a valid IP address
	@syntax:
		$IsValidIp(&lt;ip_address&gt;)
	@description:
		Returns 1 if the user &lt;ip_address&gt; string
		is a valid representation of an IP address (i.e, only has digits and dots),
		0 otherwise.
	@examples:
		<example>
			<a href="if.kvihelP">if</a>($isValidIp(127.0.0.1))<a href="server.kvihelp">server</a> 127.0.0.1 6667
		</example>
	@seealso:
		<a href="s_inetaton.kvihelp">$InetAton()</a>, <a href="s_inetntoa.kvihelp">$InetNtoa()</a>
*/
bool KviUserParser::parseFnc_ISVALIDIP(KviCommand *cmd, KviStr &buffer)
{
	KviStr ip;
	if( !processFncFinalPart(cmd, ip) ) return false;
	ip.stripWhiteSpace(); // No trailing or leading spaces in IP addresses
	buffer.append((kvi_isValidStringIp(ip.ptr()) ? '1' : '0'));
	return true;
}

/*
	@function: InetAton
	@short:
		Converts a dotted quad to a binary IP address
	@syntax:
		$InetAton(&lt;ip_address&gt;)
	@description:
		Returns a unsigned integer that represents &lt;ip_address&gt; in network byte order.<br>
		Please note that the returned number may lie out of range of <a href="s_calc.kvihelp">$calc()</a>.<br>
	@examples:
		<example>
			<a href="echo.kvihelp">echo</a> <a href="s_inetntoa.kvihelp">$InetNtoa</a>($InetAton(127.0.0.1))
		</example>
	@seealso:
		<a href="s_inetntoa.kvihelp">$InetNtoa()</a>, <a href="s_isvalidip.kvihelp">$IsValidIp()</a>
*/
bool KviUserParser::parseFnc_INETATON(KviCommand *cmd, KviStr &buffer)
{
	KviStr ip;
	if( !processFncFinalPart(cmd, ip) ) return false;
	ip.stripWhiteSpace(); // No spaces
	struct in_addr i;
	if( !kvi_stringIpToBinaryIp(ip.ptr(), &i) ) {
		cmd->setError(KVI_ERROR_InvalidIpAddress, "InetAton()", ip.ptr());
		return false;
	}
	KviStr tmp(KviStr::Format, "%u", i.s_addr);
	buffer.append(tmp);
	return true;
}

/*
	@function: InetNtoa
	@short:
		Converts a binary IP address to a dotted quad
	@syntax:
		$InetNtoa(&lt;binary_ip_address&gt;)
	@description:
		Inverse function of <a href="s_inetaton.kvihelp">$InetAton</a>.<br>
		Returns the dotted-quad equivalent of &lt;binary_ip_address&gt;.<br>
		&lt;binary_ip_address&gt; must be an unsigned number assumed in network byte order.<br>
	@examples:
		<example>
			<a href="echo.kvihelp">echo</a> $InetNtoa(<a href="s_inetaton.kvihelp">$InetAton</a>(127.0.0.1))
		</example>
	@seealso:
		<a href="s_inetaton.kvihelp">$InetAton()</a>, <a href="s_isvalidip.kvihelp">$IsValidIp()</a>
*/
bool KviUserParser::parseFnc_INETNTOA(KviCommand *cmd, KviStr &buffer)
{
	KviStr ip;
	if( !processFncFinalPart(cmd, ip) ) return false;
	ip.stripWhiteSpace(); // No spaces
	struct in_addr i;
	bool bOk = false;
	i.s_addr = ip.toULong(&bOk);
	if( !bOk ) {
		cmd->setError(KVI_ERROR_UnsignedIntegerExpected, "InetNtoa()", ip.ptr());
		return false;
	}
	if( !kvi_binaryIpToString(i, ip) ) {
		cmd->setError(KVI_ERROR_UnsignedIntegerExpected, "InetNtoa()", ip.ptr());
		return false;
	}
	buffer.append(ip);
	return true;
}

/*
	@function: Char
	@short:
		Inserts characters by ASCII code
	@syntax:
		$Char(&lt;token&gt;[, &lt;token&gt;])
	@description:
		Returns a string with all &lt;tokens&gt; concatenated
		and the following substitutions:<br>
		If &lt;token&gt; is an unsigned number, it is replaced by
		the corresponding ASCII character, otherwise the &lt;token&gt;
		is simply appended.<br>
		Inserting a NULL (ASCII 0) character is NOT allowed.<br>
	@examples:
		<example>
			echo $char(75, 86, 73, 114, 99);
			echo $char(75, 86, Ir, 99, " Millennium");
		</example>
*/
bool KviUserParser::parseFnc_CHAR(KviCommand *cmd, KviStr &buffer)
{
	KviStr code;
	KviStr token;
	if( !processFncFinalPart(cmd, code) ) return false;
	while( code.hasData() ) {
		code.getToken(token, ',');
		bool bOk = false;
		unsigned char ch = token.toUChar(&bOk);
		if( !bOk )
			buffer.append(token);
		else {
			if( ch == 0 )
				buffer.append("NULL");
			else
				buffer.append((char) ch);
		}
	}
	return true;
}

/*
	@function: Ascii
	@short:
		Returns ASCII codes of a characters
	@syntax:
		$Ascii(&lt;token&gt;)
	@description:
		Returns a string with a comma-separated list of
		the ASCII codes for each character of &lt;token&gt;<br>
	@examples:
		<example>
			echo $ascii(v);
			echo $ascii(h);
		</example>
*/
bool KviUserParser::parseFnc_ASCII(KviCommand *cmd, KviStr &buffer)
{
	KviStr token;
	if( !processFncFinalPart(cmd, token) ) return false;
	bool bNotFirst = false;
	while( token.hasData() ) {
		if( bNotFirst )
			buffer.append(',');
		else
			bNotFirst = true;
		KviStr ch(KviStr::Format, "%d", (unsigned char) *token.ptr());
		buffer.append(ch);
		token.cutLeft(1);
	}
	return true;
}

/*
	@function: BanList
	@short:
		Returns a list of ban masks on a channel
	@syntax:
		$BanList(&lt;channel&gt;[, &lt;index&gt;])
	@description:
		Without the &lt;index&gt; parameter this function returns a comma
		separated list of ban masks set on the channel.
		With the &lt;index&gt; parameter it returns the nTH ban mask,
		or an empty string if &lt;index&gt; is greater than the number
		of masks set - 1;
*/
bool KviUserParser::parseFnc_BANLIST(KviCommand *cmd, KviStr &buffer)
{
	KviStr channel;
	KviStr index;
	if( !processFncSingleParam(cmd, channel) ) return false;
	if( !processFncFinalPart(cmd, index)     ) return false;

	bool bOk;
	int idx = index.toInt(&bOk);

	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan )
		return recoverableError(cmd, KVI_ERROR_WindowNotFound, "$BANLIST");

	if( bOk ) {
		if( idx < ((int) chan->m_pBanList->count()) ) {
			KviBanMask *pMask = chan->m_pBanList->at(idx);
			if( pMask )
				buffer.append(pMask->mask);
		}
	} else {
		bool bNotFirst = false;
		for( KviBanMask *pMask = chan->m_pBanList->first(); pMask; pMask = chan->m_pBanList->next() ) {
			if( bNotFirst )
				buffer.append(',');
			else
				bNotFirst = true;
			buffer.append(pMask->mask);
		}
	}
	return true;
}

/*
	@function: BanExceptionList
	@short:
		Returns a list of banexception masks on a channel
	@syntax:
		$BanExceptionList(&lt;channel&gt;[, &lt;index&gt;])
	@description:
		Without the &lt;index&gt; parameter this function returns a comma
		separated list of ban exception masks set on the channel.
		With the &lt;index&gt; parameter it returns the nTH ban exception mask,
		or an empty string if &lt;index&gt; is greater than the number
		of masks set - 1;
		If the server does not support ban exceptions, this function
		returns always an empty string.
*/
bool KviUserParser::parseFnc_BANEXCEPTIONLIST(KviCommand *cmd, KviStr &buffer)
{
	KviStr channel;
	KviStr index;
	if( !processFncSingleParam(cmd, channel) ) return false;
	if( !processFncFinalPart(cmd, index)     ) return false;

	bool bOk;
	int idx = index.toInt(&bOk);

	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan )
		return recoverableError(cmd, KVI_ERROR_WindowNotFound, "$BANEXCEPTIONLIST");

	if( bOk ) {
		if( idx < ((int) chan->m_pExceptionList->count()) ) {
			KviBanMask *pMask = chan->m_pExceptionList->at(idx);
			if( pMask )
				buffer.append(pMask->mask);
		}
	} else {
		bool bNotFirst = false;
		for( KviBanMask *pMask = chan->m_pExceptionList->first(); pMask; pMask = chan->m_pExceptionList->next() ) {
			if( bNotFirst )
				buffer.append(',');
			else
				bNotFirst = true;
			buffer.append(pMask->mask);
		}
	}
	return true;
}

/*
	@function: InviteExceptionList
	@short:
		Returns a list of invite exception masks on a channel
	@syntax:
		$InviteExceptionList(&lt;channel&gt;[, &lt;index&gt;])
	@description:
		Without the &lt;index&gt; parameter this function returns a comma
		separated list of invite exception masks set on the channel.
		With the &lt;index&gt; parameter it returns the nTH invite exception mask,
		or an empty string if &lt;index&gt; is greater than the number
		of masks set - 1;
		If the server does not support invite exceptions, this function
		returns always an empty string.
*/
bool KviUserParser::parseFnc_INVITEEXCEPTIONLIST(KviCommand *cmd, KviStr &buffer)
{
	KviStr channel;
	KviStr index;
	if( !processFncSingleParam(cmd, channel) ) return false;
	if( !processFncFinalPart(cmd, index)     ) return false;

	bool bOk;
	int idx = index.toInt(&bOk);

	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan )
		return recoverableError(cmd, KVI_ERROR_WindowNotFound, "INVITEEXCEPTIONLIST");

	if( bOk ) {
		if( idx < ((int) chan->m_pInviteExceptionList->count()) ) {
			KviBanMask *pMask = chan->m_pInviteExceptionList->at(idx);
			if( pMask )
				buffer.append(pMask->mask);
		}
	} else {
		bool bNotFirst = false;
		KviBanMask *pMask;
		for( pMask = chan->m_pInviteExceptionList->first(); pMask; pMask = chan->m_pInviteExceptionList->next() ) {
			if( bNotFirst )
				buffer.append(',');
			else
				bNotFirst = true;
			buffer.append(pMask->mask);
		}
	}
	return true;
}

/*
	@function: AllOnChan
	@short:
		Returns a list of nicknames on a channel
	@syntax:
		$AllOnChan(&lt;channel&gt;)
	@description:
		Returns a comma separated list of nicknames
		that are on the specified channel.<br>
		If the channel window cannot be found
		this function returns an empty string.<br>
	@examples:
		<example>
			<a href="foreach.kvihelp">foreach</a>(%t, $allonchan(<a href="s_chan.kvihelp">$chan</a>))<a href="echo.kvihelp">echo</a> %t is on this channel
		</example>
	@seealso:
		<a href="s_oponchan.kvihelp">$OpOnChan()</a>, <a href="s_voiceonchan.kvihelp">$VoiceOnChan()</a>,
		<a href="s_noponchan.kvihelp">$NOpOnChan()</a>
*/
bool KviUserParser::parseFnc_ALLONCHAN(KviCommand *cmd, KviStr &buffer)
{
	KviStr window;
	if( !processFncFinalPart(cmd, window) ) return false;
	window.stripWhiteSpace();
	KviChannel *chan = m_pFrm->findChannel(window.ptr());
	if( !chan )             return true;
	if( !chan->m_pListBox ) return true;
	chan->m_pListBox->appendNickList(buffer);
	return true;
}

/*
	@function: OpOnChan
	@short:
		Returns a list of operator nicknames on a channel
	@syntax:
		$OpOnChan(&lt;channel&gt;)
	@description:
		Returns a comma separated list of nicknames
		that are on the specified channel and have operator status.<br>
		If the channel window cannot be found
		this function returns an empty string.<br>
	@examples:
		<example>
			<a href="foreach.kvihelp">foreach</a>(%t, $oponchan(<a href="s_chan.kvihelp">$chan</a>))<a href="echo.kvihelp">echo</a> %t is op on this channel
		</example>
		<a href="s_allonchan.kvihelp">$AllOnChan()</a>, <a href="s_voiceonchan.kvihelp">$VoiceOnChan()</a>,
		<a href="s_noponchan.kvihelp">$NOpOnChan()</a>
*/
bool KviUserParser::parseFnc_OPONCHAN(KviCommand *cmd, KviStr &buffer)
{
	KviStr window;
	if( !processFncFinalPart(cmd, window) ) return false;
	window.stripWhiteSpace();
	KviChannel *chan = m_pFrm->findChannel(window.ptr());
	if( !chan )             return true;
	if( !chan->m_pListBox ) return true;
	chan->m_pListBox->appendOpList(buffer);
	return true;
}

/*
	@function: NOpOnChan
	@short:
		Returns a list of non-operator nicknames on a channel
	@syntax:
		$OpOnChan(&lt;channel&gt;)
	@description:
		Returns a comma separated list of nicknames
		that are on the specified channel and NOT have operator status.<br>
		If the channel window cannot be found
		this function returns an empty string.<br>
	@examples:
		<example>
			<a href="foreach.kvihelp">foreach</a>(%t, $noponchan(<a href="s_chan.kvihelp">$chan</a>))<a href="echo.kvihelp">echo</a> %t is not an op on this channel
		</example>
		<a href="s_allonchan.kvihelp">$AllOnChan()</a>, <a href="s_voiceonchan.kvihelp">$VoiceOnChan()</a>,
		<a href="s_oponchan.kvihelp">$OpOnChan()</a>
*/
bool KviUserParser::parseFnc_NOPONCHAN(KviCommand *cmd, KviStr &buffer)
{
	KviStr window;
	if( !processFncFinalPart(cmd, window) ) return false;
	window.stripWhiteSpace();
	KviChannel *chan = m_pFrm->findChannel(window.ptr());
	if( !chan )             return true;
	if( !chan->m_pListBox ) return true;
	chan->m_pListBox->appendNOpList(buffer);
	return true;
}

/*
	@function: VoiceOnChan
	@short:
		Returns a list of voiced nicknames on a channel
	@syntax:
		$OpOnChan(&lt;channel&gt;)
	@description:
		Returns a comma separated list of nicknames
		that are on the specified channel and have voice status.<br>
		If the channel window cannot be found
		this function returns an empty string.<br>
	@examples:
		<example>
			<a href="foreach.kvihelp">foreach</a>(%t, $voiceonchan(<a href="s_chan.kvihelp">$chan</a>))<a href="echo.kvihelp">echo</a> %t has voice status
		</example>
		<a href="s_allonchan.kvihelp">$AllOnChan()</a>, <a href="s_noponchan.kvihelp">$NOpOnChan()</a>,
		<a href="s_oponchan.kvihelp">$OpOnChan()</a>
*/
bool KviUserParser::parseFnc_VOICEONCHAN(KviCommand *cmd, KviStr &buffer)
{
	KviStr window;
	if( !processFncFinalPart(cmd, window) ) return false;
	window.stripWhiteSpace();
	KviChannel *chan = m_pFrm->findChannel(window.ptr());
	if( !chan )             return true;
	if( !chan->m_pListBox ) return true;
	chan->m_pListBox->appendVoiceList(buffer);
	return true;
}

/*
	@function: NumTimeToString
	@short:
		Translates a UNIX time in seconds to a string
	@syntax:
		$NumTimeToString(&lt;time_t&gt;)
	@description:
		Translates a UNIX time in seconds to a human readable string.
*/
bool KviUserParser::parseFnc_NUMTIMETOSTRING(KviCommand *cmd, KviStr &buffer)
{
	KviStr szTime;
	if( !processFncFinalPart(cmd, szTime) ) return false;
	bool bOk = false;
	unsigned int uT = szTime.toUInt(&bOk);
	if( !bOk )
		buffer.append(_i18n_("Invalid Time"));
	else {
		QDateTime dt;
		dt.setTime_t(uT);
		buffer.append(dt.toString());
	}
	return true;
}

/*
	@function: DurationToString
	@short:
		Translates an duration in seconds to a string
	@syntax:
		$DurationToString(&lt;duration&gt;)
	@description:
		Translates a duration expressed in seconds to a human readable string
		in the form [days d hours h minutes m seconds s]
*/
bool KviUserParser::parseFnc_DURATIONTOSTRING(KviCommand *cmd, KviStr &buffer)
{
	KviStr szTime;
	if( !processFncFinalPart(cmd, szTime) ) return false;
	bool bOk = false;
	unsigned int uT = szTime.toUInt(&bOk);
	if( !bOk )
		uT = 0;
	int days  = uT / 86400;     // 86400 secs in a day
	uT = uT % 86400;
	int hours = uT /  3600;     // 3600 secs in a hour
	uT = uT % 3600;
	int mins  = uT /    60;     // 60 secs in a minute
	uT = uT % 60;               // uT == secs!
	szTime.sprintf("%d d %d h %d m %d s", days, hours, mins, uT);
	buffer.append(szTime);
	return true;
}

/*
	@function: ItemCount
	@short:
		Returns the number of listbox items in a user window
	@syntax:
		$ItemCount(&lt;window&gt;)
	@description:
		Returns the number of items in the listbox of &lt;window&gt;
*/
bool KviUserParser::parseFnc_ITEMCOUNT(KviCommand *cmd, KviStr &buffer)
{
	KviStr window;
	if( !processFncFinalPart(cmd, window) ) return false;
	KviUserWindow *uwin = m_pFrm->findUserWindow(window.ptr());
	if( !uwin )
		return recoverableError(cmd, KVI_ERROR_WindowNotFound, "$ItemCount");
	window.setNum(uwin->itemCount());
	buffer.append(window);
	return true;
}

/*
	@function: DictList
	@short:
		Returns a comma separated list of keys in the specified dictionary
	@syntax:
		$DictList(&lt;dictionary_name&gt;)
	@description:
		Returns a comma separated list of keys in the specified dictionary.<br>
		If the dictionary is not existing (equivalent to "empty") this
		function returns an empty string.<br>
	@examples:
		<example>
			<a href="foreach.kvihelp">foreach</a>(%i, <a href="s_range.kvihelp">$range</a>(0, 10))%mydict[%i]=something
			<a href="s_echo.kvihelp">echo</a> $dictlist(mydict)
			<a href="foreach.kvihelp">foreach</a>(%i, $dictlist(mydict))%mydict[%i]=
			<a href="echo.kvihelp">echo</a> $dictlist(mydict)
		</example>
	@seealso:
		<a href="s_dictcount.kvihelp">$DictCount</a>
*/
bool KviUserParser::parseFnc_DICTLIST(KviCommand *cmd, KviStr &buffer)
{
	KviStr dictname;
	if( !processFncFinalPart(cmd, dictname) ) return false;

	if( dictname.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingDictionaryName, "$DictList");
		return false;
	}

	KviVariableCache *cache;
	if( islower(*(dictname.ptr())) )
		cache = cmd->m_pLocalVarCache;
	else
		cache = g_pVarCache;

	bool bHasComma = false;
	KviDictionary *dict = cache->getDict(dictname.ptr());
	if( dict ) {
		KviVariable *var;
		for(var = dict->m_pVarCache->m_pVarList->first(); var; var = dict->m_pVarCache->m_pVarList->next() ) {
			if( bHasComma )
				buffer.append(", ");
			else
				bHasComma = true;
			buffer.append(var->szName.ptr());
		}
	}
	return true;
}

/*
	@function: DictCount
	@short:
		Returns the numer of keys in the specified dictionary
	@syntax:
		$DictCount(&lt;dictionary_name&gt;)
	@description:
		Returns the numer of keys in the specified dictionary<br>
		A return value of 0 means that the dictionary is empty (= unset).<br>
	@examples:
		<example>
			<a href="foreach.kvihelp">foreach</a>(%i, <a href="s_range.kvihelp">$range</a>(0, 10))%mydict[%i]=something
			<a href="s_echo.kvihelp">echo</a> $dictcount(mydict)
			<a href="foreach.kvihelp">foreach</a>(%i, <a href="s_dictlist.kvihelp">$dictlist</a>(mydict))%mydict[%i]=
			<a href="echo.kvihelp">echo</a> $dictcount(mydict)
		</example>
	@seealso:
		<a href="s_dictlist.kvihelp">$DictList</a>
*/
bool KviUserParser::parseFnc_DICTCOUNT(KviCommand *cmd, KviStr &buffer)
{
	KviStr dictname;
	if( !processFncFinalPart(cmd, dictname) ) return false;

	if( dictname.isEmpty() ) {
		cmd->setError(KVI_ERROR_MissingDictionaryName, "$DictCount");
		return false;
	}

	KviVariableCache *cache;
	if( islower(*(dictname.ptr())) )
		cache = cmd->m_pLocalVarCache;
	else
		cache = g_pVarCache;

	KviDictionary *dict = cache->getDict(dictname.ptr());
	if( dict ) {
		KviStr tmp;
		tmp.setNum(dict->m_pVarCache->m_pVarList->count());
		buffer.append(tmp);
	} else buffer.append('0');
	return true;
}

/*
	@function: Config
	@short:
		Returns data stored in the user config file
	@syntax:
		$Config(&lt;section&gt;, &lt;key&gt;)
	@description:
		Returns the data set by <a href="config.kvihelp">config</a> for the
		specified key in the specified section.<br>
		If there is no data associated with the key, or the section
		does not exist an empty string is returned.<br>
	@seealso:
		<a href="config.kvihelp">CONFIG</a>
*/

bool KviUserParser::parseFnc_CONFIG(KviCommand *cmd, KviStr &buffer)
{
	KviStr section;
	KviStr key;
	if( !processFncSingleParam(cmd, section) ) return false;
	if( !processFncFinalPart(cmd, key)       ) return false;
	if( section.isEmpty() )
		return recoverableError(cmd, KVI_ERROR_MissingSection, "$Config()");
	if( key.isEmpty() )
		return recoverableError(cmd, KVI_ERROR_MissingKey, "$Config()");
	g_pUserConfig->setGroup(section.ptr());
	KviStr tmp = g_pUserConfig->readEntry(key.ptr(), "");
	if( !tmp.hasData() )
		g_pUserConfig->clearKey(key.ptr()); // The key is empty... equivalent to unset
	buffer.append(tmp);
	return true;
}

/*
	@function: PluginLoaded
	@short:
		Checks whether a plugin is loaded or not
	@syntax:
		$PluginLoaded(&lt;plugin_name&gt;)
	@description:
		Returns 1 if the plugin named &lt;plugin_name&gt; is currently loaded, 0 otherwise.<br>
		&lt;plugin_name&gt; can be either a filename with path or a module name
		(The internal name that the plugin declares; for example, the libkvifserve.so declares the name "FServe";
		you can check the module name in the plugin page of the misc options dialog).<br>
		If KVIrc has been compiled without plugin support, this function always returns 0.
	@seealso:
		<a href="plugin.kvihelp">PLUGIN</a>
*/
bool KviUserParser::parseFnc_PLUGINLOADED(KviCommand *cmd, KviStr &buffer)
{
	KviStr plugname;
	if( !processFncFinalPart(cmd, plugname) ) return false;
#ifdef COMPILE_PLUGIN_SUPPORT
	KviPluginData *d = g_pPluginManager->findPlugin(plugname.ptr());
	if( d )
		buffer.append('1');
	else {
		d = g_pPluginManager->findPluginByModuleName(plugname.ptr());
		buffer.append(d ? '1' : '0');
	}
#else
	buffer.append('0');
#endif
	return true;
}

/*
	@function: IsTimer
	@short:
		Checks if a timer is running
	@syntax:
		$IsTimer(&lt;timer_name_mask&gt;)
	@description:
		Returns the number of running timers that match the &lt;timer_name_mask&gt;.<br>
		&lt;timer_name_masg&gt; is a regular expression that can contain * and ? wildcards.<br>
		If there are no matching timers running, this function returns 0.<br>
	@examples:
		Is timer "mytimer" running ?
		<example>
		<a href="if.kvihelp">if</a>($IsTimer(mytimer))<a href="echo.kvihelp">echo</a> mytimer is running
		else <a href="echo.kvihelp">echo</a> mytimer is not running
		</example>
		Find the number of timers running
		<example>
		<a href="echo.kvihelp">echo</a> $IsTimer(*)
		</examples>
	@seealso:
		<a href="timer.kvihelp">TIMER</a>
		<a href="killtimer.kvihelp">KILLTIMER</a>
*/
bool KviUserParser::parseFnc_ISTIMER(KviCommand *cmd, KviStr &buffer)
{
	KviStr timermask;
	if( !processFncFinalPart(cmd, timermask) ) return false;
	if( timermask.isEmpty() )
		return recoverableError(cmd, KVI_ERROR_MissingTimerName, "IsTimer()");
	QRegExp rexp(_CHAR_2_QSTRING(timermask.ptr()), false, true);
	int nTimers = 0;
	for( KviWindowTimer *t = m_pTimerList->first(); t; t = m_pTimerList->next() ) {
		if( rexp.search(_CHAR_2_QSTRING(t->name.ptr())) == 0 ) {
			if( rexp.matchedLength() == t->name.len() ) {
				nTimers++;
			}
		}
	}
	timermask.setNum(nTimers);
	buffer.append(timermask);
	return true;
}

/**
 * No need for docs!!!
 */
bool KviUserParser::parseFnc_SELECTED(KviCommand *cmd, KviStr &buffer)
{
	KviStr window;
	if( !processFncFinalPart(cmd, window) ) return false;
	KviWindow *wnd = m_pFrm->findWindow(window.ptr());
	if( !wnd ) return true;
	if( !wnd->m_pListBox ) return true;
	wnd->m_pListBox->appendSelectedNicknames(buffer);
	return true;
}

bool KviUserParser::parseFnc_SELECTEDMASKS(KviCommand *cmd, KviStr &buffer)
{
	KviStr window;
	if( !processFncFinalPart(cmd, window) ) return false;
	KviWindow *wnd = m_pFrm->findWindow(window.ptr());
	if( !wnd ) return true;
	if( !wnd->m_pListBox ) return true;
	wnd->m_pListBox->appendSelectedMasks(buffer);
	return true;
}

bool KviUserParser::parseFnc_SELECTEDHOSTS(KviCommand *cmd, KviStr &buffer)
{
	KviStr window;
	if( !processFncFinalPart(cmd, window) ) return false;
	KviWindow *wnd = m_pFrm->findWindow(window.ptr());
	if( !wnd ) return true;
	if( !wnd->m_pListBox ) return true;
	wnd->m_pListBox->appendSelectedHosts(buffer);
	return true;
}

bool KviUserParser::parseFnc_SELECTEDUSERNAMES(KviCommand *cmd, KviStr &buffer)
{
	KviStr window;
	if( !processFncFinalPart(cmd, window) ) return false;
	KviWindow *wnd = m_pFrm->findWindow(window.ptr());
	if( !wnd ) return true;
	if( !wnd->m_pListBox ) return true;
	wnd->m_pListBox->appendSelectedUsernames(buffer);
	return true;
}
