// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviMain"

#include <signal.h>

#include "kvi_defines.h"

#ifdef COMPILE_NEED_KDE
#include <kdeversion.h>
#if KDE_IS_VERSION(3,2,0)
#include <kcmdlineargs.h>
#endif
#endif

#include "kvi_app.h"

//
// This here is just a tiny suggestion... A nice (IMHO) way for KVIrc to
// *at least* save its settings in case it segfaults - this may sometimes happen
// for as we all know computers are the devil's tools
//
void sigsegv_handler(int);

#define KVI_ARGS_RETCODE_OK    0
#define KVI_ARGS_RETCODE_ERROR 1
#define KVI_ARGS_RETCODE_STOP  2

int parseArgs(int argc, char **argv, char **configFile, char **sessionFile, char **server, char **port, bool *createFile)
{
	if( argc > 1 ) {
		int idx = 1;
		while( idx < argc ) {
			if( kvi_strEqualCI("-v", argv[idx]) || kvi_strEqualCI("--version", argv[idx]) ) {
				debug("%s", KVI_VERSION);
				debug("Build date: %s", BUILD_DATE);
				debug("Script language tag: %s", KVI_SCRIPT_LANGUAGE_TAG);
				debug("Home page: http://packages.qa.debian.org/kvirc2");
				return KVI_ARGS_RETCODE_STOP;
			} else if( kvi_strEqualCI("-h", argv[idx]) || kvi_strEqualCI("--help", argv[idx]) ) {
				debug("Usage:");
				debug("    %s [options] [server [port]]", argv[0]);
				debug("Available options:");
				debug("    -h        : Print this help and exit");
				debug("    -v        : Print the version information and exit");
				debug("    -c <file> : Use <file> as config file instead of ~/%s", KVI_HOME_CONFIG_FILE_NAME);
				debug("                (defaults to ~/%s if <file> does not exist)", KVI_HOME_CONFIG_FILE_NAME);
				debug("    -n <file> : Use <file> as config file instead of ~/%s", KVI_HOME_CONFIG_FILE_NAME);
				debug("                (create <file> if it does not exist)");
				debug("    server    : Connect to this server after startup");
				debug("    port      : Use this port for connection");
				return KVI_ARGS_RETCODE_STOP;
			} else if( kvi_strEqualCI("-c", argv[idx]) ) {
				idx++;
				if( idx >= argc ) {
					debug("Option -c requires a config file name");
					return KVI_ARGS_RETCODE_ERROR;
				}
				*configFile = argv[idx];
				debug("Using file %s as config", argv[idx]);
			} else if( kvi_strEqualCI("-n", argv[idx]) ) {
				idx++;
				if( idx >= argc ) {
					debug("Option -n requires a config file name");
					return KVI_ARGS_RETCODE_ERROR;
				}
				*configFile = argv[idx];
				*createFile = true;
				debug("Using file %s as config", argv[idx]);
			} else if( kvi_strEqualCI("-session", argv[idx]) ) {
				idx++;
				if( idx >= argc ) {
					debug("Option -session requires a session file name");
					return KVI_ARGS_RETCODE_ERROR;
				}
				*sessionFile = argv[idx];
			} else {
				if( (*(argv[idx]) != '-') && (!(*port)) ) {
					// No dash
					KviStr tmp(argv[idx]);
					if( tmp.hasData() ) {
						if( *server ) {
							if( tmp.isUnsignedNum() )
								*port = argv[idx];
						} else (*server) = argv[idx];
					}
				}
			}
			idx++;
		}
	}

	return KVI_ARGS_RETCODE_OK;
}

#ifndef SIG_ERR
	#define SIG_ERR	((sighandler_t) -1)
#endif

int main(int argc, char **argv)
{
	char *configFile  = 0;
	char *sessionFile = 0;
	char *port        = 0;
	char *server      = 0;
	bool createFile   = false;

	if( signal(SIGSEGV, sigsegv_handler) == SIG_ERR )
		debug("Error in call to signal() - continuing anyway.");

	int retCode = parseArgs(argc, argv, &configFile, &sessionFile, &server, &port, &createFile);
	if( retCode != KVI_ARGS_RETCODE_OK )
		return ((retCode == KVI_ARGS_RETCODE_ERROR) ? (-1) : 0);

#ifdef COMPILE_NEED_KDE
#if KDE_IS_VERSION(3,2,0)
	KCmdLineArgs::init(argc, argv, "kvirc", "kvirc", "", VERSION);
#endif
#endif
	g_pApp = new KviApp(argc, argv);
	g_pApp->m_pCreateConfig = createFile;
	g_pApp->m_pConfigFile   = configFile;
	if( !g_pApp->setup() ) {
		retCode = KVI_ARGS_RETCODE_ERROR;
	} else {
		if( server )
			g_pApp->connectAfterStartup(server, port);
		retCode = g_pApp->exec();
	}
	g_pApp->destroy();
	g_pApp = 0;
	return retCode;
}

void sigsegv_handler(int sig)
{
	debug("\n");
	debug("******************************");
	debug("**** KVIrc caught SIGSEGV ****");
	debug("******************************");
	debug("\n");
	debug("Attempting to save settings...");
	debug("******************************");
	debug("\n");
	//
	// Save settings here (if possible)
	// We must be careful not to cause another SIGSEGV while saving settings. If we
	// do, the program will fall into an infinite loop. That is why I put the 'safety'
	// variable - when the program segfaults more than 5 times it will exit no
	// matter what :)
	// I do not know if causing another segfault is possible, but one never knows...
	static int cnt = 0;
	cnt++;
	if( cnt > 5 )
		goto finish;	// I know 'goto' is not too elegant but what the heck :)

	// TODO: maybe add another function to KviApp that would save the settings
	// into another file in case some data got corrupted?
	if( g_pApp )
		g_pApp->slot_saveSettings();

	debug("Please send a bug report to <kvirc2@packages.debian.org> describing how it happened.");
	debug("A backtrace would be nice. To get it, find the core dump file 'core' (try $PWD");
	debug("or $HOME first), and run:");
	debug("\tgdb -core=core");
	debug("Then, type 'bt', copy the whole output and send it to the above email address.");

finish:
	//
	// Reset the signal to its default behavior, so that the app can dump core
	//
	signal(SIGSEGV, SIG_DFL);
}
