//LabPlot : OperationsListDialog.cc

#include <math.h>
#include "OperationsListDialog.h"
#include "defs.h"

OperationsListDialog::OperationsListDialog(MainWin *m, const char *name)
	: ListDialog(m, name)
{
	setCaption(i18n("Operations Dialog"));
	KConfig *config = mw->Config();
	config->setGroup( "Operations" );

	QTabWidget *tw = new QTabWidget(vbox);
	QVBox *tab1 = new QVBox(tw);

	QHBox *hb = new QHBox(tab1);

	type = new KComboBox(hb);
	QStringList tlist;
	tlist<<i18n("add")<<i18n("substract")<<i18n("multiply")<<i18n("divide")<<i18n("greater")<<i18n("lower");
	tlist<<i18n("average");
	type->insertStringList(tlist);
	type->setCurrentItem(config->readNumEntry("Type",0));

	hb = new QHBox(tab1);
	if(s)
		new QLabel(i18n("column "),hb);
	else
		new QLabel(i18n("set "),hb);
	// TODO : spreadsheet : use selected columns if available as input values
	set1ni = new KIntNumInput(config->readNumEntry("Set1",1),hb);
	set1ni->setRange(1,INF,1,false);
	if(s)
		new QLabel(i18n("with column "),hb);
	else
		new QLabel(i18n("with set "),hb);
	// TODO : spreadsheet : use selected columns if available as input values
	set2ni = new KIntNumInput(config->readNumEntry("Set2",2),hb);
	set2ni->setRange(1,INF,1,false);

	tw->addTab(tab1,i18n("Parameter"));

	if (p) {
		Style *style=0;
		Symbol *symbol=0;
		QVBox *styletab;
		if(p->getPlot(p->API())->Type() == PSURFACE)
			styletab = surfaceStyle(tw,true);
		else
			styletab = simpleStyle(tw, style, symbol);

		tw->addTab(styletab,i18n("Style"));
	}

	QObject::connect(ok,SIGNAL(clicked()),SLOT(ok_clicked()));
	QObject::connect(apply,SIGNAL(clicked()),SLOT(apply_clicked()));
	QObject::connect(save,SIGNAL(clicked()),SLOT(saveSettings()));

	setMinimumWidth((int)(1.5*vbox->minimumSizeHint().width()));
	setMinimumHeight(50+gbox->minimumSizeHint().height()+vbox->minimumSizeHint().height());
	resize(minimumSize());
}

void OperationsListDialog::saveSettings() {
	KConfig *config = mw->Config();
	config->setGroup( "Operations" );

	config->writeEntry("Type",typecb->currentItem());
	config->writeEntry("Set1",set1ni->value());
	config->writeEntry("Set2",set2ni->value());
}

int OperationsListDialog::apply_clicked() {
	int item1 = set1ni->value()-1;
	int item2 = set2ni->value()-1;
	int oper = type->currentItem();

	if(item1<0 || item2<0) {
		KMessageBox::error(this,i18n("Negative index values are not allowed!"));
		return -1;
	}

	if(s) {
		QTable *table = s->Table();
		int rows = s->filledRows();
		s->addColumn();

		if(item1>(int)table->numCols() || item2>(int)table->numCols()) {
			KMessageBox::error(this,i18n("Selected columns are not available in the spreadsheet!"));
			return -2;
		}

		for (int i = 0;i<rows;i++) {
			double v1=table->text(i,item1).toDouble();
			double v2=table->text(i,item2).toDouble();
			double y=0;

			switch(oper) {
			case 0 : y=v1+v2; break;
			case 1 : y=v1-v2; break;
			case 2 : y=v1*v2; break;
			case 3 : y=v1/v2; break;
			case 4 : v1>v2?y=v1:y=v2; break;
			case 5 : v1<v2?y=v1:y=v2; break;
			case 6 : y=(v1+v2)/2.0; break;
			}

			table->setText(i,table->numCols()-1,QString::number(y));
		}

		return 0;
	}

	GraphList *gl = p->getPlot(p->API())->getGraphList();

	if (item1 > (int)gl->Number() || item2 > (int)gl->Number()) {
		KMessageBox::error(this, i18n("Sorry. Can not find selected data sets!"));
		return -1;
	}

	GRAPHType s1 = gl->getType(item1);
	GRAPHType s2 = gl->getType(item2);

	Point *p2d1=0, *p2d2=0;
	Point3D *p3d1=0, *p3d2=0;
	Point4D *p4d1=0, *p4d2=0;
	double *a1=0, *a2=0;
	int nr1=0, nr2=0, nx=0, ny=0;
	int dim=0;	// result dimension
	QString label1, label2;

	switch(s1) {	//first graph
	case GRAPH2D: {
		dim=2;
		Graph2D *g=gl->getGraph2D(item1);
		nr1=g->Number();
		p2d1=g->Data();
		label1=g->getLabel()->simpleTitle();
		}; break;
	case GRAPH3D: {
		dim=3;
		Graph3D *g=gl->getGraph3D(item1);
		nr1=g->Number();
		p3d1=g->Data();
		label1=g->getLabel()->simpleTitle();
		}; break;
	case GRAPH4D: {
		dim=4;
		Graph4D *g=gl->getGraph4D(item1);
		nr1=g->Number();
		p4d1=g->Data();
		label1=g->getLabel()->simpleTitle();
		}; break;
	case GRAPHM: {
		dim=1;
		GraphM *g=gl->getGraphM(item1);
		label1=g->getLabel()->simpleTitle();
		nx=g->NX(); ny=g->NY();
		a1=g->Data();
		}; break;
	default : break;
	}

	switch(s2) {	// second graph
	case GRAPH2D: {
		dim=(int)fmax(dim,2);
		Graph2D *g=gl->getGraph2D(item2);
		nr2=g->Number();
		p2d2=g->Data();
		label2=g->getLabel()->simpleTitle();
		}; break;
	case GRAPH3D: {
		dim=(int)fmax(dim,3);
		Graph3D *g=gl->getGraph3D(item2);
		nr2=g->Number();
		p3d2=g->Data();
		label2=g->getLabel()->simpleTitle();
		}; break;
	case GRAPH4D: {
		dim=(int)fmax(dim,4);
		Graph4D *g=gl->getGraph4D(item2);
		nr2=g->Number();
		p4d2=g->Data();
		label2=g->getLabel()->simpleTitle();
		}; break;
	case GRAPHM: {
		if(dim!=1) {
			KMessageBox::error(this,i18n("You can only add two matrices!"));
			return -3;
		}
		GraphM *g=gl->getGraphM(item2);
		if(g->NX() != nx || g->NY() != ny) {
			KMessageBox::error(this,i18n("Matrices must be of same size!"));
			return -4;
		}
		label2=g->getLabel()->simpleTitle();
		a2=g->Data();
		}; break;
	default : break;
	}

	int nr = (int)fmin(nr1,nr2);
	if(dim==1) nr=nx*ny;
	kdDebug()<<"	dim="<<dim<<" / nr = "<<nr<<endl;

	Point *ptr2d = new Point[nr];
	Point3D *ptr3d = new Point3D[nr];
	Point4D *ptr4d = new Point4D[nr];
	double *ptra = new double[nr];

	// set label;
	QString fun;
	switch(oper) {
	case 0:	fun = QString(i18n("sum of ")+label1+i18n(" with ")+label2); break;
	case 1:	fun = QString(i18n("difference of ")+label1+i18n(" and ")+label2); break;
	case 2:	fun = QString(i18n("product of ")+label1+i18n(" and ")+label2); break;
	case 3:	fun = QString(i18n("quotient of ")+label1+i18n(" and ")+label2); break;
	case 4:	fun = QString(i18n("maximum of ")+label1+i18n(" and ")+label2); break;
	case 5:	fun = QString(i18n("minimum of ")+label1+i18n(" and ")+label2); break;
	case 6:	fun = QString(i18n("average of ")+label1+i18n(" and ")+label2); break;
	}

	kdDebug()<<"Creating new graph"<<endl;
	double xmin=0, xmax=1, ymin=0, ymax=1, zmin=0, zmax=1, tmin=0, tmax=1;
	for (int i = 0;i<nr;i++) {
		double x=0,y=0,z=0,t=0,y1=0,y2=0,z1=0,z2=0,t1=0,t2=0;
		switch(s1) {
		case GRAPH2D: x=p2d1[i].X(); y1=p2d1[i].Y(); break;
		case GRAPH3D: x=p3d1[i].X(); y1=p3d1[i].Y(); z1=p3d1[i].Z(); break;
		case GRAPH4D: x=p4d1[i].X(); y1=p4d1[i].Y(); z1=p4d1[i].Z(); t1=p4d1[i].T(); break;
		case GRAPHM: y1=a1[i]; break;
		default: break;
		}
		switch(s2) {
		case GRAPH2D: y2=p2d2[i].Y(); break;
		case GRAPH3D: y2=p3d2[i].Y(); z2=p3d2[i].Z(); break;
		case GRAPH4D: y2=p4d2[i].Y(); z2=p4d2[i].Z(); t2=p4d2[i].T(); break;
		case GRAPHM: y2=a2[i]; break;
		default: break;
		}

		switch(oper) {
		case 0 :
			y=y1+y2;
			if(z1!=0 && z2!=0 )	// X-Y-Z + X-Y-Z
				z=1/z1+1/z2;
			else
				z=z1+z2;
			if(t1!=0 && t2!=0 )
				t=1/t1+1/t2;
			else
				t=t1+t2;
			break;
		case 1 :
			y=y2-y1;
			if(z1!=0 && z2!=0 )
				z=1/z1+1/z2;
			else
				z=z2-z1;
			if(t1!=0 && t2!=0 )
				t=1/t1+1/t2;
			else
				t=t2-t1;
			break;
		case 2 :
			y=y1*y2;
			z=z2+z1;	// ?
			t=t1+t2;
			break;
		case 3 :
			y=y1/y2;
			z=z2+z1;	// ?
			t=t1+t2;
			break;
		case 4 :
			y1>y2?y=y1:y=y2;
			z=z2+z1;	// ?
			t=t1+t2;
			break;
		case 5 :
			y1<y2?y=y1:y=y2;
			z=z2+z1;	// ?
			t=t1+t2;
			break;
		case 6 :
			y=(y1+y2)/2.0;
			if(z1!=0 && z2!=0 )
				z=1/z1+1/z2;
			else
				z=z2+z1;
			if(t1!=0 && t2!=0 )
				t=1/t1+1/t2;
			else
				t=t1+t2;
			break;
		}

		// new ranges
		if (i==0) {
			xmin=xmax=x;
			ymin=ymax=y;
			zmin=zmax=z;
			tmin=tmax=t;
		}
		else {
			x<xmin?xmin=x:0;
			x>xmax?xmax=x:0;
			y<ymin?ymin=y:0;
			y>ymax?ymax=y:0;
			z<zmin?zmin=z:0;
			z>zmax?zmax=z:0;
			t<tmin?tmin=t:0;
			t>tmax?tmax=t:0;
		}

//		kdDebug()<<"	NEW : "<<x<<' '<<y<<' '<<z<<' '<<t<<endl;
		switch(dim) {
		case 1 : ptra[i]=y; break;
		case 2 : ptr2d[i].setPoint(x,y); break;
		case 3 : ptr3d[i].setPoint(x,y,z); break;
		case 4 : ptr4d[i].setPoint(x,y,z,t); break;
		}
	}

	Style *style=0;
	Symbol *symbol=0;
	if(dim!=1) {
		style = new Style((StylesType)cb2->currentItem(),color->color(),filled->isChecked(),fcolor->color(),
			width->value(),pencb->currentItem(),brushcb->currentItem());
		style->setBoxWidth(boxwidth->value());
		style->setAutoBoxWidth(autobox->isChecked());
		style->setPointsSorting(sortpointscb->isChecked());
		symbol = new Symbol((SType)symbolcb->currentItem(),scolor->color(),ssize->value(),
		(FType)symbolfillcb->currentItem(),sfcolor->color(),sbrushcb->currentItem());
	}

	switch(dim) {
	case 1 : {
		kdDebug()<<"	new GraphM"<<endl;
		LRange range[3];
		range[0] = LRange(0,nx);
		range[1] = LRange(0,ny);
		range[2] = LRange(ymin,ymax);

		GraphM *ng = new GraphM(fun,fun,range,SSPREADSHEET,PSURFACE,style,symbol,ptra,nx,ny);
		mw->addGraphM(ng,sheetcb->currentItem());
		}; break;
	case 2 : {
		kdDebug()<<"	new Graph2D"<<endl;
		LRange range[2];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);

		Graph2D *ng = new Graph2D(fun,fun,range,SSPREADSHEET,P2D,style,symbol,ptr2d,nr);
		mw->addGraph2D(ng,sheetcb->currentItem());
		}; break;
	case 3 : {
		kdDebug()<<"	new Graph3D"<<endl;
		LRange range[3];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);
		range[2] = LRange(zmin,zmax);

		Graph3D *ng = new Graph3D(fun,fun,range,SSPREADSHEET,P2D,style,symbol,ptr3d,nr);
		mw->addGraph3D(ng,sheetcb->currentItem());
		}; break;
	case 4 : {
		kdDebug()<<"	new Graph4D"<<endl;
		LRange range[4];
		range[0] = LRange(xmin,xmax);
		range[1] = LRange(ymin,ymax);
		range[2] = LRange(zmin,zmax);
		range[3] = LRange(tmin,tmax);

		Graph4D *ng = new Graph4D(fun,fun,range,SSPREADSHEET,P2D,style,symbol,ptr4d,nr);
		mw->addGraph4D(ng,sheetcb->currentItem());
		}; break;
	}

	updateList();

	return 0;
}
