/* -*-C-*-

$Id: arp-discovery.c,v 1.7 2002/03/14 21:16:05 cph Exp $

Copyright (c) 2002 Massachusetts Institute of Technology

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.  */

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <setjmp.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <libnet.h>
#include <pcap.h>
#include "parse-ipmap.h"
#include "if-params.h"

#ifndef N_RETRIES
#define N_RETRIES 3
#endif

#define PACKET_LENGTH (LIBNET_ETH_H + LIBNET_ARP_H)

static u_char enet_bcast [ETHER_ADDR_LEN]
  = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
static u_char enet_nulls [ETHER_ADDR_LEN] = {0, 0, 0, 0, 0, 0};
static u_char enet_local [ETHER_ADDR_LEN];
static struct libnet_link_int * link_interface;
static sigjmp_buf abort_read_arp_reply;

struct arp_reply
{
  int valid_p;
  struct in_addr ip_address;
  char mac_address [18];
};

static int initialize_interface (const char *, struct if_params *);
static struct ip_map * find_ip_map_entry
  (struct in_addr *, unsigned int, struct ip_map *);
static pcap_t * initialize_libpcap (const char *);
static int initialize_libnet (const char *);
static int send_arp_requests (const char *, unsigned int, struct ip_map *);
static int send_arp_request
  (const char *, char *, struct in_addr *, struct in_addr *);
static int read_arp_reply (pcap_t *, struct arp_reply *);
static void handle_alarm (int);
static void read_arp_reply_1
  (u_char *, const struct pcap_pkthdr *, const u_char *);

int
main (int argc, const char ** argv)
{
  const char * name = 0;
  const char * ip_map_filename;
  struct ip_map * ip_map_entries;
  int n_ip_map_entries;
  unsigned int n_addresses;
  struct if_params original_if_params;
  int restore_if_params = 0;
  int result_code = 1;
  pcap_t * pd;
  unsigned int n_retries = N_RETRIES;

  if (argc != 3)
    {
      fprintf (stderr, "usage: %s INTERFACE FILENAME\n", (argv[0]));
      goto finished;
    }
  name = (argv[1]);
  ip_map_filename = (argv[2]);

  /* If file doesn't exist, act as though it did but was empty.  */
  if ((access (ip_map_filename, R_OK)) < 0)
    {
      if ((errno == ENOENT) || (errno == ENOTDIR))
	{
	  fprintf (stdout, "@NO-CHOICES@\n");
	  goto successful;
	}
      perror (ip_map_filename);
      goto finished;
    }

  n_ip_map_entries = (parse_ip_map (ip_map_filename, (&ip_map_entries)));
  if (n_ip_map_entries < 0)
    /* Error message was printed by parse_ip_map.  */
    goto finished;
  if (n_ip_map_entries == 0)
    {
      fprintf (stdout, "@NO-CHOICES@\n");
      goto successful;
    }
  {
    struct ip_map * scan = ip_map_entries;
    struct ip_map * end = (scan + n_ip_map_entries);
    n_addresses = 0;
    while (scan < end)
      n_addresses += ((scan++) -> n_addresses);
  }

  if ((initialize_interface (name, (&original_if_params))) < 0)
    goto finished;
  restore_if_params = 1;
  pd = (initialize_libpcap (name));
  if (pd == 0)
    goto finished;
  if ((initialize_libnet (name)) < 0)
    goto finished;

  while (n_retries > 0)
    {
      unsigned int i = 0;

      if ((send_arp_requests (name, n_ip_map_entries, ip_map_entries)) < 0)
	goto finished;
      while (i < (2 * n_addresses))
	{
	  struct arp_reply reply;
	  int result = (read_arp_reply (pd, (&reply)));
	  if (result < 0)	/* read timed out */
	    goto retry_done;
	  if (result > 0)
	    {
	      struct ip_map * entry
		= (find_ip_map_entry ((& (reply . ip_address)),
				      n_ip_map_entries,
				      ip_map_entries));
	      if (entry != 0)
		{
		  fprintf (stdout, "%s\n", (entry -> key));
		  goto successful;
		}
	    }
	  i += 1;
	}
      sleep (1);
    retry_done:
      n_retries -= 1;
    }
  fprintf (stdout, "@NO-RESPONSES@\n");
 successful:
  result_code = 0;
 finished:
  if (restore_if_params)
    write_interface_configuration (name, (&original_if_params));
  return (result_code);
}

#define STORE_IP(string, field, addr_flag)				\
  ((memset ((& (ifp . field)), 0, (sizeof (ifp . field)))),		\
   ((ifp . field . sin_family) = AF_INET),				\
   ((inet_aton ((string), (& (ifp . field . sin_addr))))		\
    ? (((ifp . addr_flags) |= (addr_flag)), 1)				\
    : (((ifp . addr_flags) &=~ (addr_flag)), 0)))

static int
initialize_interface (const char * name, struct if_params * original_ifp)
{
  struct if_params ifp;

  if ((read_interface_configuration (name, (&ifp))) < 0)
    return (-1);
  (*original_ifp) = ifp;
  if ((((ifp . flags) & IFF_UP) != 0)
      && (((ifp . flags) & IFF_RUNNING) != 0)
      && (IFP_ALL_ADDRESSES_VALID (&ifp)))
    return (0);
  (ifp . flags) |= (IFF_UP | IFF_RUNNING);
  if (!IFP_ALL_ADDRESSES_VALID (&ifp))
    {
      if (! ((STORE_IP ("192.168.254.254", addr, IFP_VALID_ADDR))
	     && (STORE_IP ("192.168.254.0", dstaddr, IFP_VALID_DSTADDR))
	     && (STORE_IP ("192.168.254.255", broadaddr, IFP_VALID_BROADADDR))
	     && (STORE_IP ("255.255.255.0", netmask, IFP_VALID_NETMASK))))
	{
	  fprintf (stderr, "Unable to set network interface addresses.\n");
	  return (-1);
	}
    }
  return (write_interface_configuration (name, (&ifp)));
}

static struct ip_map *
find_ip_map_entry (struct in_addr * address,
		   unsigned int n_ip_map_entries,
		   struct ip_map * ip_map_entries)
{
  struct ip_map * scan_map = ip_map_entries;
  struct ip_map * end_map = (scan_map + n_ip_map_entries);
  while (scan_map < end_map)
    {
      struct in_addr * scan_addr = (scan_map -> addresses);
      struct in_addr * end_addr = (scan_addr + (scan_map -> n_addresses));
      while (scan_addr < end_addr)
	{
	  if ((scan_addr -> s_addr) == (address -> s_addr))
	    return (scan_map);
	  scan_addr += 1;
	}
      scan_map += 1;
    }
  return (0);
}

static pcap_t *
initialize_libpcap (const char * name)
{
  char error_buffer [PCAP_ERRBUF_SIZE];
  pcap_t * pd
    = (pcap_open_live (((char *) name),
		       PACKET_LENGTH,
		       0,
		       20,
		       error_buffer));
  if (pd == 0)
    {
      fprintf (stderr, "pcap_open_live failed: %s\n", error_buffer);
      return (0);
    }
  return (pd);
}

static int
initialize_libnet (const char * name)
{
  struct sockaddr_in sin;
  char error_buffer [LIBNET_ERRBUF_SIZE];
  struct ether_addr * ea;

  if ((libnet_select_device ((&sin), ((char **) (&name)), error_buffer)) < 0)
    {
      fprintf (stderr, "libnet_select_device failed: %s\n", error_buffer);
      return (-1);
    }
  link_interface
    = (libnet_open_link_interface (((char *) name), error_buffer));
  if (link_interface == 0)
    {
      fprintf (stderr, "libnetopen_link_interface failed: %s\n", error_buffer);
      return (-1);
    }
  ea = (libnet_get_hwaddr (link_interface, name, error_buffer));
  if (ea == 0)
    {
      fprintf (stderr, "libnet_get_hwaddr failed: %s\n", error_buffer);
      return (-1);
    }
  memcpy (enet_local, (ea -> ether_addr_octet), (sizeof (enet_local)));
  return (0);
}

static int
send_arp_requests (const char * name,
		   unsigned int n_ip_map_entries,
		   struct ip_map * ip_map_entries)
{
  struct ip_map * scan_map = ip_map_entries;
  struct ip_map * end_map = (scan_map + n_ip_map_entries);
  struct in_addr null_addr;
  char buffer [PACKET_LENGTH];
  struct in_addr * scan_addr;
  struct in_addr * end_addr;

  inet_aton ("0.0.0.0", (&null_addr));
  libnet_build_ethernet
    (enet_bcast,
     enet_local,
     ETHERTYPE_ARP,
     0, 0,
     buffer);

  while (scan_map < end_map)
    {
      scan_addr = (scan_map -> addresses);
      end_addr = (scan_addr + (scan_map -> n_addresses));
      while (scan_addr < end_addr)
	{
	  if ((send_arp_request (name, buffer, (&null_addr), scan_addr)) < 0)
	    return (-1);
	  scan_addr += 1;
	}
      scan_map += 1;
    }
  return (0);
}

static int
send_arp_request (const char * name, char * buffer,
		  struct in_addr * source, struct in_addr * target)
{
  libnet_build_arp
    (ARPHRD_ETHER,
     ETHERTYPE_IP,
     ETHER_ADDR_LEN,
     4,
     ARPOP_REQUEST,
     enet_local,
     ((u_char *) (& (source -> s_addr))),
     enet_nulls,		/* This makes it a DHCP ARP request.  */
     ((u_char *) (& (target -> s_addr))),
     0, 0,
     (buffer + LIBNET_ETH_H));
  if ((libnet_write_link_layer (link_interface, name, buffer, PACKET_LENGTH))
      < 0)
    {
      perror ("libnet_write_link_layer");
      return (-1);
    }
  return (0);
}

/* Use alarm timer here to deal with fact that pcap_dispatch might not
   return.  */

static int
read_arp_reply (pcap_t * pd, struct arp_reply * reply)
{
  if (sigsetjmp (abort_read_arp_reply, 1))
    return (-1);
  signal (SIGALRM, handle_alarm);
  alarm (1);
  (void) (pcap_dispatch (pd, 0, read_arp_reply_1, ((u_char *) reply)));
  signal (SIGALRM, SIG_IGN);
  alarm (0);
  return (reply -> valid_p);
}

static void
handle_alarm (int signo)
{
  siglongjmp (abort_read_arp_reply, 1);
}

static void
read_arp_reply_1 (u_char * r,
		  const struct pcap_pkthdr * pc,
		  const u_char * packet)
{
  struct arp_reply * reply = ((struct arp_reply *) r);
  struct libnet_ethernet_hdr * p = ((struct libnet_ethernet_hdr *) packet);
  struct libnet_arp_hdr * a = ((struct libnet_arp_hdr *) (packet + ETH_H));

  if (((ntohs (p -> ether_type)) == ETHERTYPE_ARP)
      && ((ntohs (a -> ar_op)) == ARPOP_REPLY)
      && ((ntohs (a -> ar_hrd)) == ARPHRD_ETHER))
    {
      (reply -> valid_p) = 1;
      ((reply -> ip_address) . s_addr) = (* ((in_addr_t *) (& (a -> ar_spa))));
      sprintf ((reply -> mac_address),
	       "%.2x:%.2x:%.2x:%.2x:%.2x:%.2x",
	       ((a -> ar_sha) [0]),
	       ((a -> ar_sha) [1]),
	       ((a -> ar_sha) [2]),
	       ((a -> ar_sha) [3]),
	       ((a -> ar_sha) [4]),
	       ((a -> ar_sha) [5]));
    }
  else
    (reply -> valid_p) = 0;
}
