//
// The contents of this file are subject to the Mozilla Public License
// Version 1.0 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License
// at http://www.mozilla.org/MPL/
// 
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
// the License for the specific language governing rights and
// limitations under the License.
//
// This software was developed as part of the legOS project.
//
// Contributor: Pat Welch (legOS@mousebrains.com)

#ifndef _MotorPair_H_
#define _MotorPair_H_

#include <c++/Motor.H>

// This class is for manipulating a simple pair of motors, that are the drive
// motors for a rover.  i.e. you have a left and right motor which drive a
// rover.  The left motor is spinning forwards to move forwards, while the
// right motor is spinning backwards.
//
// See Motor for methods speed, direction, Forward, Reverse, Brake, and Off.
// Additional methods include:
//
//  Left()     Turns left by running both motors in reverse, so it should spin
//             about the center of the two motors.
//
//  PivotLeft() Turns left by locking up the left motor, and running the right
//              motor to pivot about the left wheel.
//
//  Right()    Turns right by running both motors in forward, so it should spin
//             about the center of the two motors.
//
//  PivotRight() Turns right by locking up the right motor, and running the 
//               left motor to pivot about the right wheel.
//
// There are corresponding speed setting methods for the above methods,
// Left(int speed), which runs the motors at the specified speed.
//

class MotorPair {
public:
  MotorPair(const Motor::Port lport, const Motor::Port rport)
    : left(lport), right(rport) {}

  void speed(const int s) const { left.speed(s); right.speed(s); }
  void direction(const MotorDirection dir) const { 
    if (dir == fwd) {
      left.direction(fwd); 
      right.direction(rev); 
    } else if (dir == rev) {
      left.direction(rev); 
      right.direction(fwd); 
    } else {
      left.direction(dir); 
      right.direction(dir); 
    }
  }
  void Forward() const { direction(fwd); }
  void Reverse() const { direction(rev); }
  void Brake() const { left.direction(brake); right.direction(brake); }
  void Off() const { left.direction(off); right.direction(off); }
  void Left() const { left.direction(fwd); right.direction(fwd); }
  void PivotLeft() const { left.Brake(); right.direction(rev); }
  void Right() const { left.direction(rev); right.direction(rev); }
  void PivotRight() const { left.direction(fwd); right.Brake(); }

  void Forward(const int s) const { Forward(); speed(s); }
  void Reverse(const int s) const { Reverse(); speed(s); }
  void Left(const int s) const { Left(); speed(s); }
  void PivotLeft(const int s) const { PivotLeft(); speed(s); }
  void Right(const int s) const { Right(); speed(s); }
  void PivotRight(const int s) const { PivotRight(); speed(s); }

  void Brake(const int d) const { Brake(); delay(d); }

  enum Limits {min = Motor::min, max = Motor::max};

private:
  const Motor left;
  const Motor right;
};

#endif // _MotorPair_H_
